/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2015, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/



#include <stdio.h>
#include <sys/types.h>
#include <dirent.h>
#include <sys/stat.h>

#include "jpeghw_api.h" // JPEGHW library
#include "jpeg_strip_adaptor.h"
#include "jpeg_page_adaptor.h"
#include "logger.h"
#include "dma_buffer.h"
#include "uio_lib_api.h"
#include "jpeg_common.h"

#include "jhw_api.h" 

#define JPEG_FILE_DEFAULT_IMAGE_HEIGHT_MAX 8400 // 14 inch default, needs backpatch based on adf height
                                                 // using jpeg_jpeghw_fixup_page_length() callback

//////////////////////////////////////////////////////////////////////////////////////////
jpeghw_error_type_t cmgr_get_quant_table(struct jpeghw_compress_struct * cinfo, uint8_t quality, uint8_t *table, uint32_t *size)
{
    if  (quality > 100)
    {
        printf("\n>>>>>%s() ERROR - invalid quality parameter, received %u but expected 0-100!\n",
                __func__, (unsigned int)quality);

        // return invalid parameter
        return e_JPEGHW_ERR_INVALID_PARAMETERS;
    }

    if (size)
    {
        if  (*size > sizeof(g_quantization_table[0]))
        {
            *size = sizeof(g_quantization_table[0]);
        }
    }

    if (table && size)
    {
        memcpy(table, &g_quantization_table[quality][0], *size);
    }

    return e_JPEGHW_SUCCESS;
}

//////////////////////////////////////////////////////////////////////////////////////////
jpeghw_error_type_t cmgr_get_huff_table(struct jpeghw_compress_struct * cinfo, uint8_t table_index, bool ac, uint8_t *bits, uint32_t *bits_size, uint8_t *val, uint32_t *val_size)
{
    if (table_index > 1)
    {
        printf("\n>>>>>%s() ERROR - invalid table_index parameter, received %u but expected 0 or 1!\n",
                __func__, (unsigned int)table_index);

        return e_JPEGHW_ERR_INVALID_PARAMETERS;
    }

    if (bits_size)
    {
        uint32_t size = (ac ? sizeof(g_ac_bits[table_index]) : sizeof(g_dc_bits[table_index]));

        if (*bits_size > size)
        {
            *bits_size = size;
        }
    }

    if (bits && bits_size)
    {
        if (ac)
        {
            memcpy(bits, g_ac_bits[table_index], *bits_size);
        }
        else
        {
            memcpy(bits, g_dc_bits[table_index], *bits_size);
        }
    }

    if (val_size)
    {
        uint32_t size = (ac ? sizeof(g_ac_val[table_index]) : sizeof(g_dc_val[table_index]));

        if (*val_size > size)
        {
            *val_size = size;
        }
    }

    if (val && val_size)
    {
        if (ac)
        {
            memcpy(val, g_ac_val[table_index], *val_size);
        }
        else
        {
            memcpy(val, g_dc_val[table_index], *val_size);
        }
    }

    return e_JPEGHW_SUCCESS;
}

//////////////////////////////////////////////////////////////////////////////////////////
#define OUTPUT_BUFFER_LENGTH 4 * 1024
struct BigBuffer_s* get_outbuf_fnc(struct jpeghw_compress_struct * jhwcinfo)
{
    struct BigBuffer_s* output_buffer = NULL;

    char *obuf = MEM_MALLOC_ALIGN(OUTPUT_BUFFER_LENGTH, 4);
    output_buffer = (struct BigBuffer_s *)dma_buffer_adopt(obuf, OUTPUT_BUFFER_LENGTH);

    return output_buffer;
}

//////////////////////////////////////////////////////////////////////////////////////////
bool outbuf_ready_fnc(struct jpeghw_compress_struct * jhwcinfo, 
                              struct BigBuffer_s* output_buffer, uint32_t bytes_in_buffer)
{
    //printf("[%d]--%s\n",gettid(),__func__);

    char *read_ptr = dma_buffer_mmap_forcpu(output_buffer);
  
    uint32_t buflen = bytes_in_buffer;
    //printf("[%d] buflen=%d\n",gettid(), buflen);

    //for(ii=0;ii<20;ii++)DBG_PRINTF_INFO("%02X ",read_ptr[ii]);DBG_PRINTF_INFO("\n");
    write((intptr_t)jhwcinfo->common.client_context, read_ptr, buflen);

    dma_buffer_unmmap_forcpu(output_buffer);

    if(output_buffer != NULL) 
    {
        (output_buffer)->freeFunc(output_buffer); 
        output_buffer =NULL; 
    }

    return true;
}

//////////////////////////////////////////////////////////////////////////////////////////
struct BigBuffer_s * need_input_fnc(struct jpeghw_decompress_struct *dinfo, uint32_t *bytes)
{
    int buflen = 16*1024;

    struct BigBuffer_s *bb = dma_buffer_malloc(0, buflen);
    char *read_ptr = dma_buffer_mmap_forcpu(bb);

    *bytes = read((intptr_t)dinfo->common.client_context, read_ptr, buflen);
    //printf("in:");for(ii=0;ii<20;ii++)printf("%02X ",read_ptr[ii]);printf("\n");
    dma_buffer_unmmap_forcpu(bb);

    return bb;
}

//////////////////////////////////////////////////////////////////////////////////////////
jpeghw_error_type_t get_jpeg_info(struct jpeghw_decompress_struct * dinfo, 
                                  struct jpeghw_jpeg_header_struct *jinfo) 
{
    FILE *fp = fdopen((intptr_t)dinfo->common.client_context, "r");

    if (fp != NULL)
    {
        uint32_t fpos = ftell(fp);
        fseek(fp, 0, SEEK_SET);
        parse_jpeg_header(fp, jinfo, false);
        fseek(fp, fpos, SEEK_SET);
    }

    return 0;
}

//////////////////////////////////////////////////////////////////////////////////////////
int read_ppm_header(int fd, int *bpp, int *width, int *height)
{
    char type[3], swidth[8], sheight[8];
    char c=0;
    int idx;
    int charsread=0;

    read(fd, type, 3);
    type[2] = 0;
 
    if(strcmp(type,"P5") == 0)
    {
        *bpp = 1;
    }
    else if(strcmp(type,"P6") == 0)
    {
        *bpp = 3;
    }
    else
    {
        *bpp = 0;
        printf("[%d]Don't recognize the file format\n",gettid());
        return -1;
    }

    read(fd, &c, 1);
    charsread++;
    idx=0;
    // gimp places a comment here
    if(c == '#')
    {
        // consume comment
        while(c != 0x0A)
        {
            if (read(fd, &c, 1) > 0)
            {
               charsread++;
                /*DBG_INFO("%c",c);*/
            }
            else
            {
                // error or EOF
                break;
            }
        }
    }
    else
    {
        swidth[idx++] = c;
    }

    c=0;
    while(c != 0x0A && c != 0x20)
    {
        read(fd, &c, 1);
        charsread++;
        swidth[idx++] = c;
        swidth[idx] = 0;
    }
    sscanf(swidth, "%d", width);

    c=0;
    idx=0;
    while(c != 0x0A && c != 0x20)
    {
        read(fd, &c, 1);
        charsread++;
        sheight[idx++] = c;
        sheight[idx] = 0;
    }
    sscanf(sheight, "%d", height);

    c=0;
    idx=0;
    while(c != 0x0A && c != 0x20)
    {
        read(fd, &c, 1);
        charsread++;
    }

    if(*width == 0 || *height == 0)
    {
        printf("[%d]file dimensions are screwed up\n",gettid());
        return -1;
    }
    
    return 0;
}

//////////////////////////////////////////////////////////////////////////////////////////
int _check_input_file(char *ppm_filename, bool color)
{
    int bpp=0, width=0, file_height=0, mcu_dim=0;

    int ppm_fd=open(ppm_filename, O_RDONLY);
    if(ppm_fd < 0)
    {
        printf("cannot open file %s\n",ppm_filename);
        return 1;
    }

    if(read_ppm_header(ppm_fd, &bpp, &width, &file_height) == -1) 
    {
       printf("Failed to read ppm header\n");
       close(ppm_fd);
       return 1;
    }

    if (bpp == 1) mcu_dim = 8;
    else          mcu_dim = 16;

    if (!color && bpp != 1)
    {
       printf("ERROR: PPM file %s is not a mono PPM.\n", ppm_filename);
       close(ppm_fd);
       return 1;
    }
    if (color && bpp != 3)
    {
       printf("ERROR: PPM file %s is not a color PPM.\n", ppm_filename);
       close(ppm_fd);
       return 1;
    }

    if (width % mcu_dim == 0)
    {
       printf("WARNING: PPM image width(%d) is a multiple of the mcu width(%d)\n", width, mcu_dim);
       printf("         You can increase code coverage by using a non mcu aligned image\n");
       printf("         PPM filename = %s\n\n", ppm_filename);
    }
    if (file_height % mcu_dim == 0)
    {
       printf("WARNING: PPM image height(%d) is a multiple of the mcu width(%d)\n", file_height, mcu_dim);
       printf("         You can increase code coverage by using a non mcu aligned image\n");
       printf("         PPM filename = %s\n\n", ppm_filename);
    }

    close(ppm_fd);
    return 0;
}

//////////////////////////////////////////////////////////////////////////////////////////
int _compress_test(char *ppm_filename, char* jpg_filename, int strip_lines, bool abort, 
             bool too_long, bool default_huff, bool default_quant, bool mcu_align, bool auto_hdr)
{
    int mcu_dim;
    struct jpeghw_error_mgr jerr;

    //printf("%s\n",__func__);

    int ppm_fd=open(ppm_filename, O_RDONLY);
    if(ppm_fd < 0)
    {
        printf("cannot open file %s, %s\n",ppm_filename, strerror(errno));
        return 1;
    }

    int jpg_fd=open(jpg_filename, O_CREAT| O_RDWR);
    if(jpg_fd < 0)
    {
        printf("cannot open file %s, %s\n",jpg_filename, strerror(errno));
        close(ppm_fd);
        return 1;
    }
    chmod(jpg_filename, 0666);

    int bpp=0, width=0, file_height=0;

    if(read_ppm_header(ppm_fd, &bpp, &width, &file_height) == -1)
    {
       printf("Failed to read ppm header\n");
       close(ppm_fd);
       close(jpg_fd);
       return 1;
    }

    if (bpp == 1) mcu_dim = 8;
    else          mcu_dim = 16;

    struct jpeghw_compress_struct cinfo;
    uint32_t flags = JPEGHW_CONFIG_FLAG_BLOCK_JPEG_CORE_AVAIL;
    if (mcu_align)
    {
       flags |= JPEGHW_CONFIG_FLAG_MCU_WIDTH_ALIGNMENT;
       flags |= JPEGHW_CONFIG_FLAG_MCU_HEIGHT_ALIGNMENT;
    }

    if(jpeghw_create_compress(&cinfo, flags) != e_JPEGHW_SUCCESS)
    {
       printf("create compress failed\n");
       close(ppm_fd);
       close(jpg_fd);
       return 1;
    }

    jpeghw_set_defaults(&cinfo.common);

    cinfo.quality = 90;
    cinfo.common.image_width = width;
    cinfo.common.image_height = file_height;
    if (too_long)
    {
       cinfo.common.image_height = JPEG_FILE_DEFAULT_IMAGE_HEIGHT_MAX;
    }
    cinfo.common.bytes_per_pixel = bpp;

    cinfo.cmgr = MEM_MALLOC(sizeof(struct jpeghw_compression_mgr));
    memset(cinfo.cmgr, 0, sizeof(struct jpeghw_compression_mgr));

    // set cmgr callbacks
    cinfo.cmgr->init = 0; //cmgr_init;
    cinfo.cmgr->get_output_buffer = get_outbuf_fnc;
    cinfo.cmgr->send_output_buffer = outbuf_ready_fnc;
    cinfo.cmgr->term = 0; //cmgr_term;

    cinfo.cmgr->get_quant_table = cmgr_get_quant_table;
    cinfo.cmgr->get_huff_table = cmgr_get_huff_table;

    if (default_huff) cinfo.cmgr->get_huff_table = 0;
    if (default_quant) cinfo.cmgr->get_quant_table = 0;

    cinfo.common.err = (struct jpeghw_error_mgr *)jpeghw_std_error(&jerr);

    // back pointer to allow callbacks to have context connection.
    cinfo.common.client_context = (void*)(intptr_t)jpg_fd;

    cinfo.common.mcu_width = mcu_dim;
    cinfo.common.mcu_height = mcu_dim;
    cinfo.common.scanline_timeout = 0;

    if (auto_hdr)
    {
       cinfo.auto_generate_jpeg_header = true;
    }
    else
    {
       cinfo.auto_generate_jpeg_header = true;
       FILE* jpg_fp = fdopen(jpg_fd, "w+");
       if (jpg_fp != NULL)
       {
        generate_jpeg_header(jpg_fp, &cinfo);
       }
    }

    if(jpeghw_start_compress(&cinfo) != e_JPEGHW_SUCCESS)
    {
       printf("start compress failed\n");
       jpeghw_destroy_compress(&cinfo);
       close(ppm_fd);
       close(jpg_fd);
       return 1;
    }

    int lines=0;
    int byte_width=width * bpp;
    int rbuf_size = 0;
    int write_lines=0;

    while(lines < file_height)
    {
        struct BigBuffer_s *ibb=0;
        int eoi=0;

        if (strip_lines > 0)
        {
           write_lines = strip_lines;
        }
        else if (strip_lines == 0) // increment stip length
        {
           static int strip_height=0;
           strip_height++;
           if (strip_height > 20) strip_height = 1;
           write_lines = strip_height;
        }
        else if (strip_lines == -1) // random stip length
        {
           write_lines = rand() % 128;
        }
        rbuf_size = write_lines * byte_width;

        if (rbuf_size == 0)
        {
            printf("Invalid rbuf_size = 0\n");
            close(ppm_fd);
            close(jpg_fd);
            return 1;
        }

        //printf("[%d]allocate %d bytes\n",gettid(),rbuf_size);
        ibb = dma_buffer_malloc(0, rbuf_size);
        if(ibb == 0)
        {
            printf("OUT OF MEMORY\n");
            close(ppm_fd);
            close(jpg_fd);
            return 1;     
        }               
        char *ibuf = dma_buffer_mmap_forcpu(ibb);
        ssize_t rv = read(ppm_fd, ibuf, rbuf_size);
        write_lines = rv/byte_width;
        //printf("---------read %d lines rs:%d rv:%d bw:%d\n",write_lines, rbuf_size,rv,byte_width);
        //int ii; for(ii=0;ii<20;ii++)printf("%02X ",ibuf[ii]);printf("\n");
        dma_buffer_unmmap_forcpu(ibb);

        lines += write_lines;
        if(lines >= file_height)
        {
            eoi = 1;
        }
        if(rv == 0)
        {
            // this is an error.  we need to detect eoi before the last buffer is sent
            // or else the "last buffer" is already sent
            printf("ERROR: read no data from input file. lines:%d height:%d\n", lines, file_height);
            break;
        }

        uint32_t lines_written = jpeghw_write_scanlines(&cinfo, ibb, write_lines, eoi);
        if (lines_written < write_lines)
        {
            printf("ERROR: not all Scanlines were processed for file . lines_written:%d write_lines:%d\n", lines_written, write_lines);

            if (lines_written == 0)
            {
                ibb = BigBuffer_Free(ibb);
            }
        }

        if (abort)
        {
           jpeghw_abort_compress(&cinfo);
           break;
        }
    }

    if(!abort && jpeghw_finish_compress(&cinfo) != e_JPEGHW_SUCCESS)
    {
        printf("finish sent back an error\n");
        exit(1);
    }

    jpeghw_destroy_compress(&cinfo);

    close(jpg_fd);
    close(ppm_fd);

    MEM_FREE_AND_NULL(cinfo.cmgr);

    return 0;
}

//////////////////////////////////////////////////////////////////////////////////////////
int _decompress_test(char *jpg_filename, char *raw_filename, bool abort, bool trim, bool dhdr)
{
    JPEG_header_struct hdr_info;
    struct jpeghw_error_mgr jerr;

    //printf("%s\n",__func__);

    int decompress_fd=open(jpg_filename, O_RDONLY);
    if(decompress_fd == -1)
    {
        printf("cannot open file %s, %s\n",jpg_filename, strerror(errno));
        return 1;
    }
    int out_fd=open(raw_filename, O_RDWR | O_CREAT);
    if(out_fd == -1)
    {
        printf("cannot open file %s, %s\n",raw_filename, strerror(errno));
        close(decompress_fd);
        return 1;
    }

    chmod(raw_filename, 0666);

    FILE *decompress_fp = fdopen(decompress_fd, "r");
    if (decompress_fp == NULL)
    {
        printf("decompress_fp failed in fdopen call!\n");
        close(out_fd);
        close(decompress_fd);
        return 1;
    }

    parse_jpeg_header(decompress_fp, &hdr_info, true);

    lseek(decompress_fd, 0, SEEK_SET);

    //printf("[%d]dim:  w:%d h:%d bbp:%d mw:%d  mh:%d mcu:%dx%d\n",gettid(),
                   //hdr_info.image_width,hdr_info.image_height,hdr_info.bytes_per_pixel,
                   //hdr_info.mcu_aligned_width, hdr_info.mcu_aligned_height,
                   //hdr_info.mcu_width, hdr_info.mcu_height);

    struct jpeghw_decompress_struct dinfo;

    memset(&dinfo, 0, sizeof(struct jpeghw_decompress_struct));

    uint32_t flags = JPEGHW_CONFIG_FLAG_BLOCK_JPEG_CORE_AVAIL;
    if (trim)
    {
       flags |= JPEGHW_CONFIG_FLAG_TRIM;
    }

    if(jpeghw_create_decompress(&dinfo, flags) != e_JPEGHW_SUCCESS)
    {
        printf("create-decompress failed\n");
        close(out_fd);
        close(decompress_fd);
        fclose(decompress_fp);
        return 1;
    }

    dinfo.common.client_context = (void*)(intptr_t)decompress_fd;

    dinfo.common.mcu_height = hdr_info.mcu_height;
    dinfo.common.mcu_width = hdr_info.mcu_width;

    dinfo.common.image_width = hdr_info.image_width;
    dinfo.common.image_height = hdr_info.image_height;
    dinfo.common.bytes_per_pixel = hdr_info.bytes_per_pixel;

    dinfo.common.err = (struct jpeghw_error_mgr *)jpeghw_std_error(&jerr);

    dinfo.dmgr = MEM_MALLOC(sizeof(struct jpeghw_decompression_mgr));
    dinfo.dmgr->get_input_buffer = need_input_fnc;
    dinfo.dmgr->init = 0;
    dinfo.dmgr->term = 0;

    dinfo.common.scanline_timeout = 0;

    dinfo.dmgr->get_jpeg_info = 0;
    if(! dhdr) dinfo.dmgr->get_jpeg_info = get_jpeg_info; // put hdr gen func here

    // make sure these functions get called for coverage
    uint32_t max_lines = jhw_get_max_decompress_lines((struct jpeghw_common_struct *)&dinfo);
    if (max_lines < 1)
    {
       printf("max decompress lines woefully small\n");
       close(out_fd);
       close(decompress_fd);
       fclose(decompress_fp);
       return 1;
    }
    jhw_set_max_decompress_lines((struct jpeghw_common_struct *)&dinfo, max_lines);

    if(jpeghw_start_decompress(&dinfo) != e_JPEGHW_SUCCESS)
    {
        printf("start-decompress failed\n");
    }
    else
    {
        uint32_t bytes_read;
        int strip_height=64;
        struct BigBuffer_s *obb=0;
        int buflen= strip_height * hdr_info.mcu_aligned_width * hdr_info.bytes_per_pixel;
        if (trim)
        {
           buflen = strip_height * hdr_info.image_width * hdr_info.bytes_per_pixel;
        }

        bool last_buffer = false;
        while(!last_buffer)
        {
            obb = dma_buffer_malloc(0, buflen);
            if(obb == 0)
            {
                printf("OUT OF MEMORY\n");
                close(out_fd);
                close(decompress_fd);
                fclose(decompress_fp);
                return 1;
            }
            char *obuf = dma_buffer_mmap_forcpu(obb);
            memset(obuf, 0x0, buflen);
            dma_buffer_unmmap_forcpu(obb);

            bytes_read = jpeghw_read_scanlines(&dinfo, obb, &last_buffer);

            obuf = dma_buffer_mmap_forcpu(obb);
            //int iii; printf("out:");for(iii=0;iii<20;iii++)printf("%02X ",obuf[iii]);printf("\n");
            write(out_fd, obuf, bytes_read);
            dma_buffer_unmmap_forcpu(obb);
    
            if(obb != NULL)
            {
                (obb)->freeFunc(obb);
                obb =NULL;
            }

            if (abort)
            {
               jpeghw_abort_decompress(&dinfo);
               break;
            }
        }

        if(!abort && jpeghw_finish_decompress(&dinfo) != e_JPEGHW_SUCCESS)
        {
            printf("finish sent back an error\n");
        }
    }
    jpeghw_destroy_decompress(&dinfo);

    fclose(decompress_fp);
    close(decompress_fd);
    close(out_fd);

    MEM_FREE_AND_NULL(dinfo.dmgr);

    return 0;
}

//////////////////////////////////////////////////////////////////////////////////////////
int jpeg_coverage_test(int argc, char *argv[])
{
    char *color_ppm=argv[1];
    char *mono_ppm=argv[2];
    char *jpg_file="temp_jpgfile.jpg";
    char *raw_file="temp_rawfile.data";
    bool abort, too_long, dhuff, dquant, mcualign, auto_hdr;
    bool trim, dhdr;
  
    _compress_test(color_ppm, jpg_file,  15, abort=false, too_long=false, 
                                      dhuff=false, dquant=false, mcualign=true,  auto_hdr=true);
    _compress_test(color_ppm, jpg_file,  15, abort=false, too_long=false, 
                                      dhuff=false, dquant=false, mcualign=false, auto_hdr=true);
    // zero strip height implies incrementing the strip height
    _compress_test(color_ppm, jpg_file,   0, abort=false, too_long=false, 
                                      dhuff=false, dquant=false, mcualign=false, auto_hdr=true);
    // -1 strip height implies random the strip height
    _compress_test(color_ppm, jpg_file,  -1, abort=false, too_long=false, 
                                      dhuff=false, dquant=false, mcualign=false, auto_hdr=true);
    _compress_test(color_ppm, jpg_file,  16, abort=true,  too_long=false, 
                                      dhuff=false, dquant=false, mcualign=false, auto_hdr=true);
    _compress_test(color_ppm, jpg_file, 128, abort=false, too_long=false, 
                                      dhuff=false, dquant=false, mcualign=false, auto_hdr=true);
    _compress_test(color_ppm, jpg_file,  25, abort=false, too_long=true,  
                                      dhuff=false, dquant=false, mcualign=false, auto_hdr=true);
    _compress_test(color_ppm, jpg_file,  25, abort=false, too_long=true,  
                                      dhuff=false, dquant=true,  mcualign=false, auto_hdr=true);
    _compress_test(color_ppm, jpg_file,  32, abort=false, too_long=false, 
                                      dhuff=false,  dquant=false, mcualign=false, auto_hdr=false);
    _compress_test(color_ppm, jpg_file,  32, abort=false, too_long=false, 
                                      dhuff=true,  dquant=false, mcualign=false, auto_hdr=true);

    _decompress_test(jpg_file, raw_file, abort=false, trim=true,  dhdr=true);
    _decompress_test(jpg_file, raw_file, abort=false, trim=false, dhdr=true);
    _decompress_test(jpg_file, raw_file, abort=false, trim=false, dhdr=false);
    _decompress_test(jpg_file, raw_file, abort=false, trim=true,  dhdr=false);
    _decompress_test(jpg_file, raw_file, abort=true,  trim=false, dhdr=false);

    _compress_test(mono_ppm, jpg_file,  15, abort=false, too_long=false, 
                                      dhuff=false, dquant=false, mcualign=true,  auto_hdr=true);
    _compress_test(mono_ppm, jpg_file, 25, abort=false, too_long=false, 
                                      dhuff=false, dquant=false, mcualign=false, auto_hdr=true);
    _compress_test(mono_ppm, jpg_file, -1, abort=false, too_long=false, 
                                      dhuff=false, dquant=false, mcualign=false, auto_hdr=true);
    _compress_test(mono_ppm, jpg_file, 16, abort=false, too_long=false, 
                                      dhuff=true,  dquant=false, mcualign=false, auto_hdr=true);
    _compress_test(mono_ppm, jpg_file, 32, abort=false, too_long=false, 
                                      dhuff=true,  dquant=true,  mcualign=false, auto_hdr=false);
    _compress_test(mono_ppm, jpg_file, 16, abort=false, too_long=false, 
                                      dhuff=false,  dquant=true,  mcualign=false, auto_hdr=true);

    _decompress_test(jpg_file, raw_file, abort=false, trim=false, dhdr=true);
    _decompress_test(jpg_file, raw_file, abort=false, trim=true,  dhdr=true);
    _decompress_test(jpg_file, raw_file, abort=true,  trim=false, dhdr=false);

    unlink(jpg_file);
    unlink(raw_file);

    return 0;
}

//////////////////////////////////////////////////////////////////////////////////////////
int main(int argc, char** argv) 
{
    char *usage_stmt = "usage: test_coverage color-ppm-filename mono-ppm-filename\n";

    if(argc != 3)
    {
        printf(usage_stmt);
        return 1;
    }
    bool color;

    if(_check_input_file(argv[1], color=true)  != OK) return 1;
    if(_check_input_file(argv[2], color=false) != OK) return 1;

    uio_lib_init(); // init uio uses /dev/uio*
    jpeghw_init();

    jpeg_coverage_test(argc, argv);

    jpeghw_terminate();

    return 0;
}

