/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014-2015, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/



#include <sys/types.h>
#include <sys/stat.h>

#include "jpeghw_api.h"
#include "jpeghw_lib.h"
#include "jpeghw_dbg.h"
#include "jpeghw_dbg.h"
#include "jhw_mem_test.h"
#include "memAPI.h"
#include "test_common.h"

#define DEFAULT_STRIP_LINES 25   // 0=entire file (could run out of memory for large files)

//////////////////////////////////////////////////////////////////////////////////////////
//#define OUTPUT_BUFFER_LENGTH 1000 * 1024
#define OUTPUT_BUFFER_LENGTH 16 * 1024
struct BigBuffer_s* get_outbuf_fnc(struct jpeghw_compress_struct * jhwcinfo)
{
    //DBG_INFO("--%s\n",__func__);
    struct BigBuffer_s* output_buffer = NULL;

    char *obuf = MEM_MALLOC_ALIGN(OUTPUT_BUFFER_LENGTH, 4);
    output_buffer = (struct BigBuffer_s *)dma_buffer_adopt(obuf, OUTPUT_BUFFER_LENGTH);
    //DBG_INFO("obuf len:%d\n", output_buffer->datalen);

    return output_buffer;
}

//////////////////////////////////////////////////////////////////////////////////////////
bool outbuf_ready_fnc(struct jpeghw_compress_struct * jhwcinfo, 
                              struct BigBuffer_s* output_buffer, uint32_t bytes_in_buffer)
{
    DBG_INFO("[%d]--%s\n",gettid(),__func__);

    INC_G_MAPPED();
    char *read_ptr = dma_buffer_mmap_forcpu(output_buffer);
  
    uint32_t buflen = bytes_in_buffer;
    DBG_INFO("[%d] buflen=%d\n",gettid(), buflen);
    //printf("%s buflen=%d\n",__func__, buflen);
    //for(ii=0;ii<20;ii++)DBG_PRINTF_INFO("%02X ",read_ptr[ii]);DBG_PRINTF_INFO("\n");
    write((int)jhwcinfo->common.client_context, read_ptr, buflen);

    DEC_G_MAPPED();
    dma_buffer_unmmap_forcpu(output_buffer);

    if(output_buffer != NULL) 
    {
        (output_buffer)->freeFunc(output_buffer); 
        output_buffer =NULL; 
    }

    return true;
}

//////////////////////////////////////////////////////////////////////////////////////////
struct BigBuffer_s * need_input_fnc(struct jpeghw_decompress_struct * jhwdinfo, uint32_t *bytes)
{
    DBG_INFO("[%d]--%s\n",gettid(),__func__);
    int buflen = 16*1024;

    struct BigBuffer_s *bb = dma_buffer_malloc(0, buflen);
    INC_G_MAPPED();
    char *read_ptr = dma_buffer_mmap_forcpu(bb);

    *bytes = fread(read_ptr, 1, buflen, (FILE*)jhwdinfo->common.client_context);
    //printf("in:");for(ii=0;ii<20;ii++)printf("%02X ",read_ptr[ii]);printf("\n");
    DEC_G_MAPPED();
    dma_buffer_unmmap_forcpu(bb);

    return bb;
}

//////////////////////////////////////////////////////////////////////////////////////////
bool scanline_timeout(struct jpeghw_common_struct *info, uint32_t time_in_seconds) 
{
   printf("T"); fflush(stdout);
   return false; // never timeout
}

//////////////////////////////////////////////////////////////////////////////////////////
int read_ppm_header(int fd, int *bpp, int *width, int *height)
{
    char type[3], swidth[8], sheight[8];
    char c=0;
    int idx;

    read(fd, type, 3);
    type[2] = 0;
    DBG_INFO("[%d]type=%s\n",gettid(),type);
 
    if(strcmp(type,"P5") == 0)
    {
        *bpp = 1;
    }
    else if(strcmp(type,"P6") == 0)
    {
        *bpp = 3;
    }
    else
    {
        *bpp = 0;
        DBG_INFO("[%d]Don't recognize the file format\n",gettid());
        return -1;
    }

    read(fd, &c, 1);
    idx=0;
    // gimp places a comment here
    if(c == '#')
    {
        DBG_INFO("[%d]Found comment\n",gettid());
        // consume comment
        while(c != 0x0A)
        {
            if (read(fd, &c, 1) > 0)
            {
                /*DBG_INFO("%c",c);*/
            }
            else
            {
                // error or EOF
                break;
            }
        }
    }
    else
    {
        swidth[idx++] = c;
    }

    c=0;
    while(c != 0x0A && c != 0x20)
    {
        read(fd, &c, 1);
        swidth[idx++] = c;
        swidth[idx] = 0;
    }
    sscanf(swidth, "%d", width);

    c=0;
    idx=0;
    while(c != 0x0A && c != 0x20)
    {
        read(fd, &c, 1);
        sheight[idx++] = c;
        sheight[idx] = 0;
    }
    sscanf(sheight, "%d", height);

    c=0;
    idx=0;
    while(c != 0x0A && c != 0x20)
    {
        read(fd, &c, 1);
    }

    if(*width == 0 || *height == 0)
    {
        DBG_INFO("[%d]file dimensions are screwed up\n",gettid());
        return -1;
    }
    
    return 0;
}

//////////////////////////////////////////////////////////////////////////////////////////
int _compress_image_file_test(char *ppm_filename, char* jpg_filename, int strip_lines)
{
    DBG_INFO("[%d]%s\n",gettid(),__func__);
    DBG_INFO("[%d]ppm-file %s  jpg-file %s\n",gettid(),ppm_filename, jpg_filename);

    int ppm_fd=open(ppm_filename, O_RDONLY);
    if(ppm_fd < 0)
    {
        printf("cannot open file %s\n",ppm_filename);
        return 1;
    }

    int jpg_fd=open(jpg_filename, O_CREAT| O_RDWR);
    if(jpg_fd < 0)
    {
        printf("cannot open file %s\n",jpg_filename);
        close(ppm_fd);
        return 1;
    }
    chmod(jpg_filename, 0666);

    int bpp=0, width=0, file_height=0;

    if(read_ppm_header(ppm_fd, &bpp, &width, &file_height) == -1)
    {
       close(ppm_fd);
       close(jpg_fd);
       return 1;
    }

    if(strip_lines == 0) strip_lines = file_height;
    if(strip_lines > file_height) strip_lines = file_height;

    DBG_INFO("[%d]bpp=%d width=%d height=%d strip-lines=%d\n",
                       gettid(),bpp, width, file_height, strip_lines);
    
    struct jpeghw_compress_struct jhwcinfo;
    int lines=0;
    int pixel_width = width;
    int byte_width = pixel_width * bpp;
    int rbuf_size = strip_lines * byte_width;
    int image_height = file_height;// + 100; // +100 tests compressor truncation
    int mcu_dim = bpp == 1? 8:16;

    memset(&jhwcinfo, 0, sizeof(struct jpeghw_compress_struct));

    jhwcinfo.cmgr = MEM_MALLOC(sizeof(struct jpeghw_compression_mgr));
    memset(jhwcinfo.cmgr, 0, sizeof(struct jpeghw_compression_mgr));
   
    jhwcinfo.common.jpeghw_context = MEM_MALLOC(sizeof(struct jpeghw_dev_struct));

    jhwcinfo.common.config_flags |= JPEGHW_CONFIG_FLAG_BLOCK_JPEG_CORE_AVAIL;
    if(jhw_open_compress(&jhwcinfo) != e_JPEGHW_SUCCESS) 
    {
        printf("open-compress failed\n");
        close(ppm_fd);
        close(jpg_fd);
        return 1;
    }

    jhwcinfo.common.client_context = (void *)jpg_fd;

    jpeghw_dev_ptr jdev = jhwcinfo.common.jpeghw_context;
    jdev->info = &jhwcinfo.common;
    jhwcinfo.common.mcu_height = jdev->info->mcu_height = 
                jhwcinfo.common.mcu_width = jdev->info->mcu_width = mcu_dim;
    jhwcinfo.common.bytes_per_pixel = jdev->info->bytes_per_pixel = bpp;

    int mcu_aligned_pixel_width = pixel_width;
    if(mcu_aligned_pixel_width % jdev->info->mcu_width != 0)
    {
       mcu_aligned_pixel_width += jdev->info->mcu_width - (mcu_aligned_pixel_width % jdev->info->mcu_width);
    }

    jhwcinfo.common.image_width = jdev->info->image_width = pixel_width;
    jdev->image_bytes_per_row = pixel_width*jdev->info->bytes_per_pixel;
    jdev->mcu_aligned_width = mcu_aligned_pixel_width;
    jdev->mcu_bytes_per_row = mcu_aligned_pixel_width*jdev->info->bytes_per_pixel;

    jhwcinfo.common.image_height = jdev->info->image_height = image_height;
    jhwcinfo.common.global_jpeg_state = e_JPEGHW_CSTATE_START;  
    jhwcinfo.auto_generate_jpeg_header = true;

    jhwcinfo.quality = 90;

    jhwcinfo.cmgr->get_output_buffer = get_outbuf_fnc;
    jhwcinfo.cmgr->send_output_buffer = outbuf_ready_fnc;
    jhwcinfo.cmgr->get_quant_table = 0;
    jhwcinfo.cmgr->get_huff_table = 0;
    jhwcinfo.common.scanline_timeout = scanline_timeout;

    bool abort = false;

    if(jhw_start_compress(&jhwcinfo) != e_JPEGHW_SUCCESS) 
    {
        printf("start-compress failed\n");
    }
    else
    {
        while(lines < file_height)
        {
            struct BigBuffer_s *ibb=0;
            int eoi=0;
            int write_lines=0;

            write_lines = strip_lines;
    
            if (rbuf_size == 0)
            {
                printf("Invalid rbuf_size = 0\n");
                close(ppm_fd);
                close(jpg_fd);
                return 1;
            }

            DBG_INFO("[%d]allocate %d bytes\n",gettid(),rbuf_size);
            ibb = dma_buffer_malloc(0, rbuf_size);
            if(ibb == 0)
            {
                printf("OUT OF MEMORY\n");
                close(ppm_fd);
                close(jpg_fd);
                return 1;     
            }               
            INC_G_MAPPED();
            char *ibuf = dma_buffer_mmap_forcpu(ibb);
            ssize_t rv = read(ppm_fd, ibuf, rbuf_size);
            write_lines = rv/byte_width;
            DBG_INFO("[%d]---------read %d bytes\n",gettid(), write_lines);
            DEC_G_MAPPED();
            dma_buffer_unmmap_forcpu(ibb);
 
            //int ii; for(ii=0;ii<20;ii++)DBG_PRINTF_INFO("%02X ",ibuf[ii]);DBG_PRINTF_INFO("\n");
            lines += write_lines;
            if(lines >= file_height)
            {
                eoi = 1;
            }
            if(rv == 0)
            {
                // this is an error.  we need to detect eoi before the last buffer is sent
                // or else the "last buffer" is already sent
                DBG_ERR("ERROR: read no data from input file.\n");
                break;
            }
   
            if(jhw_write_buffer(&jhwcinfo, ibb, write_lines, eoi) == 0)
            {
               DBG_ERR("ERROR: write buffer retuned 0.\n");
               abort = true;
            }
        }
    
        if (abort)
        {
           jhw_abort_compress(&jhwcinfo);
        }
        else
        {
           if(jhw_finish_compress(&jhwcinfo) != e_JPEGHW_SUCCESS)
           {
               printf("finish sent back an error\n");
               exit(1);
           }
        }
    }
    jhw_close_compress(&jhwcinfo);

    close(jpg_fd);
    close(ppm_fd);

    MEM_FREE_AND_NULL(jhwcinfo.cmgr);
    MEM_FREE_AND_NULL(jhwcinfo.common.jpeghw_context);

    return 0;
}

//////////////////////////////////////////////////////////////////////////////////////////
int _duo_compress_image_file_test(char *ppm_filename,
                            char* jpg1_filename, char* jpg2_filename, int strip_lines)
{
    DBG_INFO("[%d]%s\n",gettid(),__func__);
    DBG_INFO("[%d]ppm-file %s  jpg1-file %s jpg2_file %s\n",gettid(),
             ppm_filename, jpg1_filename, jpg2_filename);

    int ppm_fd=open(ppm_filename, O_RDONLY);
    if(ppm_fd < 0)
    {
        printf("cannot open file %s\n",ppm_filename);
        return 1;
    }

    unlink(jpg1_filename);
    unlink(jpg2_filename);

    int jpg1_fd=open(jpg1_filename, O_CREAT| O_RDWR);
    if(jpg1_fd < 0)
    {
        printf("cannot open file %s\n",jpg1_filename);
        close(ppm_fd);
        return 1;
    }

    int jpg2_fd=open(jpg2_filename, O_CREAT| O_RDWR);
    if(jpg2_fd < 0)
    {
        printf("cannot open file %s\n",jpg2_filename);
        close(ppm_fd);
        close(jpg1_fd);
        return 1;
    }

    int bpp=0, width=0, file_height=0;

    if(read_ppm_header(ppm_fd, &bpp, &width, &file_height) == -1)
    {
        close(ppm_fd);
        close(jpg1_fd);
        close(jpg2_fd);
        return 1;
    }

    if(strip_lines == 0) strip_lines = file_height;
    if(strip_lines > file_height) strip_lines = file_height;

    DBG_INFO("[%d]header: bpp=%d width=%d height=%d\n",gettid(),bpp, width, file_height);
    
    struct jpeghw_compress_struct jhwcinfo1, jhwcinfo2;
    int lines=0;
    int pixel_width = width;
    int byte_width = pixel_width * bpp;
    int rbuf_size = strip_lines * byte_width;
    int image_height = file_height;// + 100; // +100 tests compressor truncation
    int mcu_dim = bpp == 1? 8:16;

    memset(&jhwcinfo1, 0, sizeof(struct jpeghw_compress_struct));
    memset(&jhwcinfo2, 0, sizeof(struct jpeghw_compress_struct));

    INC_G_ALLOCATED();
    jhwcinfo1.cmgr = MEM_MALLOC(sizeof(struct jpeghw_compression_mgr));
    INC_G_ALLOCATED();
    jhwcinfo1.common.jpeghw_context = MEM_MALLOC(sizeof(struct jpeghw_dev_struct));
    INC_G_ALLOCATED();
    jhwcinfo2.cmgr = MEM_MALLOC(sizeof(struct jpeghw_compression_mgr));
    INC_G_ALLOCATED();
    jhwcinfo2.common.jpeghw_context = MEM_MALLOC(sizeof(struct jpeghw_dev_struct));

    jhwcinfo1.common.config_flags |= JPEGHW_CONFIG_FLAG_BLOCK_JPEG_CORE_AVAIL;
    jhwcinfo2.common.config_flags |= JPEGHW_CONFIG_FLAG_BLOCK_JPEG_CORE_AVAIL;
    if(! (jhw_open_compress(&jhwcinfo1) == e_JPEGHW_SUCCESS  &&
          jhw_open_compress(&jhwcinfo2) == e_JPEGHW_SUCCESS) )
    {
        printf("open-compress failed\n");
        close(ppm_fd);
        close(jpg1_fd);
        close(jpg2_fd);
        return 1;
    }

    jhwcinfo1.common.client_context = (void *)jpg1_fd;
    jhwcinfo2.common.client_context = (void *)jpg2_fd;

    jpeghw_dev_ptr jdev1 = jhwcinfo1.common.jpeghw_context;
    jpeghw_dev_ptr jdev2 = jhwcinfo2.common.jpeghw_context;
    jdev1->info = &jhwcinfo1.common;
    jdev2->info = &jhwcinfo2.common;

    jhwcinfo1.common.mcu_height = jdev1->info->mcu_height = 
                jhwcinfo1.common.mcu_width = jdev1->info->mcu_width = 
                jhwcinfo2.common.mcu_height = jdev2->info->mcu_height = 
                jhwcinfo2.common.mcu_width = jdev2->info->mcu_width = mcu_dim;
    jhwcinfo1.common.bytes_per_pixel = jdev1->info->bytes_per_pixel = 
            jhwcinfo2.common.bytes_per_pixel = jdev2->info->bytes_per_pixel = bpp;

    int mcu_aligned_pixel_width = pixel_width;
    if(mcu_aligned_pixel_width % jdev1->info->mcu_width != 0)
    {
       mcu_aligned_pixel_width += jdev1->info->mcu_width - 
                           (mcu_aligned_pixel_width % jdev1->info->mcu_width);
    }

    jhwcinfo1.common.image_width = jdev1->info->image_width = 
             jhwcinfo2.common.image_width = jdev2->info->image_width = pixel_width;
    jdev1->image_bytes_per_row = 
                   jdev2->image_bytes_per_row = pixel_width*jdev1->info->bytes_per_pixel;
    jdev1->mcu_aligned_width = 
                  jdev2->mcu_aligned_width = mcu_aligned_pixel_width;
    jdev1->mcu_bytes_per_row = 
            jdev2->mcu_bytes_per_row = mcu_aligned_pixel_width*jdev1->info->bytes_per_pixel;

    jhwcinfo1.common.image_height = jhwcinfo2.common.image_height = 
        jdev1->info->image_height = jdev2->info->image_height = image_height;
    jhwcinfo1.common.global_jpeg_state = 
             jhwcinfo2.common.global_jpeg_state = e_JPEGHW_CSTATE_START;  
    jhwcinfo1.auto_generate_jpeg_header = jhwcinfo2.auto_generate_jpeg_header = true;

    jhwcinfo1.quality = jhwcinfo2.quality = 90;

    jhwcinfo1.cmgr->get_output_buffer = jhwcinfo2.cmgr->get_output_buffer = get_outbuf_fnc;
    jhwcinfo1.cmgr->send_output_buffer = jhwcinfo2.cmgr->send_output_buffer = outbuf_ready_fnc;
    jhwcinfo1.cmgr->get_quant_table = jhwcinfo2.cmgr->get_quant_table = 0;

    if(! (jhw_start_compress(&jhwcinfo1) == e_JPEGHW_SUCCESS  &&
          jhw_start_compress(&jhwcinfo2) == e_JPEGHW_SUCCESS) )
    {
        printf("start-compress failed\n");
    }
    else
    {
        while(lines < file_height)
        {
            struct BigBuffer_s *ibb1=0, *ibb2=0;
            int eoi=0;
            int write_lines=0;

            write_lines = strip_lines;

            if (rbuf_size == 0)
            {
                printf("Invalid rbuf_size = 0\n");
                close(ppm_fd);
                close(jpg1_fd);
                close(jpg2_fd);
                return 1;
            }

            ibb1 = dma_buffer_malloc(0, rbuf_size);
            ibb2 = dma_buffer_malloc(0, rbuf_size);
            if(ibb1 == 0 || ibb2 == 0)
            {
                printf("OUT OF MEMORY\n");
                close(ppm_fd);
                close(jpg1_fd);
                close(jpg2_fd);
                return 1;     
            }               
            INC_G_MAPPED();
            char *ibuf1 = dma_buffer_mmap_forcpu(ibb1);
            char *ibuf2 = dma_buffer_mmap_forcpu(ibb2);
            ssize_t rv = read(ppm_fd, ibuf1, rbuf_size);
            memcpy(ibuf2, ibuf1, rv);
            write_lines = rv/byte_width;
            DBG_INFO("[%d]---------read %d lines\n",gettid(), write_lines);
            DEC_G_MAPPED();
            dma_buffer_unmmap_forcpu(ibb1);
            dma_buffer_unmmap_forcpu(ibb2);
 
            //int ii; for(ii=0;ii<20;ii++)DBG_PRINTF_INFO("%02X ",ibuf[ii]);DBG_PRINTF_INFO("\n");
            lines += strip_lines;
            if(lines >= file_height)
            {
                eoi = 1;
            }
            if(rv == 0)
            {
                // this is an error.  we need to detect eoi before the last buffer is sent
                // or else the "last buffer" is already sent
                DBG_ERR("ERROR: read no data from input file.\n");
                break;
            }
    
            DBG_INFO("[%d]write lines %d\n",gettid(), write_lines);
            jhw_write_buffer(&jhwcinfo1, ibb1, write_lines, eoi);
            jhw_write_buffer(&jhwcinfo2, ibb2, write_lines, eoi);
        }
    
        if(jhw_finish_compress(&jhwcinfo1) != e_JPEGHW_SUCCESS ||
           jhw_finish_compress(&jhwcinfo2) != e_JPEGHW_SUCCESS)
        {
            printf("finish sent back an error\n");
        }
    }

    close(jpg1_fd);
    close(jpg2_fd);
    close(ppm_fd);

    jhw_close_compress(&jhwcinfo1);
    jhw_close_compress(&jhwcinfo2);

    DEC_G_ALLOCATED();
    MEM_FREE_AND_NULL(jhwcinfo1.cmgr);
    DEC_G_ALLOCATED();
    MEM_FREE_AND_NULL(jhwcinfo1.common.jpeghw_context);
    DEC_G_ALLOCATED();
    MEM_FREE_AND_NULL(jhwcinfo2.cmgr);
    DEC_G_ALLOCATED();
    MEM_FREE_AND_NULL(jhwcinfo2.common.jpeghw_context);

    return 0;
}

//////////////////////////////////////////////////////////////////////////////////////////
int _decompress_image_file_test(char *jpg_filename, char* raw_filename)
{
    JPEG_header_struct hdr_info;

    DBG_INFO("[%d]%s\n",gettid(),__func__);
    DBG_INFO("[%d]jpg-file %s  raw-file:%s\n",gettid(),jpg_filename, raw_filename);

    FILE *decompress_fd=fopen(jpg_filename, "r");
    if(!decompress_fd)
    {
        DBG_INFO("[%d]cannot open file %s\n",gettid(),jpg_filename);
        return 1;
    }
    FILE *out_fd=fopen(raw_filename, "w+");
    if(!out_fd)
    {
        DBG_INFO("[%d]cannot open file %s\n",gettid(),raw_filename);
        fclose(decompress_fd);
        return 1;
    }

    DBG_INFO("[%d]parse header from %s\n",gettid(),jpg_filename);
    parse_jpeg_header(decompress_fd, &hdr_info, true);

    fseek(decompress_fd, 0, SEEK_SET);

    DBG_INFO("[%d]dim:  w:%d h:%d bbp:%d mw:%d  mh:%d mcu:%dx%d\n",gettid(),
                   hdr_info.image_width,hdr_info.image_height,hdr_info.bytes_per_pixel,
                   hdr_info.mcu_aligned_width, hdr_info.mcu_aligned_height,
                   hdr_info.mcu_width, hdr_info.mcu_height);

    struct jpeghw_decompress_struct jhwdinfo;

    memset(&jhwdinfo, 0, sizeof(struct jpeghw_decompress_struct));
    INC_G_ALLOCATED();
    jhwdinfo.dmgr = MEM_MALLOC(sizeof(struct jpeghw_decompression_mgr));
    INC_G_ALLOCATED();
    jhwdinfo.common.jpeghw_context = MEM_MALLOC(sizeof(struct jpeghw_dev_struct));

    jhwdinfo.common.config_flags |= JPEGHW_CONFIG_FLAG_BLOCK_JPEG_CORE_AVAIL;
    jhwdinfo.common.config_flags |= JPEGHW_CONFIG_FLAG_TRIM;

    if(jhw_open_decompress(&jhwdinfo) != e_JPEGHW_SUCCESS)
    {
        printf("open-decompress failed\n");
        fclose(out_fd);
        fclose(decompress_fd);
        return 1;
    }

    jhwdinfo.common.client_context = (void *)decompress_fd;

    jpeghw_dev_ptr jdev = jhwdinfo.common.jpeghw_context;
    jdev->info = &jhwdinfo.common;

    jdev->info->mcu_height = hdr_info.mcu_height;
    jdev->info->mcu_width = hdr_info.mcu_width;
    jhwdinfo.common.bytes_per_pixel = jdev->info->bytes_per_pixel = hdr_info.bytes_per_pixel;

    jhwdinfo.common.image_width = jdev->info->image_width = hdr_info.image_width;
    jhwdinfo.common.image_height = jdev->info->image_height = hdr_info.image_height;
    jdev->image_bytes_per_row =  hdr_info.image_width * hdr_info.bytes_per_pixel;
    jdev->mcu_aligned_height = hdr_info.mcu_aligned_height;
    jdev->mcu_aligned_width = hdr_info.mcu_aligned_width;
    jdev->mcu_bytes_per_row = hdr_info.mcu_aligned_width * hdr_info.bytes_per_pixel;
    jhwdinfo.common.global_jpeg_state = e_JPEGHW_DSTATE_START;

    jhwdinfo.dmgr->get_input_buffer = need_input_fnc;
    jhwdinfo.dmgr->get_jpeg_info = 0;
    jhwdinfo.common.scanline_timeout = 0; // 0 = a timeout of 10 seconds

    if(jhw_start_decompress(&jhwdinfo) != e_JPEGHW_SUCCESS)
    {
       printf("start-decompress failed\n");
    }
    else
    {
        uint32_t bytes_read;
        int strip_height=64;
        struct BigBuffer_s *obb=0;
        int buflen= strip_height * hdr_info.mcu_aligned_width * hdr_info.bytes_per_pixel;

        bool last_buffer = false;
        while(!last_buffer)
        {
            obb = dma_buffer_malloc(0, buflen);
            if(obb == 0)
            {
                printf("OUT OF MEMORY\n");
                fclose(out_fd);
                fclose(decompress_fd);
                return 1;
            }
            INC_G_MAPPED();
            char *obuf = dma_buffer_mmap_forcpu(obb);
            memset(obuf, 0x0, obb->datalen);
            DEC_G_MAPPED();
            dma_buffer_unmmap_forcpu(obb);

            jhw_read_buffer(&jhwdinfo, obb, &bytes_read, &last_buffer);

            INC_G_MAPPED();
            obuf = dma_buffer_mmap_forcpu(obb);
            //int iii; printf("out:");for(iii=0;iii<20;iii++)printf("%02X ",obuf[iii]);printf("\n");
            fwrite(obuf, 1, bytes_read, out_fd);
            DEC_G_MAPPED();
            dma_buffer_unmmap_forcpu(obb);
    
            if(obb != NULL)
            {
                (obb)->freeFunc(obb);
                obb =NULL;
            }
        }

        if(jhw_finish_decompress(&jhwdinfo) != e_JPEGHW_SUCCESS)
        {
            printf("finish sent back an error\n");
        }
    }
    jhw_close_decompress(&jhwdinfo);

    fclose(decompress_fd);
    fclose(out_fd);

    DEC_G_ALLOCATED();
    MEM_FREE_AND_NULL(jhwdinfo.dmgr);
    DEC_G_ALLOCATED();
    MEM_FREE_AND_NULL(jhwdinfo.common.jpeghw_context);

    return 0;
}

//////////////////////////////////////////////////////////////////////////////////////////
int duo_compress_image_file_test(int argc, char *argv[])
{
    DBG_INFO("[%d]duo compress_image_files_test\n",gettid());
    char *usage_stmt =
        "usage: jhw_test e input-ppm-filename output-jpeg1-filename output-jpeg1-filename number-of-strip-lines\n";

    if(argc != 6)
    {
        printf(usage_stmt);
        return 1;
    }

    int strip_lines=DEFAULT_STRIP_LINES;
    char *ppm_filename=argv[2];
    char *jpg1_filename=argv[3];
    char *jpg2_filename=argv[4];
    sscanf(argv[5], "%d", &strip_lines);

    if(_duo_compress_image_file_test(ppm_filename, 
                                  jpg1_filename, jpg2_filename, strip_lines) != 0)
    {
        printf("duo compress test failed\n");
        return 1;
    }

    char cmpcmd[256];
    sprintf(cmpcmd, "cmp %s %s 2>&1", jpg1_filename, jpg2_filename);

    FILE *ofp = popen(cmpcmd, "r");
    char cmpcmd_ret[256] = {0};
    int rv = fread(cmpcmd_ret, 1, 256, ofp);
    if (rv == 0) 
    {
        printf("Test passed\n");
    }
    else
    {
        printf("Test failed, compare failed:\n");
        printf("   %s\n", cmpcmd_ret);
    }

    pclose(ofp);

    return 0;
}

//////////////////////////////////////////////////////////////////////////////////////////
int compress_image_file_test(int argc, char *argv[])
{
    printf("compress_image_files_test\n");
    DBG_INFO("[%d]compress_image_files_test\n",gettid());
    char *usage_stmt =
        "usage: jhw_test c input-ppm-filename output-jpeg-filename number-of-strip-lines\n";

    if(argc != 5)
    {
        printf(usage_stmt);
        return 1;
    }

    int strip_lines=DEFAULT_STRIP_LINES;
    char *ppm_filename=argv[2];
    char *jpg_filename=argv[3];
    sscanf(argv[4], "%d", &strip_lines);

    return _compress_image_file_test(ppm_filename, jpg_filename, strip_lines);
}

//////////////////////////////////////////////////////////////////////////////////////////
int decompress_image_file_test(int argc, char *argv[])
{
    printf("decompress_image_files_test\n");
    DBG_INFO("[%d]decompress_image_files_test\n", gettid());
    char *usage_stmt =
        "usage: jhw_test d input-jpg-filename output-raw-filename\n";

    if(argc != 4)
    {
        printf(usage_stmt);
        return 1;
    }

    char *jpg_filename=argv[2];
    char *ppm_filename=argv[3];

    return _decompress_image_file_test(jpg_filename, ppm_filename);
}

//////////////////////////////////////////////////////////////////////////////////////////
bool compare_files(char *file1, char* file2)
{
   char cmd[1024];
   char cmdout[1024]={0};

   sprintf(cmd, "cmp %s %s", file1, file2);

   FILE *pfp = popen(cmd, "r");
   if (pfp == 0)
   {
      printf("Unable to open pipe to cmd %s\n", cmd);
      return false;
   }

   fread(cmdout, 1, 1024, pfp);
   pclose(pfp);

   if(strlen(cmdout) > 0) return false; 

   return true;
}

//////////////////////////////////////////////////////////////////////////////////////////
typedef struct filenames_s
{
    char fname1[256];
    char fname2[256];
    int iterations;
} filenames_t, *filenames_ptr;

void * _compress_thread(void * arg)
{
    filenames_ptr fnames = (filenames_ptr)arg;
    int inc;
    char cmp_file[256];
    static int cerr=0;
    static int terr=0;

    sprintf(cmp_file, "%scmp.jpg", fnames->fname1);

    unlink(cmp_file);

    _compress_image_file_test(fnames->fname1, cmp_file, DEFAULT_STRIP_LINES); 

    for(inc=0; inc<fnames->iterations; inc++)
    {
        terr++;
        DBG_INFO("[%d]----------start compress thread %d %s\n",gettid(), inc, fnames->fname1);
        //if(inc % 10 == 0) printf("."); fflush(stdout);
        printf("."); fflush(stdout);
        if(terr % 100 == 0) printf("%d",terr); fflush(stdout);
        unlink(fnames->fname2);
        _compress_image_file_test(fnames->fname1, fnames->fname2, DEFAULT_STRIP_LINES); 

        char errfile[256]={0};
        if (compare_files(cmp_file, fnames->fname2) == false)
        {
            printf("compare failed between %s and %s\n", cmp_file, fnames->fname2);
            
            sprintf(errfile, "cerr%d.jpg", cerr++);
            printf("bad file saved to %s\n", errfile);
            link(fnames->fname2, errfile);
            unlink(fnames->fname2);
            return 0;
        }
    }

    return 0;
}

void * _decompress_thread(void * arg)
{
    filenames_ptr fnames = (filenames_ptr)arg;
    int inc;

    for(inc=0; inc<fnames->iterations; inc++)
    {
        DBG_INFO("[%d]----------start decompress thread %d %s\n",gettid(), inc, fnames->fname1);
        if(inc % 10 == 0) printf("D%d ",inc); fflush(stdout);
        _decompress_image_file_test(fnames->fname1, fnames->fname2);
    }

    return 0;
}

//////////////////////////////////////////////////////////////////////////////////////////
int multicore_compress_image_files_test(int argc, char *argv[])
{
    pthread_t thd_id_1=0, thd_id_2=0;
    printf("multicore_compress_image_files_test\n");
    DBG_INFO("[%d]multicore_compress_image_files_test\n",gettid());

    char *usage_stmt =
        "usage: jhw_test m input-filename1 output-filename1 input-filename2 output-filename2 thread-iterations test-iterations\n";

    if(argc != 8)
    {
        printf(usage_stmt);
        return 1;
    }

    char *in_filename1  = argv[2];
    char *out_filename1 = argv[3];
    char *in_filename2  = argv[4];
    char *out_filename2 = argv[5];
    char *in_type1 = strrchr(in_filename1, '.');
    char *in_type2 = strrchr(in_filename2, '.');

    filenames_t fnamesA;
    strcpy(fnamesA.fname1, in_filename1);
    strcpy(fnamesA.fname2, out_filename1);
    filenames_t fnamesB;
    strcpy(fnamesB.fname1, in_filename2);
    strcpy(fnamesB.fname2, out_filename2);

    int thread_iterations = 1;
    int test_iterations = 1;
    sscanf(argv[6], "%d", &thread_iterations);
    sscanf(argv[7], "%d", &test_iterations);
    fnamesA.iterations = fnamesB.iterations = thread_iterations;
   
    // test_iterations == 0 => infinite loop
    while(test_iterations >= 1 || test_iterations == 0)
    {
        //printf("\ninteration:%d\n", test_iterations);
        if (in_type1 != NULL && strcmp(in_type1, ".ppm") == 0)
        {
            pthread_create(&(thd_id_1), 0, _compress_thread, (void *)&fnamesA);
        }
        else if(in_type1 != NULL && strcmp(in_type1, ".jpg") == 0)
        {
            pthread_create(&(thd_id_1), 0, _decompress_thread, (void *)&fnamesA);
        }
        else
        {
            printf("%s input file format not supported\n", fnamesA.fname1);
            return 1;
        }

        if(in_type2 != NULL && strcmp(in_type2, ".ppm") == 0)
        {
            pthread_create(&(thd_id_2), 0, _compress_thread, (void *)&fnamesB);
        }
        else if(in_type2 != NULL && strcmp(in_type2, ".jpg") == 0)
        {
            pthread_create(&(thd_id_2), 0, _decompress_thread, (void *)&fnamesB);
        }
        else
        {
            printf("%s input file format not supported\n", fnamesB.fname1);
            return 1;
        }

        pthread_join(thd_id_1, NULL);
        pthread_join(thd_id_2, NULL);
        DSP_MEM_STATS();

        if (test_iterations == 1) break;
        if (test_iterations > 1) test_iterations--;
    }

    printf("\ndone\n");

    return 0;
}

//////////////////////////////////////////////////////////////////////////////////////////
int main(int argc, char *argv[])
{
    jpeghw_init();

    char *usage_stmt =
        "usage: jhw_test c,d,m,e\n"
        "   c: compress test - compress one file, one core, one thread\n"
        "   d: decompress test - decompress one file, one core, one thread\n"
        "   m: multicore test - process two files, two cores, two threads\n"
        "                       iterate multiple times\n"
        "                 note: compress input ppm-file, decompress input jpg-file\n"
        "   e: duo-compress test - compress one file into two output files\n"
        "                       using two cores, one thread\n";

    if(argc < 2)
    {
        printf(usage_stmt);
        return 1;
    }

    if(strcmp(argv[1], "c") == 0)
    {
        compress_image_file_test(argc, argv);
    }
    else if(strcmp(argv[1], "d") == 0)
    {
        decompress_image_file_test(argc, argv);
    }
    else if(strcmp(argv[1], "m") == 0)
    {
        multicore_compress_image_files_test(argc, argv);
    }
    else if (strcmp(argv[1], "e") == 0) 
    {
        duo_compress_image_file_test(argc, argv);
    }
    else
    {
        printf(usage_stmt);
        return 1;
    }

    jpeghw_terminate();

    return 0;
}
