/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2015, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/



#include "jpeghw_lib.h"
#include "jpeghw_dbg.h"
#include "error_types.h"
#include <stdio.h>

void jpeghw_default_error_exit(struct jpeghw_common_struct * info);
void jpeghw_default_error_reset(struct jpeghw_common_struct * info);


/* common API library default functions */

/**
 * \brief Initialization of the JPEGHW library.
 *
 * This should only be called once at startup by the application.
 *
 **/
void jpeghw_init()
{
	DBG_INIT();
	jhw_initialize();

}

/**
 * \brief Terminate the use of the JPEGHW library.
 *
 * This should only be called once during the termination of the application.
 *
 **/
void jpeghw_terminate()
{
	DBG_TERMINATE();
	jhw_terminate();
}

/**
 * \brief Sets defaults for a jpeghw object
 *
 * \param info - pointer to jpeghw object
 *
 * This sets the standard default values for either a compression or
 * a decompression object.
 *
 * \return e_JPEGHW_SUCCESS if success, else a jpeghw_error_type_t error
 *
 **/
jpeghw_error_type_t jpeghw_set_defaults (struct jpeghw_common_struct * info)
{
	JPEGHW_VALIDATE_COMMON_INFO(info);
	info->last_error = e_JPEGHW_SUCCESS;

	if (info->image_width == 0 || info->image_height == 0 || info->bytes_per_pixel == 0)
	{
		jpeghw_error(info, e_JPEGHW_ERR_INVALID_PARAMETERS);
	}

	if (info->mcu_width == 0)
	{
		info->mcu_width = info->bytes_per_pixel==1 ? 8 : 16;
	}

	if (info->mcu_height == 0)
	{
		info->mcu_height = info->mcu_width;
	}

	if (info->type == e_JPEGHW_OBJ_COMPRESS)
	{
		struct jpeghw_compress_struct * cinfo = (struct jpeghw_compress_struct *)info;
		JPEGHW_VALIDATE_INFO(cinfo, e_JPEGHW_OBJ_COMPRESS);

		/* check for valid jpeg state */
		if (cinfo->common.global_jpeg_state != e_JPEGHW_CSTATE_START)
		{
			return jpeghw_error((struct jpeghw_common_struct *)cinfo, e_JPEGHW_ERR_BAD_STATE);
		}

		cinfo->quality = 90;
		cinfo->auto_generate_jpeg_header = true;
	}
	else if (info->type == e_JPEGHW_OBJ_DECOMPRESS)
	{
		struct jpeghw_decompress_struct * dinfo = (struct jpeghw_decompress_struct *)info;
		JPEGHW_VALIDATE_INFO(dinfo, e_JPEGHW_OBJ_DECOMPRESS);

		/* check for valid jpeg state */
		if (dinfo->common.global_jpeg_state != e_JPEGHW_DSTATE_START)
		{
			return jpeghw_error((struct jpeghw_common_struct *)dinfo, e_JPEGHW_ERR_BAD_STATE);
		}
	}

	return jpeghw_error(info, e_JPEGHW_SUCCESS);
}

/**
 * \brief Sets standard error callbacks for error manager struct
 *
 * \param err - pointer to jpeghw error manager struct
 *
 * \return pointer to updated jpeghw_error_mgr struct
 *
 **/
struct jpeghw_error_mgr* jpeghw_std_error (struct jpeghw_error_mgr * err)
{
  err->error_exit = jpeghw_default_error_exit;
  err->error_reset = jpeghw_default_error_reset;
  err->err_code = e_JPEGHW_SUCCESS;

  return err;
}

jpeghw_error_type_t jpeghw_get_quant_table(struct jpeghw_common_struct * info, uint8_t quality, uint8_t *table, uint32_t *size)
{
	jpeghw_error_type_t err = e_JPEGHW_ERROR;

	if (info == NULL)
	{
		return e_JPEGHW_ERR_INVALID_PARAMETERS;
	}

	if (info->type == e_JPEGHW_OBJ_COMPRESS)
	{
		struct jpeghw_compress_struct * cinfo = (struct jpeghw_compress_struct *)info;
		JPEGHW_VALIDATE_INFO(cinfo, e_JPEGHW_OBJ_COMPRESS);

		if(cinfo->cmgr && cinfo->cmgr->get_quant_table)
		{
			err = cinfo->cmgr->get_quant_table(cinfo, quality, table, size);
		}
	}

    if (err != e_JPEGHW_SUCCESS)
    {
    	err = jhw_get_default_quant_table(quality, table, size);
    }

    return err;
}

jpeghw_error_type_t jpeghw_get_huff_table(struct jpeghw_common_struct * info, int table_index, bool ac, uint8_t *bits, uint32_t *bits_size, uint8_t *val, uint32_t *val_size)
{
	jpeghw_error_type_t err = e_JPEGHW_ERROR;

	if (info == NULL)
	{
		return e_JPEGHW_ERR_INVALID_PARAMETERS;
	}

	if (info->type == e_JPEGHW_OBJ_COMPRESS)
	{
		struct jpeghw_compress_struct * cinfo = (struct jpeghw_compress_struct *)info;
		JPEGHW_VALIDATE_INFO(cinfo, e_JPEGHW_OBJ_COMPRESS);

		if(cinfo->cmgr && cinfo->cmgr->get_huff_table)
		{
	        err = cinfo->cmgr->get_huff_table(cinfo, table_index, ac, bits, bits_size, val, val_size);
		}
	}

    if (err != e_JPEGHW_SUCCESS)
    {
   		err = jhw_get_default_huff_table(table_index, ac, bits, bits_size, val, val_size);
    }

    return err;
}


/* internal library functions */

/**
 * \brief Performs the default behavior for an error exit.
 *
 * \param info - pointer to jpeghw object
 *
 **/
void jpeghw_default_error_exit(struct jpeghw_common_struct * info)
{
	JPEGHW_VALIDATE_COMMON_INFO(info);

	DBG_ERR("jpeghw_default_error_exit() - exit due to error %d!\n", info->err->err_code);
	exit(-1);
}


/**
 * \brief Performs the default behavior for an error reset.
 *
 * \param info - pointer to jpeghw object
 *
 **/
void jpeghw_default_error_reset(struct jpeghw_common_struct * info)
{
	JPEGHW_VALIDATE_COMMON_INFO(info);

	DBG_ERR("jpeghw_default_error_reset() - reset JPEG due to error %d!\n", info->err->err_code);

	// perform any hardware reset required.
	jhw_device_reset(info);
}

/**
 * \brief Validates the BigBuffer is setup properly.
 *
 * \param bb - pointer to BigBuffer
 *
 * \return true if valid, false otherwise
 *
 **/
bool jpeghw_valid_buffer(struct BigBuffer_s* bb)
{
	if (bb == NULL || bb->dma_alloc == NULL ||
			bb->dma_alloc->kv_addr == 0 ||
			bb->dma_alloc->kv_addr == (char *)0xdeadbeef
		)
	{
		return false;
	}

	return true;
}

/**
 * \brief Sets the error code for a jpeghw object
 *
 * \param info - pointer to jpeghw object
 * \param err_code - specific error code to set
 *
 * \return jpeghw_error_type_t error code passed into function
 *
 **/
jpeghw_error_type_t jpeghw_error(struct jpeghw_common_struct * info, jpeghw_error_type_t err_code)
{
	JPEGHW_VALIDATE_COMMON_INFO(info);

	info->last_error = err_code;

	if (info->err)
	{
		info->err->err_code = err_code;
	}
	else
	{
		// no error manager?
	}

	return err_code;
}

/**
 * \brief Sets the error code for a jpeghw object and exits
 *
 * \param info - pointer to jpeghw object
 * \param err_code - specific error code to set
 *
 * \return jpeghw_error_type_t error code passed into function
 *
 **/
jpeghw_error_type_t jpeghw_error_exit(struct jpeghw_common_struct * info, jpeghw_error_type_t err_code)
{
	JPEGHW_VALIDATE_COMMON_INFO(info);

	info->last_error = err_code;
	info->global_jpeg_state = e_JPEGHW_STATE_ERROR;

	if (info->err)
	{
		info->err->err_code = err_code;
		info->err->error_exit(info);
	}
	else
	{
		// no error manager?
	}

	return err_code;
}

/**
 * \brief Reset error code and jpeghw device hardware
 *
 * \param info - pointer to jpeghw object
 *
 * \return
 *
 **/
void jpeghw_error_reset(struct jpeghw_common_struct * info)
{
	JPEGHW_VALIDATE_COMMON_INFO(info);

	info->last_error = e_JPEGHW_SUCCESS;

	if (info->err)
	{
		info->err->err_code = e_JPEGHW_SUCCESS;
		info->err->error_reset(info);
	}
}

/**
 * \brief Allocate a new jpeghw device object and initialize it.
 *
 * \param info - pointer to jpeghw object
 *
 * \return pointer to allocated jpeghw_dev_struct struct
 *
 **/
struct jpeghw_dev_struct *jpeghw_new_dev(struct jpeghw_common_struct * info)
{
	struct jpeghw_dev_struct *dev = NULL;

	dev = (struct jpeghw_dev_struct *)MEM_MALLOC(sizeof(struct jpeghw_dev_struct));
    if(dev == NULL)
    {
    	return NULL;
    }

    memset(dev, 0, sizeof(struct jpeghw_dev_struct));

    dev->info = info;
    dev->hw_config_flags = info->config_flags;

	return dev;
}

/**
 * \brief Free an existing jpeghw device object
 *
 * \param dev - pointer to jpeghw device object
 *
 **/
void jpeghw_free_dev(struct jpeghw_dev_struct *dev)
{
	JPEGHW_VALIDATE_DEV(dev);

	if (dev->cacheBuffer != NULL)
	{
		// free cache buffer
		dev->cacheBuffer->freeFunc(dev->cacheBuffer);
	}

	MEM_FREE_AND_NULL(dev);
}

/**
 * \brief Generate a new MCU aligned BigBuffer
 *
 * \param dev - pointer to jpeghw device object
 * \param maxLines - number of scanlines to allocate in buffer
 * \param fill_char - byte used to fill the block of memory
 *
 * \return pointer allocated MCU aligned BigBuffer
 *
 **/
struct BigBuffer_s *jpeghw_new_mcu_aligned_buffer(jpeghw_dev_ptr dev, uint32_t maxLines, char fill_char)
{
	JPEGHW_VALIDATE_DEV(dev);

	struct BigBuffer_s* bb = NULL;
	uint32_t buffer_size = maxLines * dev->mcu_bytes_per_row;

	bb = dma_buffer_malloc( 0, buffer_size );
	if (!jpeghw_valid_buffer(bb))
	{
		if (bb)
		{
			bb->freeFunc(bb);
		}

		return NULL;
	}

	void *data = dma_buffer_mmap_forcpu(bb);
	if (data)
	{
		memset(data, fill_char, buffer_size);
		dma_buffer_unmmap_forcpu(bb);
	}

	return bb;
}

/**
 * \brief Consumes a BigBuffer of scanlines
 *
 * \param dev - pointer to jpeghw device object
 * \param scanlines - pointer to BigBuffer containing the scanlines to be consumed
 * \param skip_lines - number of scanlines to skip in the BigBuffer
 * \param total_lines - total number of scanlines in BigBuffer
 *
 * \return number of scanlines consumbed, or -1 if error
 *
 **/
int jpeghw_mcu_cache_consume_scanlines(jpeghw_dev_ptr dev, struct BigBuffer_s *scanlines, uint32_t skip_lines, uint32_t total_lines)
{
	JPEGHW_VALIDATE_DEV(dev);

	uint32_t lines_consumed = 0;

	if (scanlines == NULL)
	{
		return -1;
	}

	if (total_lines < skip_lines)
	{
		return -1;
	}

	void *line_ptr = dma_buffer_mmap_forcpu(scanlines);
	if (line_ptr == NULL)
	{
		return -1;
	}

	/* check to see if cache buffer needs to be flushed */
	if (dev->cacheBuffer && dev->cacheLines == dev->cacheMaxLines)
	{
		jpeghw_mcu_cache_flush_scanlines(dev, false);
	}

	/* make sure a cache buffer is available */
	if (dev->cacheBuffer == NULL)
	{
		uint32_t maxLines = dev->info->image_height;

		if (maxLines > dev->info->mcu_height)
		{
			maxLines = dev->info->mcu_height;
		}

		dev->cacheBuffer = jpeghw_new_mcu_aligned_buffer(dev, maxLines, 0xff);
		if (dev->cacheBuffer)
		{
			dev->cacheLines = 0;
			dev->cacheMaxLines = maxLines;
		}
		else
		{
			// failed to allocate cache buffer
			return -1;
		}
	}

	char *cache_ptr = (char*) dma_buffer_mmap_forcpu(dev->cacheBuffer);
	if (cache_ptr == NULL)
	{
		dma_buffer_unmmap_forcpu(scanlines);
		return -1;
	}

	/* skip past any processed lines */
	line_ptr += (skip_lines * dev->image_bytes_per_row);
	cache_ptr += (dev->cacheLines * dev->mcu_bytes_per_row);

	/* don't consume more lines than what is left in cache buffer */
	uint32_t lines_left = MIN((dev->cacheMaxLines - dev->cacheLines), (total_lines - skip_lines));

	while (lines_consumed < lines_left)
	{
		/* copy scanline to cache */
		memcpy(cache_ptr, line_ptr, dev->image_bytes_per_row);

		/* advance data pointers */
		line_ptr += dev->image_bytes_per_row;
		cache_ptr += dev->mcu_bytes_per_row;

		/* increment lines consumed */
		lines_consumed++;
		dev->cacheLines++;
	}

	dma_buffer_unmmap_forcpu(dev->cacheBuffer);
	dma_buffer_unmmap_forcpu(scanlines);

	return lines_consumed;
}

/**
 * \brief Flush scanlines from cache to jpeghw device
 *
 * \param dev - pointer to jpeghw device object
 * \param eoi - indicates end of scanline input
 *
 * \return true if success, false otherwise
 *
 **/
bool jpeghw_mcu_cache_flush_scanlines(jpeghw_dev_ptr dev, bool eoi)
{
	JPEGHW_VALIDATE_DEV(dev);

	if (eoi)
	{
		/* advance cached lines to max lines */
		dev->cacheLines = dev->cacheMaxLines;
	}

	if (dev->cacheLines < dev->cacheMaxLines)
	{
		DBG_ERR("Bad JPEG State!\n");

		// error
		return false;
	}

	if (dev->cacheBuffer != NULL)
	{
		if (dev->cacheLines > 0)
		{
			jhw_write_buffer((struct jpeghw_compress_struct *)dev->info, dev->cacheBuffer, dev->cacheLines, eoi);
		}
		else
		{
			// removing empty buffer
			dev->cacheBuffer->freeFunc(dev->cacheBuffer);
		}
	}

	dev->cacheBuffer = NULL;
	dev->cacheLines = 0;
	dev->cacheMaxLines = 0;

	return true;
}

/**
 * \brief Cleanup the jpeghw object
 *
 * \param info - pointer to jpeghw object
 *
 **/
void jpeghw_finish(struct jpeghw_common_struct * info)
{
	JPEGHW_VALIDATE_COMMON_INFO(info);
	info->last_error = e_JPEGHW_SUCCESS;

	 /* reset state for possible reuse of object */
	if (info->type == e_JPEGHW_OBJ_COMPRESS)
	{
		info->global_jpeg_state = e_JPEGHW_CSTATE_START;
	} else if (info->type == e_JPEGHW_OBJ_DECOMPRESS)
	{
		info->global_jpeg_state = e_JPEGHW_DSTATE_START;
	} else
	{
		info->global_jpeg_state = e_JPEGHW_STATE_UNKNOWN;
	}

}
