/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2015, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/



#include "jpeghw_lib.h"
#include "jpeghw_dbg.h"


/**
 * \brief Initialize a JPEG decompression object
 *
 * The JPEG decompression object is a ‘struct’ that can be local,
 * static or allocated via a malloc().  If the ‘struct’ is local,
 * then the calling routine which declares the local MUST execute
 * the entire JPEG decompression sequence within the routine.
 *
 * \param dinfo - pointer to JPEG decompression object
 * \param flags - JPEGHW_CONFIG_FLAG_xxx flags
 *
 * \return e_JPEGHW_SUCCESS if success, else a jpeghw_error_type_t error
 *
 **/
jpeghw_error_type_t jpeghw_create_decompress(struct jpeghw_decompress_struct * dinfo, uint32_t flags)
{
	DBG_INFO("Start jpeghw_create_decompress() call\n");

	if (dinfo == NULL)
	{
		DBG_ERR("dinfo is NULL!\n");
		return e_JPEGHW_ERR_INVALID_PARAMETERS;
	}

	// initialize structure
	memset(dinfo, 0, sizeof(struct jpeghw_decompress_struct));

	/* set struct type */
	dinfo->common.type = e_JPEGHW_OBJ_DECOMPRESS;
	dinfo->common.config_flags = flags;

	dinfo->common.jpeghw_context = (void *)jpeghw_new_dev(&dinfo->common);

	if (dinfo->common.jpeghw_context == NULL)
	{
		DBG_ERR("jpeghw_context is NULL!\n");
		return jpeghw_error_exit((struct jpeghw_common_struct *)dinfo, e_JPEGHW_ERR_OUT_OF_MEMORY);
	}

	/* attempt to acquire compression hardware */
	jpeghw_error_type_t ret = jhw_open_decompress(dinfo);
	if (ret != e_JPEGHW_SUCCESS)
	{
		DBG_ERR("jhw_open_decompress() call failed!\n");
		return jpeghw_error((struct jpeghw_common_struct *)dinfo, ret);
	}

	/* good to go */
	dinfo->common.global_jpeg_state = e_JPEGHW_DSTATE_START;

	DBG_INFO("Exit jpeghw_create_decompress() call\n");

	return jpeghw_error((struct jpeghw_common_struct *)dinfo, e_JPEGHW_SUCCESS);
}

/**
 * \brief Release the JPEG decompression object
 *
 * When done with a decompression object, frees all subsidiary memory.
 *
 * \param dinfo - pointer to JPEG decompression object
 * \param flags - JPEGHW_CONFIG_FLAG_xxx flags
 *
 **/
void jpeghw_destroy_decompress(struct jpeghw_decompress_struct * dinfo)
{
	DBG_INFO("Start jpeghw_destroy_decompress() call\n");

	JPEGHW_VALIDATE_INFO(dinfo, e_JPEGHW_OBJ_DECOMPRESS);
	dinfo->common.last_error = e_JPEGHW_SUCCESS;

	struct jpeghw_dev_struct *dev = (struct jpeghw_dev_struct *)dinfo->common.jpeghw_context;
	JPEGHW_VALIDATE_DEV(dev);

	/* release compression hardware */
	jhw_close_decompress(dinfo);

	dinfo->common.type = 0;
	dinfo->common.global_jpeg_state = 0;
	dinfo->common.last_error = e_JPEGHW_SUCCESS;

	jpeghw_free_dev(dev);
	dinfo->common.jpeghw_context = NULL;

	DBG_INFO("Exit jpeghw_destroy_decompress() call\n");
}

/**4/abcfip
 * \brief Abort the current decompression cycle
 *
 * \param dinfo - pointer to JPEG decompression object
 *
 * \return e_JPEGHW_SUCCESS if success, else a jpeghw_error_type_t error
 *
 **/
jpeghw_error_type_t jpeghw_abort_decompress(struct jpeghw_decompress_struct * dinfo)
{
	DBG_INFO("Start jpeghw_abort_decompress() call\n");

	JPEGHW_VALIDATE_INFO(dinfo, e_JPEGHW_OBJ_DECOMPRESS);
	dinfo->common.last_error = e_JPEGHW_SUCCESS;

	/* check to make sure we have not already aborted */
	if (dinfo->common.global_jpeg_state == e_JPEGHW_STATE_ABORTED)
	{
		DBG_ERR("Bad JPEG State!\n");
		return jpeghw_error((struct jpeghw_common_struct *)dinfo, e_JPEGHW_ERR_BAD_STATE);
	}

	if (dinfo->dmgr && dinfo->dmgr->term)
	{
		dinfo->dmgr->term(dinfo);
	}

	jhw_abort_decompress(dinfo);

	dinfo->common.global_jpeg_state = e_JPEGHW_STATE_ABORTED;

	DBG_INFO("Exit jpeghw_abort_decompress() call\n");

	return jpeghw_error((struct jpeghw_common_struct *)dinfo, e_JPEGHW_SUCCESS);
}

/**
 * \brief Begin decompression cycle
 *
 * Begin the JPEG decompression cycle which initializes the internal state, allocates
 * working storage, and emits the first few bytes of the JPEG datastream header.
 *
 * \param dinfo - pointer to JPEG decompression object
 *
 * \return e_JPEGHW_SUCCESS if success, else a jpeghw_error_type_t error
 *
 **/
jpeghw_error_type_t jpeghw_start_decompress(struct jpeghw_decompress_struct * dinfo)
{
	DBG_INFO("Start jpeghw_start_decompress() call\n");

	JPEGHW_VALIDATE_INFO(dinfo, e_JPEGHW_OBJ_DECOMPRESS);
	dinfo->common.last_error = e_JPEGHW_SUCCESS;

	struct jpeghw_dev_struct *dev = (struct jpeghw_dev_struct *)dinfo->common.jpeghw_context;
	JPEGHW_VALIDATE_DEV(dev);

	/* check for valid error and decompression managers */
	if (dinfo->common.err == NULL || dinfo->dmgr == NULL)
	{
		DBG_ERR("common.err or dmgr is NULL!\n");
		return jpeghw_error((struct jpeghw_common_struct *)dinfo, e_JPEGHW_ERR_INVALID_PARAMETERS);
	}

	/* check for valid image width, height, and color components */
	if (dinfo->common.image_width == 0 || dinfo->common.image_height == 0 ||
			dinfo->common.bytes_per_pixel < 1 || dinfo->common.bytes_per_pixel > 4)
	{
		DBG_ERR("Invalid image width, height or color components!\n");
		return jpeghw_error((struct jpeghw_common_struct *)dinfo, e_JPEGHW_ERR_INVALID_PARAMETERS);
	}

	/* check for valid mcu width and height */
	if ( (dinfo->common.mcu_width != 8 && dinfo->common.mcu_width != 16) ||
		  (dinfo->common.mcu_height != 8 && dinfo->common.mcu_height != 16) )
	{
		DBG_ERR("Invalid MCU width or height!\n");
		return jpeghw_error((struct jpeghw_common_struct *)dinfo, e_JPEGHW_ERR_INVALID_PARAMETERS);
	}

	/* check for valid jpeg state */
	if (dinfo->common.global_jpeg_state != e_JPEGHW_DSTATE_START)
	{
		DBG_ERR("Bad JPEG State!\n");
		return jpeghw_error((struct jpeghw_common_struct *)dinfo, e_JPEGHW_ERR_BAD_STATE);
	}

	/* initialize dev structure */
	dev->image_bytes_per_row = dinfo->common.bytes_per_pixel * dinfo->common.image_width;

	/* calculate mcu aligned width */
	dev->mcu_aligned_width = dinfo->common.image_width;
	if (dinfo->common.mcu_width>0)
	{
		uint32_t mcu_adj = dinfo->common.image_width % dinfo->common.mcu_width;

		if ( mcu_adj > 0 )
		{
			dev->mcu_aligned_width += (dinfo->common.mcu_width - mcu_adj);
		}
	}

	/* calculate mcu aligned height */
	dev->mcu_aligned_height = dinfo->common.image_height;
	if (dinfo->common.mcu_height>0)
	{
		uint32_t mcu_adj = dinfo->common.image_height % dinfo->common.mcu_height;

		if ( mcu_adj > 0 )
		{
			dev->mcu_aligned_height += (dinfo->common.mcu_height - mcu_adj);
		}
	}

	dev->mcu_bytes_per_row = dinfo->common.bytes_per_pixel * dev->mcu_aligned_width;

	/* call decompression manager initialization */
	if (dinfo->dmgr && dinfo->dmgr->init)
	{
		dinfo->dmgr->init(dinfo);
	}

	jpeghw_error_type_t ret = jhw_start_decompress(dinfo);
	if (ret != e_JPEGHW_SUCCESS)
	{
		DBG_ERR("jpeghw_start_decompress() call failed!\n");

		if (dinfo->dmgr && dinfo->dmgr->term)
		{
			dinfo->dmgr->term(dinfo);
		}

		return jpeghw_error((struct jpeghw_common_struct *)dinfo, ret);
	}

	dinfo->out_bytes = 0;
	dinfo->out_scanlines = 0;
	dinfo->common.global_jpeg_state = e_JPEGHW_DSTATE_READY;

	dev->hw_config_flags |= jhwd_get_config_flags();

	DBG_INFO("Exit jpeghw_start_decompress() call\n");

	return jpeghw_error((struct jpeghw_common_struct *)dinfo, e_JPEGHW_SUCCESS);
}

/**
 * \brief Read all the decompressed image data
 *
 * Read all the decompressed image data by providing a buffer of
 * of scanlines to be read and the return value is the number of bytes
 * actually read.  Image data is returned in top-to-bottom scanline order.
 *
 * \param dinfo - pointer to JPEG decompression object
 * \param scanlines - pointer to BigBuffer containing the image data
 * \param done      - indicates end of image data
 *
 * \return total bytes of image data read
 *
 **/
uint32_t jpeghw_read_scanlines(struct jpeghw_decompress_struct * dinfo, struct BigBuffer_s *scanlines, bool *done)
{
	jpeghw_error_type_t err = e_JPEGHW_SUCCESS;
	uint32_t bytes_read = 0;
	bool last_buffer = false;
	DBG_INFO("Start jpeghw_read_scanlines() call\n");

	JPEGHW_VALIDATE_INFO(dinfo, e_JPEGHW_OBJ_DECOMPRESS);
	dinfo->common.last_error = e_JPEGHW_SUCCESS;

	struct jpeghw_dev_struct *dev = (struct jpeghw_dev_struct *)dinfo->common.jpeghw_context;
	JPEGHW_VALIDATE_DEV(dev);

	if (scanlines == NULL)
	{
		DBG_ERR("scanlines are NULL!\n");
		jpeghw_error((struct jpeghw_common_struct *)dinfo, e_JPEGHW_ERR_INVALID_PARAMETERS);

		return 0;
	}

	if (done)
	{
		*done = false;
	}

	/* check for valid jpeg state */
	if (dinfo->common.global_jpeg_state != e_JPEGHW_DSTATE_READY)
	{
		DBG_ERR("Bad JPEG State!\n");
		jpeghw_error((struct jpeghw_common_struct *)dinfo, e_JPEGHW_ERR_BAD_STATE);

		if (done)
		{
			*done = true;
		}

		return 0;
	}

	struct BigBuffer_s *mcu_scanlines = NULL;
	uint32_t height = scanlines->datalen / dev->image_bytes_per_row;
	uint32_t buflen = height * dev->mcu_bytes_per_row;

	if (buflen != scanlines->datalen)
	{

		mcu_scanlines = dma_buffer_malloc(0, buflen);
	}

	if (mcu_scanlines)
	{
		uint32_t mcu_bytes_read = 0;

		err = jhw_read_buffer(dinfo, mcu_scanlines, &mcu_bytes_read, &last_buffer);
		if (err != e_JPEGHW_SUCCESS)
		{
			jpeghw_error((struct jpeghw_common_struct *)dinfo, err);

			if (done)
			{
				*done = true;
			}

			BigBuffer_Free(mcu_scanlines);

			return 0;
		}

		void *mcu_data_ptr = dma_buffer_mmap_forcpu(mcu_scanlines);
		if (mcu_data_ptr)
		{
			void *image_data_ptr = dma_buffer_mmap_forcpu(scanlines);
			if (image_data_ptr)
			{
				uint32_t i;
				for (i=0; i<height; i++)
				{
					uint32_t offset = i * dev->image_bytes_per_row;
					uint32_t offset_mcu = i * dev->mcu_bytes_per_row;

					memcpy(image_data_ptr+offset, mcu_data_ptr+offset_mcu, dev->image_bytes_per_row);
					bytes_read += dev->image_bytes_per_row;
				}

				dma_buffer_unmmap_forcpu(scanlines);
			}
			else
			{
				DBG_ERR("Unable to map scanlines!\n");
				jpeghw_error((struct jpeghw_common_struct *)dinfo, e_JPEGHW_ERR_OUT_OF_MEMORY);

				if (done)
				{
					*done = true;
				}
			}

			dma_buffer_unmmap_forcpu(mcu_scanlines);
		}
		else
		{
			DBG_ERR("Unable to map mcu_scanlines!\n");
			jpeghw_error((struct jpeghw_common_struct *)dinfo, e_JPEGHW_ERR_OUT_OF_MEMORY);

			if (done)
			{
				*done = true;
			}
		}

		BigBuffer_Free(mcu_scanlines);
	}
	else
	{
		err = jhw_read_buffer(dinfo, scanlines, &bytes_read, &last_buffer);
		if (err != e_JPEGHW_SUCCESS)
		{
			jpeghw_error((struct jpeghw_common_struct *)dinfo, err);

			if (done)
			{
				*done = true;
			}

			return 0;
		}
	}

	dinfo->out_bytes += bytes_read;
	dinfo->out_scanlines += height;

	/* call progress monitor hook if present */
	if (dinfo->common.progress_monitor != NULL)
	{
		(*dinfo->common.progress_monitor) ((struct jpeghw_common_struct *) dinfo);
	}

	if (done)
	{
		*done = last_buffer;
	}

	DBG_INFO("Exit jpeghw_read_scanlines() call\n");

	return bytes_read;
}

/**
 * \brief Complete the decompression cycle
 *
 * After all the image data has been read, release any
 * working memory associated with the JPEG decompression object
 *
 * \param dinfo - pointer to JPEG decompression object
 *
 * \return e_JPEGHW_SUCCESS if success, else a jpeghw_error_type_t error
 *
 **/
jpeghw_error_type_t jpeghw_finish_decompress(struct jpeghw_decompress_struct * dinfo)
{
	DBG_INFO("Start jpeghw_finish_decompress() call\n");

	JPEGHW_VALIDATE_INFO(dinfo, e_JPEGHW_OBJ_DECOMPRESS);

	if (dinfo->common.global_jpeg_state != e_JPEGHW_DSTATE_READY)
	{
		// attempt to abort the decompress
		jpeghw_abort_decompress(dinfo);

		DBG_ERR("Bad JPEG State!\n");
		return jpeghw_error((struct jpeghw_common_struct *)dinfo, e_JPEGHW_ERR_BAD_STATE);
	}

	jpeghw_error_type_t ret = e_JPEGHW_SUCCESS;

	if (dinfo->common.last_error != e_JPEGHW_SUCCESS)
	{
		ret = jpeghw_abort_decompress(dinfo);
		if (ret != e_JPEGHW_SUCCESS)
		{
			DBG_ERR("jpeghw_abort_decompress() call failed!\n");
		}
	}
	else
	{
		ret = jhw_finish_decompress(dinfo);
		if (ret != e_JPEGHW_SUCCESS)
		{
			DBG_ERR("jhw_finish_decompress() call failed!\n");
		}
	}

	if (dinfo->dmgr && dinfo->dmgr->term)
	{
		dinfo->dmgr->term(dinfo);
	}

	jpeghw_finish((struct jpeghw_common_struct *) dinfo);

	DBG_INFO("Exit jpeghw_finish_decompress() call\n");

	return jpeghw_error((struct jpeghw_common_struct *)dinfo, ret);
}
