/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2015, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/



#include "jpeghw_lib.h"
#include "jpeghw_dbg.h"


/**
 * \brief Initialize a JPEG compression object
 *
 * The JPEG compression object is a ‘struct’ that can be local,
 * static or allocated via a malloc().  If the ‘struct’ is local,
 * then the calling method which declares the local MUST execute
 * the entire JPEG compression sequence within the method.
 *
 * \param cinfo - pointer to JPEG compression object
 * \param flags - JPEGHW_CONFIG_FLAG_xxx flags
 *
 * \return e_JPEGHW_SUCCESS if success, else a jpeghw_error_type_t error
 *
 **/
jpeghw_error_type_t jpeghw_create_compress(struct jpeghw_compress_struct * cinfo, uint32_t flags)
{
	DBG_INFO("Start jpeghw_create_compress() call\n");

	if (cinfo == NULL)
	{
		DBG_ERR("cinfo is NULL!\n");
		return e_JPEGHW_ERR_INVALID_PARAMETERS;
	}

	// initialize structure
	memset(cinfo, 0, sizeof(struct jpeghw_compress_struct));

	/* set struct type */
	cinfo->common.type = e_JPEGHW_OBJ_COMPRESS;
	cinfo->common.config_flags = flags;

	cinfo->common.jpeghw_context = (void *)jpeghw_new_dev(&cinfo->common);

	if (cinfo->common.jpeghw_context == NULL)
	{
		DBG_ERR("jpeghw_context is NULL!\n");
		return jpeghw_error_exit((struct jpeghw_common_struct *)cinfo, e_JPEGHW_ERR_OUT_OF_MEMORY);
	}

	/* attempt to acquire compression hardware */
	jpeghw_error_type_t ret = jhw_open_compress(cinfo);
	if (ret != e_JPEGHW_SUCCESS)
	{
		DBG_ERR("jhw_open_compress() call failed!\n");
		return jpeghw_error((struct jpeghw_common_struct *)cinfo, ret);
	}

	/* good to go */
	cinfo->common.global_jpeg_state = e_JPEGHW_CSTATE_START;

	DBG_INFO("Exit jpeghw_create_compress() call\n");

	return jpeghw_error((struct jpeghw_common_struct *)cinfo, e_JPEGHW_SUCCESS);
}

/**
 * \brief Release the JPEG compression object
 *
 * When done with a compression object, frees all subsidiary memory.
 *
 * \param cinfo - pointer to JPEG compression object
 * \param flags - JPEGHW_CONFIG_FLAG_xxx flags
 *
 **/
void jpeghw_destroy_compress(struct jpeghw_compress_struct * cinfo)
{
	DBG_INFO("Start jpeghw_destroy_compress() call\n");

	JPEGHW_VALIDATE_INFO(cinfo, e_JPEGHW_OBJ_COMPRESS);
	cinfo->common.last_error = e_JPEGHW_SUCCESS;

	struct jpeghw_dev_struct *dev = (struct jpeghw_dev_struct *)cinfo->common.jpeghw_context;
	JPEGHW_VALIDATE_DEV(dev);

	/* release compression hardware */
	jhw_close_compress(cinfo);

	cinfo->common.type = 0;
	cinfo->common.global_jpeg_state = 0;
	cinfo->common.last_error = e_JPEGHW_SUCCESS;

	jpeghw_free_dev(dev);
	cinfo->common.jpeghw_context = NULL;

	DBG_INFO("Exit jpeghw_destroy_compress() call\n");
}

/**
 * \brief Abort the current compression cycle
 *
 * \param cinfo - pointer to JPEG compression object
 *
 * \return e_JPEGHW_SUCCESS if success, else a jpeghw_error_type_t error
 *
 **/
jpeghw_error_type_t jpeghw_abort_compress(struct jpeghw_compress_struct * cinfo)
{
	DBG_INFO("Start jpeghw_abort_compress() call\n");

	JPEGHW_VALIDATE_INFO(cinfo, e_JPEGHW_OBJ_COMPRESS);
	cinfo->common.last_error = e_JPEGHW_SUCCESS;

	/* check to make sure we have not already aborted */
	if (cinfo->common.global_jpeg_state == e_JPEGHW_STATE_ABORTED)
	{
		DBG_ERR("Bad JPEG State!\n");
		return jpeghw_error((struct jpeghw_common_struct *)cinfo, e_JPEGHW_ERR_BAD_STATE);
	}

	if (cinfo->cmgr && cinfo->cmgr->term)
	{
		cinfo->cmgr->term(cinfo);
	}

	jhw_abort_compress(cinfo);

	cinfo->common.global_jpeg_state = e_JPEGHW_STATE_ABORTED;

	DBG_INFO("Exit jpeghw_abort_compress() call\n");

	return jpeghw_error((struct jpeghw_common_struct *)cinfo, e_JPEGHW_SUCCESS);
}

/**
 * \brief Begin compression cycle
 *
 * Begin the JPEG compression cycle which initializes the internal state, allocates
 * working storage, and emits the first few bytes of the JPEG datastream header.
 *
 * \param cinfo - pointer to JPEG compression object
 *
 * \return e_JPEGHW_SUCCESS if success, else a jpeghw_error_type_t error
 *
 **/
jpeghw_error_type_t jpeghw_start_compress(struct jpeghw_compress_struct * cinfo)
{
	DBG_INFO("Start jpeghw_start_compress() call\n");

	JPEGHW_VALIDATE_INFO(cinfo, e_JPEGHW_OBJ_COMPRESS);
	cinfo->common.last_error = e_JPEGHW_SUCCESS;

	struct jpeghw_dev_struct *dev = (struct jpeghw_dev_struct *)cinfo->common.jpeghw_context;
	JPEGHW_VALIDATE_DEV(dev);

	/* check for valid error and compression managers */
	if (cinfo->common.err == NULL || cinfo->cmgr == NULL)
	{
		DBG_ERR("common.err or cmgr is NULL!\n");
		return jpeghw_error((struct jpeghw_common_struct *)cinfo, e_JPEGHW_ERR_INVALID_PARAMETERS);
	}

	/* check for valid image width, height, and color components */
	if (cinfo->common.image_width == 0 || cinfo->common.image_height == 0 ||
			cinfo->common.bytes_per_pixel < 1 || cinfo->common.bytes_per_pixel > 4)
	{
		DBG_ERR("Invalid image width, height or color components!\n");
		return jpeghw_error((struct jpeghw_common_struct *)cinfo, e_JPEGHW_ERR_INVALID_PARAMETERS);
	}

	/* check for valid mcu width and height */
	if ( (cinfo->common.mcu_width != 8 && cinfo->common.mcu_width != 16) ||
		  (cinfo->common.mcu_height != 8 && cinfo->common.mcu_height != 16) )
	{
		DBG_ERR("Invalid MCU width or height!\n");
		return jpeghw_error((struct jpeghw_common_struct *)cinfo, e_JPEGHW_ERR_INVALID_PARAMETERS);
	}

	/* check for valid jpeg state */
	if (cinfo->common.global_jpeg_state != e_JPEGHW_CSTATE_START)
	{
		DBG_ERR("Bad JPEG State!\n");
		return jpeghw_error((struct jpeghw_common_struct *)cinfo, e_JPEGHW_ERR_BAD_STATE);
	}

	/* initialize dev structure */
	dev->image_bytes_per_row = cinfo->common.bytes_per_pixel * cinfo->common.image_width;

	/* calculate mcu aligned width */
	dev->mcu_aligned_width = cinfo->common.image_width;
	if (cinfo->common.mcu_width>0)
	{
		uint32_t mcu_adj = cinfo->common.image_width % cinfo->common.mcu_width;

		if ( mcu_adj > 0 )
		{
			dev->mcu_aligned_width += (cinfo->common.mcu_width - mcu_adj);
		}
	}

	/* calculate mcu aligned height */
	dev->mcu_aligned_height = cinfo->common.image_height;
	if (cinfo->common.mcu_height>0)
	{
		uint32_t mcu_adj = cinfo->common.image_height % cinfo->common.mcu_height;

		if ( mcu_adj > 0 )
		{
			dev->mcu_aligned_height += (cinfo->common.mcu_height - mcu_adj);
		}
	}

	dev->mcu_bytes_per_row = cinfo->common.bytes_per_pixel * dev->mcu_aligned_width;

	/* call compression manager initialization */
	if (cinfo->cmgr && cinfo->cmgr->init)
	{
		cinfo->cmgr->init(cinfo);
	}

	jpeghw_error_type_t ret = jhw_start_compress(cinfo);
	if (ret != e_JPEGHW_SUCCESS)
	{
		DBG_ERR("jhw_start_compress() call failed!\n");

		if (cinfo->cmgr && cinfo->cmgr->term)
		{
			cinfo->cmgr->term(cinfo);
		}

		return jpeghw_error((struct jpeghw_common_struct *)cinfo, ret);
	}

	cinfo->in_scanlines = 0;
	cinfo->in_bytes = 0;
	cinfo->common.global_jpeg_state = e_JPEGHW_CSTATE_SCANNING;

	dev->hw_config_flags |= jhwc_get_config_flags();

	DBG_INFO("Exit jpeghw_start_compress() call\n");

	return jpeghw_error((struct jpeghw_common_struct *)cinfo, e_JPEGHW_SUCCESS);
}

/**
 * \brief Write all the compressed image data
 *
 * Writes all the required image data by passing one or
 * more scanlines at a time, up to the total image height.
 *
 * \param cinfo     - pointer to JPEG compression object
 * \param scanlines - pointer to BigBuffer containing the scanlines to compress
 * \param num_lines - number of scanlines in the BigBuffer
 * \param eoi       - indicates end of scanline input
 *
 * \return number of scanlines processed
 *
 **/
uint32_t jpeghw_write_scanlines(struct jpeghw_compress_struct * cinfo, struct BigBuffer_s* scanlines, uint32_t num_lines, bool eoi)
{
	DBG_INFO("Start jpeghw_write_scanlines() call\n");

	JPEGHW_VALIDATE_INFO(cinfo, e_JPEGHW_OBJ_COMPRESS);
	cinfo->common.last_error = e_JPEGHW_SUCCESS;

	uint32_t line_ctr = 0;
	struct jpeghw_dev_struct *dev = (struct jpeghw_dev_struct *)cinfo->common.jpeghw_context;
	JPEGHW_VALIDATE_DEV(dev);

	/* check for valid jpeg state */
	if (cinfo->common.global_jpeg_state != e_JPEGHW_CSTATE_SCANNING)
	{
		DBG_ERR("Bad JPEG State!\n");
		jpeghw_error((struct jpeghw_common_struct *)cinfo, e_JPEGHW_ERR_BAD_STATE);

		return 0;
	}

	if (cinfo->in_scanlines >= cinfo->common.image_height)
	{
		DBG_ERR("Data Overrun!\n");
		jpeghw_error((struct jpeghw_common_struct *)cinfo, e_JPEGHW_ERR_DATA_OVERRUN);

		return 0;
	}

	/* ignore any extra scanlines at bottom of image. */
	num_lines = MIN(num_lines, (cinfo->common.image_height - cinfo->in_scanlines));

	if (num_lines > 0)
	{
		// determine if mcu alignment is required
		if (    (dev->cacheBuffer) ||
				(dev->hw_config_flags & JPEGHW_CONFIG_FLAG_MCU_WIDTH_ALIGNMENT &&
				(cinfo->common.image_width % cinfo->common.mcu_width  != 0) ) ||
				(dev->hw_config_flags & JPEGHW_CONFIG_FLAG_MCU_HEIGHT_ALIGNMENT &&
				(num_lines % cinfo->common.mcu_height  != 0) )
			)
		{
			line_ctr = 0;

			do
			{
				int lines;

				lines = jpeghw_mcu_cache_consume_scanlines(dev, scanlines, line_ctr, num_lines);
				if (lines == -1)
				{
					// error
					break;
				}
				else
				{
					line_ctr += lines;
				}

			} while (line_ctr < num_lines);

			BigBuffer_Free(scanlines);
		}
		// assume the HW can take care of any alignment issues
		else
		{
			line_ctr = jhw_write_buffer(cinfo, scanlines, num_lines, eoi);
			if (line_ctr == 0)
			{
				DBG_WARN("jhw_write_buffer() call failed!\n");

				if (cinfo->common.global_jpeg_state == e_JPEGHW_STATE_ERROR)
				{
					DBG_ERR("e_JPEGHW_STATE_ERROR encountered!\n");
				}
			}
			else if (!eoi && line_ctr > num_lines)
			{
				DBG_WARN("Returned more lines than requested!!!\n");
			}
		}

		cinfo->in_scanlines += line_ctr;
		cinfo->in_bytes += (line_ctr * dev->image_bytes_per_row);
	}
	else
	{
		DBG_WARN("No Scanlines\n");
	}

	/* call progress monitor hook if present */
	if (cinfo->common.progress_monitor != NULL)
	{
		(*cinfo->common.progress_monitor) ((struct jpeghw_common_struct *) cinfo);
	}

	DBG_INFO("Exit jpeghw_write_scanlines() call\n");

	return line_ctr;
}

/**
 * \brief Complete the compression cycle
 *
 * After all the image data has been written, complete
 * the JPEG compression cycle by ensuring that all the
 * buffered data has been written to the data destination
 * and then release any working memory associated with
 * the JPEG compression object.
 *
 * \param cinfo - pointer to JPEG compression object
 *
 * \return e_JPEGHW_SUCCESS if success, else a jpeghw_error_type_t error
 *
 **/
jpeghw_error_type_t jpeghw_finish_compress(struct jpeghw_compress_struct * cinfo)
{
	DBG_INFO("Start jpeghw_finish_compress() call\n");

	JPEGHW_VALIDATE_INFO(cinfo, e_JPEGHW_OBJ_COMPRESS);

	struct jpeghw_dev_struct *dev = (struct jpeghw_dev_struct *)cinfo->common.jpeghw_context;
	JPEGHW_VALIDATE_DEV(dev);

	if (cinfo->common.global_jpeg_state != e_JPEGHW_CSTATE_SCANNING)
	{
		// attempt to abort the compress
		jpeghw_abort_compress(cinfo);

		DBG_ERR("Bad JPEG State!\n");
		return jpeghw_error((struct jpeghw_common_struct *)cinfo, e_JPEGHW_ERR_BAD_STATE);
	}

	jpeghw_error_type_t ret = e_JPEGHW_SUCCESS;

	if (cinfo->common.last_error != e_JPEGHW_SUCCESS)
	{
		ret = jpeghw_abort_compress(cinfo);
	}
	else
	{
		// if cache buffer was used, flush it
		if (dev->cacheBuffer != NULL)
		{
			jpeghw_mcu_cache_flush_scanlines(dev, true);
		}

		ret = jhw_finish_compress(cinfo);
		if (ret != e_JPEGHW_SUCCESS)
		{
			DBG_ERR("jhw_finish_compress() call failed!\n");
		}
	}

	if (cinfo->cmgr && cinfo->cmgr->term)
	{
		cinfo->cmgr->term(cinfo);
	}

	jpeghw_finish((struct jpeghw_common_struct *) cinfo);

	DBG_INFO("Exit jpeghw_finish_compress() call\n");

	return jpeghw_error((struct jpeghw_common_struct *)cinfo, ret);
}
