/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2015-2016, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/



#include <stdio.h>
#include <sys/types.h>
#include <dirent.h>

#include "jpeghw_api.h" // JPEGHW library
#include "jpeg_strip_adaptor.h"
#include "jpeg_page_adaptor.h"

#include "logger.h"
#include "dma_buffer.h"
#include "uio_lib_api.h"

#include "jpeg_common.h"


//#define MANUAL_JPEG_HEADER_PARSE 1

#define DBG_PRFX "jpeg_decompress: "
#define LOGGER_MODULE_MASK DEBUG_LOGGER_MODULE_DEVICES | LOGGER_SUBMODULE_BIT(20)

struct file_blob_s {
	char filename[256];
	char page_name[256];
};

struct jpeg_jpeghw_file_page_adaptor {
	jpeg_page_adaptor_BaseClassMembers
	;
	//< put this macro in your subclass
	/// jpeghw
	struct jpeghw_decompress_struct decompress_info;   //< jpeghw compress

	struct jpeghw_decompression_mgr jdmgr;
	struct jpeghw_error_mgr jerr;
	/// jpeghw end

	JPEG_header_struct jpeginfo;

	// add my local page variables here.
	FILE* src_fp;
	char dst_filename[256]; /**< generated filename */
	FILE* dst_fp; /**< downstream dst stream handle */
	pthread_t output_thd_id; /**< output image data thread */

	bool error;
	bool retry;
};

struct jpeg_jpeghw_file_strip_adaptor {
	jpeg_strip_adaptor_BaseClassMembers
	;  // put this macro in your subclass

	char *filename; /** src filename */
	struct BigBuffer_s* bb; //< input data as dma_buffer.

};

const char* output_path = ".";

// externs
extern void logger_init();


#define JPEG_MEM_DST_MGR_BUFFER_SIZE (16 * 1024)   ///< might want to have this be configurable.
void dmgr_init(struct jpeghw_decompress_struct * dinfo) {
	struct jpeg_jpeghw_file_page_adaptor *jpa =
			(struct jpeg_jpeghw_file_page_adaptor*) dinfo->common.client_context;

	DBG_ASSERT(jpa);

	if (jpa->src_fp)
	{
		// reset to beginning of file
		fseek(jpa->src_fp, 0, SEEK_SET);
	}
}

void dmgr_term(struct jpeghw_decompress_struct * dinfo) {
	struct jpeg_jpeghw_file_page_adaptor *jpa =
			(struct jpeg_jpeghw_file_page_adaptor*) dinfo->common.client_context;

	DBG_ASSERT(jpa);
}

struct BigBuffer_s* dmgr_get_input_buffer(struct jpeghw_decompress_struct * dinfo, uint32_t *bytes_in_buffer)
{
	DBG_ASSERT(dinfo);
	DBG_ASSERT(bytes_in_buffer);

	struct jpeg_jpeghw_file_page_adaptor *jpa =
			(struct jpeg_jpeghw_file_page_adaptor*) dinfo->common.client_context;

	DBG_ASSERT(jpa);

	if (jpa->src_fp == NULL)
	{
		return NULL;
	}

	*bytes_in_buffer = 0;
	struct BigBuffer_s *input_buffer = dma_buffer_malloc(0, JPEG_MEM_DST_MGR_BUFFER_SIZE);
	if (input_buffer)
	{
		char *read_ptr = dma_buffer_mmap_forcpu(input_buffer);
		if (read_ptr)
		{
			*bytes_in_buffer = fread(read_ptr, 1, JPEG_MEM_DST_MGR_BUFFER_SIZE, jpa->src_fp);
		}

		dma_buffer_unmmap_forcpu(input_buffer);
	}

	return input_buffer;
}


#ifdef MANUAL_JPEG_HEADER_PARSE
jpeghw_error_type_t dmgr_get_jpeg_info(struct jpeghw_decompress_struct * dinfo, struct jpeghw_jpeg_header_struct *jinfo) {
	struct jpeg_jpeghw_file_page_adaptor *jpa =
			(struct jpeg_jpeghw_file_page_adaptor*) dinfo->common.client_context;

	DBG_ASSERT(jpa);

	memcpy(jinfo, &jpa->jpeginfo, sizeof(jpa->jpeginfo));
}
#endif


bool my_read_scanline_timeout(
		struct jpeghw_common_struct *info, uint32_t time_in_seconds) {
	bool ret = false;

	DBG_ASSERT(info);

	struct jpeg_jpeghw_file_page_adaptor *jpa =
			(struct jpeg_jpeghw_file_page_adaptor*) info->client_context;

	DBG_ASSERT(jpa);

	if (time_in_seconds > 20)
	{
		DBG_PRINTF_ERR("%s() Decompression read scanline timed out!\n", __func__);
		ret = true;
	}

	return ret;
}

void my_error_exit(struct jpeghw_common_struct * cinfo) {
	printf(">>>>>jpeg internal exit");
	// do a longjmp here
	exit(-1);
}

void jpeg_jpeghw_file_attach_page(struct jpeg_page_adaptor *base_page, void *blob) {
	struct jpeg_jpeghw_file_page_adaptor* page =
			(struct jpeg_jpeghw_file_page_adaptor *) base_page;
	int fpos = 0;

	if (page == NULL)
	{
		DBG_PRINTF_ERR("%s() Invalid page parameter!\n", __func__);
		return;
	}

	if (blob == NULL)
	{
		DBG_PRINTF_ERR("%s() Invalid file_blob variable!\n", __func__);
		return;
	}

	struct file_blob_s *file_blob = (struct file_blob_s *) blob;

	DBG_PRINTF_INFO("%s() page %p file %p filename '%s'\n", __func__, page,
			file_blob, file_blob->filename);

	page->src_fp = fopen(file_blob->filename, "r");
	if (page->src_fp)
	{ // good file (stream)
		bool valid_stream = false;

		fpos = parse_jpeg_header(page->src_fp, &page->jpeginfo, false);
		if (fpos != FAIL)
		{
			page->decompress_info.common.image_width = page->jpeginfo.image_width;
			page->decompress_info.common.image_height = page->jpeginfo.image_height;
			page->decompress_info.common.bytes_per_pixel = page->jpeginfo.bytes_per_pixel;
			page->decompress_info.common.mcu_width = page->jpeginfo.mcu_width;
			page->decompress_info.common.mcu_height = page->jpeginfo.mcu_height;

			valid_stream = true;
		}

		jpeghw_set_defaults(&page->decompress_info.common);

		if (valid_stream)
		{
            const char* fmt = "%s/%s.%d.ppm";            
            pid_t pid = getpid();

			snprintf(page->dst_filename, sizeof(page->dst_filename),
					fmt, output_path, file_blob->page_name, pid);

			page->dst_fp = fopen(page->dst_filename, "w"); // create/truncate dst
			DBG_ASSERT(page->dst_fp);

			jpeghw_start_decompress(&page->decompress_info);
		}
		else
		{
			DBG_PRINTF_ERR("%s() '%s' NOT a valid stream format!\n", __func__, file_blob->filename);
		}
	}
}

void jpeg_jpeghw_file_send_page(struct jpeg_page_adaptor *base_page, void *blob) {
	DBG_PRINTF_INFO("%s(%p)\n", __func__, base_page);
	struct jpeg_jpeghw_file_page_adaptor* page =
			(struct jpeg_jpeghw_file_page_adaptor *) base_page;

	if (page->dst_fp)
	{
		generate_ppm_header(page->dst_fp, &page->decompress_info);
	}
}

void jpeg_jpeghw_file_close_page(struct jpeg_page_adaptor *base_page) {
	struct jpeg_jpeghw_file_page_adaptor *jpa =
			(struct jpeg_jpeghw_file_page_adaptor *) base_page;

	if (jpa == NULL)
	{
		return;
	}

	if (jpeghw_finish_decompress(&jpa->decompress_info) != e_JPEGHW_SUCCESS)
	{
		printf(">>>>>%s() ERROR - decompress failed!\n", __func__);
		jpa->error = true;
	}

	if (jpa->src_fp)
	{
		fclose(jpa->src_fp);
		jpa->src_fp = NULL;
	}

	if (jpa->dst_fp)
	{
		fclose(jpa->dst_fp);
		jpa->dst_fp = NULL;
	}
}

void jpeg_jpeghw_file_free_in_strip(struct jpeg_strip_adaptor* base_strip) {
	struct jpeg_jpeghw_file_strip_adaptor *strip =
			(struct jpeg_jpeghw_file_strip_adaptor *) base_strip;
	DBG_PRINTF_INFO("%s() strip_adaptor %p\n", __func__, strip);

	if (strip->bb) {
		DBG_PRINTF_INFO("%s() relinquish strip jsa %p bb %p\n", __func__, strip,
				strip->bb);

		strip->bb = BigBuffer_Free(strip->bb); // reclaim payload
	}
}

void jpeg_send_strip_jpeghw(struct jpeg_page_adaptor *base_page,
		struct jpeg_strip_adaptor *base_strip, void *blob)
{
	if (!base_page || !base_strip || !blob)
		return;

	struct jpeg_jpeghw_file_strip_adaptor *strip =
			(struct jpeg_jpeghw_file_strip_adaptor *) base_strip;
	struct jpeg_jpeghw_file_page_adaptor* page =
			(struct jpeg_jpeghw_file_page_adaptor *) base_page;

	strip->bytes_per_pixel = page->decompress_info.common.bytes_per_pixel;
	int mcu_height = 48;
	strip->width_pix_in = page->decompress_info.common.image_width;
	strip->out_strip_width = page->decompress_info.common.image_width;
	strip->lines_in = page->decompress_info.common.image_height;
	strip->out_strip_height = (page->decompress_info.common.image_height + (mcu_height - 1))
			& ~(mcu_height - 1); // round up

	bool done = false;
	while (!done)
	{
		uint32_t buflen= mcu_height * strip->out_strip_width * strip->bytes_per_pixel;

        strip->bb = dma_buffer_malloc(0, buflen);
        if(strip->bb)
        {
			uint32_t bytes_read = 0;

			bytes_read = jpeghw_read_scanlines(&page->decompress_info, strip->bb, &done);
			if (bytes_read)
			{
				char *obuf = dma_buffer_mmap_forcpu(strip->bb);
				if (obuf)
				{
					fwrite(obuf, 1, bytes_read, page->dst_fp);
					dma_buffer_unmmap_forcpu(strip->bb);
				}
			}
			else
			{
				if (!done)
				{
					printf(">>>>>%s() ERROR - no bytes were processed for file %s!\n", __func__, page->dst_filename);
					done = true;
				}
			}

			(strip->bb)->freeFunc(strip->bb);
			strip->bb = NULL;
        }
	}

	// after output with lines computation?   need progress object ?
	strip->free_in_strip(base_strip);
}

static int is_valid_pgm_file(const struct dirent *dir) {
	const char *s = dir->d_name;
	int len = strlen(s) - 4;

	if (len >= 0)
	{
		if (strncmp(s+len, ".jpg", 4) == 0)
		{
			return 1;
		}
	}

	return 0;
}

static struct jpeg_jpeghw_file_page_adaptor *construct_jpeg_jpeghw_file_page_adaptor()
{
	struct jpeg_jpeghw_file_page_adaptor *jpa = 0;

	jpa = (struct jpeg_jpeghw_file_page_adaptor *) MEM_CALLOC(
			sizeof(struct jpeg_jpeghw_file_page_adaptor), 1);

	if (!jpa)
		return 0;

	jpa = (struct jpeg_jpeghw_file_page_adaptor*) construct_jpeg_page_adaptor(
			(struct jpeg_page_adaptor*) jpa);

	if (!jpa)
		return 0;

	bool warning = false;
	jpeghw_error_type_t ret = e_JPEGHW_ERROR;
	do
	{
		if (ret == e_JPEGHW_ERR_NOT_AVAILABLE)
		{
			if (!warning)
			{
				printf("WARNING: Unable to acquire compressor/decompressor, retrying...\n");
				warning = true;
			}

			posix_sleep_us(100);
		}

		ret = jpeghw_create_decompress(&jpa->decompress_info, JPEGHW_CONFIG_FLAG_BLOCK_JPEG_CORE_AVAIL);

	} while (ret == e_JPEGHW_ERR_NOT_AVAILABLE);

	if (ret == e_JPEGHW_SUCCESS)
	{
		jpa->adopt_from_page = jpeg_jpeghw_file_attach_page;
		jpa->jpeg_send_page = jpeg_jpeghw_file_send_page;
		jpa->page_close = jpeg_jpeghw_file_close_page;

		// set dmgr callbacks
		jpa->decompress_info.dmgr = &jpa->jdmgr;
		jpa->decompress_info.dmgr->init = dmgr_init;
		jpa->decompress_info.dmgr->get_input_buffer = dmgr_get_input_buffer;
#ifdef MANUAL_JPEG_HEADER_PARSE
		jpa->decompress_info.dmgr->get_jpeg_info = dmgr_get_jpeg_info;
#endif
		jpa->decompress_info.dmgr->term = dmgr_term;

		// optional scanline timeout manager
		jpa->decompress_info.common.scanline_timeout = my_read_scanline_timeout;

		// optional error manager
		jpa->jerr.error_exit = my_error_exit;
		jpa->decompress_info.common.err = jpeghw_std_error(&jpa->jerr);

		// back pointer to allow callbacks to have context connection.
		jpa->decompress_info.common.client_context = (void *)jpa;
	}
	else
	{
		memFree(jpa);
		jpa = NULL;
	}

	return (struct jpeg_jpeghw_file_page_adaptor *) jpa;
}

static void destroy_jpeg_jpeghw_file_page_adaptor(
		struct jpeg_jpeghw_file_page_adaptor *jpa) {

	DBG_ASSERT(jpa);

	if ( jpa  )
	{
		jpeghw_destroy_decompress(&jpa->decompress_info);

		memFree(jpa);
	}
}

static struct jpeg_jpeghw_file_strip_adaptor *construct_jpeg_jpeghw_file_strip_adaptor()
{
	struct jpeg_jpeghw_file_strip_adaptor *jsa = 0;

	jsa = (struct jpeg_jpeghw_file_strip_adaptor *) MEM_MALLOC(
			sizeof(struct jpeg_jpeghw_file_strip_adaptor));
	if (!jsa)
		return NULL;

	jsa = (struct jpeg_jpeghw_file_strip_adaptor*) construct_jpeg_strip_adaptor(
			(struct jpeg_strip_adaptor*) jsa);

	jsa->send_strip = jpeg_send_strip_jpeghw;

	jsa->free_in_strip = jpeg_jpeghw_file_free_in_strip; // default
	return (struct jpeg_jpeghw_file_strip_adaptor *) jsa;
}

static void destroy_jpeg_jpeghw_file_strip_adaptor(struct jpeg_jpeghw_file_strip_adaptor *jfsa)
{
	DBG_ASSERT(jfsa);
	if (jfsa)
	{
		memFree(jfsa);
	}
}

int main_jpeghw_adaptor(int argc, char *argv[]) {
	struct dirent **eps;
	int n;
	int filearg=0;
	int cnt;

	DBG_PRINTF_INFO("%s() argc %d argv[%d] %s\n", __func__, argc, 1, argv[1]);

	// optional setup a cached output buffer allocator
	// strip_adaptor->alloc_new_bb = dma_buffer_cached_malloc;

	// one directory per page with "page_..." sorted strips of pgm files

	if (argc == 2) {
		filearg = 1;
	} else {
		DBG_PRINTF_ERR("usage mode files : program /dir/path/\n");
		return 0;
	}

	n = scandir(argv[filearg], &eps, is_valid_pgm_file, alphasort);
	if (n < 0)
	{
		DBG_PRINTF_ERR("ERROR: Couldn't open the directory!\n");
		return 0;
	}

	// process all files in the directory
	cnt = 0;
	while (cnt < n)
	{
		struct file_blob_s file_blob;
		char *strip_name = NULL;

		strip_name = eps[cnt]->d_name;

		memset(&file_blob.filename, 0, sizeof(file_blob.filename));
		memset(&file_blob.page_name, 0, sizeof(file_blob.page_name));

		int len = sizeof(file_blob.filename)-1;
		strncpy(file_blob.filename, argv[filearg], len);

		len = sizeof(file_blob.filename) - strlen(file_blob.filename)-1;
		strncat(file_blob.filename, "/", len);

		len = sizeof(file_blob.filename) - strlen(file_blob.filename)-1;
		strncat(file_blob.filename, strip_name, len);

		strcpy(file_blob.page_name, strip_name);

		DBG_PRINTF_INFO("%s\n", file_blob.filename);

		printf("%s(%s) - Start decompress\n", __func__, file_blob.filename);

		struct jpeg_jpeghw_file_page_adaptor *page_adaptor = construct_jpeg_jpeghw_file_page_adaptor();
		struct jpeg_jpeghw_file_strip_adaptor *strip_adaptor =	construct_jpeg_jpeghw_file_strip_adaptor();

		if (page_adaptor && strip_adaptor)
		{

			// setup new page
			jpeg_recv_page((struct jpeg_page_adaptor *)page_adaptor, (void *)&file_blob);

			jpeg_send_strip((struct jpeg_page_adaptor *)page_adaptor,
					(struct jpeg_strip_adaptor *)strip_adaptor,
					&file_blob);

			jpa_from_page_close((struct jpeg_page_adaptor *)page_adaptor);
		}

		destroy_jpeg_jpeghw_file_strip_adaptor(strip_adaptor);
		destroy_jpeg_jpeghw_file_page_adaptor(page_adaptor);

		printf("%s(%s) - Finish decompress\n", __func__, file_blob.filename);

		cnt++; // next file argument?
	}

	return 0;
}

/// test-sample() -- take a list of options/files and apply appropriate encode operations.
int main(int argc, char** argv) {
	memInitMemory(0, 1024 * 1024 * 128); // init memory pool uses dmaalloc.ko, implies 1 process for now.
	logger_init();  // controlled by /home/root/logger_config_file if present
	uio_lib_init(); // init uio uses /dev/uio*
	jpeghw_init();

	main_jpeghw_adaptor(argc, argv);

	jpeghw_terminate();

	return 0;
}

// eof jpeg_decompress.c
