/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2015, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/



#include <stdio.h>
#include <sys/types.h>
#include <dirent.h>

#include "jpeghw_api.h" // JPEGHW library
#include "jpeg_strip_adaptor.h"
#include "jpeg_page_adaptor.h"

#include "logger.h"
#include "dma_buffer.h"
#include "uio_lib_api.h"

#include "jpeg_common.h"


#define DBG_PRFX "jpeg_compress: "
#define LOGGER_MODULE_MASK DEBUG_LOGGER_MODULE_DEVICES | LOGGER_SUBMODULE_BIT(20)

static int filenum = 0;
#define JPEG_FILE_DEFAULT_IMAGE_HEIGHT_MAX 8400 // 14 inch default, needs backpatch based on adf height
												 // using jpeg_jpeghw_fixup_page_length() callback

struct jpeghw_fifo_record {
	struct BigBuffer_s* bb; // this must be defined first
	uint32_t bb_size;
};

struct file_blob_s {
	char filename[256];
	char page_name[256];
	bool is_last_strip;     //< eop marker
};

struct jpeg_jpeghw_file_page_adaptor {
	jpeg_page_adaptor_BaseClassMembers
	;
	//< put this macro in your subclass
	/// jpeghw
	struct jpeghw_compress_struct compress_info;   //< jpeghw compress

	struct jpeghw_compression_mgr jcmgr;
	struct jpeghw_error_mgr jerr;
	/// jpeghw end

	uint32_t image_height;
	PPM_header_struct ppminfo;

	// add my local page variables here.
	char dst_filename[256]; /**< generated filename */
	FILE* dst_fp; /**< downstream dst stream handle */

	int quality;

	bool auto_generate_jpeg_header;
	bool error;
};

struct jpeg_jpeghw_file_strip_adaptor {
	jpeg_strip_adaptor_BaseClassMembers
	;  // put this macro in your subclass

	char *filename; /** src filename */
	struct BigBuffer_s* bb; //< input data as dma_buffer.

};

const char* output_path = ".";

// externs
extern void logger_init();

#define JPEG_MEM_DST_MGR_BUFFER_SIZE (16 * 1024)   ///< might want to have this be configurable.
void cmgr_init(struct jpeghw_compress_struct * cinfo) {
	struct jpeg_jpeghw_file_page_adaptor *jpa =
			(struct jpeg_jpeghw_file_page_adaptor*) cinfo->common.client_context;

	DBG_ASSERT(jpa);
}

void cmgr_term(struct jpeghw_compress_struct * cinfo) {
	struct jpeg_jpeghw_file_page_adaptor *jpa =
			(struct jpeg_jpeghw_file_page_adaptor*) cinfo->common.client_context;

	DBG_ASSERT(jpa);
}

struct BigBuffer_s* cmgr_get_output_buffer(struct jpeghw_compress_struct * cinfo) {
	struct jpeg_jpeghw_file_page_adaptor *jpa =
			(struct jpeg_jpeghw_file_page_adaptor*) cinfo->common.client_context;

	DBG_ASSERT(jpa);

	struct BigBuffer_s* output_buffer = dma_buffer_malloc(0, JPEG_MEM_DST_MGR_BUFFER_SIZE);

	return output_buffer;
}

jpeghw_error_type_t cmgr_get_quant_table(struct jpeghw_compress_struct * cinfo, uint8_t quality, uint8_t *table, uint32_t *size)
{
	if  (quality > 100)
	{
		printf("\n>>>>>%s() ERROR - invalid quality parameter, received %u but expected 0-100!\n",
				__func__, (unsigned int)quality);

		// return invalid parameter
		return e_JPEGHW_ERR_INVALID_PARAMETERS;
	}

	if (size)
	{
		if  (*size > sizeof(g_quantization_table[0]))
		{
			*size = sizeof(g_quantization_table[0]);
		}
	}

	if (table && size)
	{
		memcpy(table, &g_quantization_table[quality][0], *size);
	}

	return e_JPEGHW_SUCCESS;
}

jpeghw_error_type_t cmgr_get_huff_table(struct jpeghw_compress_struct * cinfo, uint8_t table_index, bool ac, uint8_t *bits, uint32_t *bits_size, uint8_t *val, uint32_t *val_size)
{
	if (table_index > 1)
	{
		printf("\n>>>>>%s() ERROR - invalid table_index parameter, received %u but expected 0 or 1!\n",
				__func__, (unsigned int)table_index);

		return e_JPEGHW_ERR_INVALID_PARAMETERS;
	}

	if (bits_size)
	{
		uint32_t size = (ac ? sizeof(g_ac_bits[table_index]) : sizeof(g_dc_bits[table_index]));

		if (*bits_size > size)
		{
			*bits_size = size;
		}
	}

	if (bits && bits_size)
	{
		if (ac)
		{
			memcpy(bits, g_ac_bits[table_index], *bits_size);
		}
		else
		{
    		memcpy(bits, g_dc_bits[table_index], *bits_size);
		}
	}

	if (val_size)
	{
		uint32_t size = (ac ? sizeof(g_ac_val[table_index]) : sizeof(g_dc_val[table_index]));

		if (*val_size > size)
		{
			*val_size = size;
		}
	}

	if (val && val_size)
	{
		if (ac)
		{
			memcpy(val, g_ac_val[table_index], *val_size);
		}
		else
		{
    		memcpy(val, g_dc_val[table_index], *val_size);
		}
	}

	return e_JPEGHW_SUCCESS;
}

bool cmgr_send_output_buffer(struct jpeghw_compress_struct * cinfo, struct BigBuffer_s* output_buffer, uint32_t bytes_in_buffer) {
	struct jpeg_jpeghw_file_page_adaptor *jpa =
			(struct jpeg_jpeghw_file_page_adaptor*) cinfo->common.client_context;

	DBG_ASSERT(jpa);
	DBG_ASSERT(output_buffer);

	char *data = dma_buffer_mmap_forcpu(output_buffer);
	if (data)
	{
		fwrite(data, 1, bytes_in_buffer, jpa->dst_fp);
		dma_buffer_unmmap_forcpu(output_buffer);
	}

	BigBuffer_Free(output_buffer);

	return true;
}


bool my_write_scanline_timeout(
		struct jpeghw_common_struct *info, uint32_t time_in_seconds) {
	bool ret = false;

	DBG_ASSERT(info);

	struct jpeg_jpeghw_file_page_adaptor *jpa =
			(struct jpeg_jpeghw_file_page_adaptor*) info->client_context;

	DBG_ASSERT(jpa);

	if (time_in_seconds > 30)
	{
		DBG_PRINTF_ERR("%s() Compression write scanline timed out!\n", __func__);
		ret = true;
	}

	return ret;
}

void my_error_exit(struct jpeghw_common_struct * cinfo) {
	printf(">>>>>jpeg internal exit");
	// do a longjmp here
	exit(-1);
}

void jpeg_jpeghw_file_attach_page(struct jpeg_page_adaptor *base_page, void *blob) {
	struct jpeg_jpeghw_file_page_adaptor* page =
			(struct jpeg_jpeghw_file_page_adaptor *) base_page;
	int fpos = 0;

	if (page == NULL)
	{
		DBG_PRINTF_ERR("%s() Invalid page parameter!\n", __func__);
		return;
	}

	if (blob == NULL)
	{
		DBG_PRINTF_ERR("%s() Invalid file_blob variable!\n", __func__);
		return;
	}

	struct file_blob_s *file_blob = (struct file_blob_s *) blob;
	char *filename = file_blob->filename;


	DBG_PRINTF_INFO("%s() page %p file %p filename '%s'\n", __func__, page,
			file_blob, filename);

	FILE *src_fp = fopen(filename, "r");
	if (src_fp)
	{ // good file (stream)
		bool valid_stream = false;

		fpos = parse_ppm_header(src_fp, &page->ppminfo);
		if (fpos != FAIL)
		{
			page->compress_info.common.image_width = page->ppminfo.image_width;

			page->compress_info.common.image_height = page->image_height == 0 ?
					JPEG_FILE_DEFAULT_IMAGE_HEIGHT_MAX : page->image_height;

			page->compress_info.common.bytes_per_pixel = page->ppminfo.bytes_per_pixel;
			page->compress_info.common.mcu_width = page->ppminfo.bytes_per_pixel == 1 ? 8 : 16;
			page->compress_info.common.mcu_height = page->compress_info.common.mcu_width;

			valid_stream = true;
		}

		jpeghw_set_defaults(&page->compress_info.common);

		page->compress_info.auto_generate_jpeg_header = page->auto_generate_jpeg_header;

		if (valid_stream)
		{
            const char* fmt = "%s/tfile.%d.%d.jpg";
            pid_t pid = getpid();

			snprintf(page->dst_filename, sizeof(page->dst_filename),
					fmt, output_path, pid, filenum);
            filenum++;

			page->dst_fp = fopen(page->dst_filename, "w"); // create/truncate dst
			DBG_ASSERT(page->dst_fp);

			page->compress_info.quality = page->quality;
			jpeghw_start_compress(&page->compress_info);
		}
		else
		{
			DBG_PRINTF_ERR("%s() '%s' NOT a valid stream format!\n", __func__, filename);
		}

		fclose(src_fp);
	}
}

void jpeg_jpeghw_file_send_page(struct jpeg_page_adaptor *base_page, void *blob) {
	DBG_PRINTF_INFO("%s(%p)\n", __func__, base_page);
	struct jpeg_jpeghw_file_page_adaptor* page =
			(struct jpeg_jpeghw_file_page_adaptor *) base_page;

	if (false == page->auto_generate_jpeg_header)
	{
		// manually append to exiting JPEG header
		generate_jpeg_header(page->dst_fp, &page->compress_info);
	}
}

void jpeg_jpeghw_file_close_page(struct jpeg_page_adaptor *base_page) {
	struct jpeg_jpeghw_file_page_adaptor *jpa =
			(struct jpeg_jpeghw_file_page_adaptor *) base_page;

	if (jpa == NULL)
	{
		return;
	}

	if (jpeghw_finish_compress(&jpa->compress_info) != e_JPEGHW_SUCCESS)
	{
		printf(">>>>>%s() ERROR - compress failed!\n", __func__);
		jpa->error = true;
	}

	if (jpa->dst_fp)
	{
		fclose(jpa->dst_fp);
	}
}

// update img height in results file with given src lines
int jpeg_jpeghw_fixup_page_length(struct jpeg_page_adaptor* base_page) {

	if (!base_page)
	{
		return FAIL;
	}

	uint32_t lines_encoded = base_page->strip_out_height;
	struct jpeg_jpeghw_file_page_adaptor* page =
			(struct jpeg_jpeghw_file_page_adaptor *) base_page;

	if (lines_encoded != page->compress_info.common.image_height)
	{
		FILE *dst_file = fopen(page->dst_filename, "r+"); // read/write dst
		if (dst_file) { // good file (stream)
			lines_encoded = page_jpeg_fixup_page_length(dst_file, lines_encoded);
		}

		if (dst_file)
			fclose(dst_file);
	}

	return lines_encoded;
}

void jpeg_jpeghw_file_attach_from_strip(struct jpeg_page_adaptor *base_page,
		struct jpeg_strip_adaptor *base_strip, void *blob)
{
	if (!base_page || !base_strip || !blob)
		return;

	struct jpeg_jpeghw_file_page_adaptor* page =
			(struct jpeg_jpeghw_file_page_adaptor *) base_page;
	struct jpeg_jpeghw_file_strip_adaptor *jfsa =
			(struct jpeg_jpeghw_file_strip_adaptor *) base_strip;
	struct file_blob_s *file_blob = (struct file_blob_s *) blob;

	int size;
	int strip_bytes_read;

	bool valid_stream = false;
	FILE *src_fp = fopen(file_blob->filename, "r");
	if (src_fp)
	{ // good file (stream)
		PPM_header_struct ppminfo;

		int fpos = parse_ppm_header(src_fp, &ppminfo);
		if (fpos != FAIL)
		{
			if (page->compress_info.common.image_width == ppminfo.image_width &&
					page->compress_info.common.bytes_per_pixel == ppminfo.bytes_per_pixel)
			{
				jfsa->bytes_per_pixel = page->compress_info.common.bytes_per_pixel;
				jfsa->width_pix_in = page->compress_info.common.image_width;
				jfsa->out_strip_width = page->compress_info.common.image_width;
				jfsa->lines_in = ppminfo.image_height;
				jfsa->out_strip_height = (jfsa->lines_in + (page->compress_info.common.mcu_height - 1))
						& ~(page->compress_info.common.mcu_height - 1); // round up

				valid_stream = true;
			}
		}

		if (valid_stream)
		{
			jfsa->is_end_of_page = file_blob->is_last_strip;
			size = jfsa->width_pix_in * jfsa->lines_in * jfsa->bytes_per_pixel; // reality

			DBG_PRINTF_INFO("%s() img (%d x %d x %d) -> %d-byte strip\n", __func__,
					jfsa->width_pix_in, jfsa->bytes_per_pixel, jfsa->out_strip_height,
					size);

			jfsa->bb = dma_buffer_malloc(0, size);

			char *data = dma_buffer_mmap_forcpu(jfsa->bb);
			if (data)
			{
				strip_bytes_read = fread(data, 1, size, src_fp);
				if (strip_bytes_read != size)
				{
					jfsa->out_strip_height = strip_bytes_read
							/ (jfsa->width_pix_in * jfsa->bytes_per_pixel); // last strip is often short.
					DBG_PRINTF_INFO("%s() LAST %d-line strip of %d-line page\n",
							__func__, jfsa->out_strip_height, jfsa->lines_in);
					jfsa->is_end_of_page = true;
				}
			}

			dma_buffer_unmmap_forcpu(jfsa->bb);
		}
		else
		{
			DBG_PRINTF_ERR("%s() '%s' NOT a valid stream format!\n", __func__, file_blob->filename);
		}

		fclose(src_fp);
	}

	DBG_PRINTF_INFO("%s() strip w/lines in(%d) out(%d)\n", __func__,
			jfsa->lines_in, jfsa->out_strip_height);
}


void jpeg_jpeghw_file_free_in_strip(struct jpeg_strip_adaptor* base_strip) {
	struct jpeg_jpeghw_file_strip_adaptor *strip =
			(struct jpeg_jpeghw_file_strip_adaptor *) base_strip;
	DBG_PRINTF_INFO("%s() strip_adaptor %p\n", __func__, strip);

	if (strip->bb) {
		DBG_PRINTF_INFO("%s() relinquish strip jsa %p bb %p\n", __func__, strip,
				strip->bb);

		strip->bb = BigBuffer_Free(strip->bb); // reclaim payload
	}
}

void jpeg_recv_strip_jpeghw(struct jpeg_page_adaptor *base_page,
		struct jpeg_strip_adaptor *base_strip, void *blob)
{
	if (!base_page || !base_strip || !blob)
		return;

	struct jpeg_jpeghw_file_strip_adaptor *strip =
			(struct jpeg_jpeghw_file_strip_adaptor *) base_strip;
	struct jpeg_jpeghw_file_page_adaptor* page =
			(struct jpeg_jpeghw_file_page_adaptor *) base_page;

	strip->adopt_bb_from_blob(base_page, base_strip, blob); // binds jsa->in_area.bb

	if (strip->bb)
	{
		uint32_t lines = jpeghw_write_scanlines(&page->compress_info, strip->bb,
							strip->lines_in, strip->is_end_of_page);

		if (lines < strip->lines_in) {
			printf(
					">>>>>%s() WARNING - not all Scanlines were processed for file %s (%d/%d)!\n",
					__func__, page->dst_filename, lines, strip->lines_in);
			page->error = true;

			if (lines == 0) {
				strip->bb = BigBuffer_Free(strip->bb);
			}
		} else {
			jpa_from_page_at_output_eoi(base_page, strip->lines_in);
		}

		strip->bb = NULL; // passed ownership downstream
	}

	// after output with lines computation?   need progress object ?
	strip->free_in_strip(base_strip);
}

static int is_valid_pgm_file(const struct dirent *dir) {
	const char *s = dir->d_name;
	int len = strlen(s) - 4;

	if (len >= 0)
	{
		if (strncmp(s+len, ".pgm", 4) == 0)
		{
			return 1;
		}
		else if (strncmp(s+len, ".pbm", 4) == 0)
		{
			return 1;
		}
		else if (strncmp(s+len, ".ppm", 4) == 0)
		{
			return 1;
		}
	}

	return 0;
}

static struct jpeg_jpeghw_file_page_adaptor *construct_jpeg_jpeghw_file_page_adaptor()
{
	struct jpeg_jpeghw_file_page_adaptor *jpa = 0;

	jpa = (struct jpeg_jpeghw_file_page_adaptor *) MEM_CALLOC(
			sizeof(struct jpeg_jpeghw_file_page_adaptor), 1);
	if (!jpa)
		return 0;

	jpa = (struct jpeg_jpeghw_file_page_adaptor*) construct_jpeg_page_adaptor(
			(struct jpeg_page_adaptor*) jpa);

	bool warning = false;
	jpeghw_error_type_t ret = e_JPEGHW_ERROR;
	do
	{
		if (ret == e_JPEGHW_ERR_NOT_AVAILABLE)
		{
			if (!warning)
			{
				printf("WARNING: Unable to acquire compressor, retrying...\n");
				warning = true;
			}

			posix_sleep_us(100);
		}

		ret = jpeghw_create_compress(&jpa->compress_info, JPEGHW_CONFIG_FLAG_BLOCK_JPEG_CORE_AVAIL);

	} while (ret == e_JPEGHW_ERR_NOT_AVAILABLE);

	if (ret == e_JPEGHW_SUCCESS)
	{
		jpa->adopt_from_page = jpeg_jpeghw_file_attach_page;
		jpa->jpeg_send_page = jpeg_jpeghw_file_send_page;
		jpa->page_close = jpeg_jpeghw_file_close_page;
		jpa->fixup_jpg_page_length = jpeg_jpeghw_fixup_page_length;

		jpa->image_height = JPEG_FILE_DEFAULT_IMAGE_HEIGHT_MAX;
		jpa->quality = 90;
		jpa->auto_generate_jpeg_header = true;

		// set cmgr callbacks
		jpa->compress_info.cmgr = &jpa->jcmgr;
		jpa->compress_info.cmgr->init = cmgr_init;
		jpa->compress_info.cmgr->get_output_buffer = cmgr_get_output_buffer;
		jpa->compress_info.cmgr->send_output_buffer = cmgr_send_output_buffer;
		jpa->compress_info.cmgr->term = cmgr_term;

		jpa->compress_info.cmgr->get_quant_table = cmgr_get_quant_table;
		jpa->compress_info.cmgr->get_huff_table = cmgr_get_huff_table;

		// optional scanline timeout manager
		jpa->compress_info.common.scanline_timeout = my_write_scanline_timeout;

		// optional error manager
		jpa->jerr.error_exit = my_error_exit;
		jpa->compress_info.common.err = jpeghw_std_error(&jpa->jerr);

		// back pointer to allow callbacks to have context connection.
		jpa->compress_info.common.client_context = (void *)jpa;
	}
	else
	{
		memFree(jpa);
		jpa = NULL;
	}

	return (struct jpeg_jpeghw_file_page_adaptor *) jpa;
}

static void destroy_jpeg_jpeghw_file_page_adaptor(
		struct jpeg_jpeghw_file_page_adaptor *jpa) {

	DBG_ASSERT(jpa);

	if ( jpa  )
	{
		jpeghw_destroy_compress(&jpa->compress_info);

		memFree(jpa);
	}
}

static struct jpeg_jpeghw_file_strip_adaptor *construct_jpeg_jpeghw_file_strip_adaptor()
{
	struct jpeg_jpeghw_file_strip_adaptor *jsa = 0;

	jsa = (struct jpeg_jpeghw_file_strip_adaptor *) MEM_MALLOC(
			sizeof(struct jpeg_jpeghw_file_strip_adaptor));
	if (!jsa)
		return NULL;

	jsa = (struct jpeg_jpeghw_file_strip_adaptor*) construct_jpeg_strip_adaptor(
			(struct jpeg_strip_adaptor*) jsa);

	jsa->adopt_bb_from_blob = jpeg_jpeghw_file_attach_from_strip;
	jsa->recv_strip = jpeg_recv_strip_jpeghw;

	jsa->free_in_strip = jpeg_jpeghw_file_free_in_strip; // default

	return (struct jpeg_jpeghw_file_strip_adaptor *) jsa;
}

static void destroy_jpeg_jpeghw_file_strip_adaptor(struct jpeg_jpeghw_file_strip_adaptor *jfsa)
{
	DBG_ASSERT(jfsa);
	if (jfsa)
	{
		memFree(jfsa);
	}
}

#define NUM_COMPRESSORS 2
int main_jpeghw_adaptor_strips(int argc, char *argv[]) {
	struct dirent **eps[NUM_COMPRESSORS];
	int n[NUM_COMPRESSORS];
	int filearg[NUM_COMPRESSORS];
	int ncomp, cnt;
	struct jpeg_jpeghw_file_page_adaptor *page_adaptor[NUM_COMPRESSORS];
	struct jpeg_jpeghw_file_strip_adaptor *strip_adaptor[NUM_COMPRESSORS];

	int quality = 90;
	int num_compressors = 1;
	int image_height = JPEG_FILE_DEFAULT_IMAGE_HEIGHT_MAX;
	bool auto_generate_jpeg_header = true;

	DBG_PRINTF_INFO("%s() argc %d argv[%d] %s\n", __func__, argc, 1, argv[1]);

	// optional setup a cached output buffer allocator
	// strip_adaptor->alloc_new_bb = dma_buffer_cached_malloc;

	// one directory per page with "page_..." sorted strips of pgm files

	for (ncomp=0; ncomp<NUM_COMPRESSORS; ncomp++)
	{
		filearg[ncomp] = 0;
		n[ncomp] = 0;
		eps[ncomp] = NULL;
		page_adaptor[ncomp] = NULL;
		strip_adaptor[ncomp] = NULL;
	}

	if (argc == 2) {
		filearg[0] = 1;
	} else if (argc == 3) {
		num_compressors = 2;
		filearg[0] = 1;
		filearg[1] = 2;
	}
	else {
		DBG_PRINTF_ERR("usage mode files : program /dir/path1/ </dir/path2/> \n");
		return 0;
	}

	for (ncomp=0; ncomp<num_compressors; ncomp++)
	{
		if (filearg[ncomp] != 0)
		{
			n[ncomp] = scandir(argv[filearg[ncomp]], &eps[ncomp], is_valid_pgm_file, alphasort);
			if (n[ncomp] < 0)
			{
				DBG_PRINTF_ERR("ERROR: Couldn't open the directory (%s)!\n", argv[filearg[ncomp]]);
				return 0;
			}

			if (n[ncomp] == 0)
			{
				DBG_PRINTF_ERR("ERROR: Empty directory (%s)!\n", argv[filearg[ncomp]]);
				return 0;
			}

			page_adaptor[ncomp] = construct_jpeg_jpeghw_file_page_adaptor(quality);
			if (page_adaptor[ncomp])
			{
				page_adaptor[ncomp]->image_height = image_height;
				page_adaptor[ncomp]->quality = quality;
				page_adaptor[ncomp]->auto_generate_jpeg_header = auto_generate_jpeg_header;
			}

			strip_adaptor[ncomp] = construct_jpeg_jpeghw_file_strip_adaptor();
		}
	}

	int max_cnt = 0;
	for (ncomp=0; ncomp<num_compressors; ncomp++)
	{
		if (max_cnt < n[ncomp])
		{
			max_cnt = n[ncomp];
		}
	}

    for (cnt = 0; cnt < max_cnt; ++cnt)
    { // next strip/file argument?
        for (ncomp = 0; ncomp < num_compressors; ncomp++)
        {
			if (page_adaptor[ncomp] && strip_adaptor[ncomp] && cnt < n[ncomp])
			{
				struct file_blob_s file_blob;
				char *strip_name = NULL;

				strip_name = eps[ncomp][cnt]->d_name;

				memset(&file_blob.filename, 0, sizeof(file_blob.filename));
				memset(&file_blob.page_name, 0, sizeof(file_blob.page_name));
				file_blob.is_last_strip = false;

				int len = sizeof(file_blob.filename)-1;
				strncpy(file_blob.filename, argv[filearg[ncomp]], len);

				len = sizeof(file_blob.filename) - strlen(file_blob.filename)-1;
				strncat(file_blob.filename, "/", len);

				len = sizeof(file_blob.filename) - strlen(file_blob.filename)-1;
				strncat(file_blob.filename, strip_name, len);

				DBG_PRINTF_INFO("%s\n", file_blob.filename);

				len = strlen(strip_name) - 4;
				if (len > 0)
				{
					strncpy(file_blob.page_name, strip_name, len);
					file_blob.page_name[len] = 0;
				}
				else
				{
					strcpy(file_blob.page_name, strip_name);
				}

				file_blob.is_last_strip = (cnt==(n[ncomp]-1));

				if (cnt == 0)
				{
					// setup new page
					jpeg_recv_page((struct jpeg_page_adaptor *)page_adaptor[ncomp], (void *)&file_blob);
				}

				jpeg_recv_strip((struct jpeg_page_adaptor *)page_adaptor[ncomp],
						(struct jpeg_strip_adaptor *)strip_adaptor[ncomp],
						&file_blob);
			}
        }
    }

	for (ncomp=0; ncomp<num_compressors; ncomp++)
	{
		jpa_from_page_close((struct jpeg_page_adaptor *)page_adaptor[ncomp]);
		jpa_fixup_jpg_page((struct jpeg_page_adaptor *)page_adaptor[ncomp]);
	}

	for (ncomp=0; ncomp<num_compressors; ncomp++)
	{
		destroy_jpeg_jpeghw_file_strip_adaptor(strip_adaptor[ncomp]);
		destroy_jpeg_jpeghw_file_page_adaptor(page_adaptor[ncomp]);
	}

	return 0;
}

/// test-sample() -- take a list of options/files and apply appropriate encode operations.
int main(int argc, char** argv) {
	memInitMemory(0, 1024 * 1024 * 128); // init memory pool uses dmaalloc.ko, implies 1 process for now.
	logger_init();  // controlled by /home/root/logger_config_file if present
	uio_lib_init(); // init uio uses /dev/uio*
	jpeghw_init();

	main_jpeghw_adaptor_strips(argc, argv);

	jpeghw_terminate();

	return 0;
}

// eof jpeg_compress.c
