/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2015, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/


#include <stdio.h>
#include <stdint.h>
#include <stdbool.h>
#include <ctype.h>
#include <sys/types.h>

#include "memAPI.h"
#include "logger.h"
#include "list.h"

#include "jpeghw_api.h" // JPEGHW library
#include "jpeg_common.h"

#define DBG_PRFX "jpeg_common: "
#define LOGGER_MODULE_MASK DEBUG_LOGGER_MODULE_DEVICES | LOGGER_SUBMODULE_BIT(20)


//--------------------------------------------------------------------------------
// parsing routines
//--------------------------------------------------------------------------------
static char *_jpeghw_get_token(FILE *fp, char *token, int length)
{
	int ch = EOF;
	char *ptr = NULL;

	// skip past any whitespace
	for (;;)
	{
		while (isspace(ch = fgetc(fp)));

		// also skip any comments
		if (ch == '#')
		{
			// skip comment to EOL or EOF
			while (ch != '\n' && ch != EOF)
			{
				ch = fgetc(fp);
			}

			// if EOF, done
			if (ch == EOF)
				break;
		}
		else
		{
			// not a comment, so done
			break;
		}
	}

	ptr = token;

	// if not EOF, start filling in token string
	if (ch != EOF)
	{
		do
		{
			*ptr++ = ch;
			ch = fgetc(fp);
		}
		while ( !isspace(ch) && ch != '#' && ch != EOF && (ptr-token) < (length-1) );
	}

	// append NULL at end of token string
	*ptr = 0;

	return token;
}

//--------------------------------------------------------------------------------
// init_info
//--------------------------------------------------------------------------------
static void _jpeghwc_init_info(PPM_header_struct* ppminfo)
{
    ppminfo->image_height        = 0;
    ppminfo->image_width         = 0;
    ppminfo->bytes_per_pixel     = 0;

}

static int jpeghwd_color_id [4];
static void _jpeghwd_init_info(JPEG_header_struct* jpeginfo)
{
    memset(jpeginfo, 0, sizeof(JPEG_header_struct));
}

//--------------------------------------------------------------------------------
// calc_zz
//--------------------------------------------------------------------------------
static int _jpeghwd_calc_zz(int in_index)
{
    int out_index;

    switch(in_index)
    {
      case 0: out_index  =  0; break;
      case 1: out_index  =  1; break;
      case 2: out_index  =  8; break;
      case 3: out_index  = 16; break;
      case 4: out_index  =  9; break;
      case 5: out_index  =  2; break;
      case 6: out_index  =  3; break;
      case 7: out_index  = 10; break;
      case 8: out_index  = 17; break;
      case 9: out_index  = 24; break;
      case 10: out_index = 32; break;
      case 11: out_index = 25; break;
      case 12: out_index = 18; break;
      case 13: out_index = 11; break;
      case 14: out_index =  4; break;
      case 15: out_index =  5; break;
      case 16: out_index = 12; break;
      case 17: out_index = 19; break;
      case 18: out_index = 26; break;
      case 19: out_index = 33; break;
      case 20: out_index = 40; break;
      case 21: out_index = 48; break;
      case 22: out_index = 41; break;
      case 23: out_index = 34; break;
      case 24: out_index = 27; break;
      case 25: out_index = 20; break;
      case 26: out_index = 13; break;
      case 27: out_index =  6; break;
      case 28: out_index =  7; break;
      case 29: out_index = 14; break;
      case 30: out_index = 21; break;
      case 31: out_index = 28; break;
      case 32: out_index = 35; break;
      case 33: out_index = 42; break;
      case 34: out_index = 49; break;
      case 35: out_index = 56; break;
      case 36: out_index = 57; break;
      case 37: out_index = 50; break;
      case 38: out_index = 43; break;
      case 39: out_index = 36; break;
      case 40: out_index = 29; break;
      case 41: out_index = 22; break;
      case 42: out_index = 15; break;
      case 43: out_index = 23; break;
      case 44: out_index = 30; break;
      case 45: out_index = 37; break;
      case 46: out_index = 44; break;
      case 47: out_index = 51; break;
      case 48: out_index = 58; break;
      case 49: out_index = 59; break;
      case 50: out_index = 52; break;
      case 51: out_index = 45; break;
      case 52: out_index = 38; break;
      case 53: out_index = 31; break;
      case 54: out_index = 39; break;
      case 55: out_index = 46; break;
      case 56: out_index = 53; break;
      case 57: out_index = 60; break;
      case 58: out_index = 61; break;
      case 59: out_index = 54; break;
      case 60: out_index = 47; break;
      case 61: out_index = 55; break;
      case 62: out_index = 62; break;
      case 63: out_index = 63; break;
      default:
        DBG_PRINTF_ERR("ERROR in %s : Can't calc_zz on value %d.\n",__func__, in_index);
        return FAIL;
        break;
    }

    return out_index;
}

//--------------------------------------------------------------------------------
// parse_sof
//--------------------------------------------------------------------------------
static uint8_t* _jpeghwd_parse_sof(JPEG_header_struct* jpeginfo, uint8_t *buff_ptr, uint32_t buff_len)
{
    SOF_struct *sof_part;
    SOF_comp_spec_struct *comp_part;
    int i;
    int max_h_m1;
    int max_v_m1;
    int h_mcu;
    int v_mcu;
    uint8_t* height_address = 0; // return the 'height' address/offset

    if (sizeof(SOF_struct) > buff_len)
    {
        DBG_PRINTF_ERR("ERROR in %s : buffer length < SOF_struct size!\n",__func__);
        return 0;
    }

    sof_part = (SOF_struct *)buff_ptr;

    // check precision for 8 bit
    if(sof_part->P != 8)
    {
        DBG_PRINTF_ERR("ERROR in %s : Precision of %d not supported (only 8 supported).\n",__func__, sof_part->P);
        return 0;
    }
    if(sof_part->Nf > 4)
    {
        DBG_PRINTF_ERR("ERROR in %s : Support 4 color max (Nf <= 4) (not supported Nf = %d).\n",
                __func__, sof_part->Nf);
        return 0;
    }

    jpeginfo->image_width   = (sof_part->X_msb<<8) + sof_part->X_lsb;
    jpeginfo->image_height  = (sof_part->Y_msb<<8) + sof_part->Y_lsb;
    jpeginfo->bytes_per_pixel = (sof_part->Nf);

    // capture address of img height (lines)
    height_address = &sof_part->Y_msb;

    DBG_PRINTF_INFO("*** SOF parsing ***\n");
    DBG_PRINTF_INFO("SOF precision = %d\n", sof_part->P);
    DBG_PRINTF_INFO("SOF image width = %d\n", jpeginfo->image_width);
    DBG_PRINTF_INFO("SOF image height = %d (@ %p)\n", jpeginfo->image_height, height_address);
    DBG_PRINTF_INFO("SOF num_planes_m1 = %d\n", jpeginfo->bytes_per_pixel -1);

    for(i=0; i<4; i++)
    {
    	jpeghwd_color_id[i] = 0;
    }

    for(i=0; i<sof_part->Nf; i++)
    {
        if ( (sizeof(SOF_struct) + (sizeof(SOF_comp_spec_struct)*i) + sizeof(SOF_comp_spec_struct))  > buff_len )
        {
            DBG_PRINTF_ERR("ERROR in %s : buffer length < SOF_comp_spec_struct size!\n",__func__);
            return 0;
        }

        comp_part = (SOF_comp_spec_struct *)(buff_ptr + sizeof(SOF_struct) +
                                             (sizeof(SOF_comp_spec_struct)*i) );

        DBG_PRINTF_INFO("SOF component ID = %d\n", comp_part->C);
        DBG_PRINTF_INFO("SOF H = %d\n", ((comp_part->HV>>4)&0xF));
        DBG_PRINTF_INFO("SOF V = %d\n", ((comp_part->HV)&0xF));
        DBG_PRINTF_INFO("SOF T = %d\n", comp_part->T);

        if (i < 4)
        {
        	jpeghwd_color_id[i] = comp_part->C;
			if(i==0)
			{
				jpeginfo->h0_m1           = ((comp_part->HV>>4)&0xF)-1;
				jpeginfo->v0_m1           = ((comp_part->HV)&0xF)-1;
				jpeginfo->dequant_q_map_0 = comp_part->T;
			}
			else if (i==1)
			{
				jpeginfo->h1_m1           = ((comp_part->HV>>4)&0xF)-1;
				jpeginfo->v1_m1           = ((comp_part->HV)&0xF)-1;
				jpeginfo->dequant_q_map_1 = comp_part->T;
			}
			else if (i==2)
			{
				jpeginfo->h2_m1           = ((comp_part->HV>>4)&0xF)-1;
				jpeginfo->v2_m1           = ((comp_part->HV)&0xF)-1;
				jpeginfo->dequant_q_map_2 = comp_part->T;
			}
			else if (i==3)
			{
				jpeginfo->h3_m1           = ((comp_part->HV>>4)&0xF)-1;
				jpeginfo->v3_m1           = ((comp_part->HV)&0xF)-1;
				jpeginfo->dequant_q_map_3 = comp_part->T;
			}
			else
			{
				DBG_PRINTF_ERR("ERROR in %s : Encountered color component %d.\n",__func__, comp_part->C);
				return 0;
			}
        }
        else
        {
            DBG_PRINTF_ERR("ERROR in %s : 'i' is out of range - i= %d.\n",__func__, i);
            return 0;
        }
    }

    // calculate the scale values
    max_h_m1 = 0;
    if(jpeginfo->h0_m1 > max_h_m1) max_h_m1 = jpeginfo->h0_m1;
    if(jpeginfo->h1_m1 > max_h_m1) max_h_m1 = jpeginfo->h1_m1;
    if(jpeginfo->h2_m1 > max_h_m1) max_h_m1 = jpeginfo->h2_m1;
    if(jpeginfo->h3_m1 > max_h_m1) max_h_m1 = jpeginfo->h3_m1;

    max_v_m1 = 0;
    if(jpeginfo->v0_m1 > max_v_m1) max_v_m1 = jpeginfo->v0_m1;
    if(jpeginfo->v1_m1 > max_v_m1) max_v_m1 = jpeginfo->v1_m1;
    if(jpeginfo->v2_m1 > max_v_m1) max_v_m1 = jpeginfo->v2_m1;
    if(jpeginfo->v3_m1 > max_v_m1) max_v_m1 = jpeginfo->v3_m1;

    if((max_h_m1+1)%(jpeginfo->h0_m1+1))
    {
        DBG_PRINTF_INFO("ERROR in %s : Non integer sscale for h0.\n",__func__);
        exit(EXIT_FAILURE);
    }
    if((max_h_m1+1)%(jpeginfo->h1_m1+1))
    {
        DBG_PRINTF_INFO("ERROR in %s : Non integer scale for h1.\n",__func__);
        exit(EXIT_FAILURE);
    }
    if((max_h_m1+1)%(jpeginfo->h2_m1+1))
    {
        DBG_PRINTF_INFO("ERROR in %s : Non integer scale for h2.\n",__func__);
        exit(EXIT_FAILURE);
    }
    if((max_h_m1+1)%(jpeginfo->h3_m1+1))
    {
        DBG_PRINTF_INFO("ERROR in %s : Non integer scale for h3.\n",__func__);
        exit(EXIT_FAILURE);
    }

    if((max_v_m1+1)%(jpeginfo->v0_m1+1))
    {
        DBG_PRINTF_INFO("ERROR in %s : Non integer scale for v0.\n",__func__);
        exit(EXIT_FAILURE);
    }
    if((max_v_m1+1)%(jpeginfo->v1_m1+1))
    {
        DBG_PRINTF_INFO("ERROR in %s : Non integer scale for v1.\n",__func__);
        exit(EXIT_FAILURE);
    }
    if((max_v_m1+1)%(jpeginfo->v2_m1+1))
    {
        DBG_PRINTF_INFO("ERROR in %s : Non integer scale for v2.\n",__func__);
        exit(EXIT_FAILURE);
    }
    if((max_v_m1+1)%(jpeginfo->v3_m1+1))
    {
        DBG_PRINTF_INFO("ERROR in %s : Non integer scale for v3.\n",__func__);
        exit(EXIT_FAILURE);
    }

    // ITU-T81 Section A.2.2 says monochrome images must be 8x8 regardless of H and V values
    if((jpeginfo->bytes_per_pixel == 1) && (jpeginfo->h0_m1>0 || jpeginfo->v0_m1>0))
    {
      DBG_PRINTF_INFO("Warning in %s:  Input image set subsampling for monochrome image."
               "  Reverting to no sub-sampling!\n",__func__);
        jpeginfo->h0_m1        = 0;
        jpeginfo->h1_m1        = 0;
        jpeginfo->h2_m1        = 0;
        jpeginfo->h3_m1        = 0;

        jpeginfo->v0_m1        = 0;
        jpeginfo->v1_m1        = 0;
        jpeginfo->v2_m1        = 0;
        jpeginfo->v3_m1        = 0;

        max_h_m1           = 0;
        max_v_m1           = 0;
    }

    // calculate how many mcuss
    // use the max_h and max_v to calc
    // make sure we have image dimensions
    if(jpeginfo->image_height == 0 || jpeginfo->image_width == 0)
    {
        DBG_PRINTF_ERR("ERROR in %s : Can't calculate num_mcu_m1 since image height or image_width is 0.\n",__func__);
        DBG_PRINTF_INFO("image_height = %d, image_width = %d.\n",jpeginfo->image_height, jpeginfo->image_width);
        exit(EXIT_FAILURE);
    }
    h_mcu = ((jpeginfo->image_width/8) / (max_h_m1+1));
    // if we have an odd sized image, might need an extra mcu
    if((h_mcu*(max_h_m1+1)*8) < jpeginfo->image_width)
    {
        h_mcu++;
    }

    v_mcu = ((jpeginfo->image_height/8) / (max_v_m1+1));
    // if we have an odd sized image, might need an extra mcu
    if((v_mcu*(max_v_m1+1)*8) < jpeginfo->image_height)
    {
        v_mcu++;
    }
    jpeginfo->num_mcu_m1 = (h_mcu * v_mcu) - 1;

    jpeginfo->mcu_aligned_width  = (h_mcu*(max_h_m1+1)*8);
    jpeginfo->mcu_aligned_height = (v_mcu*(max_v_m1+1)*8);

    // capture the mcu width & height
    jpeginfo->mcu_width = (max_h_m1+1)*8;
    jpeginfo->mcu_height = (max_v_m1+1)*8;

    return height_address;;
}

//--------------------------------------------------------------------------------
// parse_sos
//--------------------------------------------------------------------------------
static int _jpeghwd_parse_sos(JPEG_header_struct* jpeginfo, uint8_t *buff_ptr, uint32_t buff_len)
{
    SOS_struct *sos_part;
    SOS_comp_spec_struct *comp_part;
    int i;
    int found_bit;
    int bit_sel;
    int dc_sel[4];
    int ac_sel[4];

    for(i=0; i<4; i++)
    {
        dc_sel[i] = 0;
        ac_sel[i] = 0;
    }

    if (sizeof(SOS_struct) > buff_len)
    {
        DBG_PRINTF_ERR("ERROR in %s : buffer length < SOS_struct size!\n",__func__);
        return FAIL;
    }

    sos_part = (SOS_struct *)buff_ptr;

    if(sos_part->Ns > 4)
    {
        DBG_PRINTF_ERR("ERROR in %s : Support 4 color max (Nf <= 4)"
                 " (not supported Nf = %d).\n",__func__, sos_part->Ns);
        return FAIL;
    }

    for(i=0; i<sos_part->Ns; i++)
    {
        if ( (sizeof(SOS_struct) + (sizeof(SOS_comp_spec_struct)*i) + sizeof(SOS_comp_spec_struct))  > buff_len )
        {
            DBG_PRINTF_ERR("ERROR in %s : buffer length < SOF_comp_spec_struct size!\n",__func__);
            return FAIL;
        }

        comp_part = (SOS_comp_spec_struct *)(buff_ptr + sizeof(SOS_struct) +
                                             (sizeof(SOS_comp_spec_struct)*i));

        found_bit = 0;
        bit_sel   = 0;
        while ((found_bit == 0) && (bit_sel < 4))
        {
            if(jpeghwd_color_id[bit_sel] == comp_part->Cs)
            {
                found_bit = 1;
            }
            else
            {
                bit_sel++;
            }
        }

        if(bit_sel > 3)
        {
            DBG_PRINTF_ERR("ERROR in %s : in SOS, could not match color ID 0x%02x.\n",
                                           __func__, comp_part->Cs);
            return FAIL;
        }
        else
        {
            dc_sel[bit_sel] = (comp_part->TdTa>>4) & 1;
            ac_sel[bit_sel] = comp_part->TdTa & 1;
        }

        jpeginfo->huff_dc_table_sel =
                      (dc_sel[3]<<3) + (dc_sel[2]<<2) + (dc_sel[1]<<1) + dc_sel[0];
        jpeginfo->huff_ac_table_sel =
                      (ac_sel[3]<<3) + (ac_sel[2]<<2) + (ac_sel[1]<<1) + ac_sel[0];
    }

    return OK;
}

//--------------------------------------------------------------------------------
// parse_dqt
//--------------------------------------------------------------------------------
static int _jpeghwd_parse_dqt(JPEG_header_struct* jpeginfo, uint8_t *buff_ptr, uint32_t buff_len)
{
    DQT_struct *dqt_part;
    DQT_comp_spec_struct *comp_part;
    int i;
    int j;
    int table_num;
    int zz_i;
    int num_tables;
    int dqt_len;

    if (sizeof(DQT_struct) > buff_len)
    {
        DBG_PRINTF_ERR("ERROR in %s : buffer length < DQT_struct size!\n",__func__);
        return FAIL;
    }

    dqt_part = (DQT_struct *)buff_ptr;
    dqt_len  = (dqt_part->Lq_msb<<8) + dqt_part->Lq_lsb;
    // calculate number of dqt tables
    num_tables = ((dqt_len)-2)/65;

    DBG_PRINTF_INFO("*** DQT parsing ***\n");
    DBG_PRINTF_INFO("DQT Length = %d\n", dqt_len);
    DBG_PRINTF_INFO("DQT Num Tables = %d\n", num_tables);

    for(j=0; j<num_tables; j++)
    {
        if ( (sizeof(DQT_struct) + (sizeof(DQT_comp_spec_struct)*j) + sizeof(DQT_comp_spec_struct))  > buff_len )
        {
            DBG_PRINTF_ERR("ERROR in %s : buffer length < SOF_comp_spec_struct size!\n",__func__);
            return FAIL;
        }

        comp_part = (DQT_comp_spec_struct *)(buff_ptr + sizeof(DQT_struct) + (sizeof(DQT_comp_spec_struct)*j));
        if((comp_part->PqTq >> 4) > 4)
        {
            DBG_PRINTF_ERR("ERROR in %s : Only Pq == 0 (8 bit precision) supported.\n",__func__);
            return FAIL;
        }

        table_num = comp_part->PqTq & 0x3;
        DBG_PRINTF_INFO("DQT Table Num= %d\n", table_num);
        for(i=0; i<64; i++)
        {
           zz_i = _jpeghwd_calc_zz(i);
           jpeginfo->dequant_q_array[zz_i+(table_num*64)] = comp_part->Q[i];  // in zig-zag order in header
        }
    }

    return OK;
}

//--------------------------------------------------------------------------------
// parse_dht
//--------------------------------------------------------------------------------
static int _jpeghwd_parse_dht(JPEG_header_struct* jpeginfo, uint8_t *buff_ptr, uint32_t buff_len)
{
    DHT_struct *dht_part;
    DHT_comp_spec_struct *comp_part;
    uint32_t i,j;
    uint32_t table_i;
    uint32_t ptr_base;
    uint32_t ac_n_dc;
    uint32_t table_num;
    uint32_t len;
    uint32_t dht_len;
    uint32_t index;

    if (sizeof(DHT_struct) > buff_len)
    {
        DBG_PRINTF_ERR("ERROR in %s : buffer length < DHT_struct size!\n",__func__);
        return FAIL;
    }

    dht_part = (DHT_struct *)buff_ptr;
    dht_len  = (dht_part->Lh_msb<<8) + dht_part->Lh_lsb;

    DBG_PRINTF_INFO("*** DHT parsing ***\n");
    DBG_PRINTF_INFO("DHT Marker Len = %d\n", dht_len);

    index = sizeof(DHT_struct);
    while(index < dht_len)
    {
        if ( index + sizeof(DHT_comp_spec_struct)  > buff_len )
        {
            DBG_PRINTF_ERR("ERROR in %s : buffer length < SOF_comp_spec_struct size!\n",__func__);
            return FAIL;
        }

        comp_part = (DHT_comp_spec_struct *)(buff_ptr + index);

        table_num = comp_part->TcTh & 0xF;
        ac_n_dc   = (comp_part->TcTh>>4) & 1;
        DBG_PRINTF_INFO("DHT Table Num = %d\n", table_num);
        DBG_PRINTF_INFO("DHT ac_n_dc = %d\n", ac_n_dc);

        if(table_num > 1)
        {
            DBG_PRINTF_INFO("ERROR in %s : Only two AC/DC tables supported (not %d).\n",__func__,table_num);
            exit(EXIT_FAILURE);
        }

        // set the index into the array
        table_i = 0;  // index into table in header

        ptr_base = 0;
        if(table_num) ptr_base += 32;
        if(ac_n_dc)  ptr_base += 16;

        DBG_PRINTF_INFO("DHT ptr_base = %d\n", ptr_base);

        for(i=0; i<16; i++)
        {
            len = comp_part->Li[i];

            DBG_PRINTF_INFO("DHT len[%d] = %d\n", i, len);
            if(len > 0)
            {
                for(j=0; j<len; j++)
                {
                    DBG_PRINTF_INFO("DHT len %d  Entry %d = 0x%x "
                             " (table_i = %d)\n",
                             len, j, comp_part->V[table_i], table_i);

                    if (ac_n_dc)
                    {
                        jpeginfo->huff_ac_bits[table_num][i] = comp_part->Li[i];

                    	if (table_i < sizeof(jpeginfo->huff_ac_array[table_num]))
                    	{
                    		jpeginfo->huff_ac_array[table_num][table_i] = comp_part->V[table_i];
                    	}
                    }
                    else
                    {
                        jpeginfo->huff_dc_bits[table_num][i] = comp_part->Li[i];

                    	if (table_i < sizeof(jpeginfo->huff_dc_array[table_num]))
                    	{
                    		jpeginfo->huff_dc_array[table_num][table_i] = comp_part->V[table_i];
                    	}
                    }

                    table_i++;
                }
            }
        }

        // go to next table
        index += 17;  // {Tc, Th}, and 16 lengths
        for(i=0; i<16; i++)
            index += comp_part->Li[i];

    }

    return OK;
}

//--------------------------------------------------------------------------------
// parse_dri
//--------------------------------------------------------------------------------
static int _jpeghwd_parse_dri(JPEG_header_struct* jpeginfo, uint8_t *buff_ptr, uint32_t buff_len)
{
    DRI_struct *dri_part;
    uint16_t restart_interval;

    DBG_PRINTF_INFO("*** DRI parsing ***\n");

    if (sizeof(DRI_struct) > buff_len)
    {
        DBG_PRINTF_ERR("ERROR in %s : buffer length < DRI_struct size!\n",__func__);
        return FAIL;
    }

    dri_part = (DRI_struct *)buff_ptr;
    restart_interval = (dri_part->Ri_msb<<8) + dri_part->Ri_lsb;
    // restart interval > 0 means enabling restart intervals
    // restart interval of 0 means disabling restart intervals
    if(restart_interval > 0)
    {
        jpeginfo->restart_enable      = 1;
        jpeginfo->restart_interval_m1 = (restart_interval-1);
    }
    else
    {
        jpeginfo->restart_enable      = 0;
        jpeginfo->restart_interval_m1 = 0;
    }

    return OK;
}


int parse_ppm_header(FILE *fp, PPM_header_struct* ppminfo)
{
	char str[256] = {0};
	int fpos = FAIL;
	int n = 0;

    DBG_PRINTF_INFO("%s\n", __func__);

	if (ppminfo == NULL)
	{
		DBG_PRINTF_ERR("%s() ppminfo is NULL!\n", __func__);
		return FAIL;
	}

	_jpeghwc_init_info(ppminfo);

	if (fp == NULL)
	{
		DBG_PRINTF_ERR("%s() fmemopen failure!\n", __func__);
		return FAIL;
	}

	_jpeghw_get_token(fp, str, sizeof(str));

	if (strcmp(str, "P6") && strcmp(str, "P5"))
	{
		DBG_PRINTF_ERR("%s() NOT a valid PPM file!\n", __func__);
		return FAIL;
	}

	ppminfo->bytes_per_pixel = strcmp(str, "P6") ? 1 : 3;

	if (sscanf(_jpeghw_get_token(fp, str, sizeof(str)), "%d", &n) != 1)
	{
		DBG_PRINTF_ERR("%s() NOT a valid PPM file!\n", __func__);
		return FAIL;
	}

	ppminfo->image_width = n;

	if (sscanf(_jpeghw_get_token(fp, str, sizeof(str)), "%d", &n) != 1)
	{
		DBG_PRINTF_ERR("%s() NOT a valid PPM file!\n", __func__);
		return FAIL;
	}

	ppminfo->image_height = n;

	if (sscanf(_jpeghw_get_token(fp, str, sizeof(str)), "%d", &n) != 1)
	{
		DBG_PRINTF_ERR("%s() NOT a valid PPM file!\n", __func__);
		return FAIL;
	}

	if (n != 255)
	{
		DBG_PRINTF_ERR("%s() NOT 8-bit components - pnm max = %d!\n", __func__, n);
		return FAIL;
	}

	fpos = ftell(fp);

	return fpos;
}

int parse_jpeg_header(FILE *fp, JPEG_header_struct* jpeginfo, bool stop_after_sof)
{
    int fpos = 0;
    int byte_count;
    uint8_t buffer[16*1024];
    int buffer_len = sizeof(buffer);

    DBG_PRINTF_INFO("%s\n", __func__);

	if (jpeginfo == NULL)
	{
		DBG_PRINTF_ERR("%s() jpeginfo is NULL!\n", __func__);
		return FAIL;
	}

	_jpeghwd_init_info(jpeginfo);

	if (fp == NULL)
	{
		DBG_PRINTF_ERR("%s() invalid file handle!\n", __func__);
		return FAIL;
	}

	fread(buffer, 1, buffer_len, fp);
	fseek(fp, 0, SEEK_SET);

    // loop through the data
    while(fpos<buffer_len)
    {
        DBG_PRINTF_INFO("fpos=%d.\n", fpos);
        DBG_PRINTF_INFO("buffer[%d] = 0x%02x\n", fpos, buffer[fpos]);

        // if we might have found a header
        if(buffer[fpos] == 0xFF && buffer[fpos+1] != FILL_BYTE)
        {
            if (buffer[fpos+1] == EOI_BYTE)
            {
                DBG_PRINTF_INFO("EOI at offset 0x%x\n",fpos);
                fseek(fp, fpos, SEEK_SET);
                return fpos; // found EOI so done
            }
            else if (buffer[fpos+1] == SOI_BYTE  ||
                     buffer[fpos+1] == RST0_BYTE ||
                     buffer[fpos+1] == RST1_BYTE ||
                     buffer[fpos+1] == RST2_BYTE ||
                     buffer[fpos+1] == RST3_BYTE ||
                     buffer[fpos+1] == RST4_BYTE ||
                     buffer[fpos+1] == RST5_BYTE ||
                     buffer[fpos+1] == RST6_BYTE ||
                     buffer[fpos+1] == RST7_BYTE)
            {
                DBG_PRINTF_INFO("SOI or RST at offset 0x%x\n",fpos);
                fpos += 2;
            }
            else
            {
                DBG_PRINTF_INFO("fpos = %d\n", fpos);
                byte_count = (buffer[fpos+2]<<8)+buffer[fpos+3]+2;

                if(buffer[fpos+1] == SOS_BYTE)
                {
                    DBG_PRINTF_INFO("SOS at offset 0x%x\n",fpos);
                    _jpeghwd_parse_sos(jpeginfo, (uint8_t *)(buffer+fpos), (buffer_len-fpos));
                    fpos += byte_count;
                    jpeginfo->ecs_offset = fpos;
                    DBG_PRINTF_INFO("ecs_offset 0x%x (%d)\n", jpeginfo->ecs_offset, jpeginfo->ecs_offset);
                    fseek(fp, fpos, SEEK_SET);
                    return fpos;
                }
                else if(buffer[fpos+1] == SOF_BASELINE_BYTE)
                {
                    DBG_PRINTF_INFO("SOF at offset 0x%x\n",fpos);
                    _jpeghwd_parse_sof(jpeginfo, (uint8_t *)(buffer+fpos), (buffer_len-fpos));

					if(stop_after_sof)
					{
						fseek(fp, fpos, SEEK_SET);
						return fpos;
					}
                }
                else if(((buffer[fpos+1] & SOF_OTHER_MASK) == SOF_OTHER_BYTE)
                         && (buffer[fpos+1] != DHT_BYTE))
                {
                    DBG_PRINTF_INFO("Illegal SOF of 0xFF%02x!!!!!\n", buffer[fpos+1]);
                    return FAIL;
                }
                else if(buffer[fpos+1] == DQT_BYTE)
                {
                    DBG_PRINTF_INFO("DQT at offset 0x%x\n",fpos);
                    _jpeghwd_parse_dqt(jpeginfo, (uint8_t *)(buffer+fpos), (buffer_len-fpos));
                }
                else if (buffer[fpos+1] == DHT_BYTE)
                {
                    DBG_PRINTF_INFO("DHT at offset 0x%x\n",fpos);
                    _jpeghwd_parse_dht(jpeginfo, (uint8_t *)(buffer+fpos), (buffer_len-fpos));
                }
                else if (buffer[fpos+1] == DRI_BYTE)
                {
                    DBG_PRINTF_INFO("DRI at offset 0x%x\n",fpos);
                    _jpeghwd_parse_dri(jpeginfo, (uint8_t *)(buffer+fpos), (buffer_len-fpos));
                }
                else
                {
                    DBG_PRINTF_INFO("Skipping 0xFF%02x header of %d bytes at offset %d.\n",
                             buffer[fpos+1], byte_count,fpos);
                }
                fpos += byte_count;
            }
        }
        else
        {
        	fpos += 1;
        }
    }

    return FAIL;
}


static int parse_jpeg_height_offset(FILE *fp, JPEG_header_struct* jpeginfo)
{
    int fpos = 0;
    int byte_count;
    uint8_t buffer[16*1024];
    int buffer_len = sizeof(buffer);
    int height_offset = FAIL;

    DBG_PRINTF_INFO("%s\n", __func__);

	if (jpeginfo == NULL)
	{
		DBG_PRINTF_ERR("%s() jpeginfo is NULL!\n", __func__);
		return FAIL;
	}

	_jpeghwd_init_info(jpeginfo);

	fread(buffer, 1, buffer_len, fp);
	fseek(fp, 0, SEEK_SET);

    // loop through the data
    while(fpos<buffer_len)
    {
        DBG_PRINTF_INFO("fpos=%d.\n", fpos);
        DBG_PRINTF_INFO("buffer[%d] = 0x%02x\n", fpos, buffer[fpos]);

        // if we might have found a header
        if(buffer[fpos] == 0xFF && buffer[fpos+1] != FILL_BYTE)
        {
            if (buffer[fpos+1] == EOI_BYTE)
            {
                DBG_PRINTF_INFO("EOI at offset 0x%x\n",fpos);
                break;
            }
            else if (buffer[fpos+1] == SOI_BYTE  ||
                     buffer[fpos+1] == RST0_BYTE ||
                     buffer[fpos+1] == RST1_BYTE ||
                     buffer[fpos+1] == RST2_BYTE ||
                     buffer[fpos+1] == RST3_BYTE ||
                     buffer[fpos+1] == RST4_BYTE ||
                     buffer[fpos+1] == RST5_BYTE ||
                     buffer[fpos+1] == RST6_BYTE ||
                     buffer[fpos+1] == RST7_BYTE)
            {
                DBG_PRINTF_INFO("SOI or RST at offset 0x%x\n",fpos);
                fpos += 2;
            }
            else
            {
                DBG_PRINTF_INFO("fpos = %d\n", fpos);
                byte_count = (buffer[fpos+2]<<8)+buffer[fpos+3]+2;

                if(buffer[fpos+1] == SOS_BYTE)
                {
                    DBG_PRINTF_INFO("SOS at offset 0x%x\n",fpos);
                    _jpeghwd_parse_sos(jpeginfo, (uint8_t *)(buffer+fpos), (buffer_len-fpos));
                    fpos += byte_count;
                    jpeginfo->ecs_offset = fpos;
                    DBG_PRINTF_INFO("ecs_offset 0x%x (%d)\n", jpeginfo->ecs_offset, jpeginfo->ecs_offset);
                    fseek(fp, fpos, SEEK_SET);
                    height_offset = fpos;
                }
                else if(buffer[fpos+1] == SOF_BASELINE_BYTE)
                {
                    DBG_PRINTF_INFO("SOF at offset 0x%x\n",fpos);
                    uint8_t* offset_address = _jpeghwd_parse_sof(jpeginfo, (uint8_t *)(buffer+fpos), (buffer_len-fpos));
                    if (offset_address)
                    {
                        fseek(fp, offset_address-buffer, SEEK_SET);
                    	height_offset = offset_address-buffer;
                    	break;
                    }
                }
                else if(((buffer[fpos+1] & SOF_OTHER_MASK) == SOF_OTHER_BYTE)
                         && (buffer[fpos+1] != DHT_BYTE))
                {
                    DBG_PRINTF_INFO("Illegal SOF of 0xFF%02x!!!!!\n", buffer[fpos+1]);
                    return FAIL;
                }
                else if(buffer[fpos+1] == DQT_BYTE)
                {
                    DBG_PRINTF_INFO("DQT at offset 0x%x\n",fpos);
                    _jpeghwd_parse_dqt(jpeginfo, (uint8_t *)(buffer+fpos), (buffer_len-fpos));
                }
                else if (buffer[fpos+1] == DHT_BYTE)
                {
                    DBG_PRINTF_INFO("DHT at offset 0x%x\n",fpos);
                    _jpeghwd_parse_dht(jpeginfo, (uint8_t *)(buffer+fpos), (buffer_len-fpos));
                }
                else if (buffer[fpos+1] == DRI_BYTE)
                {
                    DBG_PRINTF_INFO("DRI at offset 0x%x\n",fpos);
                    _jpeghwd_parse_dri(jpeginfo, (uint8_t *)(buffer+fpos), (buffer_len-fpos));
                }
                else
                {
                    DBG_PRINTF_INFO("Skipping 0xFF%02x header of %d bytes at offset %d.\n",
                             buffer[fpos+1], byte_count,fpos);
                }
                fpos += byte_count;
            }
        }
        else
        {
        	fpos += 1;
        }
    }

    return height_offset;
}


static void jpeg_write_marker(FILE *fp, uint8_t marker, uint16_t length)
{
	fputc(HEADER_BYTE, fp);
	fputc(marker, fp);

	if (length)
	{
		uint8_t highByte = (uint8_t)((length>>8) & 0xFF);
		uint8_t lowByte = (uint8_t)(length & 0xFF);

		fputc(highByte, fp);
		fputc(lowByte, fp);
	}
}

static void jpeg_write_header_section(FILE *fp, uint8_t marker, struct jpeghw_compress_struct *cinfo)
{
	switch(marker)
	{
	case SOI_BYTE:
		{
			jpeg_write_marker(fp, SOI_BYTE, 0);
		}
		break;

	case JFIF_BYTE:
		{
			uint32_t length = 2 + 5 + 2 + 1 + 2 + 2 + 1 + 1 + 0;
			uint8_t JifiHead[5] = {'J' , 'F' , 'I' , 'F' , 0x00};
			uint8_t ver_major = 0x01;
			uint8_t ver_minor = 0x02;
			uint8_t density_units = 0x01;
			uint8_t x_density_msb = (uint8_t)((cinfo->common.image_width>>8) & 0xFF);
			uint8_t x_density_lsb = (uint8_t)(cinfo->common.image_width & 0xFF);
			uint8_t y_density_msb = (uint8_t)((cinfo->common.image_height>>8) & 0xFF);
			uint8_t y_density_lsb = (uint8_t)(cinfo->common.image_height & 0xFF);

			jpeg_write_marker(fp, JFIF_BYTE, length);
			fwrite(JifiHead, 1, sizeof(JifiHead), fp);

			fputc(ver_major, fp); // Version Major
			fputc(ver_minor, fp); // Version Minor

			fputc(density_units, fp); // Density Units
			fputc(x_density_msb, fp); // Horizontal pixel density msb
			fputc(x_density_lsb, fp); // Horizontal pixel density lsb
			fputc(y_density_msb, fp); // Vertical pixel density msb
			fputc(y_density_lsb, fp); // Vertical pixel density lsb

			fputc(0x00, fp); // Thumbnail width
			fputc(0x00, fp); // Thumbnail height
		}
		break;

	case DQT_BYTE:
		{
			uint8_t quantization_table[256];
			uint32_t table_size = sizeof(quantization_table);

			memset(quantization_table, 0, table_size);
			if (jpeghw_get_quant_table(&cinfo->common, cinfo->quality, &quantization_table[0], &table_size) == e_JPEGHW_SUCCESS)
			{
				int i;
				int table_num = (cinfo->common.bytes_per_pixel == 1) ? 1 : 2;
				uint16_t length = ((table_num * (1+64)) + 2);

				jpeg_write_marker(fp, DQT_BYTE, length);

				for (i=0; i<table_num; i++)
				{
					DQT_comp_spec_struct dqt_comp;
					int offset = 64*i;

					dqt_comp.PqTq = i;
					memcpy(dqt_comp.Q, &quantization_table[offset+i], sizeof(dqt_comp.Q));
					fwrite(&dqt_comp, 1, sizeof(dqt_comp), fp);
				}

			}

		}
		break;

	case SOF_BASELINE_BYTE:
		{
			int i;
			uint8_t precision = 8; // 8-bit sample precision
			uint8_t lines_msb = (uint8_t)((cinfo->common.image_height>>8) & 0xFF);
			uint8_t lines_lsb = (uint8_t)(cinfo->common.image_height & 0xFF);
			uint8_t samples_msb = (uint8_t)((cinfo->common.image_width>>8) & 0xFF);
			uint8_t samples_lsb = (uint8_t)(cinfo->common.image_width & 0xFF);
			uint8_t component_num = (uint8_t)(cinfo->common.bytes_per_pixel);
			uint16_t length = (component_num * 3) + 8;

			jpeg_write_marker(fp, SOF_BASELINE_BYTE, length);

			fputc(precision, fp); // sample precision

			fputc(lines_msb, fp); // number of lines
			fputc(lines_lsb, fp);

			fputc(samples_msb, fp); // number of samples per line
			fputc(samples_lsb, fp);

			fputc(component_num, fp); // number of image componnents in frame

			if (component_num == 1)
			{
				SOF_comp_spec_struct sof_comp;

				sof_comp.C = 0x01;
				sof_comp.HV = 0x11;
				sof_comp.T = 0x00;

				fwrite(&sof_comp, 1, sizeof(sof_comp), fp);
			}
			else
			{
				for (i=0; i<component_num; i++)
				{
					SOF_comp_spec_struct sof_comp;

					sof_comp.C = i+1;

					switch (i)
					{
					case 0:
						sof_comp.HV = 0x22;
						sof_comp.T = 0x00;
						break;

					case 1:
						sof_comp.HV = 0x11;
						sof_comp.T = 0x01;
						break;

					case 2:
						sof_comp.HV = 0x11;
						sof_comp.T = 0x01;
						break;

					default:
						sof_comp.HV = 0;
						sof_comp.T = 0;
						break;
					}

					fwrite(&sof_comp, 1, sizeof(sof_comp), fp);
				}
			}
		}
		break;

	case DHT_BYTE:
		{
			int i;
			int table_num = (cinfo->common.bytes_per_pixel == 1 ? 1 : 2);
			int table_length = 0;
			uint8_t bits[16];
			uint8_t vals[384];
			uint32_t bits_size, vals_size;

			// get total table length
			for (i=0; i<table_num; i++)
			{
				// DC Table
				bits_size = sizeof(bits);
				vals_size = sizeof(vals);
				if (jpeghw_get_huff_table(&cinfo->common, i, false, NULL, &bits_size, NULL, &vals_size)
						== e_JPEGHW_SUCCESS);
				{
					table_length += (1 + (bits_size + vals_size));
				}

				// AC Table
				bits_size = sizeof(bits);
				vals_size = sizeof(vals);
				if (jpeghw_get_huff_table(&cinfo->common, i, true, NULL, &bits_size, NULL, &vals_size)
						== e_JPEGHW_SUCCESS);
				{
					table_length += (1 + (bits_size + vals_size));
				}
			}

			jpeg_write_marker(fp, DHT_BYTE, (table_length+2));

			// get lengths
			for (i=0; i<table_num; i++)
			{
				uint8_t TcTh;

				// DC Table
				bits_size = sizeof(bits);
				vals_size = sizeof(vals);
				if (jpeghw_get_huff_table(&cinfo->common, i, false, &bits[0], &bits_size, &vals[0], &vals_size)
						== e_JPEGHW_SUCCESS);
				{
					TcTh = (0x00 | (i & 0x0F)) ;

					fputc(TcTh, fp);
					fwrite(&bits, 1, bits_size, fp);
					fwrite(&vals, 1, vals_size, fp);
				}

				// AC Table
				bits_size = sizeof(bits);
				vals_size = sizeof(vals);
				if (jpeghw_get_huff_table(&cinfo->common, i, true, &bits[0], &bits_size, &vals[0], &vals_size)
						== e_JPEGHW_SUCCESS);
				{
					TcTh = (0x10 | (i & 0x0F)) ;

					fputc(TcTh, fp);
					fwrite(&bits, 1, bits_size, fp);
					fwrite(&vals, 1, vals_size, fp);
				}
			}
		}
		break;

	case SOS_BYTE:
		{
			uint8_t i;
			uint8_t component_num = (uint8_t)cinfo->common.bytes_per_pixel;
			uint32_t length = 3 + (component_num * 2) + 3;

			jpeg_write_marker(fp, SOS_BYTE, length);
			fputc(component_num, fp); // Ns

			for (i=0; i<component_num; i++)
			{
				uint8_t TdTa = (i == 0 ? 0x00 : 0x11);

				fputc(i+1, fp);  // Cs
				fputc(TdTa, fp); // Td Ta
			}

			fputc(0x00, fp); // Ss
			fputc(0x3F, fp); // Se
			fputc(0x00, fp); // AhAl
		}
		break;

	default:
		// skipping
		break;
	}
}


void generate_ppm_header(FILE *fp, struct jpeghw_decompress_struct* dinfo)
{
	char header[256];

	if (fp == NULL || dinfo == NULL)
	{
		return;
	}

	sprintf(header, "%s\n%d %d\n255\n",
			(dinfo->common.bytes_per_pixel == 1 ? "P5" : "P6"),
			dinfo->common.image_width,
			dinfo->common.image_height);

	int len = strlen(header);
	fwrite(header, 1, len, fp);
}

void generate_jpeg_header(FILE *fp, struct jpeghw_compress_struct* cinfo)
{
	if (cinfo && fp)
	{
		jpeg_write_header_section(fp, SOI_BYTE, cinfo);
		jpeg_write_header_section(fp, JFIF_BYTE, cinfo);
		jpeg_write_header_section(fp, DQT_BYTE, cinfo);
		jpeg_write_header_section(fp, SOF_BASELINE_BYTE, cinfo);
		jpeg_write_header_section(fp, DHT_BYTE, cinfo);
		jpeg_write_header_section(fp, SOS_BYTE, cinfo);
	}
}


static uint16_t swap_uint16(uint16_t val) {
	return (val << 8) | (val >> 8);
}

int page_jpeg_fixup_page_length(FILE *dst_file, uint32_t lines_encoded)
{
	JPEG_header_struct jpeginfo;

	int fpos = parse_jpeg_height_offset(dst_file, &jpeginfo);
	if (fpos != FAIL) {
		uint16_t current_lines;
		uint16_t little_endian_lines;

		if (fpos > 0) {
			fseek(dst_file, fpos, SEEK_SET);

			if (sizeof(little_endian_lines)
					== fread(&little_endian_lines, 1,
							sizeof(little_endian_lines), dst_file)) {
				current_lines = swap_uint16(little_endian_lines);

				printf("%s() update %u->%u lines\n", __func__,
						(unsigned int)current_lines, (unsigned int)lines_encoded);

				little_endian_lines = swap_uint16(lines_encoded);

				fseek(dst_file, fpos, SEEK_SET);
				fwrite(&little_endian_lines, 1,
						sizeof(little_endian_lines), dst_file);
			}
		}
	}

	return lines_encoded;
}



/* FIFO routines */
void testFifo_Init(testFifo* pFifoHead, uint32_t maxFifoCount)
{
    // check for NULL FIFO head
	DBG_ASSERT(pFifoHead);

	if (pFifoHead == NULL)
		return;

    if (pthread_mutex_init( &pFifoHead->mutex, NULL ) != 0)
    {
    	// DBG_ASSERT - failed to init mutex
    	DBG_ASSERT(false);
    }

    ATInitList(&pFifoHead->list);

    pFifoHead->maxCount = maxFifoCount;

    sem_init(&pFifoHead->fill_sem, 0, 0);
    if (pFifoHead->maxCount)
    {
        sem_init(&pFifoHead->empty_sem, 0, pFifoHead->maxCount);
    }
}

void testFifo_Destroy(testFifo* pFifoHead)
{
    // check for NULL FIFO head
	DBG_ASSERT(pFifoHead);

    if (!testFifo_IsEmpty(pFifoHead))
    {
        // DBG_ASSERT - should only destroy empty FIFO lists
    	DBG_ASSERT(false);

        do {
   			posix_sleep_us(10);
        } while (!testFifo_IsEmpty(pFifoHead));
    }

    pthread_mutex_destroy(&pFifoHead->mutex);
    sem_destroy(&pFifoHead->fill_sem);
    if (pFifoHead->maxCount)
    {
        sem_destroy(&pFifoHead->empty_sem);
        pFifoHead->maxCount = 0;
    }
}

int testFifo_Add(testFifo* pFifoHead, void* data, bool wait)
{
    int ret = 0;

    // check for NULL FIFO head
    DBG_ASSERT(pFifoHead);

    if (data == NULL)
    	return -1;

    if (pFifoHead->maxCount)
    {
        if (wait)
        {
            ret = sem_wait(&pFifoHead->empty_sem);
        }
        else
        {
            ret = sem_trywait(&pFifoHead->empty_sem);
        }
    }

    if (ret == 0)
    {
        testFifoRecord* record = (testFifoRecord*)MEM_MALLOC(sizeof(testFifoRecord));

        if (record)
        {
        	record->data = data;

			if (pthread_mutex_lock( &pFifoHead->mutex ) == 0)
			{
				ATInsertHeadList(&pFifoHead->list, &record->list_node);
				pthread_mutex_unlock( &pFifoHead->mutex );

				sem_post(&pFifoHead->fill_sem);
			}
			else
			{
				MEM_FREE_AND_NULL(record);
				ret = -1;
			}
        }
    }

    return ret;
}

void* testFifo_Remove(testFifo* pFifoHead, bool wait)
{
    int ret = 1;
    void *data = NULL;

    // check for NULL FIFO head
    DBG_ASSERT(pFifoHead);

    if (wait)
    {
    	ret = sem_wait(&pFifoHead->fill_sem);
    }
    else
    {
    	ret = sem_trywait(&pFifoHead->fill_sem);
    }

    if (ret == 0)
    {
        testFifoRecord *record;
        ATLISTENTRY *pNode;

        if (pthread_mutex_lock( &pFifoHead->mutex ) == 0)
        {
            pNode = ATRemoveTailList(&pFifoHead->list);
            pthread_mutex_unlock( &pFifoHead->mutex );

            if (pNode != NULL)
            {
				record = CONTAINING_RECORD(pNode, testFifoRecord, list_node);
				if (record != NULL)
				{
					data = record->data;

					MEM_FREE_AND_NULL(pNode);
				}
			}

            if (pFifoHead->maxCount)
            {
                sem_post(&pFifoHead->empty_sem);
            }
        }
        else
        {
        	data = NULL;
        }
    }

    return data;
}

bool testFifo_IsEmpty(testFifo* pFifoHead)
{
    bool result = false;

    // check for NULL FIFO head
    DBG_ASSERT(pFifoHead);

	if (pthread_mutex_lock( &pFifoHead->mutex ) == 0)
	{
		result = ATIsListEmpty(&pFifoHead->list);
		pthread_mutex_unlock( &pFifoHead->mutex );
	}

    return result;
}
