/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2015, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/



#ifndef _JPEGHW_API_H
#define _JPEGHW_API_H

#include <stdio.h>
#include <stdint.h>
#include <stdbool.h>
#include <string.h>

#include "dma_buffer.h"

#define JPEGHW_LIB_VERSION 10 /* version 1.0 */

/* JPEGHW configuration flags */
#define JPEGHW_CONFIG_FLAG_TRIM						0x00000001  // force trimming on width and height
#define JPEGHW_CONFIG_FLAG_MCU_WIDTH_ALIGNMENT 		0x00000002  // force MCU width alignment
#define JPEGHW_CONFIG_FLAG_MCU_HEIGHT_ALIGNMENT 	0x00000004  // force MCU height alignment
#define JPEGHW_CONFIG_FLAG_BLOCK_JPEG_CORE_AVAIL 	0x00000008  // blocking wait for available JPEG core
#define JPEGHW_CONFIG_FLAG_INPUT_YUV444			 	0x00000010  // Let the input data source be YUV444. (YCbCr instead of RGB888)

/* Error types - used for reporting errors.  Handling/checking for errors
 * should be by checking for 'success' or 'not success', for example:
 *
 *   if (function() == e_JPEGHW_SUCCESS) {
 *   	// no error
 *   }
 *
 * or:
 *
 *   if (err = function() != e_JPEGHW_SUCCESS) {
 *     // handle err
 *   }
 *
 */
typedef enum
{
    e_JPEGHW_SUCCESS = 				 0,
    e_JPEGHW_ERROR = 				-1,
    e_JPEGHW_ERR_INVALID_PARAMETERS = 	  -2,
    e_JPEGHW_ERR_BAD_VERSION = 			-3,
    e_JPEGHW_ERR_BAD_STRUCT_SIZE = 		-4,
    e_JPEGHW_ERR_BAD_OBJECT = 			-5,
    e_JPEGHW_ERR_BAD_STATE = 			-6,
    e_JPEGHW_ERR_NOT_AVAILABLE = 		-7,
    e_JPEGHW_ERR_DATA_OVERRUN = 		-8,
    e_JPEGHW_ERR_OUT_OF_MEMORY = 		-9,
    e_JPEGHW_ERR_BUFFER_TOO_SMALL = 	-10,
    e_JPEGHW_ERR_TIMEOUT = 	-11,
} jpeghw_error_type_t;

/* Object types */
typedef enum
{
    e_JPEGHW_OBJ_COMPRESS = 1,
    e_JPEGHW_OBJ_DECOMPRESS,
} jpeghw_object_type_t;

/* State types */
typedef enum
{
    e_JPEGHW_STATE_UNKNOWN = 0,
    e_JPEGHW_STATE_ERROR,
    e_JPEGHW_STATE_ABORTED,

    e_JPEGHW_CSTATE_START,
    e_JPEGHW_CSTATE_SCANNING,

    e_JPEGHW_DSTATE_START,
    e_JPEGHW_DSTATE_READY,
    e_JPEGHW_DSTATE_SCANNING,
    e_JPEGHW_DSTATE_STOPPING,
} jpeghw_state_type_t;


struct jpeghw_common_struct;
struct jpeghw_compress_struct;
struct jpeghw_decompress_struct;
struct jpeghw_jpeg_header_struct;


typedef void (*jpeghw_error_exit_t)(struct jpeghw_common_struct * cinfo);
typedef void (*jpeghw_error_message_t)(struct jpeghw_common_struct * cinfo, char * buffer);
typedef void (*jpeghw_error_reset_t)(struct jpeghw_common_struct * cinfo);

struct jpeghw_error_mgr {
	jpeghw_error_exit_t error_exit;
	jpeghw_error_reset_t error_reset;

	jpeghw_error_type_t err_code;
};


/* Object for compression */
typedef void  (*jpeghw_compress_init_t)(struct jpeghw_compress_struct * cinfo);
typedef jpeghw_error_type_t (*jpeghw_compress_quant_table_t)(struct jpeghw_compress_struct * cinfo, uint8_t quality, uint8_t *table, uint32_t *size);
typedef jpeghw_error_type_t (*jpeghw_compress_huff_table_t)(struct jpeghw_compress_struct * cinfo, uint8_t table_index, bool ac, uint8_t *bits, uint32_t *bits_size, uint8_t *val, uint32_t *val_size);
typedef bool  (*jpeghw_compress_output_buffer_t)(struct jpeghw_compress_struct * cinfo, struct BigBuffer_s* output_buffer, uint32_t bytes_in_buffer );
typedef struct BigBuffer_s*  (*jpeghw_compress_get_buffer_t)(struct jpeghw_compress_struct * cinfo );
typedef void  (*jpeghw_compress_term_t)(struct jpeghw_compress_struct * cinfo);

struct jpeghw_compression_mgr {
	jpeghw_compress_quant_table_t get_quant_table;
	jpeghw_compress_huff_table_t get_huff_table;

	jpeghw_compress_init_t init;
	jpeghw_compress_output_buffer_t send_output_buffer;
	jpeghw_compress_get_buffer_t get_output_buffer;
	jpeghw_compress_term_t term;
};

/* Object for decompression */
typedef jpeghw_error_type_t (*jpeghw_decompress_jpeg_info_t)(struct jpeghw_decompress_struct * dinfo, struct jpeghw_jpeg_header_struct *jinfo);
typedef void (*jpeghw_decompress_init_t)(struct jpeghw_decompress_struct * dinfo);
typedef struct BigBuffer_s* (*jpeghw_decompress_input_buffer_t)(struct jpeghw_decompress_struct * dinfo, uint32_t *bytes_in_buffer);
typedef void (*jpeghw_decompress_term_t)(struct jpeghw_decompress_struct * dinfo);

struct jpeghw_decompression_mgr {
	jpeghw_decompress_jpeg_info_t get_jpeg_info;

	jpeghw_decompress_init_t init;
	jpeghw_decompress_input_buffer_t get_input_buffer;
	jpeghw_decompress_term_t term;
};


typedef void (*jpeghw_progress_monitor_t)(struct jpeghw_common_struct * info);
typedef bool (*jpeghw_scanline_timeout_t)(struct jpeghw_common_struct * info, uint32_t time_in_seconds);

struct jpeghw_common_struct {
	/* These fields must be filled in by the application
	 * before calling jpeghw_start_(de)compress()
	 */
	struct jpeghw_error_mgr *err; /* error handling (optional) */
	jpeghw_progress_monitor_t progress_monitor; /* progress monitor (NULL if none) */
	jpeghw_scanline_timeout_t scanline_timeout; /* scanline timeout handling (optional) */

	/* Image description
	 *
	 * These fields that must be filled in by application before calling function
	 * jpeghw_set_defaults().
	 *
	 * For decompression, these fields are typically filled in by the application
	 * after reading the JPEG source header, and these values can also be used by
	 * the application to decide how to process the image.
	 */
	uint32_t 		image_width; /* width of the JPEG image */
	uint32_t 		image_height; /* height of the JPEG image */
	int				bytes_per_pixel; /* # of color components in the JPEG image */

	uint16_t 		mcu_width;  /* must be 8 or 16 */
	uint16_t 		mcu_height; /* must be 8 or 16 */

	/* available for use by the application */
	void * client_context;

	/* The application may examine these fields but must not modify them.
	 */
	jpeghw_object_type_t type; /* type of object (compression/decompression) */
	jpeghw_state_type_t global_jpeg_state; /* keeps track of the current state while compression/decompression of the JPEG */
	jpeghw_error_type_t last_error; /* store last error */
	uint32_t			config_flags; /* JPEGHW configuration flags */

	/* Remaining fields should not be touched by the application.
	 */
	void * jpeghw_context;
};


struct jpeghw_compress_struct {
	struct jpeghw_common_struct common; /* NOTE: this must be defined first */

	/* Compress data manager (required)
	 * Must be filled in by application before
	 * calling function jpeghw_start_compress()
	 */
	struct jpeghw_compression_mgr *cmgr;

	/* Compression parameters - fields that must be set before calling
	 * jpeghw_start_compress(). It is recommended that the application first
	 * call jpeghw_set_defaults() to ensure everything is initialized to a
	 * reasonable default before the application makes any specific changes
	 * to the following fields.
	 */
	uint8_t			quality; /* JPEG image quality percentage */
	bool 			auto_generate_jpeg_header; /* have the hardware generate the jpeg header */

	/* State variables: these variables indicate the progress of compression.
	 * The application may examine these but must not modify them.
	 */
	uint32_t 		in_scanlines; /* 1 .. input_height */
	uint32_t		in_bytes; /* number of bytes input so far */
};


struct jpeghw_decompress_struct {
	struct jpeghw_common_struct common; /* NOTE: this must be defined first */

	/* Decompress data manager (required)
	 * Must be filled in by application before
	 * calling function jpeghw_start_decompress()
	 */
	struct jpeghw_decompression_mgr *dmgr; /* decompress data manager (required) */

	/* State variables: these variables indicate the progress of decompression.
	 * The application may examine these but must not modify them.
	 */
	uint32_t 		out_scanlines; /* 0 .. (out_height-1) */
	uint32_t		out_bytes; /* number of bytes output so far */
};

/* JPEG header for decompression */
struct jpeghw_jpeg_header_struct
{
	uint32_t 	image_width;
	uint32_t 	image_height;
	uint32_t 	bytes_per_pixel;
	uint32_t	mcu_width;
	uint32_t	mcu_height;
	uint32_t 	mcu_aligned_width;
	uint32_t 	mcu_aligned_height;

	uint32_t   	ecs_offset;

	uint32_t	h0_m1;
	uint32_t	v0_m1;
	uint32_t	h1_m1;
	uint32_t	v1_m1;
	uint32_t	h2_m1;
	uint32_t	v2_m1;
	uint32_t	h3_m1;
	uint32_t	v3_m1;
	uint32_t	num_mcu_m1;

    uint32_t	restart_enable;
    uint32_t	restart_interval_m1;

    uint32_t	dequant_q_map_0;
    uint32_t	dequant_q_map_1;
    uint32_t	dequant_q_map_2;
    uint32_t	dequant_q_map_3;

    uint32_t	huff_dc_table_sel;
    uint32_t	huff_ac_table_sel;
    uint8_t 	huff_dc_bits [2][16];
    uint8_t 	huff_dc_array [2][12];
    uint8_t 	huff_ac_bits [2][16];
    uint8_t 	huff_ac_array [2][162];
    uint8_t 	dequant_q_array [256];
};



/* Initialize and terminate jpeg session */
void jpeghw_init();
void jpeghw_terminate();

/* common library API functions */
jpeghw_error_type_t jpeghw_set_defaults (struct jpeghw_common_struct * info);
struct jpeghw_error_mgr* jpeghw_std_error (struct jpeghw_error_mgr * err);
jpeghw_error_type_t jpeghw_get_quant_table(struct jpeghw_common_struct * info, uint8_t quality, uint8_t *table, uint32_t *size);
jpeghw_error_type_t jpeghw_get_huff_table(struct jpeghw_common_struct * info, int table_index, bool ac, uint8_t *bits, uint32_t *bits_size, uint8_t *val, uint32_t *val_size);

/* Initialization of JPEG (de)compression objects */
jpeghw_error_type_t jpeghw_create_compress(struct jpeghw_compress_struct * cinfo, uint32_t flags);
jpeghw_error_type_t jpeghw_create_decompress(struct jpeghw_decompress_struct * dinfo, uint32_t flags);

/* Destruction of JPEG (de)compression objects */
void jpeghw_destroy_compress(struct jpeghw_compress_struct * cinfo);
void jpeghw_destroy_decompress(struct jpeghw_decompress_struct * dinfo);

/* Abort an incomplete (de)compression operation if there is a
 * need to retain the JPEG object.  NOTE: If there is no need to
 * retain the JPEG object then one can simply abort and release it
 * by calling jpeg_destroy_(de)compress()
 */
jpeghw_error_type_t jpeghw_abort_compress(struct jpeghw_compress_struct * cinfo);
jpeghw_error_type_t jpeghw_abort_decompress(struct jpeghw_decompress_struct * dinfo);

/* Main functions for JPEG compression operations */
jpeghw_error_type_t jpeghw_start_compress(struct jpeghw_compress_struct * cinfo);
uint32_t jpeghw_write_scanlines(struct jpeghw_compress_struct * cinfo, struct BigBuffer_s* scanlines, uint32_t num_lines, bool eoi);
jpeghw_error_type_t jpeghw_finish_compress(struct jpeghw_compress_struct * cinfo);

/* Main functions for JPEG decompression operations */
jpeghw_error_type_t jpeghw_start_decompress(struct jpeghw_decompress_struct * dinfo);
uint32_t jpeghw_read_scanlines(struct jpeghw_decompress_struct * dinfo, struct BigBuffer_s* scanlines, bool *done);
jpeghw_error_type_t jpeghw_finish_decompress(struct jpeghw_decompress_struct * dinfo);

#endif  // _JPEGHW_API_H
