/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014-2015, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/



#include <string.h>

#include "dma_buffer.h"
#include "jhw_dev.h"
#include "jhw_dma.h"
#include "jhw_uio.h"


int jhwd_destroy_odma_list(struct jpeghw_decompress_struct * jhwdinfo, bool wait_for_complete)
{
    DBG_INFO("[%d]%s\n",gettid(), __func__);
    struct jpeg_dev_info_s * dev = JHWD_GET_DEV(jhwdinfo);

    jhw_destroy_dma_descriptors(dev->odma_desc_list, dev->number_odma_descriptors);

    return OK;
}

int _jhwd_wait_for_odma(struct jpeghw_decompress_struct * jhwdinfo)
{
   struct jpeg_dev_info_s * dev = JHWD_GET_DEV(jhwdinfo);
   jpeg_event_t jpeg_event;
   // if the timeout callback is NULL the user does not want to manage timeouts
   // therefore set the timeout to 10 seconds
   // otherwise call the user every second while we are waiting
   uint32_t timeout_in_seconds = (jhwdinfo->common.scanline_timeout == NULL) ? 
                  DEFAULT_SCANLINE_TIMEOUT : 1; // default seconds if no timeout callback
   uint32_t timeout_count = 0;

   do
   {
       DBG_INFO("[%d]--->>wait for omsg mqd:%d for %d seconds...\n",
                                gettid(), dev->omqd.mqd, timeout_in_seconds);

       jpeg_event.event_flags = 0;
       int rv = posix_wait_for_message(dev->omqd.mqd, (char *)&jpeg_event,
               sizeof(jpeg_event_t), (timeout_in_seconds*USEC_PER_SECOND));
       if(rv == ETIMEDOUT)
       {
           DBG_WARN("[%d]%s(): WARNING - posix_wait_for_message timed out\n",gettid(), __func__);

           if (jhwdinfo->common.scanline_timeout == NULL)
           {
               jpeg_event.event_flags = JPEG_DECODE_TIMEOUT;
           }
           else
           {
               timeout_count++;

               // ask the user if he wants to timeout or not.  true = timeout
               if (jhwdinfo->common.scanline_timeout(
                               (struct jpeghw_common_struct *)jhwdinfo, timeout_count))
               {
                   jpeg_event.event_flags = JPEG_DECODE_TIMEOUT;
               }
           }
       }
       else
       {
           timeout_count = 0;
       }

       DBG_INFO("[%d]--->>received omsg 0x%08x\n",gettid(), jpeg_event.event_flags);
   }
   while(! (jpeg_event.event_flags  & JPEG_DMA_OUT ||
               jpeg_event.event_flags  & JPEG_DECODED ||
               jpeg_event.event_flags  & JPEG_DECODE_ERROR ||
               jpeg_event.event_flags  & JPEG_DECODE_TIMEOUT ||
               jpeg_event.event_flags  & TERMINATE_THRD));

   return jpeg_event.event_flags;
}

int jhwd_init_odma_list(struct jpeghw_decompress_struct * jhwdinfo)
{
    DBG_INFO("[%d]%s\n",gettid(), __func__);
    struct jpeg_dev_info_s * dev = JHWD_GET_DEV(jhwdinfo);

    dev->decode_complete = false;
    dev->line_counter = 0;

    dev->dma_list_idx = 0;
    dev->odma_desc_list = jhw_new_dma_descriptors(dev->number_odma_descriptors);
    if(dev->odma_desc_list == NULL)
    {
        DBG_ERR("[%d]Failed to allocate memory for odma descriptor list\n",gettid());
        jpeghw_error(&jhwdinfo->common, e_JPEGHW_ERR_OUT_OF_MEMORY);
        return FAIL;
    }

    return OK;
}

int jhwd_get_odma_data(struct jpeghw_decompress_struct * jhwdinfo, 
      struct BigBuffer_s *big_buffer, uint32_t *bytes_read, bool *last_buffer)
{
    jpeghw_dev_ptr   jdev = JPEGHWD_GET_DEV(jhwdinfo);
    struct jpeg_dev_info_s * dev = JHWD_GET_DEV(jhwdinfo);

    uint32_t dest_len = big_buffer->datalen;
    uint32_t dest_lines = dest_len / dev->byte_width;
    uint32_t number_of_mcu_aligned_chunks = dest_lines / jdev->info->mcu_height;
    int idx;
    uint32_t mcu_chunk_len = jdev->info->mcu_height * dev->byte_width;

    DBG_INFO("[%d]%s()\n",gettid(), __func__);

    if(big_buffer == NULL || last_buffer == NULL)
    {
        DBG_ERR("[%d]%s big_buffer or last_buffer is NULL\n",gettid(), __func__);
        jpeghw_error(&jhwdinfo->common, e_JPEGHW_ERR_INVALID_PARAMETERS);
        return FAIL;
    }

    *last_buffer = false;

    if(dest_lines % jdev->info->mcu_height != 0)
    {
        DBG_ERR("[%d]%s big_buffer height is not a multiple of MCU height\n",gettid(), __func__);
        *last_buffer = true;
        jpeghw_error(&jhwdinfo->common, e_JPEGHW_ERR_INVALID_PARAMETERS);
        return FAIL;
    }
    if(dest_lines > dev->max_number_deccompressed_lines)
    {
        DBG_ERR("[%d]%s big_buffer height is too large for DMA descriptorst\n",gettid(), __func__);
        *last_buffer = true;
        jpeghw_error(&jhwdinfo->common, e_JPEGHW_ERR_INVALID_PARAMETERS);
        return FAIL;
    }
    if(dest_len != mcu_chunk_len * number_of_mcu_aligned_chunks)
    {
        DBG_ERR("[%d]%s chunk lens don't add up to big_buffer len\n",gettid(), __func__);
        *last_buffer = true;
        jpeghw_error(&jhwdinfo->common, e_JPEGHW_ERR_INVALID_PARAMETERS);
        return FAIL;
    }
    // if decode has already completed why are they still asking for data 
    if(dev->decode_complete)
    {
        DBG_ERR("[%d]%s decode had already completed\n",gettid(), __func__);
        *last_buffer = true;
        jpeghw_error(&jhwdinfo->common, e_JPEGHW_ERR_BAD_STATE);
        return  FAIL;
    }

#ifdef ASIC_JPEG64
    uint64_t top_of_chain = (uintptr_t)(uintptr_t)dev->odma_desc_list[0]->hwaddr;
#else
    uint32_t top_of_chain = (uintptr_t)dev->odma_desc_list[0]->hwaddr;
#endif

    INC_G_SMAPPED();
    jpegdescsize_t source_addr = (uintptr_t)(dma_buffer_map_single(big_buffer, DMA_FROM_DEVICE));

    jhw_dma_desc_t *desc = 0;
    for(idx=0; idx<number_of_mcu_aligned_chunks; idx++)
    {
        desc = dev->odma_desc_list[idx];

        desc->big_buffer = big_buffer;
        desc->control_reg = READY_HW_OWN;
        desc->length = mcu_chunk_len;
        desc->source_addr = source_addr;

        // no buffer splitting
        desc->src_addr_low_0 = desc->source_addr;
        desc->src_addr_high_0 = desc->source_addr + desc->length -1;
        desc->src_addr_low_1 = desc->source_addr;
        desc->src_addr_high_1 = desc->source_addr + desc->length -1;

        if(idx > 0)
        {
            dev->odma_desc_list[idx-1]->next_desc_addr = (uintptr_t)desc->hwaddr;
        }
        source_addr += mcu_chunk_len;
    }

    if (desc != 0)
    {
        desc->next_desc_addr = 0; // terminate end of chain
        desc->control_reg |= INT_ON_FINISH;  // enable interrupt for last descriptor
    }
    else
    {
        DBG_ERR("[%d]%s desc is 0!\n",gettid(), __func__);
        *last_buffer = true;
        jpeghw_error(&jhwdinfo->common, e_JPEGHW_ERR_BAD_OBJECT);
        return  FAIL;
    }

    posix_sleep_us(1);


    POGO_ODMA_UDMA_REGS_t* odma = (POGO_ODMA_UDMA_REGS_t*) dev->reg.pogo_odma_udma;

#ifdef ASIC_JPEG64
    // Write lower and upper 32-bits of the top of chain 64-bit address
    odma->ULDSR = (uint32_t)top_of_chain;
    odma->UUDSR = (uint64_t)top_of_chain >> 32;
    DBG_INFO("[%d]%s  set descriptors into odma-ULDSR/UUDSR:0x%08x/0x%08x\n", gettid(), __func__, odma->ULDSR, odma->UUDSR);

    // A write of any value to UDSR will cause the DMA to
    // start by fetching the descriptor at the address pointed to by
    // the Upper and Lower Address Registers
    odma->UDSR = (uint32_t)top_of_chain;
#else
    odma->UDR = top_of_chain;
    DBG_INFO("[%d]%s  set descriptors into odma-UDR:0x%08x\n", gettid(), __func__, odma->UDR);
#endif

    int ev = _jhwd_wait_for_odma(jhwdinfo);

    DEC_G_SMAPPED();
    dma_buffer_unmap_single(big_buffer, DMA_FROM_DEVICE);

    if(ev  & JPEG_DMA_OUT || ev & JPEG_DECODED || ev & JPEG_DECODE_ERROR ||
    		ev & JPEG_DECODE_TIMEOUT || ev & TERMINATE_THRD)
    {
        // calculate the number of lines produced via this interrupt
        int idx;
        int lines = 0;
        for(idx=0; idx<dev->number_odma_descriptors; idx++)
        {
            if(!(dev->odma_desc_list[idx]->control_reg & READY_HW_OWN))
            {
                lines += jdev->info->mcu_height;
            }
            if(dev->odma_desc_list[idx]->next_desc_addr == 0) break;
        }
        dev->line_counter += lines;
        if(bytes_read != NULL)
        {
            *bytes_read = lines * dev->byte_width;
        }
    }

    if(ev & JPEG_DECODED || ev & JPEG_DECODE_ERROR ||
                   ev & JPEG_DECODE_TIMEOUT || ev & TERMINATE_THRD)
    {
        *last_buffer = true;
        if(ev & JPEG_DECODED) DBG_INFO("[%d]JPEG DECODED\n",gettid());
        if(ev & JPEG_DECODE_ERROR) DBG_INFO("[%d]JPEG DECODED WITH HUFFMAN ERRORS\n",gettid());
        if(ev & JPEG_DECODE_TIMEOUT) DBG_ERR("[%d]JPEG DECODED WITH TIMEOUT ERRORS\n",gettid());
        if(ev & TERMINATE_THRD) DBG_ERR("[%d]TERMINATE THREAD\n",gettid());

        dev->decode_complete = true;
    }

    return OK;
}

