/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014-2015, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/



#include <string.h>

#include "dma_buffer.h"
#include "jhw_dev.h"
#include "jhw_dma.h"
#include "jhw_uio.h"

#define DEC_WAIT_VAL 100   // usecs
#define DEC_TIMEOUT_VAL 1000000 / DEC_WAIT_VAL // timeout=1 sec


int jhwd_destroy_idma_list(struct jpeghw_decompress_struct * jhwdinfo, bool wait_for_complete)
{
    DBG_INFO("[%d]%s\n",gettid(), __func__);
    struct jpeg_dev_info_s * dev = JHWD_GET_DEV(jhwdinfo);
    int retval = OK;
    int event_flags = TERMINATE_THRD;
    jpeg_event_t jpeg_event;
    jpeg_event.event_flags = event_flags;
    jpeg_event.dev_info_ptr = dev;

    // prevents hangs if the hardware is not responding   
    if(wait_for_complete)
    {
        int inc;
        for(inc=0; inc<DEC_TIMEOUT_VAL; inc++)
        {
            if(dev->decode_complete) break;
            posix_sleep_us(DEC_WAIT_VAL);
        }
        if(inc == DEC_TIMEOUT_VAL)
        {
            mq_send(dev->imqd.mqd, (char *)&jpeg_event, sizeof(jpeg_event_t), 0);
            DBG_ERR("[%d]Never received the DECODE-COMPLETE\n",gettid());
            retval = FAIL;
        }
    }
    else
    {
        DBG_ERR("[%d]IDMA Abort\n",gettid());
        mq_send(dev->imqd.mqd, (char *)&jpeg_event, sizeof(jpeg_event_t), 0);
    }

    // destroy the descriptors and wait for thread to complete
    pthread_join(dev->idma_thd_id, NULL);
    DEC_G_ALLOCATED();
    MEM_FREE_UNCACHED(dev->idma_desc->hwaddr, dev->idma_desc);
    DBG_INFO("[%d]%s idma thread closed\n",gettid(), __func__);

    return retval;
}

int jhwd_init_idma_list(struct jpeghw_decompress_struct * jhwdinfo)
{
    DBG_INFO("[%d]%s\n",gettid(), __func__);
    struct jpeg_dev_info_s * dev = JHWD_GET_DEV(jhwdinfo);
    void *idma_hwaddr;

    dev->decode_complete = false;
    INC_G_ALLOCATED();
    dev->idma_desc = (jhw_dma_desc_t *)MEM_MALLOC_UNCACHED(&idma_hwaddr, sizeof(jhw_dma_desc_t), e_32_byte);
    if(dev->idma_desc == NULL)
    {
        DBG_ERR("[%d]Failed to allocate memory for idma descriptor\n",gettid());
        jpeghw_error(&jhwdinfo->common, e_JPEGHW_ERR_OUT_OF_MEMORY);
        return FAIL;
    }
    memset(dev->idma_desc, 0, sizeof(jhw_dma_desc_t));
    dev->idma_desc->hwaddr = idma_hwaddr;
    DBG_INFO("[%d]%s   new idma desc:%p hwaddr:%p to idma_desc\n", gettid(), __func__, dev->idma_desc, dev->idma_desc->hwaddr);

    return OK;
}

static void * _jhwd_start_idma(void * arg)
{
    struct jpeghw_decompress_struct * jhwdinfo = (struct jpeghw_decompress_struct *)arg;
    struct jpeg_dev_info_s * dev = JHWD_GET_DEV(jhwdinfo);
    DBG_INFO("[%d]%s dev:0x%08x\n",gettid(), __func__, (unsigned int)dev);
    uint32_t buflen=0;

    if(jhwdinfo->dmgr == NULL)
    {
        DBG_ERR("[%d]%s compression mgr is NULL\n",gettid(), __func__);
        jpeghw_error(&jhwdinfo->common, e_JPEGHW_ERR_INVALID_PARAMETERS);
        return 0;
    }
    if(jhwdinfo->dmgr->get_input_buffer == NULL)
    {
        DBG_ERR("[%d]%s get_input_buffer callback is NULL\n",gettid(), __func__);
        jpeghw_error(&jhwdinfo->common, e_JPEGHW_ERR_INVALID_PARAMETERS);
        return 0;
    }

    while(1) // until image is decoded
    {
        INC_G_ALLOCATED();  // assuming the user allocated this big_buffer
        struct BigBuffer_s *big_buffer = jhwdinfo->dmgr->get_input_buffer(jhwdinfo, &buflen);
        if(big_buffer == NULL || buflen == 0)
        {
            DBG_INFO("[%d]End of data.  STOP THREAD\n",gettid());
            if(big_buffer != NULL)
            {
                DEC_G_ALLOCATED();  // the user did not allocated big_buffer so dec counter
                big_buffer = JHW_FREE_MAPPED_BIG_BUFFER(big_buffer);
            }
            return NULL;
        }

        dev->idma_desc->big_buffer = big_buffer;
        dev->idma_desc->control_reg = READY_HW_OWN | INT_ON_FINISH; 
        dev->idma_desc->length = dev->idma_desc->big_buffer->datalen;
        INC_G_SMAPPED();
        dev->idma_desc->source_addr =
                (uintptr_t)(dma_buffer_map_single(dev->idma_desc->big_buffer, DMA_TO_DEVICE));
        dev->idma_desc->next_desc_addr = 0;

        DBG_INFO("[%d]buflen(%d) big_buffer len(%d)\n",gettid(), buflen, dev->idma_desc->length);
        if(buflen > dev->idma_desc->length)
        {
            DBG_ERR("[%d]buflen(%d) is larger than big_buffer len(%d)\n",gettid(), buflen, dev->idma_desc->length);
            ASSERT(0);
        }

        JPEG_IDMA_UDMA_REGS_t* idma = (JPEG_IDMA_UDMA_REGS_t*) dev->reg.jpeg_idma_udma;
        posix_sleep_us(1); // hold for a few instruction cycles

#ifdef ASIC_JPEG64
        uint64_t addr = (uintptr_t)dev->idma_desc->hwaddr;
        DBG_INFO("[%d]%s  hwaddr:0x%x\n", gettid(), __func__, addr);

        // Write lower and upper 32-bits of starting descriptor 64-bit address
        idma->ULDSR = (uint32_t)addr;
        idma->UUDSR = (uint64_t)addr >> 32;
        DBG_INFO("[%d]%s  set descriptors into jpeg-idma-ULDSR/UUDSR:0x%08x/0x%08x buflen:%d\n", gettid(), __func__, idma->ULDSR, idma->UUDSR, buflen);

        // A write of any value to UDSR will cause the DMA to
        // start by fetching the descriptor at the address pointed to by
        // the Upper and Lower Descriptor Address Registers
        idma->UDSR = (uint32_t)addr;
#else
        idma->UDR = (uintptr_t)dev->idma_desc->hwaddr;
        DBG_INFO("[%d]  set descriptors into jpeg-idma-UDR:0x%08x buflen:%d\n",gettid(), idma->UDR, buflen);
#endif

        jpeg_event_t jpeg_event;
        uint32_t timeout_in_seconds = 1;
        
        do 
        {
            DBG_INFO("[%d]--->>wait for imsg mqd:%d for %d seconds...\n",gettid(), dev->imqd.mqd, timeout_in_seconds);

            jpeg_event.event_flags = 0;
            int rv = posix_wait_for_message(dev->imqd.mqd, (char *)&jpeg_event,
                                             sizeof(jpeg_event_t), (timeout_in_seconds*USEC_PER_SECOND));
            if(rv == ETIMEDOUT)
            {
                DBG_WARN("[%d]%s(): WARNING - posix_wait_for_message timed out\n",gettid(), __func__);
            }

            DBG_INFO("[%d]--->>received imsg 0x%08x\n",gettid(), jpeg_event.event_flags);
        }
        while(! (jpeg_event.event_flags & JPEG_DMA_IN || jpeg_event.event_flags & TERMINATE_THRD ||
        		jpeg_event.event_flags & JPEG_DECODED || jpeg_event.event_flags & JPEG_DECODE_ERROR));

        if(jpeg_event.event_flags & JPEG_DECODED ||
        		jpeg_event.event_flags & JPEG_DECODE_ERROR || jpeg_event.event_flags & TERMINATE_THRD)
        {
            if(jpeg_event.event_flags & TERMINATE_THRD) DBG_ERR("[%d]TERMINATE THREAD\n",gettid());
            if(jpeg_event.event_flags & JPEG_DECODE_ERROR || jpeg_event.event_flags & TERMINATE_THRD)
            {
               jpeghw_error(&jhwdinfo->common, e_JPEGHW_ERROR);
            }
            big_buffer = JHW_FREE_MAPPED_BIG_BUFFER(big_buffer);
            return NULL;
        }

        big_buffer = JHW_FREE_MAPPED_BIG_BUFFER(big_buffer);
    }
}

int jhwd_start_idma(struct jpeghw_decompress_struct * jhwdinfo)
{
    struct jpeg_dev_info_s * dev = JHWD_GET_DEV(jhwdinfo);

    DBG_INFO("[%d]%s dev:0x%08x\n",gettid(), __func__, (unsigned int)dev);
    int ret = pthread_create(&(dev->idma_thd_id), 0, _jhwd_start_idma, (void *)jhwdinfo);
    //int ret = 0;
    DBG_INFO("[%d] thread started ret:%d\n",gettid(), ret);

    if(ret == 0) return OK;
    
    jpeghw_error(&jhwdinfo->common, e_JPEGHW_ERROR);
    return FAIL;
}
