/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014-2015, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/



#include "jhw_api.h"
#include "jhw_dev.h"
#include "jhw_asic.h"
#include "jhw_core.h"

// soft reset the JPEG core
static void _jhw_reset_core(struct jpeg_dev_info_s * dev) // jpeg control block register
{
    JPEG_TOP_REGS_t*   _top = (JPEG_TOP_REGS_t*) dev->reg.top; // block top (io) reg
    JPEG_CORE_REGS_t* _core = (JPEG_CORE_REGS_t*) dev->reg.core; // block core reg

    DBG_INFO("[%d]%s(%#x) _core %p top %p \n",gettid(), __func__, (unsigned int)dev, _core, _top);

    //dbg DBG("%s() dev %#x base %#x io %#x\n", __func__, dev, _core, _top);
    if (_top && _core)
    {
        // jpeg core/header
        _core->CR = JPEG_CORE_CR_CLR_HDR_DATA_REPLACE_VAL(_core->CR, 1);
        posix_sleep_us(3);
        _core->CR = JPEG_CORE_CR_CLR_HDR_DATA_REPLACE_VAL(_core->CR, 0);

        // jpeg io
        _top->CR = JPEG_TOP_CR_SOFT_RESET_REPLACE_VAL(_top->CR, 1);
        posix_sleep_us(3); // threadx cpu_spin_delay(3);
        _top->CR = JPEG_TOP_CR_SOFT_RESET_REPLACE_VAL(_top->CR, 0);
    }
}

// reset a given UDMA channel
static int _jhw_reset_dma(UDMA_REGS_t* udma)
{
    volatile uint32_t udma_status = udma->USR;
    DBG_INFO("[%d]%s dma-USR:0x%08x\n",gettid(), __func__, udma_status);
    
    udma->UCR = UDMA_USR_BUSY_REPLACE_VAL(udma->UCR, 0); 
    posix_sleep_us(3); // hold for a few instruction cycles
    udma->UCR = UDMA_USR_BUSY_REPLACE_VAL(udma->UCR, 1); 
    
    int tock = 0;
    while (udma_status & UDMA_USR_BUSY_MASK) 
    { 
        udma_status = udma->USR;
        if (tock++ > 10) 
        { // declare dead if no response in 10 cycles
            DBG_ERR("[%d]%s stuck!? udma->USR:0x%08x\n",gettid(), __func__, udma->USR);
            return FAIL;
        }
    }
    return OK;
}

static bool _jhw_init_jpeg_info(struct jpeghw_decompress_struct * jhwdinfo, struct jpeghw_jpeg_header_struct *jinfo)
{
    int i, j, t;

    DBG_INFO("Start %s() call\n", __func__);

    struct jpeg_dev_info_s * dev = JHWD_GET_DEV(jhwdinfo);

    if (dev == NULL || jinfo == NULL)
    {
        jpeghw_error(&jhwdinfo->common, e_JPEGHW_ERR_INVALID_PARAMETERS);
        return false;
    }

    device_core_info_ptr core_info = &dev->core_info;

    /* set JPEG header fields */
    core_info->ecs_offset = jinfo->ecs_offset;

    core_info->h0_m1 = jinfo->h0_m1;
    core_info->v0_m1 = jinfo->v0_m1;
    core_info->h1_m1 = jinfo->h1_m1;
    core_info->v1_m1 = jinfo->v1_m1;
    core_info->h2_m1 = jinfo->h2_m1;
    core_info->v2_m1 = jinfo->v2_m1;
    core_info->h3_m1 = jinfo->h3_m1;
    core_info->v3_m1 = jinfo->v3_m1;

    core_info->num_mcu_m1 = jinfo->num_mcu_m1;

    core_info->restart_enable = jinfo->restart_enable;
    core_info->restart_interval_m1 = jinfo->restart_interval_m1;

    core_info->dequant_q_map_0 = jinfo->dequant_q_map_0;
    core_info->dequant_q_map_1 = jinfo->dequant_q_map_1;
    core_info->dequant_q_map_2 = jinfo->dequant_q_map_2;
    core_info->dequant_q_map_3 = jinfo->dequant_q_map_3;

    core_info->huff_dc_table_sel = jinfo->huff_dc_table_sel;
    core_info->huff_ac_table_sel = jinfo->huff_ac_table_sel;

    /* create Huffman SRAM contents */
    uint32_t sram_i = 0;
    uint32_t ptr_base = 0;

    for (t=0; t<4; t++)
    {
        uint8_t *bits = NULL;
        uint8_t *val = NULL;
        uint32_t bits_size = 0;

        uint32_t table_num = 0;
        uint32_t tree_val = 0;
        uint32_t code_valid = 0;
        uint32_t len_code_addr_base = 0;
        bool ac = false;

        switch (t)
        {
            case 0: // DC table 0
                table_num = 0;
                sram_i = len_code_addr_base = 0;
                ptr_base = 0;
                ac = false;
                break;

            case 1: // AC table 0
                table_num = 0;
                sram_i = len_code_addr_base = 12;
                ptr_base += 16;
                ac = true;
                break;

            case 2: // DC table 1
                table_num = 1;
                sram_i = len_code_addr_base = 12+162;
                ptr_base += 16;
                ac = false;
                break;

            case 3: // AC table 1
                table_num = 1;
                sram_i = len_code_addr_base = 12+162+12;
                ptr_base += 16;
                ac = true;
                break;
        }

        // set Table bits and values
        if (ac)
        {
            bits = &jinfo->huff_ac_bits[table_num][0];
            bits_size = sizeof(jinfo->huff_ac_bits[table_num]);

            val = &jinfo->huff_ac_array[table_num][0];
        }
        else
        {
            bits = &jinfo->huff_dc_bits[table_num][0];
            bits_size = sizeof(jinfo->huff_dc_bits[table_num]);

            val = &jinfo->huff_dc_array[table_num][0];
        }

        if (bits && val)
        {
            uint32_t table_i = 0;

            // loop through the bits and values to calculate the sram values
            // tree_val is the current node as we work through the Huffman tree
            // code_valid is the code length enable values for Code Enable registers
            // v_i is the index into the val table
            // huff_ptr_array is the array to load into HUFF_PTR RAM
            // huff_fcode_array is the array to load into HUFF_FCODE RAM
            // huff_lcode_array is the array to load into HUFF_LCODE RAM
            // huff_sym_array is the array to load into HUFF_SYM RAM
            // huff_len_code_array is the array to load into LEN_CODE RAM
            for (i=0; i<bits_size; i++)
            {
                uint8_t len;
                uint32_t rrrr, ssss;

                len = bits[i];
                if (len > 0)
                {
                    code_valid = code_valid | (1<<i);
                    core_info->huff_ptr_array[ptr_base+i] = sram_i;
                    for (j=0; j<len; j++)
                    {
                        uint32_t len_code_addr = len_code_addr_base;

                        if (j==0)
                        {
                            core_info->huff_fsymbol_array[ptr_base+i] = tree_val;
                        }

                        if (j==(len-1))
                        {
                            core_info->huff_lsymbol_array[ptr_base+i] = tree_val;
                        }

                        core_info->huff_sym_array[sram_i] = val[table_i];

                        rrrr = (core_info->huff_sym_array[sram_i] >> 4) & 0xF;
                        ssss = core_info->huff_sym_array[sram_i] & 0xF;

                        if (ac)
                        {
                            if (rrrr == 0 && ssss == 0)
                            {
                                // do nothing - offset of 0
                            }
                            else if (rrrr == 0xF && ssss == 0)
                            {
                                len_code_addr += 1; // offset 1
                            }
                            else
                            {
                                len_code_addr += 1 + (10 * rrrr) + ssss;
                            }
                        }
                        else
                        {
                            len_code_addr += ssss;
                        }

                        core_info->huff_code_array[len_code_addr] = ((i)<<16) | (tree_val);

                        sram_i++;
                        table_i++;
                        tree_val += 1;
                    }
                }

                tree_val = (tree_val << 1);
            }
        }
        else
        {
            DBG_ERR("%s() ERROR - unable to set bits and val!\n", __func__);
            jpeghw_error(&jhwdinfo->common, e_JPEGHW_ERROR);
            return false;
        }

        // ac_code_en_1 and ac_code_en_0 are the values to program in
        // Huffman Decode Code Enable AC Register
        // dc_code_en_1 and dc_code_en_0 are the values to program in
        // Huffman Decode Code Enable DC Register
        if(ac)
        {
          if(table_num == 1)
              core_info->ac_code_en_1 = code_valid;
          else
              core_info->ac_code_en_0 = code_valid;
        }
        else
        {
          if(table_num == 1)
              core_info->dc_code_en_1 = code_valid;
          else
              core_info->dc_code_en_0 = code_valid;
        }
    }


    /* copy quant table */
    uint32_t q_size = sizeof(core_info->dequant_q_array);
    if (q_size > sizeof(jinfo->dequant_q_array))
    {
        q_size = sizeof(jinfo->dequant_q_array);
    }
    memcpy(core_info->dequant_q_array, jinfo->dequant_q_array, q_size);

    return true;
}


// initialize the JPEG registers for decompression
static int _jhwd_set_decompression_regs(struct jpeghw_decompress_struct * jhwdinfo)
{
    DBG_INFO("[%d]%s\n",gettid(), __func__);
    jpeghw_dev_ptr   jdev = JPEGHWD_GET_DEV(jhwdinfo);
    struct jpeg_dev_info_s * dev = JHWD_GET_DEV(jhwdinfo);
    device_core_info_ptr core_info = &dev->core_info;
    struct jpeghw_jpeg_header_struct jinfo;
    int bytes_per_pixel = jdev->info->bytes_per_pixel;
    uint32_t byte_width = dev->byte_width;

    DBG_INFO("[%d] assign register base addrs\n",gettid());
   // decoding dma shadow registers
    JPEG_CORE_REGS_t*      _core = (JPEG_CORE_REGS_t*)      dev->reg.core;
    JPEG_IDMA_UDMA_REGS_t* _idma = (JPEG_IDMA_UDMA_REGS_t*) dev->reg.jpeg_idma_udma;
    POGO_ODMA_UDMA_REGS_t* _odma = (POGO_ODMA_UDMA_REGS_t*) dev->reg.pogo_odma_udma;
    POGO_ODMA_CORE_REGS_t* _ocor = (POGO_ODMA_CORE_REGS_t*) dev->reg.pogo_odma_core;

    DBG_INFO("[%d] disable all interrupts\n",gettid());
    // disable all interrupt sources and then soft reset the jpeg block
    _core->IRQ_EN = 0;
    _idma->UIER   = 0;
    _odma->UIER   = 0;
    _ocor->OIER   = 0;

    DBG_INFO("[%d] reset hardware\n",gettid());
    // soft reset
    _jhw_reset_core(dev);
    jhwd_reset_idma(jhwdinfo);
    jhwd_reset_odma(jhwdinfo);

    _core->CR = 0x00000000; // reset mode
    _core->IRQ_ACK = _core->IRQ_PEND; // ack interrupts 
    _core->ECS_OFFSET = 0;

    _core->CR |= JPEG_CORE_CR_CLR_HDR_DATA_MASK;
    posix_sleep_us(3);
    _core->CR &= ~JPEG_CORE_CR_CLR_HDR_DATA_MASK;
    DBG_INFO("[%d]%s() decoding .. header reset\n",gettid(), __func__);

    memset(&jinfo, 0, sizeof(jinfo));
    if(jhwdinfo->dmgr && jhwdinfo->dmgr->get_jpeg_info && 
       jhwdinfo->dmgr->get_jpeg_info(jhwdinfo, &jinfo) == e_JPEGHW_SUCCESS)
    {
        _jhw_init_jpeg_info(jhwdinfo, &jinfo);

        bytes_per_pixel = jinfo.bytes_per_pixel;
        byte_width = jinfo.image_width*jinfo.bytes_per_pixel;

        // program the core jpeg block with the image info
        _core->IMAGE = JPEG_CORE_IMAGE_NUM_PLANES_M1_REPLACE_VAL(_core->IMAGE, bytes_per_pixel-1);
        _core->DIM = JPEG_CORE_DIM_Y_REPLACE_VAL(_core->DIM, jinfo.mcu_aligned_height);
        _core->DIM = JPEG_CORE_DIM_X_REPLACE_VAL(_core->DIM, jinfo.mcu_aligned_width);
        DBG_INFO("[%d]%s() src (%d x %d x %d)\n",gettid(), __func__, jinfo.mcu_aligned_width,
                jinfo.mcu_aligned_height, bytes_per_pixel);

        _core->SUB = core_info->v3_m1 << JPEG_CORE_SUB_V3_M1_SHIFT |
                     core_info->h3_m1 << JPEG_CORE_SUB_H3_M1_SHIFT |
                     core_info->v2_m1 << JPEG_CORE_SUB_V2_M1_SHIFT |
                     core_info->h2_m1 << JPEG_CORE_SUB_H2_M1_SHIFT |
                     core_info->v1_m1 << JPEG_CORE_SUB_V1_M1_SHIFT |
                     core_info->h1_m1 << JPEG_CORE_SUB_H1_M1_SHIFT |
                     core_info->v0_m1 << JPEG_CORE_SUB_V0_M1_SHIFT |
                     core_info->h0_m1 << JPEG_CORE_SUB_H0_M1_SHIFT;

        _core->HUFF            = core_info->huff_ac_table_sel << JPEG_CORE_HUFF_AC_TABLE_SEL_SHIFT |
                                 core_info->huff_dc_table_sel << JPEG_CORE_HUFF_DC_TABLE_SEL_SHIFT;

        _core->HUFF_CODE_EN_DC = core_info->dc_code_en_1 << JPEG_CORE_HUFF_CODE_EN_DC_DC_CODE_EN_1_SHIFT |
                                 core_info->dc_code_en_0 << JPEG_CORE_HUFF_CODE_EN_DC_DC_CODE_EN_0_SHIFT;

        _core->HUFF_CODE_EN_AC = core_info->ac_code_en_1 << JPEG_CORE_HUFF_CODE_EN_AC_AC_CODE_EN_1_SHIFT |
                                 core_info->ac_code_en_0 << JPEG_CORE_HUFF_CODE_EN_AC_AC_CODE_EN_0_SHIFT;


        _core->RESTART         = core_info->restart_enable << JPEG_CORE_RESTART_RESTART_ENABLE_SHIFT |
                                 core_info->restart_interval_m1 << JPEG_CORE_RESTART_RESTART_INTERVAL_M1_SHIFT;

        _core->QUANT           = core_info->dequant_q_map_3 << JPEG_CORE_QUANT_Q_MAP_3_SHIFT |
                                 core_info->dequant_q_map_2 << JPEG_CORE_QUANT_Q_MAP_2_SHIFT |
                                 core_info->dequant_q_map_1 << JPEG_CORE_QUANT_Q_MAP_1_SHIFT |
                                 core_info->dequant_q_map_0 << JPEG_CORE_QUANT_Q_MAP_0_SHIFT;

        _core->ECS_OFFSET      = core_info->ecs_offset;

        /***************************************
        Setting SRAM access enable bit of CR register. This allows loading of Huffman tables
        **********************************/
        _core->CR |= JPEG_CORE_CR_SRAM_ACCESS_ENABLE_MASK;

        /**********************************
        Writing HUFF_DEC_FSYMBOL table
        **********************************/
        int i;
        for(i=0; i<64; i++)
        {
            _core->HUFF_FCODE[i] = core_info->huff_fsymbol_array[i];
        }
        /**********************************
        Writing HUFF_DEC_LSYMBOL Table
        **********************************/
        for(i=0; i<64; i++)
        {
            _core->HUFF_LCODE[i] = core_info->huff_lsymbol_array[i];
        }
        /***********************************
        HUFF_DEC_PTR Table
        **********************************        */
        for(i=0; i<64; i++)
        {
            _core->HUFF_PTR[i] = core_info->huff_ptr_array[i];
        }
        /**********************************
        HUFF_SYM Table
        **********************************/
        for(i=0; i<348; i++)
        {
            _core->HUFF_SYM[i] = core_info->huff_sym_array[i];
        }

        /**********************************
        HUFF_LEN_Table
        **********************************/
        for(i=0; i<348; i++)
        {
            _core->LEN_CODE[i] = core_info->huff_code_array[i];
        }

        /**********************************
        Writing Quantization table
        **********************************/
        // retrieve the user quant table and use it if it's there
        for(i=0; i<sizeof(core_info->dequant_q_array); i++)
        {
            _core->Q_TABLE[i] = core_info->dequant_q_array[i];
        }

        /**********************************
        Clearing the sram access enable bit.
        **********************************        */
        _core->CR &= ~JPEG_CORE_CR_SRAM_ACCESS_ENABLE_MASK;
    }
    else
    {
        // enable header generation
        _core->CR = JPEG_CORE_CR_HEADER_ENABLE_REPLACE_VAL(_core->CR, 1);
    } 

    _core->PAD = 0;
    // if the image width/height are not mcu aligned then the extra pixels the 
    // decoder produces to align with the mcu will be this value.
    // This value is the pixel value that will be used to pad the image on the right and bottom. 
    // In a 1-plane image, bits [7:0] will be used. 
    // In a 3-plane image, Y = bits[23:16], Cb = bits[15:8], and Cr = bits[7:0]. 
    // In a 4-plane image, C = bits[31:24], M = bits[23:16], Y = bits[15:8], K = bits[7:0] 
    // NOTE: This value is prior to the CSC. 
    _core->PAD_VAL = 0xFF8080; // white

    // dma line (byte) width
    _ocor->OLWR  = byte_width;
    DBG_INFO("[%d]%s() olwr %d\n",gettid(), __func__, _ocor->OLWR);

    // csc setup -- for 24-bit RGB or 8-bit grayscale
    {
        CSC_JEDI_REGS_t* _csc = (CSC_JEDI_REGS_t*) dev->reg.csc;
        register uint32_t csc_setting = 0; // decode: csc_setting accumulator
        if (bytes_per_pixel == 1 || bytes_per_pixel == 4) // decode: 1 or 4 plane(s)
        {
            csc_setting = CSC_JEDI_CCR_PREOFFBYPASS_REPLACE_VAL(csc_setting,  1);
            csc_setting = CSC_JEDI_CCR_POSTOFFBYPASS_REPLACE_VAL(csc_setting, 1);
            csc_setting = CSC_JEDI_CCR_OFFSETBYPASS_REPLACE_VAL(csc_setting,  1);
            csc_setting = CSC_JEDI_CCR_BYPASSALL_REPLACE_VAL(csc_setting,     1);
        }
        else // decode: 3-plane 24-bit data
        {
            csc_setting = CSC_JEDI_CCR_PREOFFBYPASS_REPLACE_VAL(csc_setting,  0);
            csc_setting = CSC_JEDI_CCR_POSTOFFBYPASS_REPLACE_VAL(csc_setting, 1);
            csc_setting = CSC_JEDI_CCR_OFFSETBYPASS_REPLACE_VAL(csc_setting,  0);
            csc_setting = CSC_JEDI_CCR_BYPASSALL_REPLACE_VAL(csc_setting,     0);

            _csc-> MCR0 = 0x1000;
            _csc-> MCR1 = 0x0000;
            _csc-> MCR2 = 0x166E;
            _csc-> MCR3 = 0x1000;
            _csc-> MCR4 = 0x3A7E;
            _csc-> MCR5 = 0x3493;
            _csc-> MCR6 = 0x1000;
            _csc-> MCR7 = 0x1C5A;
            _csc-> MCR8 = 0x3FFF;
        }

        _csc->CCR = csc_setting; // commit -> csc block register
        DBG_INFO("[%d]%s() csc %#x (reg %#x)\n",gettid(), __func__,
                 csc_setting, ((CSC_JEDI_REGS_t*)(dev->reg.csc))->CCR);
    }

    // (decode) depogo core configuration
    {
        DEPOGO_REGS_t* _depogo = (DEPOGO_REGS_t*) dev->reg.depogo;
        uint32_t depogo_settings = 0; // decode: pogo settings accumulator
        depogo_settings = DEPOGO_CFG_BPP_REPLACE_VAL(depogo_settings, 7); // 8-bpp all modes
        depogo_settings = DEPOGO_CFG_MONOCHAN_REPLACE_VAL(depogo_settings, 0);
        if (bytes_per_pixel == 1) // mono plane
        {
            depogo_settings = DEPOGO_CFG_FMT_REPLACE_VAL(depogo_settings, POGO_FMT_MONO);
            depogo_settings = DEPOGO_CFG_COLORSWAP_REPLACE_VAL(depogo_settings, 0); // no-swap
        }
        else
        if (bytes_per_pixel == 3) // 3-plane
        {
            depogo_settings = DEPOGO_CFG_FMT_REPLACE_VAL(depogo_settings, DEPOGO_FMT_24);
            depogo_settings = DEPOGO_CFG_COLORSWAP_REPLACE_VAL(depogo_settings, 1); // swap
        }
        else // 4-plane (rgbx or xrgb)
        {
            depogo_settings = DEPOGO_CFG_FMT_REPLACE_VAL(depogo_settings, POGO_FMT_RGBx); // packed rgbx
            depogo_settings = DEPOGO_CFG_COLORSWAP_REPLACE_VAL(depogo_settings, 1);       // swap
        }

        _depogo->Cfg = depogo_settings; // commit depogo settings
        DBG_INFO("[%d]%s() (dec)depogo cfg %d (reg %#x)\n",gettid(), __func__, depogo_settings, _depogo->Cfg);
    }

    { // finish odma setup
        uint32_t odma_cr_settings = 0;
        odma_cr_settings = UDMA_UCR_OWNPOLARITY_REPLACE_VAL(odma_cr_settings,     0);
        odma_cr_settings = UDMA_UCR_OWNWRITEDISABLE_REPLACE_VAL(odma_cr_settings, 0);
        odma_cr_settings = UDMA_UCR_BEATS_REPLACE_VAL(odma_cr_settings,           2); // 16-beats
        odma_cr_settings = UDMA_UCR_ENABLE_REPLACE_VAL(odma_cr_settings,          1); // enable
        _odma->UCR = odma_cr_settings; // commit
        DBG_INFO("[%d]%s() odma/pogo cr %#x (reg %#x)\n",gettid(), __func__, odma_cr_settings, _odma->UCR);
    }

    { // finalize idma setup
        uint32_t idma_cr_settings = 0;
        idma_cr_settings = UDMA_UCR_OWNPOLARITY_REPLACE_VAL(idma_cr_settings,     0);
        idma_cr_settings = UDMA_UCR_OWNWRITEDISABLE_REPLACE_VAL(idma_cr_settings, 0);
        idma_cr_settings = UDMA_UCR_BEATS_REPLACE_VAL(idma_cr_settings,           2); // 16-beats
        idma_cr_settings = UDMA_UCR_ENABLE_REPLACE_VAL(idma_cr_settings,          1); // enable
        _idma->UCR = idma_cr_settings; // commit
        DBG_INFO("[%d]%s() idma/jpeg cr %d (%#x)\n",gettid(), __func__, idma_cr_settings, _idma->UCR);
    }

    // decode mode
    _core->CR &= ~JPEG_CORE_CR_ENCODE_MODE_MASK; // set decode mode
    _core->CR &= ~JPEG_CORE_CR_AVG_N_DECIMATE_MASK; // make sure it is clear

    // clear/acknowledge any outstanding idma, odma (pogo and jpeg) interrupts
    _idma->UICR    = 0xffffffff;
    _odma->UICR    = 0xffffffff;
    _ocor->OICR    = 0xffffffff;
    _core->IRQ_ACK = 0xffffffff;

    // enable decoding interrupts
    _idma->UIER   = 0x00000002; // all idma interrupts
    _odma->UIER   = 0x00000002; // all odma interrupts
    _ocor->OIER   = 0x00000003; // err + EOI odma core interrupts
    _core->IRQ_EN = 0xf000001e; // core interrupts 

    // enable decode int+ (sans encoded)
    _core->IRQ_EN = 0xffffffff & ~JPEG_CORE_IRQ_PEND_ENCODE_DONE_IRQ_PENDING_MASK; // all sans encoding

    _odma->UIFR = 0x00000000;
    _idma->UIFR = 0x00000000;

    return OK;
}

// Initialze the JPEG hardware for decompression
static int _jhwd_setup_decompressor(struct jpeghw_decompress_struct * jhwdinfo)
{
    DBG_INFO("[%d]%s\n",gettid(), __func__);

    int setup_ok = _jhwd_set_decompression_regs(jhwdinfo); // enforce hw/dev availability
    
    return setup_ok; // NOT settings_ok
}

// reset the pogo idma 
int jhwd_reset_idma(struct jpeghw_decompress_struct * jhwdinfo)
{
    DBG_INFO("[%d]%s\n",gettid(), __func__);
    struct jpeg_dev_info_s * dev = JHWD_GET_DEV(jhwdinfo);

    return _jhw_reset_dma((UDMA_REGS_t*)dev->reg.jpeg_idma_udma);
}

// reset the jpeg odma 
int jhwd_reset_odma(struct jpeghw_decompress_struct * jhwdinfo)
{
    DBG_INFO("[%d]%s\n",gettid(), __func__);
    struct jpeg_dev_info_s * dev = JHWD_GET_DEV(jhwdinfo);

    return _jhw_reset_dma((UDMA_REGS_t*)dev->reg.pogo_odma_udma);
}

// reset the jpeg core
void jhwd_reset_core(struct jpeghw_common_struct * jhwinfo)
{
    struct jpeg_dev_info_s * dev = JHW_GET_DEV(jhwinfo);
    _jhw_reset_core(dev);
}

// Initialze the JPEG hardware for decompression
int jhwd_init_decompression_hw(struct jpeghw_decompress_struct * jhwdinfo)
{
    DBG_INFO("[%d]%s\n",gettid(), __func__);

    return _jhwd_setup_decompressor(jhwdinfo);
}

