/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014-2015, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/



#include "jhw_api.h"
#include "jhw_dev.h"
#include "jhw_uio.h"
#include "jhw_core.h"
#include "mqueue.h"
#include "jpeghw_lib.h"


// open idma and odma message queues to be used to communcate interrupt status 
// from the core ISR to the firmware.
static jpeghw_error_type_t _jhwd_open_message_queues(struct jpeghw_decompress_struct * jhwdinfo)
{
    struct jpeg_dev_info_s * dev = JHWD_GET_DEV(jhwdinfo);
    dev->imqd.mqd = jhw_open_message_queue(dev, IDMA_D_MSGQ_NAME, dev->imqd.name);
    dev->omqd.mqd = jhw_open_message_queue(dev, ODMA_D_MSGQ_NAME, dev->omqd.name);

    if(dev->imqd.mqd < 0 || dev->omqd.mqd < 0)
    {
        return e_JPEGHW_ERROR;
    }
    return e_JPEGHW_SUCCESS;//
}

// return decompression configuration flags.  The flags describe the capabilites
// available for this driver.  The flag are described below
uint32_t jhwd_get_config_flags()
{
    uint32_t flags = 0;

    // setting the flag implies
    // the hardware cannot pad width to achive mcu alignment so
    // the hardware cannot trim mcu aligned width to achive true
    // image width so the client is responsible for trimming
    flags |= JPEGHW_CONFIG_FLAG_MCU_WIDTH_ALIGNMENT;

    // setting the flag implies
    // the hardware cannot trim end-of-image strips that are mcu
    // aligned so the client is responsible for trimming
    flags |= JPEGHW_CONFIG_FLAG_MCU_HEIGHT_ALIGNMENT;

    return flags;
}

// create a jpeg_dev_info structure
// open decompression access to the hardware via uio devices.
// allocate memory for the jpeg_dev_info structure and attatch it 
// to the jpeghw_decompress structure.
jpeghw_error_type_t jhw_open_decompress(struct jpeghw_decompress_struct * jhwdinfo)
{
    struct jpeg_dev_info_s * dev = NULL;
    jpeghw_error_type_t jerr;

    DBG_INFO("[%d]%s()\n",gettid(), __func__);
    if(jhwdinfo == NULL)
    {
        DBG_ERR("[%d]%s jhwdinfo is NULL\n",gettid(), __func__);
        return e_JPEGHW_ERR_INVALID_PARAMETERS;
    }

    if (! jhw_is_initialized()) 
    {
        DBG_ERR("[%d]JHW driver is not initialized, cannot continue.\n",gettid());
        return e_JPEGHW_ERROR;
    }

    INC_G_ALLOCATED();
    dev = (struct jpeg_dev_info_s *)MEM_MALLOC(sizeof(struct jpeg_dev_info_s));
    if(dev == NULL)
    {
        DBG_ERR("[%d]%s out of memory\n",gettid(), __func__);
        jhwdinfo->common.global_jpeg_state = e_JPEGHW_STATE_ERROR;
        return e_JPEGHW_ERR_OUT_OF_MEMORY;
    }
    memset(dev, 0, sizeof(struct jpeg_dev_info_s));

    ((jpeghw_dev_ptr)jhwdinfo->common.jpeghw_context)->jhw_dev = dev;
    dev->type = DECOMPRESSOR;

    // open the next available uio device associated with the JPEG block.
    // if JPEGHW_CONFIG_FLAG_BLOCK_JPEG_CORE_AVAIL bit is set in the config_flags
    // then open hangs waiting for a core to free up
    jpeghw_error_type_t ret = jhw_open_available_core(dev, 
                      (jhwdinfo->common.config_flags & JPEGHW_CONFIG_FLAG_BLOCK_JPEG_CORE_AVAIL));

    if(ret != e_JPEGHW_SUCCESS)
    {
        DBG_ERR("[%d]%s faile to open access to the core\n",gettid(), __func__);
        DEC_G_ALLOCATED();
        MEM_FREE_AND_NULL(dev);
        jpeghw_error(&jhwdinfo->common, e_JPEGHW_ERR_INVALID_PARAMETERS);
        jhwdinfo->common.global_jpeg_state = e_JPEGHW_STATE_ERROR;
        return ret;
    }

    // Set default values
    dev->cookie = DEV_COOKIE;
    dev->max_number_deccompressed_lines = MAX_DECOMPRESSED_LINES;
    dev->decode_trim = jhwdinfo->common.config_flags & JPEGHW_CONFIG_FLAG_TRIM;

    // attach the compressor to the uio device
    if(jhw_attach_to_device(dev) != OK)
    {
        DBG_ERR("[%d]%s attach to device failed\n",gettid(), __func__);
        jhwdinfo->common.global_jpeg_state = e_JPEGHW_STATE_ERROR;
        jpeghw_error(&jhwdinfo->common, e_JPEGHW_ERR_INVALID_PARAMETERS);
        return e_JPEGHW_ERROR;
    }

    // open the idma and odma comm message queues
    jerr= _jhwd_open_message_queues(jhwdinfo);
    if(jerr != e_JPEGHW_SUCCESS)
    {
        DBG_ERR("[%d]%s open message queue failed\n",gettid(), __func__);
        jhwdinfo->common.global_jpeg_state = e_JPEGHW_STATE_ERROR;
        jpeghw_error(&jhwdinfo->common, e_JPEGHW_ERR_INVALID_PARAMETERS);
        return jerr;
    }

    return e_JPEGHW_SUCCESS;
}

// start the decompression.
// sanity check jpeghw_common structure values
// open the message queues used to communicate interrupt status
// intialize the JPEG core registers, intialize the IDMA and ODMA registers.
// start the IDMA thread that will via callbacks keep the IDMAs full of data
jpeghw_error_type_t jhw_start_decompress(struct jpeghw_decompress_struct * jhwdinfo)
{
    DBG_INFO("[%d]===%s()===\n",gettid(), __func__);

    JHWD_VALIDATE(jhwdinfo);
    struct jpeg_dev_info_s * dev = JHWD_GET_DEV(jhwdinfo);
    jpeghw_dev_ptr   jdev = JPEGHWD_GET_DEV(jhwdinfo);

    // sanity check some parameters
    if(jdev->info->image_width == 0 || jdev->info->image_height == 0)
    {
        DBG_ERR("[%d]%s image width/height are out of bounds\n",gettid(), __func__);
        jhwdinfo->common.global_jpeg_state = e_JPEGHW_STATE_ERROR;
        jpeghw_error(&jhwdinfo->common, e_JPEGHW_ERR_INVALID_PARAMETERS);
        return e_JPEGHW_ERR_INVALID_PARAMETERS;
    }

    if((jdev->info->mcu_height != 8 && jdev->info->mcu_width !=  16) ||
               (jdev->info->mcu_height != 8 && jdev->info->mcu_width != 16))
    {
        DBG_ERR("[%d]%s MCU width/height are out of bounds\n",gettid(), __func__);
        jhwdinfo->common.global_jpeg_state = e_JPEGHW_STATE_ERROR;
        jpeghw_error(&jhwdinfo->common, e_JPEGHW_ERR_INVALID_PARAMETERS);
        return e_JPEGHW_ERR_INVALID_PARAMETERS;
    }

    if(jdev->info->bytes_per_pixel != 1 && jdev->info->bytes_per_pixel != 3)
    {
        DBG_ERR("[%d]%s Bytes per pixel out of bounds\n",gettid(), __func__);
        jhwdinfo->common.global_jpeg_state = e_JPEGHW_STATE_ERROR;
        jpeghw_error(&jhwdinfo->common, e_JPEGHW_ERR_INVALID_PARAMETERS);
        return e_JPEGHW_ERR_INVALID_PARAMETERS;
    }

    if(dev->max_number_deccompressed_lines % jdev->info->mcu_height != 0)
    {
        DBG_ERR("[%d]%s max decompressed lines is not mcu aligned\n",gettid(), __func__);
        jhwdinfo->common.global_jpeg_state = e_JPEGHW_STATE_ERROR;
        jpeghw_error(&jhwdinfo->common, e_JPEGHW_ERR_INVALID_PARAMETERS);
        return e_JPEGHW_ERR_INVALID_PARAMETERS;
    }

    dev->byte_width = jdev->mcu_bytes_per_row;
    dev->number_odma_descriptors = dev->max_number_deccompressed_lines / jdev->info->mcu_height;

    // intialize the jpeg hardware block
    if(jhwd_init_decompression_hw(jhwdinfo) != OK)
    {
        DBG_ERR("[%d]%s init hardware failed\n",gettid(), __func__);
        jhwdinfo->common.global_jpeg_state = e_JPEGHW_STATE_ERROR;
        return e_JPEGHW_ERROR;
    }

    // intialize the jpeg DMAs
    if(jhwd_init_odma_list(jhwdinfo) != OK)
    {
        DBG_ERR("[%d]%s init odma failed\n",gettid(), __func__);
        jhwdinfo->common.global_jpeg_state = e_JPEGHW_STATE_ERROR;
        return e_JPEGHW_ERROR;
    }
    if(jhwd_init_idma_list(jhwdinfo) != OK)
    {
        DBG_ERR("[%d]%s init idma failed\n",gettid(), __func__);
        jhwdinfo->common.global_jpeg_state = e_JPEGHW_STATE_ERROR;
        return e_JPEGHW_ERROR;
    }

    // turn on the interrupts
    if(jhw_enable_device_interrupts(dev) != OK)
    {
        DBG_ERR("[%d]%s enable device interrupts failed\n",gettid(), __func__);
        jhwdinfo->common.global_jpeg_state = e_JPEGHW_STATE_ERROR;
        jpeghw_error(&jhwdinfo->common, e_JPEGHW_ERROR);
        return e_JPEGHW_ERROR;
    }

    // start the idma thread used to feed the IDMA via callbacks
    // to the user.  the callback is defined in the dmgr structure.
    if(jhwd_start_idma(jhwdinfo) != OK)
    {
        DBG_ERR("[%d]%s start idma failed\n",gettid(), __func__);
        jhwdinfo->common.global_jpeg_state = e_JPEGHW_STATE_ERROR;
        return e_JPEGHW_ERROR;
    }

    return e_JPEGHW_SUCCESS;
}

// detatch and close access to the uio device.
// close the DMA comm message queues and remove them from the system
// free the memory allocated for the jepg_dev_info structure in the open call
static void _jhw_close(struct jpeghw_common_struct * jhwinfo)
{
    struct jpeg_dev_info_s * dev = JHW_GET_DEV(jhwinfo);

    DBG_INFO("[%d]%s()\n",gettid(), __func__);
    mq_close(dev->imqd.mqd);
    mq_close(dev->omqd.mqd);
    unlink(dev->imqd.name);
    unlink(dev->omqd.name);
    jhw_detach_from_device(dev);
    jhw_close_core(dev);

    DEC_G_ALLOCATED();
    MEM_FREE_AND_NULL(dev);
}

// close decompression.
jpeghw_error_type_t jhw_close_decompress(struct jpeghw_decompress_struct * jhwdinfo)
{
    DBG_INFO("[%d]%s()\n",gettid(), __func__);
    // make sure the input parameter is valid
    if(jhwdinfo == NULL)
    {
        jpeghw_error(&jhwdinfo->common, e_JPEGHW_ERR_INVALID_PARAMETERS);
        return e_JPEGHW_ERR_INVALID_PARAMETERS;
    }
    JHWD_VALIDATE(jhwdinfo);

    _jhw_close((struct jpeghw_common_struct *)jhwdinfo);
    return e_JPEGHW_SUCCESS;
}

// finish a decompression.
// destroy the DMA lists created in the start-decompression call
// reset the DMA hardware
jpeghw_error_type_t jhw_finish_decompress(struct jpeghw_decompress_struct * jhwdinfo)
{
    DBG_INFO("[%d]%s()\n",gettid(), __func__);
    // make sure the input parameter is valid
    JHWD_VALIDATE(jhwdinfo);

    // the true in the 2nd parameter below tells the destroy calls to wait
    // for decode complete.  wait for the the last buffer to be decompressed
    // and returned to the user.
    jhwd_destroy_odma_list(jhwdinfo, true);
    jhwd_destroy_idma_list(jhwdinfo, true);
    jhwd_reset_core((struct jpeghw_common_struct *)jhwdinfo);
    jhwd_reset_idma(jhwdinfo);
    jhwd_reset_odma(jhwdinfo);
    return e_JPEGHW_SUCCESS;
}

// much like the finish decompression above except there is no wait
// for completion of the DMA list destruction.
jpeghw_error_type_t jhw_abort_decompress(struct jpeghw_decompress_struct * jhwdinfo)
{
    DBG_INFO("[%d]%s()\n",gettid(), __func__);
    // make sure the input parameter is valid
    JHWD_VALIDATE(jhwdinfo);

    // the false in the 2nd parameter below tells the destroy calls to not wait
    // for decode complete.  
    jhwd_destroy_idma_list(jhwdinfo, false);
    jhwd_destroy_odma_list(jhwdinfo, false);

    return e_JPEGHW_SUCCESS;
}

// trims extra data from image.  The extra data is there because when jpeg encodes
// it must align with mcu dimensions.  
static void _jhwd_trim_buffer(struct jpeghw_decompress_struct * jhwdinfo, 
         struct BigBuffer_s *big_buffer, uint32_t *bytes_read, bool last_buffer)
{
    DBG_INFO("[%d]%s() last-buffer:%d\n",gettid(), __func__, last_buffer);
    jpeghw_dev_ptr   jdev = JPEGHWD_GET_DEV(jhwdinfo);

    // see if it's already mcu width aligned, if so leave
    if (!last_buffer && jdev->info->image_width % jdev->info->mcu_width == 0)
    {
       return;
    }

    uint32_t image_width = jdev->info->image_width;
    uint32_t image_byte_width = image_width * jdev->info->bytes_per_pixel;
    uint32_t image_height = jdev->info->image_height;
    uint32_t decoded_width = jdev->mcu_aligned_width;
    uint32_t decoded_byte_width = decoded_width * jdev->info->bytes_per_pixel;
    uint32_t decoded_height = jdev->mcu_aligned_height;
    uint32_t line=0;
    uint32_t last_line = *bytes_read/decoded_byte_width;
 
    DBG_INFO("[%d] iw:%d ih:%d dw:%d dh:%d bpp:%d\n", gettid(), image_width, image_height, 
             decoded_width, decoded_height, jdev->info->bytes_per_pixel);
    DBG_INFO("[%d] trim %d bytes from lines and %d lines from image\n", gettid(),
             decoded_byte_width - image_byte_width, decoded_height - image_height);

    char *buf = dma_buffer_mmap_forcpu(big_buffer);

    if (last_buffer)
    {
       last_line -= decoded_height - image_height;
    }

    DBG_INFO("buf:%p last-line:%d\n", buf, last_line);

    // compress the data to remove the end non-image data
    for (line = 1; line < last_line; line++)
    {
        char *srcloc = buf + (line * decoded_byte_width);
        char *dstloc = buf + (line * image_byte_width);

        //DBG_INFO("%p <- %p 0x%02x\n", dstloc, srcloc, image_byte_width);
        memcpy(dstloc, srcloc, image_byte_width);
        *bytes_read -= decoded_byte_width - image_byte_width;
    }
    *bytes_read -= decoded_byte_width - image_byte_width;

    dma_buffer_unmmap_forcpu(big_buffer);
}

// read enough data from the output DMA to fill the given BigBuffer.
// the paramter bytes_read will return the actual number of bytes residing
// in the BigBuffer.
// the parameter last_buffer will return true if the current BigBuffer
// contains the last data for this given decompression
jpeghw_error_type_t jhw_read_buffer(struct jpeghw_decompress_struct * jhwdinfo,
      struct BigBuffer_s *big_buffer, uint32_t *bytes_read, bool *last_buffer)
{
    DBG_INFO("[%d]%s()\n",gettid(), __func__);
    JHWD_VALIDATE(jhwdinfo);
    struct jpeg_dev_info_s * dev = JHWD_GET_DEV(jhwdinfo);

    if(jhwd_get_odma_data(jhwdinfo, big_buffer, bytes_read, last_buffer) != OK)
    {
        return e_JPEGHW_ERROR;
    }

    if (dev->decode_trim)
    {
        _jhwd_trim_buffer(jhwdinfo, big_buffer, bytes_read, *last_buffer);
    }

    return e_JPEGHW_SUCCESS;
}
  
