/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014-2015, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/



#include <string.h>

#include "dma_buffer.h"
#include "jhw_dev.h"
#include "jhw_dma.h"
#include "jhw_uio.h"

#define ENC_WAIT_VAL 100   // usecs
#define ENC_TIMEOUT_VAL 30000000 / ENC_WAIT_VAL // timeout=30 sec

int jhwc_destroy_odma_list(struct jpeghw_compress_struct * jhwcinfo, bool wait_for_complete)
{
    DBG_INFO("[%d]%s wait %d\n",gettid(), __func__, wait_for_complete);
    struct jpeg_dev_info_s * dev = JHWC_GET_DEV(jhwcinfo);
    int retval = OK;
    int event_flags = TERMINATE_THRD;
    jpeg_event_t jpeg_event;
    jpeg_event.event_flags = event_flags;
    jpeg_event.dev_info_ptr = dev;

    // prevents hangs if the hardware is not responding   
    if(wait_for_complete)
    {
        int inc;
        for(inc=0; inc<ENC_TIMEOUT_VAL; inc++)
        {
            if(dev->encode_complete) break;

            DBG_INFO("[%d]  wait sleep for %dus\n",gettid(), ENC_WAIT_VAL);

            posix_sleep_us(ENC_WAIT_VAL);
        }

        if(inc == ENC_TIMEOUT_VAL)
        {
            DBG_INFO("[%d]  send omsg 0x%08x len:%d mqd:%d\n",gettid(), event_flags, 
                                      sizeof(jpeg_event_t), dev->omqd.mqd);
            mq_send(dev->omqd.mqd, (char *)&jpeg_event, sizeof(jpeg_event_t), 0);
            DBG_ERR("[%d]Never received the ENCODE-COMPLETE\n",gettid());
            retval = FAIL;
        }
    }
    else
    {
        DBG_ERR("[%d]ODMA Abort\n",gettid());
        mq_send(dev->omqd.mqd, (char *)&jpeg_event, sizeof(jpeg_event_t), 0);
    }
    
    //DBG_INFO("Wait for thread to complete\n");
    pthread_join(dev->odma_thd_id, NULL);
    DEC_G_ALLOCATED();
    MEM_FREE_UNCACHED(dev->odma_desc->hwaddr, dev->odma_desc);
    DBG_INFO("[%d]%s odma thread closed\n",gettid(), __func__);

    return retval;
}

int jhwc_init_odma_list(struct jpeghw_compress_struct * jhwcinfo)
{
    DBG_INFO("[%d]%s\n",gettid(), __func__);
    struct jpeg_dev_info_s * dev = JHWC_GET_DEV(jhwcinfo);
    void *odma_hwaddr;

    dev->encode_complete = false;
    INC_G_ALLOCATED();
    dev->odma_desc = (jhw_dma_desc_t *)MEM_MALLOC_UNCACHED(&odma_hwaddr, sizeof(jhw_dma_desc_t), e_32_byte);
    if(dev->odma_desc == NULL)
    {
        DBG_ERR("[%d]Failed to allocate memory for odma descriptor\n",gettid());
        jpeghw_error(&jhwcinfo->common, e_JPEGHW_ERR_INVALID_PARAMETERS);
        return FAIL;
    }
    memset(dev->odma_desc, 0, sizeof(jhw_dma_desc_t));
    dev->odma_desc->hwaddr = odma_hwaddr;
    DBG_INFO("[%d]%s   new odma desc:%p hwaddr:%p to odma_desc\n", gettid(), __func__, dev->odma_desc, dev->odma_desc->hwaddr);

    return OK;
}

static void * _jhwc_start_odma(void * arg)
{
    struct jpeghw_compress_struct * jhwcinfo = (struct jpeghw_compress_struct *)arg;
    struct jpeg_dev_info_s * dev = JHWC_GET_DEV(jhwcinfo);
    DBG_INFO("[%d]%s dev:0x%08x\n",gettid(), __func__, (unsigned int)dev);

    if(jhwcinfo->cmgr == NULL)
    {
        DBG_ERR("[%d]%s compression mgr is NULL\n",gettid(), __func__);
        jpeghw_error(&jhwcinfo->common, e_JPEGHW_ERR_INVALID_PARAMETERS);
        return 0;
    }
    if(jhwcinfo->cmgr->send_output_buffer == NULL)
    {
        DBG_ERR("[%d]%s send_output_buffer callback is NULL\n",gettid(), __func__);
        jpeghw_error(&jhwcinfo->common, e_JPEGHW_ERR_INVALID_PARAMETERS);
        return 0;
    }

    u_int32_t encode_flag = 0;

    while(1) // until image is encoded
    {
        INC_G_ALLOCATED();  // assuming the user allocated this big_buffer
        struct BigBuffer_s *big_buffer = jhwcinfo->cmgr->get_output_buffer(jhwcinfo);
        if(big_buffer == NULL)
        {
            DBG_INFO("[%d]Unexpected end of data.  TERMINATE THREAD\n",gettid());
            jpeghw_error(&jhwcinfo->common, e_JPEGHW_ERR_INVALID_PARAMETERS);
            return NULL;
        }

        dev->odma_desc->big_buffer = big_buffer;
        dev->odma_desc->control_reg = READY_HW_OWN | INT_ON_FINISH; 
        dev->odma_desc->length = dev->odma_desc->big_buffer->datalen;
        INC_G_SMAPPED();
		dev->odma_desc->source_addr =
				(uintptr_t)(dma_buffer_map_single(dev->odma_desc->big_buffer, DMA_FROM_DEVICE));
        dev->odma_desc->next_desc_addr = 0;

        POGO_ODMA_UDMA_REGS_t* odma = (POGO_ODMA_UDMA_REGS_t*) dev->reg.jpeg_odma_udma;
        posix_sleep_us(1); // hold for a few instruction cycles

#ifdef ASIC_JPEG64
        uint64_t addr = (uintptr_t)dev->odma_desc->hwaddr;
        DBG_INFO("[%d]%s  hwaddr:0x%x\n", gettid(), __func__, addr);

        // Write lower and upper 32-bits of starting descriptor 64-bit address
        odma->ULDSR = (uint32_t)addr;
        odma->UUDSR = (uint64_t)addr >> 32;

        DBG_INFO("[%d]%s  set descriptor(len:%d) into odma-ULDSR:0x%08x odma->UUDSR:0x%08x\n", gettid(), __func__,
                         dev->odma_desc->big_buffer->datalen, odma->ULDSR, odma->UUDSR);

        // A write of any value to UDSR will cause the DMA to
        // start by fetching the descriptor at the address pointed to by
        // the Upper and Lower Descriptor Address Registers
        odma->UDSR = (uint32_t)addr;
#else
        odma->UDR = (uintptr_t)dev->odma_desc->hwaddr;
        DBG_INFO("[%d]%s  set descriptor(len:%d) into odma-UDR:0x%08x\n", gettid(), __func__,
                         dev->odma_desc->big_buffer->datalen, odma->UDR);
#endif

        jpeg_event_t jpeg_event;
        uint32_t timeout_in_seconds = 1;
        
        do
        {
            DBG_INFO("[%d]<<---wait for omsg mqd:%d for %d seconds...\n",gettid(), dev->omqd.mqd, timeout_in_seconds);

            jpeg_event.event_flags = 0;
            int rv = posix_wait_for_message(dev->omqd.mqd, (char *)&jpeg_event,
                                           sizeof(jpeg_event_t), (timeout_in_seconds*USEC_PER_SECOND));
            if(rv == ETIMEDOUT)
            {
            	DBG_WARN("[%d]%s(): WARNING - posix_wait_for_message timed out\n",gettid(), __func__);
            }

            if (jpeg_event.event_flags & JPEG_ENCODED) {
            	encode_flag = JPEG_ENCODED;
            }
            if (encode_flag == JPEG_ENCODED) {
            	if (jpeg_event.event_flags & JPEG_DMA_OUT) {
            		encode_flag |= JPEG_DMA_OUT;
            	}
            }

            DBG_INFO("[%d]<<---received omsg mqd:%d 0x%08x\n",gettid(), 
                                                  dev->omqd.mqd, jpeg_event.event_flags);
        }
        while(! ((jpeg_event.event_flags & JPEG_DMA_OUT) ||
                 (jpeg_event.event_flags && ((encode_flag & (JPEG_DMA_OUT | JPEG_ENCODED)) == (JPEG_DMA_OUT | JPEG_ENCODED))) ||
                 (jpeg_event.event_flags & TERMINATE_THRD) ||
                 (jpeg_event.event_flags & JPEG_ENCODE_ERROR)) );

        if(jpeg_event.event_flags & TERMINATE_THRD || jpeg_event.event_flags & JPEG_ENCODE_ERROR)
        {
            DEC_G_ALLOCATED();  // assuming the user freed this big_buffer
            DBG_INFO("[%d]TERMINATE THREAD\n",gettid());
            jpeghw_error(&jhwcinfo->common, e_JPEGHW_ERROR);
            return NULL;
        }

        if(jpeg_event.event_flags & JPEG_DMA_OUT)
        {
            DEC_G_SMAPPED();
            dma_buffer_unmap_single(dev->odma_desc->big_buffer, DMA_FROM_DEVICE);

            DBG_INFO("[%d]      bytes in buffer:%d\n",gettid(), dev->core_info.jpeg_odma_oblr);
            DEC_G_ALLOCATED();  // assuming the user freed this big_buffer
            if(jhwcinfo->cmgr->send_output_buffer(jhwcinfo, 
                    dev->odma_desc->big_buffer, dev->core_info.jpeg_odma_oblr) == false)
            {
                DBG_INFO("[%d]USER TERMINATED THREAD\n",gettid());
                return NULL;
            }
        }

        if(jpeg_event.event_flags & JPEG_ENCODED)
        {
            DBG_INFO("[%d]JPEG ENCODED\n",gettid());
            dev->encode_complete = true;
            return NULL;
        }
    }
}

int jhwc_start_odma(struct jpeghw_compress_struct * jhwcinfo)
{
    struct jpeg_dev_info_s * dev = JHWC_GET_DEV(jhwcinfo);

    DBG_INFO("[%d]%s dev:0x%08x\n",gettid(), __func__, (unsigned int)dev);
    int ret = pthread_create(&(dev->odma_thd_id), 0, _jhwc_start_odma, (void *)jhwcinfo);
    DBG_INFO("[%d] thread started ret:%d\n",gettid(), ret);

    if(ret == 0) return OK;
    
    jpeghw_error(&jhwcinfo->common, e_JPEGHW_ERROR);
    return FAIL;
}
