/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014-2015, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/



#include <string.h>

#include "dma_buffer.h"
#include "jhw_dev.h"
#include "jhw_dma.h"
#include "jhw_uio.h"

// wait for an IDMA interrupt
//static int _jhwc_wait_for_idma(struct jpeg_dev_info_s * dev)
static int _jhwc_wait_for_idma(struct jpeghw_compress_struct *jhwcinfo)
{
    struct jpeg_dev_info_s * dev = JHWD_GET_DEV(jhwcinfo);
    jpeg_event_t jpeg_event;
    jpeg_event.event_flags = 0;
    // if the timeout callback is NULL the user does not want to manage timeouts
    // therefore set the timeout to 10 seconds
    // otherwise call the user every second while we are waiting
    uint32_t timeout_in_seconds = (jhwcinfo->common.scanline_timeout == NULL) ? 
                   DEFAULT_SCANLINE_TIMEOUT : 1; // default seconds if no timeout callback
    uint32_t timeout_count = 0;

    do
    {
        DBG_INFO("[%d]--->>wait for imsg mqd:%d for %d seconds...\n",
                                       gettid(), dev->imqd.mqd, timeout_in_seconds);

        jpeg_event.event_flags = 0;
        int rv = posix_wait_for_message(dev->imqd.mqd, (char *)&jpeg_event,
                               sizeof(jpeg_event_t), (timeout_in_seconds*USEC_PER_SECOND));
        if(rv == ETIMEDOUT)
        {
            DBG_WARN("[%d]%s(): WARNING - posix_wait_for_message timed out\n",gettid(), __func__);

            if (jhwcinfo->common.scanline_timeout == NULL)
            {
                jpeg_event.event_flags = JPEG_ENCODE_TIMEOUT;
            }
            else
            {
                timeout_count++;

                // ask the user if he wants to timeout or not.  true = timeout
                if (jhwcinfo->common.scanline_timeout(
                                (struct jpeghw_common_struct *)jhwcinfo, timeout_count))
                {
                    jpeg_event.event_flags = JPEG_ENCODE_TIMEOUT;
                }
            }
        }
        else
        {
            timeout_count = 0;
        }

        DBG_INFO("[%d]--->>received imsg 0x%08x\n",gettid(), jpeg_event.event_flags);
    }
    while(! (jpeg_event.event_flags & JPEG_DMA_IN ||
                jpeg_event.event_flags & JPEG_ENCODED ||
                jpeg_event.event_flags & JPEG_ENCODE_ERROR ||
                jpeg_event.event_flags & JPEG_ENCODE_TIMEOUT ||
                jpeg_event.event_flags & POGO_DMA_HW_OWN));

    return jpeg_event.event_flags;
}

//static int _jhwc_wait_for_idma_hw_own(struct jpeg_dev_info_s * dev)
static int _jhwc_wait_for_idma_hw_own(struct jpeghw_compress_struct *jhwcinfo)
{
    int ev = 0;
    while(! (ev & POGO_DMA_HW_OWN || ev & JPEG_ENCODE_ERROR ||
             ev & JPEG_ENCODED || ev & JPEG_ENCODE_TIMEOUT))
    {
        ev = _jhwc_wait_for_idma(jhwcinfo);
    }
    return ev;
}

// free any memory allocated to old and older big_buffers.
// free all memory allocated for the creation of a DMA descriptor list.
int jhwc_destroy_idma_list(struct jpeghw_compress_struct * jhwcinfo, bool wait_for_complete)
{
    DBG_INFO("[%d]%s\n",gettid(), __func__);
    struct jpeg_dev_info_s * dev = JHWC_GET_DEV(jhwcinfo);
   
    if (wait_for_complete) {
		struct mq_attr attr;
		do
		{
			mq_getattr(dev->imqd.mqd, &attr);
			if (attr.mq_curmsgs > 0) {
				// wait for last big buffer to complete
				_jhwc_wait_for_idma(jhwcinfo);
			}
		} while(attr.mq_curmsgs > 1);
	}
	
    // free old big buffers
    dev->bb_info.old_big_buffer = JHW_FREE_MAPPED_BIG_BUFFER(dev->bb_info.old_big_buffer);
    dev->bb_info.older_big_buffer = JHW_FREE_MAPPED_BIG_BUFFER(dev->bb_info.older_big_buffer);
    dev->bb_info.split_buffer = JHW_FREE_MAPPED_BIG_BUFFER(dev->bb_info.split_buffer);
    dev->bb_info.too_small_big_buffer =
          JHW_FREE_BIG_BUFFER(dev->bb_info.too_small_big_buffer); // just in case it exists

    // destroy the list of descriptors
    jhw_destroy_dma_descriptors(dev->idma_desc_list, dev->number_idma_descriptors);

    return OK;
}

// create an IDMA descriptor list containing "number_idma_descriptors" 
// number of descriptors
int jhwc_init_idma_list(struct jpeghw_compress_struct * jhwcinfo)
{
    DBG_INFO("[%d]%s\n",gettid(), __func__);
    struct jpeg_dev_info_s * dev = JHWC_GET_DEV(jhwcinfo);

    if(dev->number_idma_descriptors == 0)
    {
        jpeghw_error(&jhwcinfo->common, e_JPEGHW_ERR_INVALID_PARAMETERS);
        return FAIL;
    }
    dev->dma_list_idx = 0;
    dev->line_counter = 0;
    dev->bb_info.old_big_buffer = NULL;
    dev->bb_info.older_big_buffer = NULL;
    dev->bb_info.split_buffer = NULL;

    dev->idma_desc_list = jhw_new_dma_descriptors(dev->number_idma_descriptors);
    if(dev->idma_desc_list == NULL)
    {
        DBG_ERR("[%d]Failed to allocate memory for idma descriptor list\n",gettid());
        jpeghw_error(&jhwcinfo->common, e_JPEGHW_ERR_OUT_OF_MEMORY);
        return FAIL;
    }

    return OK;
}

// Add the data contained in big_buffer to the IDMA.  
int jhwc_add_idma_data(struct jpeghw_compress_struct * jhwcinfo, 
                       struct BigBuffer_s *big_buffer, bool last_buffer)
{
    jpeghw_dev_ptr   jdev = JPEGHWC_GET_DEV(jhwcinfo);
    struct jpeg_dev_info_s * dev = JHWC_GET_DEV(jhwcinfo);
    dev->dma_list_idx = 0;

    // sanity check
    if(big_buffer == NULL)
    {
        DBG_ERR("[%d]%s big_buffer is NULL\n",gettid(), __func__);
        jpeghw_error(&jhwcinfo->common, e_JPEGHW_ERR_INVALID_PARAMETERS);
        return FAIL;
    }

    if (dev == NULL)
    {
        DBG_ERR("[%d]%s dev is NULL\n",gettid(), __func__);
        jpeghw_error(&jhwcinfo->common, e_JPEGHW_ERR_INVALID_PARAMETERS);
        return FAIL;
    }

    // if encode has already completed why are they still sending data
    if(dev->encode_complete)
    {
        DBG_ERR("[%d]%s encode had already completed\n",gettid(), __func__);
        jpeghw_error(&jhwcinfo->common, e_JPEGHW_ERR_BAD_STATE);
        return  FAIL;
    }

    uint32_t idx=0;
    DBG_INFO("[%d]%s\n",gettid(), __func__);

    INC_G_SMAPPED();
    char *bb_source_addr = (char *)dma_buffer_map_single(big_buffer, DMA_TO_DEVICE);
    char *source_addr = bb_source_addr;
    uint32_t bb_source_lines = big_buffer->datalen / dev->byte_width;
    uint32_t source_lines = bb_source_lines;
    uint32_t mcu_aligned_chunk_len = dev->byte_width * jdev->info->mcu_height;

#ifdef ASIC_JPEG64
    uint64_t top_of_chain = (uintptr_t)dev->idma_desc_list[dev->dma_list_idx]->hwaddr;
#else
    uint32_t top_of_chain = (uintptr_t)dev->idma_desc_list[dev->dma_list_idx]->hwaddr;
#endif

    uint32_t borrow_lines = 0;
    bool descriptors_ready = false;
    int overrun_counter = 0;

    DBG_INFO("[%d]   srcaddr=0x%08x\n",gettid(), (unsigned int)source_addr);
    DBG_INFO("[%d]   source lines =%d mcu-v =%d\n",gettid(), source_lines, jdev->info->mcu_height);
    DBG_INFO("[%d]   remainder lines =%d\n",gettid(), dev->bb_info.remainder_lines);
    DBG_INFO("[%d]   remainder offset =%d\n",gettid(), dev->bb_info.remainder_offset);

    DBG_INFO("chunklen:%d byte_width:%d mcu-h:%d \n", 
              mcu_aligned_chunk_len, dev->byte_width, jdev->info->mcu_height);
    // Is there a big buffer pending deletion?
    if(dev->bb_info.old_big_buffer != NULL)
    {
        if(dev->bb_info.remainder_lines + source_lines < jdev->info->mcu_height)
        {
            DBG_ERR("[%d]--!!!---Remainder + big buffer < mcu height.  Aborting...\n",gettid());
            jpeghw_error(&jhwcinfo->common, e_JPEGHW_ERR_BAD_OBJECT);
            return FAIL;
        }

        // wait for the previous data to be consumed by the dma
        int evi = _jhwc_wait_for_idma_hw_own(jhwcinfo);
        if(evi == JPEG_ENCODED) return OK;
        if(evi == JPEG_ENCODE_ERROR)
        {
           jpeghw_error(&jhwcinfo->common, e_JPEGHW_ERROR);
           return FAIL;
        }
        if(evi == JPEG_ENCODE_TIMEOUT)
        {
           jpeghw_error(&jhwcinfo->common, e_JPEGHW_ERR_TIMEOUT);
           return TIMEOUT;
        }

        // Are there any lines remaining from the last call?  i.e. was the buffer of the last
        // call not properly aligned with the mcu height.  If so we need to program a split 
        // descriptor to pull data from both buffers, the old one and the current one
        if(dev->bb_info.remainder_lines > 0)
        {
            DBG_INFO("[%d]Set the remainder and borrowed lines in the dma split regs.\n",gettid());

            // Borrow lines from the new big buffer and combine them with the remainder
            // from the last call.  Use the pogo dma split registers that facilitate
            // pulling data from two locations to satisfy one dma transfer.
            borrow_lines = jdev->info->mcu_height - dev->bb_info.remainder_lines;
            DBG_INFO("[%d]  remainder lines =%d  borrow lines =%d\n",gettid(), 
                                        dev->bb_info.remainder_lines, borrow_lines);
            jhw_dma_desc_t *desc = dev->idma_desc_list[dev->dma_list_idx];

            DBG_INFO("[%d]write to split desc idx %d top-of-chain:0x%p\n",gettid(),dev->dma_list_idx,(void*)top_of_chain);

            descriptors_ready = true;

            DBG_INFO("[%d]  remainder:%d offset:%d old-lines:%d\n",gettid(),
                     dev->bb_info.remainder_lines,dev->bb_info.remainder_offset,
                     dev->bb_info.old_big_buffer->datalen/dev->byte_width);

            #ifdef USE_SPLIT_DESCRIPTORS

            desc->source_addr = (jpegdescsize_t)(dev->bb_info.buf_hwaddr + dev->bb_info.remainder_offset);
            desc->length = mcu_aligned_chunk_len;
            desc->control_reg |= READY_HW_OWN;

            // buffer splitting - tell this descriptor to pull from two buffers
            desc->src_addr_low_0 = desc->source_addr;
            desc->src_addr_high_0 = desc->source_addr +
                                       (dev->bb_info.remainder_lines * dev->byte_width) -1;
            desc->src_addr_low_1 = (uintptr_t)source_addr;
            desc->src_addr_high_1 = (uintptr_t)(source_addr + (borrow_lines * dev->byte_width) -1);

            DBG_INFO("[%d]  remainder-addr:%p borrow-addr:%p\n",gettid(),
                                     (void *)desc->source_addr, (void *)source_addr);

            #else

            // create a new contiguous buffer that will contain the remainder from
            // the last buffer and the lines borrowed from the current buffer in 
            // order to fullfill the mcu height requirement for the POGO DMAs
            INC_G_ALLOCATED();
            struct BigBuffer_s *nbb = dma_buffer_malloc(0, mcu_aligned_chunk_len);

            char *rbuf = dma_buffer_mmap_forcpu(dev->bb_info.old_big_buffer);
            char *bbuf = dma_buffer_mmap_forcpu(big_buffer);
            char *nbuf = dma_buffer_mmap_forcpu(nbb);

            // fill the new buffer with the old remainder and new borrowed lines
            uint32_t rlen = dev->bb_info.remainder_lines * dev->byte_width;
            memcpy(nbuf, rbuf + dev->bb_info.remainder_offset, rlen);
            memcpy(nbuf + rlen, bbuf, borrow_lines * dev->byte_width);

            dma_buffer_unmmap_forcpu(dev->bb_info.old_big_buffer);
            dma_buffer_unmmap_forcpu(big_buffer);
            dma_buffer_unmmap_forcpu(nbb);

            // program the descriptor with the new buffer
            desc->source_addr = (jpegdescsize_t)dma_buffer_map_single(nbb, DMA_TO_DEVICE);
            desc->length = mcu_aligned_chunk_len;
            desc->control_reg |= READY_HW_OWN;

            // no buffer splitting
            desc->src_addr_low_0 = desc->source_addr;
            desc->src_addr_high_0 = (desc->source_addr + desc->length -1);
            desc->src_addr_low_1 = desc->source_addr;
            desc->src_addr_high_1 = (desc->source_addr + desc->length -1);

            // free the old split buffer if it exists
            dev->bb_info.split_buffer = JHW_FREE_MAPPED_BIG_BUFFER(dev->bb_info.split_buffer);
            dev->bb_info.split_buffer = nbb;

            #endif

            source_addr += borrow_lines * dev->byte_width;
            // source_lines have to reflect the lines borrowed from source_lines
            source_lines -= borrow_lines;

            // update the idma list index
            dev->dma_list_idx++;
            dev->dma_list_idx = (dev->dma_list_idx % dev->number_idma_descriptors);
            overrun_counter++; // increment because we consumed a descriptor
        }
    }
    DBG_INFO("[%d]   new source lines=%d\n",gettid(), source_lines);

    // free the memory from the oldest buffer if any exists
    DBG_INFO("[%d]%s   free older big-buffer\n",gettid(), __func__);
    dev->bb_info.older_big_buffer = JHW_FREE_MAPPED_BIG_BUFFER(dev->bb_info.older_big_buffer);

    // save big_buffer information for next time this function is called
    dev->bb_info.older_big_buffer = dev->bb_info.old_big_buffer;
    DBG_INFO("Save %p into old buffer %p\n", big_buffer, dev);
    dev->bb_info.old_big_buffer = big_buffer;
    dev->bb_info.buf_hwaddr = (uintptr_t)bb_source_addr;
    // remainder_lines is calculated from source_lines.  if lines were borrowed
    // in order to create a split descriptor those lines should not be used to determine
    // if we have a current remainder left
    dev->bb_info.remainder_lines = source_lines % jdev->info->mcu_height;
    // the remainder offset is calculated from the beginning of the original big_buffer
    // so use bb_source_lines, not source_lines
    dev->bb_info.remainder_offset = (bb_source_lines - 
                                     dev->bb_info.remainder_lines) * dev->byte_width;

    POGO_IDMA_UDMA_REGS_t* idma = (POGO_IDMA_UDMA_REGS_t*) dev->reg.pogo_idma_udma;
    uint32_t number_of_mcu_aligned_chunks = source_lines / jdev->info->mcu_height;
    DBG_INFO("sourcelines=%d \n",source_lines  );
    DBG_INFO("[%d]   dev=%p chunklen=%d numchunks=%d\n",gettid(), dev,
                           mcu_aligned_chunk_len, number_of_mcu_aligned_chunks);
    DBG_INFO("[%d]   source_lines=%d chunklen=%d numchunks=%d remainder:%d\n",gettid(), source_lines,
              mcu_aligned_chunk_len, number_of_mcu_aligned_chunks, dev->bb_info.remainder_lines);

    // Add data to the dma chain in chunks each having a height equal to the
    // mcu height (this is a requirement of the hardware).  
    // the dma chain is a circular chain (next ptrs point to subsequent descriptor)
    // with the last descriptor's next ptr pointing to the first descriptor
    // When each descriptor is filled the OWN bit is set to the dma hw.  If we
    // have more data than the chain can hold (i.e. we run into a descriptor with
    // the OWN bit already set to dma hw) then we send what we have intialize to the
    // hardware by setting the UDR register and wait until the dma has consumed all of 
    // those descriptors.  Now we continue filling descriptors until we run out of data or
    // hit the OWN bit again.  continue this process until all data is consumed.
    for(idx=0; idx<number_of_mcu_aligned_chunks; idx++)
    {
        jhw_dma_desc_t *desc = dev->idma_desc_list[dev->dma_list_idx];

        // the overrun_counter is a fail safe if for some reason the output hardware
        // is consuming as fast as we build input dmas
        DBG_INFO("0x%08x %d\n", desc->control_reg & READY_HW_OWN,overrun_counter);
        if(desc->control_reg & READY_HW_OWN || overrun_counter == dev->number_idma_descriptors)
        {
            overrun_counter = 0;
            DBG_INFO("[%d] stop and wait for JPEG block to catch up idx=%d...\n",gettid(), dev->dma_list_idx);
            DBG_INFO("[%d]   load top_of_chain(0x%p) into UDR\n",gettid(), (void*)top_of_chain);

            // send what we have so far, then stop and wait for JPEG block to catch up
            posix_sleep_us(1);
#ifdef ASIC_JPEG64
            DBG_INFO("[%d]%s  top_of_chain:0x%x\n", gettid(), __func__, top_of_chain);

            // Write lower and upper 32-bits of the top of chain 64-bit address
            idma->ULDSR = (uint32_t)top_of_chain;
            idma->UUDSR = (uint64_t)top_of_chain >> 32;
            DBG_INFO("[%d]%s  set descriptors into idma-ULDSR/UUDSR:0x%08x/0x%08x\n", gettid(), __func__, idma->ULDSR, idma->UUDSR);

            // A write of any value to UDSR will cause the DMA to
            // start by fetching the descriptor at the address pointed to by
            // the Upper and Lower Address Registers
            idma->UDSR = (uint32_t)top_of_chain;
#else
            idma->UDR = top_of_chain;
            DBG_INFO("[%d]%s  set descriptors into idma-UDR:0x%08x\n", gettid(), __func__, idma->UDR);
#endif
            if(_jhwc_wait_for_idma_hw_own(jhwcinfo) == JPEG_ENCODE_ERROR)
            {
               jpeghw_error(&jhwcinfo->common, e_JPEGHW_ERROR);
               return FAIL;
            }

            top_of_chain = (uintptr_t)desc->hwaddr;
            DBG_INFO("[%d] running again...\n",gettid());
        }

        DBG_INFO("[%d]write to desc idx %d top-of-chain:%p\n",gettid(),dev->dma_list_idx,(void *)top_of_chain);
        descriptors_ready = true;

        // start addressing at the point after borrowed data (if any exists)
        desc->source_addr = (uintptr_t)(source_addr + (idx * mcu_aligned_chunk_len));
        desc->length = mcu_aligned_chunk_len;
        desc->control_reg |= READY_HW_OWN;

        // no buffer splitting in here
        desc->src_addr_low_0 = desc->source_addr;
        desc->src_addr_high_0 = desc->source_addr + desc->length -1;
        desc->src_addr_low_1 = desc->source_addr;
        desc->src_addr_high_1 = desc->source_addr + desc->length -1;

        // update the idma list index
        dev->dma_list_idx++;
        dev->dma_list_idx = (dev->dma_list_idx % dev->number_idma_descriptors);
        overrun_counter++; 
    }

    if(descriptors_ready)
    {
        DBG_INFO("[%d]   load top_of_chain(0x%p) into UDR\n", gettid(), (void *)top_of_chain);
        // tell the dma to process the rest of the data, we are outa here
        posix_sleep_us(1);
#ifdef ASIC_JPEG64
        // Write lower and upper 32-bits of the top of chain 64-bit address
        idma->ULDSR = (uint32_t)top_of_chain;
        idma->UUDSR = (uint64_t)top_of_chain >> 32;
        DBG_INFO("[%d]%s  set descriptors into idma-ULDSR/UUDSR:0x%08x/0x%08x\n", gettid(), __func__, idma->ULDSR, idma->UUDSR);

        // A write of any value to UDSR will cause the DMA to
        // start by fetching the descriptor at the address pointed to by
        // the Upper and Lower Address Registers
        idma->UDSR = (uint32_t)top_of_chain;
#else
        idma->UDR = top_of_chain;
        DBG_INFO("[%d]%s  set descriptors into idma-UDR:0x%08x\n", gettid(), __func__, idma->UDR);
#endif
    }
    else
    {
        // no descriptors read so send a fake descriptors read message
        jpeg_event_t jpeg_event;
        jpeg_event.event_flags = POGO_DMA_HW_OWN;
        jpeg_event.dev_info_ptr = dev;
        DBG_INFO("[%d]  no descriptors ready... send imsg 0x%08x\n",gettid(), jpeg_event.event_flags);
        mq_send(dev->imqd.mqd, (char *)&jpeg_event, sizeof(jpeg_event_t), 0);
        return OK;
    }

    return OK;
}
