/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014-2015, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/



#include "jhw_api.h"
#include "jhw_dev.h"
#include "jhw_asic.h"
#include "jhw_core.h"

/*
 *  \brief detail literals -- subsampling, huffman
 */
#define DEFAULT_H0_M1                     1
#define DEFAULT_V0_M1                     1
#define DEFAULT_H1_M1                     0
#define DEFAULT_V1_M1                     0
#define DEFAULT_H2_M1                     0
#define DEFAULT_V2_M1                     0
#define DEFAULT_H3_M1                     0
#define DEFAULT_V3_M1                     0
#define DEFAULT_PAD_X                     0
#define DEFAULT_PAD_VAL                   0x0
#define DEFAULT_HUFF_DC_TABLE_SEL         0x6
#define DEFAULT_HUFF_AC_TABLE_SEL         0x6
#define DEFAULT_DC_CODE_EN_0              0x1fe
#define DEFAULT_DC_CODE_EN_1              0x7fe
#define DEFAULT_AC_CODE_EN_0              0xcffe
#define DEFAULT_AC_CODE_EN_1              0xeffe
#define DEFAULT_RESTART_ENABLE            0
#define DEFAULT_RESTART_INTERVAL_M1       0
#define DEFAULT_Q_MAP_0                   0x0
#define DEFAULT_Q_MAP_1                   0x1
#define DEFAULT_Q_MAP_2                   0x1
#define DEFAULT_Q_MAP_3                   0x0

// display contents of core registers
//static void _show_core_regs(JPEG_CORE_REGS_t *r)
//{
//    DBG_INFO(" Core Regs:\n");
//    DBG_INFO("  IRQ_EN        :0x%08x\n", r->IRQ_EN);
//    DBG_INFO("  IRQ_PEND      :0x%08x\n", r->IRQ_PEND);
//    DBG_INFO("  IRQ_ACK       :0x%08x\n", r->IRQ_ACK);
//    DBG_INFO("  CR            :0x%08x\n", r->CR);
//    DBG_INFO("  IMAGE         :0x%08x\n", r->IMAGE);
//    DBG_INFO("  DIM           :0x%08x\n", r->DIM);
//    DBG_INFO("  PAD           :0x%08x\n", r->PAD);
//    DBG_INFO("  PAD_VAL       :0x%08x\n", r->PAD_VAL);
//    DBG_INFO("  SUB           :0x%08x\n", r->SUB);
//    DBG_INFO("  HUF_CODE_EN_DC:0x%08x\n", r->HUFF_CODE_EN_DC);
//    DBG_INFO("  HUF_CODE_EN_AC:0x%08x\n", r->HUFF_CODE_EN_AC);
//    DBG_INFO("  RESTART       :0x%08x\n", r->RESTART);
//    DBG_INFO("  QUANT         :0x%08x\n", r->QUANT);
//    DBG_INFO("  STATUS0       :0x%08x\n", r->STATUS0);
//    DBG_INFO("  STATUS1       :0x%08x\n", r->STATUS1);
//    DBG_INFO("  STATUS2       :0x%08x\n", r->STATUS2);
//    DBG_INFO("  STATUS3       :0x%08x\n", r->STATUS3);
//    DBG_INFO("  STATUS4       :0x%08x\n", r->STATUS4);
//    DBG_INFO("  STATUS5       :0x%08x\n", r->STATUS5);
//    DBG_INFO("  STATUS6       :0x%08x\n", r->STATUS6);
//    DBG_INFO("  STATUS7       :0x%08x\n", r->STATUS7);
//    DBG_INFO("  STATUS8       :0x%08x\n", r->STATUS8);
//    DBG_INFO("  STATUS9       :0x%08x\n", r->STATUS9);
//}

// initialize the internal huffman registers and tables
static bool _jhwc_init_huffman(struct jpeghw_compress_struct * jhwcinfo, JPEG_CORE_REGS_t* core_reg)
{
    DBG_INFO("[%d]%s\n",gettid(), __func__);
    struct jpeg_dev_info_s * dev = JHWC_GET_DEV(jhwcinfo);
    device_core_info_ptr core_info = &dev->core_info;
    core_reg->CR |= JPEG_CORE_CR_SRAM_ACCESS_ENABLE_MASK;

    uint32_t reg_temp;

    core_info->huff_dc_table_sel = DEFAULT_HUFF_DC_TABLE_SEL; // 0x6
    core_info->huff_ac_table_sel = DEFAULT_HUFF_AC_TABLE_SEL; // 0x6

    //HUFFMAN TABLE Select Register
    reg_temp = 0;
    reg_temp = JPEG_CORE_HUFF_DC_TABLE_SEL_REPLACE_VAL(reg_temp,core_info->huff_dc_table_sel);
    reg_temp = JPEG_CORE_HUFF_AC_TABLE_SEL_REPLACE_VAL(reg_temp,core_info->huff_ac_table_sel);
    core_reg->HUFF = reg_temp;

    //HUFFMAN DC Register
    reg_temp = 0;
    reg_temp = JPEG_CORE_HUFF_CODE_EN_DC_DC_CODE_EN_1_REPLACE_VAL(reg_temp,core_info->dc_code_en_1);
    reg_temp = JPEG_CORE_HUFF_CODE_EN_DC_DC_CODE_EN_0_REPLACE_VAL(reg_temp,core_info->dc_code_en_0);
    core_reg->HUFF_CODE_EN_DC = reg_temp;

    //HUFFMAN AC Register
    reg_temp = 0;
    reg_temp = JPEG_CORE_HUFF_CODE_EN_AC_AC_CODE_EN_1_REPLACE_VAL(reg_temp,core_info->ac_code_en_1);
    reg_temp = JPEG_CORE_HUFF_CODE_EN_AC_AC_CODE_EN_0_REPLACE_VAL(reg_temp,core_info->ac_code_en_0);
    core_reg->HUFF_CODE_EN_AC = reg_temp;

    // HUFFMAN RESTART
    reg_temp = 0;
    reg_temp = JPEG_CORE_RESTART_RESTART_ENABLE_REPLACE_VAL(reg_temp,DEFAULT_RESTART_ENABLE);
    reg_temp = JPEG_CORE_RESTART_RESTART_INTERVAL_M1_REPLACE_VAL(reg_temp,DEFAULT_RESTART_INTERVAL_M1);
    core_reg->RESTART = reg_temp;

    // QUANTIZIZER - QTABLE MAPPING
    reg_temp = 0;
    reg_temp = JPEG_CORE_QUANT_Q_MAP_0_REPLACE_VAL(reg_temp,DEFAULT_Q_MAP_0);
    reg_temp = JPEG_CORE_QUANT_Q_MAP_1_REPLACE_VAL(reg_temp,DEFAULT_Q_MAP_1);
    reg_temp = JPEG_CORE_QUANT_Q_MAP_2_REPLACE_VAL(reg_temp,DEFAULT_Q_MAP_2);
    reg_temp = JPEG_CORE_QUANT_Q_MAP_3_REPLACE_VAL(reg_temp,DEFAULT_Q_MAP_3);
    core_reg->QUANT = reg_temp;

    // set ECS offset to zero just to be safe it is not used Encode operations
    core_reg->ECS_OFFSET = 0;

    /***************************************
    Setting SRAM access enable bit of CR register. This allows loading of Huffman tables
    **********************************/
    core_reg->CR |= JPEG_CORE_CR_SRAM_ACCESS_ENABLE_MASK;

    /**********************************
    Writing HUFF_DEC_FSYMBOL table
    **********************************/
    int i;
    for(i=0; i<64; i++)
    {
      core_reg->HUFF_FCODE[i] = core_info->huff_fsymbol_array[i];
    }
    /**********************************
    Writing HUFF_DEC_LSYMBOL Table
    **********************************/
    for(i=0; i<64; i++)
    {
        core_reg->HUFF_LCODE[i] = core_info->huff_lsymbol_array[i];
    }
    /***********************************
    HUFF_DEC_PTR Table
    **********************************        */
    for(i=0; i<64; i++)
    {
        core_reg->HUFF_PTR[i] = core_info->huff_ptr_array[i];
    }
    /**********************************
    HUFF_SYM Table
    **********************************/
    for(i=0; i<348; i++)
    {
        core_reg->HUFF_SYM[i] = core_info->huff_sym_array[i];
    }

    /**********************************
    HUFF_LEN_Table
    **********************************/
    for(i=0; i<348; i++)
    {
        core_reg->LEN_CODE[i] = core_info->huff_code_array[i];
    }

    /**********************************
    Writing Quantization table
    **********************************/
    // retrieve the user quant table and use it if it's there
	for(i=0; i<sizeof(core_info->dequant_q_array); i++)
	{
		core_reg->Q_TABLE[i] = core_info->dequant_q_array[i];
	}

    /**********************************
    Clearing the sram access enable bit.
    **********************************        */
    core_reg->CR &= ~JPEG_CORE_CR_SRAM_ACCESS_ENABLE_MASK;
    return true;
}

// soft reset the JPEG core
static void _jhw_reset_core(struct jpeg_dev_info_s * dev) // jpeg control block register
{
    JPEG_TOP_REGS_t*   _top = (JPEG_TOP_REGS_t*) dev->reg.top; // block top (io) reg
    JPEG_CORE_REGS_t* _core = (JPEG_CORE_REGS_t*) dev->reg.core; // block core reg

    DBG_INFO("[%d]%s(%#x) _core %p top %p \n",gettid(), __func__, (unsigned int)dev, _core, _top);

    //dbg DBG("%s() dev %#x base %#x io %#x\n", __func__, dev, _core, _top);
    if (_top && _core)
    {
        // jpeg core/header
        _core->CR = JPEG_CORE_CR_CLR_HDR_DATA_REPLACE_VAL(_core->CR, 1);
        posix_sleep_us(3);
        _core->CR = JPEG_CORE_CR_CLR_HDR_DATA_REPLACE_VAL(_core->CR, 0);

        // jpeg io
        _top->CR = JPEG_TOP_CR_SOFT_RESET_REPLACE_VAL(_top->CR, 1);
        posix_sleep_us(3); // threadx cpu_spin_delay(3);
        _top->CR = JPEG_TOP_CR_SOFT_RESET_REPLACE_VAL(_top->CR, 0);
    }
}

// initialize the JPEG registers for compression
static int _jhwc_set_compression_regs(struct jpeghw_compress_struct * jhwcinfo)
{
    DBG_INFO("[%d]%s\n",gettid(), __func__);
    jpeghw_dev_ptr   jdev = JPEGHWC_GET_DEV(jhwcinfo);
    struct jpeg_dev_info_s * dev = JHWC_GET_DEV(jhwcinfo);
    device_core_info_ptr core_info = &dev->core_info;
    int bytes_per_pixel = jdev->info->bytes_per_pixel;

    DBG_INFO("[%d] assign register base addrs\n",gettid());
    // encoding dma shadow registers
    JPEG_CORE_REGS_t*      _core = (JPEG_CORE_REGS_t*)      dev->reg.core;
    JPEG_IDMA_UDMA_REGS_t* _idma = (JPEG_IDMA_UDMA_REGS_t*) dev->reg.pogo_idma_udma;
    JPEG_ODMA_UDMA_REGS_t* _odma = (JPEG_ODMA_UDMA_REGS_t*) dev->reg.jpeg_odma_udma;
    POGO_IDMA_CORE_REGS_t* _icor = (POGO_IDMA_CORE_REGS_t*) dev->reg.pogo_idma_core;


    DBG_INFO("[%d] disable all interrupts\n",gettid());
    // disable all interrupt sources and then soft reset the jpeg block
    _core->IRQ_EN = 0;
    _idma->UIER = 0;
    _icor->IIER = 0;
    _odma->UIER = 0;

    DBG_INFO(" reset hardware\n");
    // soft resetjpeg_hw
    _jhw_reset_core(dev);
    jhwc_reset_idma(jhwcinfo);
    jhwc_reset_odma(jhwcinfo);

    _core->CR = 0x00000000; // blank core mode

    if (jhwcinfo->auto_generate_jpeg_header) // generate jpeg header?
    {
        _core->CR |= JPEG_CORE_CR_CLR_HDR_DATA_MASK; // clr header data
        posix_sleep_us(3); // hold for a few instruction cycles
        _core->CR &= ~JPEG_CORE_CR_CLR_HDR_DATA_MASK;
        DBG_INFO("[%d]%s() encoding .. header reset\n",gettid(), __func__);

        // enable header generation
        _core->CR = JPEG_CORE_CR_HEADER_ENABLE_REPLACE_VAL(_core->CR, 1);
    }

    DBG_INFO("[%d]%s() mcu:( %d x %d ) image:( %d x %d )\n",gettid(), __func__,
        jdev->info->mcu_width, jdev->info->mcu_height, jdev->info->image_width, jdev->info->image_height);

    _icor->IRHR = jdev->info->mcu_height;

    // program the core jpeg block with the image info
    _core->IMAGE = JPEG_CORE_IMAGE_NUM_PLANES_M1_REPLACE_VAL(_core->IMAGE, bytes_per_pixel-1);
    _core->DIM = JPEG_CORE_DIM_Y_REPLACE_VAL(_core->DIM, jdev->info->image_height);
    _core->DIM = JPEG_CORE_DIM_X_REPLACE_VAL(_core->DIM, jdev->info->image_width);
    _icor->ILWR = dev->byte_width;
    DBG_INFO("[%d]%s() src (%d x %d x %d)\n",gettid(), __func__, jdev->info->image_width,
                    jdev->info->image_height, bytes_per_pixel);

    DBG_INFO("[%d]%s() (ilwr %d) * (mcu irhr %d) -> mcu-dma-area %d\n", gettid(), __func__,
    				_icor->ILWR, _icor->IRHR, _icor->ILWR * _icor->IRHR);

    _core->PAD = core_info->pad;
    _core->PAD_VAL = core_info->pad_value;

    _core->SUB = core_info->v3_m1 << JPEG_CORE_SUB_V3_M1_SHIFT |
                 core_info->h3_m1 << JPEG_CORE_SUB_H3_M1_SHIFT |
                 core_info->v2_m1 << JPEG_CORE_SUB_V2_M1_SHIFT |
                 core_info->h2_m1 << JPEG_CORE_SUB_H2_M1_SHIFT |
                 core_info->v1_m1 << JPEG_CORE_SUB_V1_M1_SHIFT |
                 core_info->h1_m1 << JPEG_CORE_SUB_H1_M1_SHIFT |
                 core_info->v0_m1 << JPEG_CORE_SUB_V0_M1_SHIFT |
                 core_info->h0_m1 << JPEG_CORE_SUB_H0_M1_SHIFT;

    // init header
    if (jhwcinfo->auto_generate_jpeg_header)
    {
        _core->HUFF            = core_info->huff_ac_table_sel << JPEG_CORE_HUFF_AC_TABLE_SEL_SHIFT |
                                 core_info->huff_dc_table_sel << JPEG_CORE_HUFF_DC_TABLE_SEL_SHIFT;
        _core->HUFF_CODE_EN_DC = core_info->dc_code_en_1 << JPEG_CORE_HUFF_CODE_EN_DC_DC_CODE_EN_1_SHIFT |
                                 core_info->dc_code_en_0 << JPEG_CORE_HUFF_CODE_EN_DC_DC_CODE_EN_0_SHIFT;
        _core->HUFF_CODE_EN_AC = core_info->ac_code_en_1 << JPEG_CORE_HUFF_CODE_EN_AC_AC_CODE_EN_1_SHIFT |
                                 core_info->ac_code_en_0 << JPEG_CORE_HUFF_CODE_EN_AC_AC_CODE_EN_0_SHIFT;
        _core->RESTART         = core_info->restart_enable << JPEG_CORE_RESTART_RESTART_ENABLE_SHIFT |
                                 core_info->restart_interval_m1 << JPEG_CORE_RESTART_RESTART_INTERVAL_M1_SHIFT;
        _core->QUANT           = core_info->dequant_q_map_3 << JPEG_CORE_QUANT_Q_MAP_3_SHIFT |
                                 core_info->dequant_q_map_2 << JPEG_CORE_QUANT_Q_MAP_2_SHIFT |
                                 core_info->dequant_q_map_1 << JPEG_CORE_QUANT_Q_MAP_1_SHIFT |
                                 core_info->dequant_q_map_0 << JPEG_CORE_QUANT_Q_MAP_0_SHIFT;
        _core->ECS_OFFSET      = 0; // core_info->ecs_offset;
    }

    _jhwc_init_huffman(jhwcinfo, _core); // encode: setup jpeg image quality

    // setup for 24-bit RGB or 8-bit grayscale
    {
        CSC_JEDI_REGS_t* _csc = (CSC_JEDI_REGS_t*) dev->reg.csc;
        uint32_t csc_setting = 0; // encode: csc_setting accumulator
        if (bytes_per_pixel == 1 || bytes_per_pixel == 4)
        {
            // configure the CSC
            csc_setting = CSC_JEDI_CCR_PREOFFBYPASS_REPLACE_VAL(csc_setting , 1);
            csc_setting = CSC_JEDI_CCR_POSTOFFBYPASS_REPLACE_VAL(csc_setting, 1);
            csc_setting = CSC_JEDI_CCR_OFFSETBYPASS_REPLACE_VAL(csc_setting , 1);
            csc_setting = CSC_JEDI_CCR_BYPASSALL_REPLACE_VAL(csc_setting    , 1);
        }
        else // 3-plane data
        {
            // configure the CSC
            csc_setting = CSC_JEDI_CCR_PREOFFBYPASS_REPLACE_VAL(csc_setting,  1);
            csc_setting = CSC_JEDI_CCR_POSTOFFBYPASS_REPLACE_VAL(csc_setting, 0);
            csc_setting = CSC_JEDI_CCR_OFFSETBYPASS_REPLACE_VAL(csc_setting,  0);
            csc_setting = CSC_JEDI_CCR_BYPASSALL_REPLACE_VAL(csc_setting,     0);

            if ( jdev->info->config_flags & JPEGHW_CONFIG_FLAG_INPUT_YUV444 ) // input source is YCbCr444 format.
            {
			    csc_setting = CSC_JEDI_CCR_POSTOFFBYPASS_REPLACE_VAL(csc_setting, 1);
			    csc_setting = CSC_JEDI_CCR_OFFSETBYPASS_REPLACE_VAL(csc_setting,  1);
			    csc_setting = CSC_JEDI_CCR_BYPASSALL_REPLACE_VAL(csc_setting,     1);
            }

            _csc->MCR0 = 0x04c8;
            _csc->MCR1 = 0x0964;
            _csc->MCR2 = 0x01d2;
            _csc->MCR3 = 0x3d4d;
            _csc->MCR4 = 0x3ab3;
            _csc->MCR5 = 0x0800;
            _csc->MCR6 = 0x0800;
            _csc->MCR7 = 0x394d;
            _csc->MCR8 = 0x3eb3;
        }
        _csc->CCR = csc_setting; // commit csc settings
        DBG_INFO("[%d]%s() csc %#x (reg %#x)\n",gettid(), __func__, csc_setting, _csc->CCR);
    }

    // (encode) pogo core configuration
    {
        POGO_JPEG_REGS_t* _pogo = (POGO_JPEG_REGS_t*) dev->reg.pogo;
        uint32_t pogo_settings = 0; // clear
        _pogo->Cfg = pogo_settings; // reset pogo settings
        // configure the encode dma pogoizer
        pogo_settings = POGO_JPEG_CFG_BPP_REPLACE_VAL(pogo_settings, 7); // 8-bpp (bits per pixel) all modes
        pogo_settings = DEPOGO_CFG_MONOCHAN_REPLACE_VAL(pogo_settings, 0);
        if (bytes_per_pixel == 1)
        {
            pogo_settings = POGO_JPEG_CFG_FMT_REPLACE_VAL(pogo_settings, POGO_FMT_MONO);
            pogo_settings = POGO_JPEG_CFG_COLORSWAP_REPLACE_VAL(pogo_settings, 0); // ignore colorswap
        }
        else if (bytes_per_pixel == 3) // 3-byte 24-bit packed pixel
        {
            pogo_settings = POGO_JPEG_CFG_FMT_REPLACE_VAL(pogo_settings, POGO_FMT_24); // 3-byte pixel
            pogo_settings = POGO_JPEG_CFG_COLORSWAP_REPLACE_VAL(pogo_settings, 1); // enable channel byte swap
        }
        else // rgbx (4-plane)
        {
            pogo_settings = POGO_JPEG_CFG_FMT_REPLACE_VAL(pogo_settings, POGO_FMT_RGBx); // 4-byte pixel
            pogo_settings = POGO_JPEG_CFG_COLORSWAP_REPLACE_VAL(pogo_settings, 1); // enable channel byte swap
        }

        _pogo->Cfg = pogo_settings; // commit pogo settings
        DBG_INFO("[%d]%s() pogo cfg %#x (reg %#x)\n",gettid(), __func__, pogo_settings, _pogo->Cfg);
    }

    // finish idma/pogo setup
    {
        uint32_t idma_cr_settings = 0;
        idma_cr_settings = UDMA_UCR_OWNPOLARITY_REPLACE_VAL(idma_cr_settings,     0);
        idma_cr_settings = UDMA_UCR_OWNWRITEDISABLE_REPLACE_VAL(idma_cr_settings, 0);
        idma_cr_settings = UDMA_UCR_BEATS_REPLACE_VAL(idma_cr_settings,           1); // 8-beats
        idma_cr_settings = UDMA_UCR_ENABLE_REPLACE_VAL(idma_cr_settings,          1); // enable
        _idma->UCR = idma_cr_settings; // commit
        DBG_INFO("[%d]%s() idma/pogo %d (UCR: %#x)\n",gettid(), __func__, idma_cr_settings, _idma->UCR);
    }

    // finish odma/jpeg setup
    {
        uint32_t odma_cr_settings = 0;
        odma_cr_settings = UDMA_UCR_OWNPOLARITY_REPLACE_VAL(odma_cr_settings,     0);
        odma_cr_settings = UDMA_UCR_OWNWRITEDISABLE_REPLACE_VAL(odma_cr_settings, 0);
        odma_cr_settings = UDMA_UCR_BEATS_REPLACE_VAL(odma_cr_settings,           1); // 8-beats
        odma_cr_settings = UDMA_UCR_ENABLE_REPLACE_VAL(odma_cr_settings,          1); // enable
        _odma->UCR = odma_cr_settings; // commit
        DBG_INFO("[%d]%s() odma/jpeg %d (UCR %#x)\n",gettid(), __func__, odma_cr_settings, _odma->UCR);
    }

    // encode mode
    _core->CR |= JPEG_CORE_CR_ENCODE_MODE_MASK; // set encode mode
    //_core->CR &= ~JPEG_CORE_CR_AVG_N_DECIMATE_MASK; // make sure it is clear

    // clear/acknowledge any outstanding idma, udma, odma, or jpeg interrupts
    _idma->UICR    = 0xffffffff;
    _odma->UICR    = 0xffffffff;
    _icor->IICR    = 0xffffffff;
    _core->IRQ_ACK = 0xffffffff;

    DBG_INFO(" enable all interrupts\n");
    // encode: enable desired interrupt sources
    _idma->UIER   = 0x00000001; // idma interrupts
    _icor->IIER   = 0x00000001; // idma pogo core interrupts
    _odma->UIER   = 0x00000002; // odma interrupts
    _core->IRQ_EN = 0xf000001d; // core interrupts

    //_core->IRQ_FORCE = 0x30000017; //  force testing
    //_core->IRQ_FORCE = 0x00000001; //  force testing
    _odma->UIFR = 0x00000000;
    _idma->UIFR = 0x00000000;
    //_odma->UIFR = 0x00000002;
    //_idma->UIFR = 0x00000004;

    //show_core_regs(dev);
    return OK;
}

// intialize the JPEG subsampling registers
static int _jhwc_setup_subsampling(struct jpeghw_compress_struct * jhwcinfo)
{
    DBG_INFO("[%d]%s\n",gettid(), __func__);
    jpeghw_dev_ptr   jdev = JPEGHWC_GET_DEV(jhwcinfo);
    struct jpeg_dev_info_s * dev = JHWC_GET_DEV(jhwcinfo);

    // subsampling
    dev->core_info.h0_m1 = 0;
    dev->core_info.v0_m1 = 0;
    dev->core_info.h1_m1 = 0;
    dev->core_info.v1_m1 = 0;
    dev->core_info.h2_m1 = 0;
    dev->core_info.v2_m1 = 0;
    dev->core_info.h3_m1 = 0;
    dev->core_info.v3_m1 = 0;

    if(jdev->info->bytes_per_pixel > 1)
    {
        // chroma subsampling
        dev->core_info.h0_m1 = jdev->info->mcu_width  == 8?0:1; // supports 8 or 16 mcu-w
        dev->core_info.v0_m1 = jdev->info->mcu_height == 8?0:1; // suuports 8 or 16 mcu-h
    }

    return OK;
}

// Initialze the JPEG hardware for compression
static int _jhwc_setup_compressor(struct jpeghw_compress_struct * jhwcinfo)
{
    DBG_INFO("[%d]%s\n",gettid(), __func__);

    _jhwc_setup_subsampling(jhwcinfo);
    int setup_ok = _jhwc_set_compression_regs(jhwcinfo); // enforce hw/dev availability

    return setup_ok; // NOT settings_ok
}

// reset a given UDMA channel
static int _jhw_reset_dma(UDMA_REGS_t* udma)
{
    volatile uint32_t udma_status = udma->USR;
    DBG_INFO("[%d]%s dma-USR:0x%08x\n",gettid(), __func__, udma_status);

    udma->UCR = UDMA_USR_BUSY_REPLACE_VAL(udma->UCR, 0);
    posix_sleep_us(3); // hold for a few instruction cycles
    udma->UCR = UDMA_USR_BUSY_REPLACE_VAL(udma->UCR, 1);

    int tock = 0;
    while (udma_status & UDMA_USR_BUSY_MASK)
    {
        udma_status = udma->USR;
        if (tock++ > 10)
        { // declare dead if no response in 10 cycles
            DBG_ERR("[%d]%s stuck!? udma->USR:0x%08x\n",gettid(), __func__, udma->USR);
            return FAIL;
        }
    }
    return OK;
}

// reset the pogo idma
int jhwc_reset_idma(struct jpeghw_compress_struct * jhwcinfo)
{
    DBG_INFO("[%d]%s\n",gettid(), __func__);
    struct jpeg_dev_info_s * dev = JHWC_GET_DEV(jhwcinfo);

    return _jhw_reset_dma((UDMA_REGS_t*)dev->reg.pogo_idma_udma);
}

// reset the jpeg odma
int jhwc_reset_odma(struct jpeghw_compress_struct * jhwcinfo)
{
    DBG_INFO("[%d]%s\n",gettid(), __func__);
    struct jpeg_dev_info_s * dev = JHWC_GET_DEV(jhwcinfo);

    return _jhw_reset_dma((UDMA_REGS_t*)dev->reg.jpeg_odma_udma);
}

// reset the jpeg core
void jhwc_reset_core(struct jpeghw_common_struct * jhwinfo)
{
    struct jpeg_dev_info_s * dev = JHW_GET_DEV(jhwinfo);
    _jhw_reset_core(dev);
}

// Initialze the JPEG hardware for compression
int jhwc_init_compression_hw(struct jpeghw_compress_struct * jhwcinfo)
{
    DBG_INFO("[%d]%s\n",gettid(), __func__);

    return _jhwc_setup_compressor(jhwcinfo);
}

// Set the DIM-Y register to a new height.  This facilitates late
// initialization of the image height when it is not known when the
// compression begins.  Otherwise the hardware will hang expecting
// more data.
void jhwc_update_DIM_Y(struct jpeghw_compress_struct * jhwcinfo, int height)
{
    DBG_INFO("[%d]%s adjusting height register to %d\n",gettid(), __func__, height);

    struct jpeg_dev_info_s * dev = JHWC_GET_DEV(jhwcinfo);

    JPEG_CORE_REGS_t* _core = (JPEG_CORE_REGS_t*) dev->reg.core;
    _core->DIM = JPEG_CORE_DIM_Y_REPLACE_VAL(_core->DIM, height);
}

