/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014-2016, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/



#define DBG_PRFX "jpeg.hw.enc: "

#include <stdio.h>
#include <string.h>
#include <sys/file.h>

#include "map_mem.h"
#include "uio_lib_api.h"
#include "jhw_dev.h"
#include "jhw_uio.h"

#define JPEG_UIO_DEV "jpeg_uio"

// jpeg core/hw register collection
static struct jpeg_dev_regs_s jpeg_core[] = {
    { // 1st core
        .core           = JPEG_0_BASE,
        .top            = JPEG_0_TOP_BASE,
        .csc            = JPEG_0_CSC_BASE,
        .pogo           = JPEG_0_POGO_BASE,
        .depogo         = JPEG_0_DEPOGO_BASE,
        .pogo_idma_udma = JPEG_0_POGO_IDMA_UDMA_BASE,
        .pogo_idma_core = JPEG_0_POGO_IDMA_CORE_BASE,
        .jpeg_idma_udma = JPEG_0_JPEG_IDMA_UDMA_BASE,
        .jpeg_idma_core = JPEG_0_JPEG_IDMA_CORE_BASE,
        .pogo_odma_udma = JPEG_0_POGO_ODMA_UDMA_BASE,
        .pogo_odma_core = JPEG_0_POGO_ODMA_CORE_BASE,
        .jpeg_odma_udma = JPEG_0_JPEG_ODMA_UDMA_BASE,
        .jpeg_odma_core = JPEG_0_JPEG_ODMA_CORE_BASE,
    },

#if JPEG_DEVICES_AVAILABLE == 2
    { // 2nd core
        .core           = JPEG_1_BASE,
        .top            = JPEG_1_TOP_BASE,
        .csc            = JPEG_1_CSC_BASE,
        .pogo           = JPEG_1_POGO_BASE,
        .depogo         = JPEG_1_DEPOGO_BASE,
        .pogo_idma_udma = JPEG_1_POGO_IDMA_UDMA_BASE,
        .pogo_idma_core = JPEG_1_POGO_IDMA_CORE_BASE,
        .jpeg_idma_udma = JPEG_1_JPEG_IDMA_UDMA_BASE,
        .jpeg_idma_core = JPEG_1_JPEG_IDMA_CORE_BASE,
        .pogo_odma_udma = JPEG_1_POGO_ODMA_UDMA_BASE,
        .pogo_odma_core = JPEG_1_POGO_ODMA_CORE_BASE,
        .jpeg_odma_udma = JPEG_1_JPEG_ODMA_UDMA_BASE,
        .jpeg_odma_core = JPEG_1_JPEG_ODMA_CORE_BASE,
    },
#endif // 2nd core
         
#if JPEG_DEVICES_AVAILABLE > 2
#error insufficent register settings
#endif // >1 core
};

struct jpeg_dev_regs_s g_jpeg_mapped_regs[JPEG_DEVICES_AVAILABLE];

// map the register base addresses into user memory
static void _jhw_memmap_registers(jpeg_dev_regs_ptr regs)
{
    int fd = open("/dev/mem", (O_RDWR | O_SYNC | O_CLOEXEC));

    regs->core   = (uintptr_t)mapMem(regs->core, sizeof(JPEG_CORE_REGS_t), &regs->core_base, fd);
    regs->top    = (uintptr_t)mapMem(regs->top, sizeof(JPEG_TOP_REGS_t), &regs->top_base, fd);
    regs->csc    = (uintptr_t)mapMem(regs->csc, sizeof(CSC_JEDI_REGS_t), &regs->csc_base, fd);
    regs->pogo   = (uintptr_t)mapMem(regs->pogo, sizeof(POGO_JPEG_REGS_t), &regs->pogo_base, fd);
    regs->depogo = (uintptr_t)mapMem(regs->depogo, sizeof(DEPOGO_REGS_t), &regs->depogo_base, fd);

    regs->jpeg_idma_udma = (uintptr_t)mapMem(regs->jpeg_idma_udma,
                           sizeof(JPEG_IDMA_UDMA_REGS_t), &regs->jpeg_idma_udma_base, fd);
    regs->jpeg_odma_udma = (uintptr_t)mapMem(regs->jpeg_odma_udma,
                           sizeof(JPEG_ODMA_UDMA_REGS_t), &regs->jpeg_odma_udma_base, fd);
    regs->pogo_idma_udma = (uintptr_t)mapMem(regs->pogo_idma_udma,
                           sizeof(POGO_IDMA_UDMA_REGS_t), &regs->pogo_idma_udma_base, fd);
    regs->pogo_odma_udma = (uintptr_t)mapMem(regs->pogo_odma_udma,
                           sizeof(POGO_ODMA_UDMA_REGS_t), &regs->pogo_odma_udma_base, fd);

    regs->pogo_idma_core = (uintptr_t)mapMem(regs->pogo_idma_core,
                           sizeof(POGO_IDMA_CORE_REGS_t), &regs->pogo_idma_core_base, fd);
    regs->pogo_odma_core = (uintptr_t)mapMem(regs->pogo_odma_core,
                           sizeof(POGO_ODMA_CORE_REGS_t), &regs->pogo_odma_core_base, fd);
    regs->jpeg_idma_core = (uintptr_t)mapMem(regs->jpeg_idma_core,
                           sizeof(JPEG_IDMA_CORE_REGS_t), &regs->jpeg_idma_core_base, fd);
    regs->jpeg_odma_core = (uintptr_t)mapMem(regs->jpeg_odma_core,
                           sizeof(JPEG_ODMA_CORE_REGS_t), &regs->jpeg_odma_core_base, fd);

    close(fd);
}

// unmap the register base addresses
static void _jhw_unmemmap_registers(jpeg_dev_regs_ptr regs)
{
    unMapMem(regs->core_base, sizeof(JPEG_CORE_REGS_t));
    unMapMem(regs->top_base, sizeof(JPEG_TOP_REGS_t));
    unMapMem(regs->csc_base, sizeof(CSC_JEDI_REGS_t));
    unMapMem(regs->pogo_base, sizeof(POGO_JPEG_REGS_t));
    unMapMem(regs->depogo_base, sizeof(DEPOGO_REGS_t));

    unMapMem(regs->jpeg_idma_udma_base, sizeof(JPEG_IDMA_UDMA_REGS_t));
    unMapMem(regs->jpeg_odma_udma_base, sizeof(JPEG_ODMA_UDMA_REGS_t));
    unMapMem(regs->pogo_idma_udma_base, sizeof(POGO_IDMA_UDMA_REGS_t));
    unMapMem(regs->pogo_odma_udma_base, sizeof(POGO_ODMA_UDMA_REGS_t));

    unMapMem(regs->pogo_idma_core_base, sizeof(POGO_IDMA_CORE_REGS_t));
    unMapMem(regs->pogo_odma_core_base, sizeof(POGO_ODMA_CORE_REGS_t));
    unMapMem(regs->jpeg_idma_core_base, sizeof(JPEG_IDMA_CORE_REGS_t));
    unMapMem(regs->jpeg_odma_core_base, sizeof(JPEG_ODMA_CORE_REGS_t));
}

// map jpeg registers from their hardware address into application virtual addresses
// otherwise we would not be able to access the registers
void jhw_iomap_registers()
{
    DBG_INFO("[%d]%s\n",gettid(), __func__);
    int core_idx=0;

    for(core_idx=0; core_idx < JPEG_DEVICES_AVAILABLE; core_idx++)
    {
        memcpy(&g_jpeg_mapped_regs[core_idx], &jpeg_core[core_idx], sizeof(struct jpeg_dev_regs_s));
        _jhw_memmap_registers(&g_jpeg_mapped_regs[core_idx]);
    }
}

// unmap the jpeg registers so we don't create a memory leak
void jhw_iounmap_registers()
{
    DBG_INFO("[%d]%s\n",gettid(), __func__);
    int core_idx=0;

    for(core_idx=0; core_idx < JPEG_DEVICES_AVAILABLE; core_idx++)
    {
        memcpy(&g_jpeg_mapped_regs[core_idx], &jpeg_core[core_idx], sizeof(struct jpeg_dev_regs_s));
        _jhw_unmemmap_registers(&g_jpeg_mapped_regs[core_idx]);
    }
}

// map appropriate jpeg registers for a given core into dev regs
static void _jhw_iomap_registers(struct jpeg_dev_info_s * dev)
{
    int core_idx=0;

    if (!dev)
    	return;

    DBG_INFO("[%d]%s dev_num:%d\n",gettid(), __func__, dev->device_number);

    for(core_idx=0; core_idx < JPEG_DEVICES_AVAILABLE; core_idx++)
    {
    	DBG_INFO("[%d]  jpeg_core%d:%#x\n",gettid(), core_idx, jpeg_core[core_idx].core);
    }

    if (dev->device_number < 0 || dev->device_number >= JPEG_DEVICES_AVAILABLE)
    {
    	DBG_ERR("[%d]%s() Invalid device number\n",gettid(), __func__);
    	return;
    }

    memcpy(&dev->reg, &g_jpeg_mapped_regs[dev->device_number], sizeof(struct jpeg_dev_regs_s));
}

// open the first available uio_jpeg device (there will be the same number of uio devices
// in the system as the number of jpeg cores in the hardware)
// if wait == true, then wait until a device becomes available 
// otherwise return immediately: if a device was opened then return success
//                               if a device was not opend then return unavailable error
static jpeghw_error_type_t _jhw_get_available_uio_dev(struct jpeg_dev_info_s * dev, bool wait)
{
        jpeghw_error_type_t rc = e_JPEGHW_ERROR;
    int dev_number;

    if(dev == NULL)
            return e_JPEGHW_ERROR;

    if(dev && dev->uio_info.uio) return OK; // lookup current  REVISIT

    DBG_INFO("[%d]%s\n",gettid(),__func__);
    char dev_name[16] = { "<unknown>" };

    do {
        for(dev_number = 0; !dev->uio_info.uio && (dev_number < JPEG_DEVICES_AVAILABLE); dev_number++)
        {
            snprintf(dev_name, sizeof(dev_name)-1, "%s%d", JPEG_UIO_DEV, dev_number);
            //DBG_INFO("%s() try %s dev\n", __func__, dev_name);
            dev->uio_info.uio = (void *)uio_open_dev(dev_name);
            if (dev->uio_info.uio)
            {
                int jpeg_fd = uio_get_dev_fd(dev->uio_info.uio);
                //DBG_INFO("%s() jpeg_fd %d\n", __func__, jpeg_fd);

                if (flock(jpeg_fd, LOCK_EX|LOCK_NB) == OK)
                {
                    dev->device_number = dev_number;
                    dev->uio_info.uio_fd = jpeg_fd;
                    strncpy(dev->uio_info.uio_dev_name, dev_name, sizeof(dev->uio_info.uio_dev_name));

                    rc = e_JPEGHW_SUCCESS;
                    wait = false;
                    break;
                }
                else
                {
                    if(errno != 11)DBG_INFO("[%d]flock err: %s fd:%d\n",gettid(), strerror(errno),jpeg_fd);
                    uio_close_dev(dev->uio_info.uio);
                    dev->uio_info.uio = NULL;
                    rc = e_JPEGHW_ERR_NOT_AVAILABLE;
                }
            }
        }

        if (wait)
        {
           posix_sleep_us(1000); // wait 1ms
        }
    } while (wait);

    if(rc == e_JPEGHW_SUCCESS) DBG_INFO("[%d]%s() opened %s dev t:%d\n",gettid(), 
                                                      __func__, dev_name, dev->type);
    else  DBG_ERR("[%d]%s() NO devices available\n",gettid(), __func__);
    return rc;
}

// close the core identified by the dev->uio_info sturcture
jpeghw_error_type_t _jhw_close_uio_dev(struct jpeg_dev_info_s * dev)
{
    DBG_INFO("[%d]%s %s t:%d\n",gettid(),__func__, dev->uio_info.uio_dev_name, dev->type);

        // cleanup
    if (dev->uio_info.uio)
    {
        int jpeg_fd = uio_get_dev_fd(dev->uio_info.uio);
        DBG_INFO("[%d] jpeg_uio_fd = %d\n",gettid(), jpeg_fd);

        if (jpeg_fd != -1)
        {
            int err = flock(jpeg_fd, LOCK_UN);
            if(err != 0) DBG_INFO("[%d]flock %s fd:%d\n",gettid(), strerror(errno), jpeg_fd);
        }

        uio_close_dev(dev->uio_info.uio);
        dev->uio_info.uio = NULL;
    }

    return e_JPEGHW_SUCCESS;
}

// display contents of err as a string
static int _jhw_int_show_err(int *err, const char *func)
{
    if(*err) switch (*err) 
    {
        default:
          DBG_ERR("[%d]%s() error %d\n",gettid(), func, *err); break;
        case -1:
          DBG_ERR("[%d]%s() NO dev\n",gettid(), func); break;
        case -2:
          DBG_ERR("[%d]%s() NO uio fd\n",gettid(), func); break;
        case -3:
          DBG_ERR("[%d]%s() ALREADY attached/detached\n",gettid(), func); *err = 0; break;
        case -4:
          DBG_ERR("[%d]%s() write to uio dev failed\n",gettid(), func); break;
    }

    return OK;
}

// attach this instance of the driver to uio interrupts
static int _jhw_int_attach(struct jpeg_dev_info_s * dev, uint32_t priority, jhw_isr_cb_t isr)
{
    int rc = -5; 
    DBG_INFO("[%d]%s() dev %s uio %p\n",gettid(), __func__, 
                           dev->uio_info.uio_dev_name, dev->uio_info.uio);

    if (dev->uio_info.uio) 
    {
        priority = 0;
        rc = uio_int_attach( dev->uio_info.uio, priority, isr, dev); 
        _jhw_int_show_err(&rc, __func__);

        if (!rc) 
            DBG_INFO("[%d]%s(%s) isr %p ok\n",gettid(), __func__, dev->uio_info.uio_dev_name, isr); 
    }

    return rc;
}

// dettach this instance of the driver from uio interrupts
static int _jhw_int_detach(struct jpeg_dev_info_s * dev)
{
    int rc = -5; 

    if (dev && dev->uio_info.uio) 
    {
        DBG_INFO("[%d]%s() dev %s uio %p\n",gettid(), __func__, 
                               dev->uio_info.uio_dev_name, dev->uio_info.uio);

        rc = uio_int_detach( dev->uio_info.uio );
        _jhw_int_show_err(&rc, __func__);

        if (!rc) DBG_INFO("[%d]%s(%s) ok\n",gettid(), __func__, dev->uio_info.uio_dev_name);
    }

    return rc;
}

// enable uio interrupts for this instance of the driver
static int _jhw_int_enable(struct jpeg_dev_info_s * dev)
{
    int rc = -5; 

    if (dev && dev->uio_info.uio) 
    {
        DBG_INFO("[%d]%s() dev %s uio %p\n",gettid(), __func__, 
                               dev->uio_info.uio_dev_name, dev->uio_info.uio);

        rc = uio_int_enable( dev->uio_info.uio );
        _jhw_int_show_err(&rc, __func__);

        if (!rc) DBG_INFO("[%d]%s(%s) ok\n",gettid(), __func__, dev->uio_info.uio_dev_name);
    }

    return rc;
}

// disable uio interrupts for this instance of the driver
static int _jhw_int_disable(struct jpeg_dev_info_s * dev)
{
    int rc = -5; 

    if (dev && dev->uio_info.uio) 
    {
        DBG_INFO("[%d]%s() dev %s uio %p\n",gettid(), __func__, dev->uio_info.uio_dev_name, dev->uio_info.uio);

        rc = uio_int_disable( dev->uio_info.uio );
        _jhw_int_show_err(&rc, __func__);

        if (!rc) DBG_INFO("[%d]%s(%s) ok\n",gettid(), __func__, dev->uio_info.uio_dev_name);
    }

    return rc;
}

// the ISR for the compression
static void _jhwc_dev_isr(int32_t interrupt_count, void *isr_context)
{
    uint32_t event_flags=0;
    bool idma_intr=false, odma_intr=false;
    struct jpeg_dev_info_s *dev = (struct jpeg_dev_info_s *)isr_context;

    pthread_mutex_lock(&dev->isr_mutex);

    JPEG_CORE_REGS_t*      _core = (JPEG_CORE_REGS_t*)      dev->reg.core;
    JPEG_TOP_REGS_t*       _top  = (JPEG_TOP_REGS_t*)       dev->reg.top;
    JPEG_ODMA_CORE_REGS_t* _ocor = (JPEG_ODMA_CORE_REGS_t*) dev->reg.jpeg_odma_core;
    JPEG_ODMA_UDMA_REGS_t* _odma = (JPEG_ODMA_UDMA_REGS_t*) dev->reg.jpeg_odma_udma;
    POGO_IDMA_CORE_REGS_t* _icor = (POGO_IDMA_CORE_REGS_t*) dev->reg.pogo_idma_core;
    POGO_IDMA_UDMA_REGS_t* _idma = (POGO_IDMA_UDMA_REGS_t*) dev->reg.pogo_idma_udma;

    uint32_t core_pending_intr = _core->IRQ_PEND;
    uint32_t top_pending_intr  = _top->IRQ_PEND;
    uint32_t icor_pending_intr = _icor->IIPR;
    uint32_t ocor_pending_intr = _ocor->OIPR;
    uint32_t idma_pending_intr = _idma->UIPR;
    uint32_t odma_pending_intr = _odma->UIPR;
    uint32_t idma_burstlen_intr = _idma->UBLR;

    DBG_INFO("[%d]->>-%s cnt:%d\n",gettid(), __func__, interrupt_count);
    DBG_INFO("[%d]    Core :0x%08x Top :0x%08x\n",gettid(), core_pending_intr, top_pending_intr);
    DBG_INFO("[%d]    Icore:0x%08x Idma:0x%08x\n",gettid(), icor_pending_intr, idma_pending_intr);
    DBG_INFO("[%d]    Ocore:0x%08x Odma:0x%08x\n",gettid(), ocor_pending_intr, odma_pending_intr);
    DBG_INFO("[%d]    idma->UIPR 0x%08x &idma->UCR 0x%08x\n",gettid(), (unsigned int)&_idma->UIPR, (unsigned int)&_idma->UCR);
    DBG_INFO("[%d]    idma USR:0x%08x odma USR:0x%08x\n",gettid(), _idma->USR, _odma->USR);

    if(core_pending_intr & JPEG_CORE_IRQ_EN_ENCODE_DONE_IRQ_EN_MASK)
    {
        core_pending_intr &= ~JPEG_CORE_IRQ_EN_ENCODE_DONE_IRQ_EN_MASK;
        _core->IRQ_ACK = JPEG_CORE_IRQ_EN_ENCODE_DONE_IRQ_EN_MASK;
        event_flags |= JPEG_ENCODED;   
        idma_intr = true;
        odma_intr = true;
    }

    if(top_pending_intr)
    {
        // handle any pending pogo/idma interrupts
        if (top_pending_intr & JPEG_TOP_IRQ_PEND_POGO_IDMA_IRQ_PENDING_MASK)
        {
            DBG_INFO("[%d]     JPEG_TOP_IRQ_PEND_POGO_IDMA_IRQ_PENDING_MASK\n",gettid());
            idma_intr = true;
            if (idma_pending_intr)
            {
                if (idma_pending_intr & UDMA_UIPR_OWN_MASK)  // OWN interrupt
                {
                    DBG_INFO("[%d]     iUDMA_UIPR_OWN_MASK\n",gettid());
                    _idma->UICR = UDMA_UIPR_OWN_MASK; // ack
                    idma_pending_intr &= ~UDMA_UIPR_OWN_MASK; 
                    event_flags |= POGO_DMA_HW_OWN;
                }

                if (idma_pending_intr & UDMA_UIPR_DESC_MASK) // Descriptor complete
                {
                    DBG_INFO("[%d]     iUDMA_UIPR_DESC_MASK\n",gettid());
                    _idma->UICR = UDMA_UIPR_DESC_MASK; // ack
                    idma_pending_intr &= ~UDMA_UIPR_DESC_MASK; // clr
                    if (idma_burstlen_intr & UDMA_UBLR_BURSTLAST_MASK) // last link in the dma chain?
                    {
                        // ?? event_flags |= POGO_DMA_IN; // post idma chain done
                    }
                }

                if (idma_pending_intr & UDMA_UIPR_CLEARCOMPLETE_MASK) // 4 clear complete
                {
                    // ignore
                    DBG_INFO("[%d]     iUDMA_UIPR_CLEARCOMPLETE_MASK\n",gettid());
                    _idma->UICR = UDMA_UIPR_CLEARCOMPLETE_MASK; // ack
                    idma_pending_intr &= ~UDMA_UIPR_CLEARCOMPLETE_MASK; // clr
                }

                // bresp (transitional, ignored)
                if (idma_pending_intr & UDMA_UIPR_BRESP_MASK) // 8 bresp
                {
                    // ignore
                    DBG_INFO("[%d]     iUDMA_UIPR_BRESP_MASK\n",gettid());
                    _idma->UICR = UDMA_UIPR_BRESP_MASK; // ack
                    idma_pending_intr &= ~UDMA_UIPR_BRESP_MASK; // clr
                }

                // rresp (transitional, ignored)
                if (idma_pending_intr & UDMA_UIPR_RRESP_MASK) // 16 rresp
                {
                    // ignore
                    DBG_INFO("[%d]     iUDMA_UIPR_RRESP_MASK\n",gettid());
                    _idma->UICR = UDMA_UIPR_RRESP_MASK; // ack
                    idma_pending_intr &= ~UDMA_UIPR_RRESP_MASK; // clr
                }

                // out-or-range err
                if (idma_pending_intr & UDMA_UIPR_OUTOFRANGE_MASK) // 32
                {
                    DBG_INFO("[%d]     iUDMA_UIPR_OUTOFRANGE_MASK\n",gettid());
                    _idma->UICR = UDMA_UIPR_OUTOFRANGE_MASK; // ack
                    idma_pending_intr &= ~UDMA_UIPR_OUTOFRANGE_MASK; // clr
                    event_flags |= POGO_DMA_ERR_RANGE;
                    event_flags |= POGO_DMA_ERR_IN;   
                }

            } // _idma->UIPR (udma)

            // any unhandled idma/pogo remaining?
            if (idma_pending_intr)
            {
                DBG_ERR("[%d]%s() idma/pogo unhandled interrupt %p\n",gettid(), 
                                      __func__, (void *)idma_pending_intr);
                _idma->UICR = idma_pending_intr; 
            }

            // idma/pogo/core pending
            if (icor_pending_intr)
            {
                idma_intr = true;
                // end-of-strip
                if (icor_pending_intr & POGO_IDMA_CORE_IIPR_EOS_MASK) // 1 end-of-desc
                {
                    DBG_INFO("[%d]     POGO_IDMA_CORE_IIPR_EOS_MASK\n",gettid());
                    _icor->IICR = POGO_IDMA_CORE_IIPR_EOS_MASK; // ack
                    icor_pending_intr &= ~POGO_IDMA_CORE_IIPR_EOS_MASK; // clr
                    event_flags |= POGO_DMA_EOS; 
                }

                // length err
                if (icor_pending_intr & POGO_IDMA_CORE_IIPR_LENGTHERR_MASK)
                {
                    DBG_INFO("[%d]     POGO_IDMA_CORE_IIPR_LENGTHERR_MASK\n",gettid());
                    icor_pending_intr &= ~POGO_IDMA_CORE_IIPR_LENGTHERR_MASK; // clr
                    _icor->IICR = POGO_IDMA_CORE_IIPR_LENGTHERR_MASK; // ack
                    event_flags |= POGO_DMA_ERR_LENGTH;
                    event_flags |= POGO_DMA_ERR_IN;   
                    if (idma_burstlen_intr & UDMA_UBLR_BURSTLAST_MASK) // last link in the dma chain?
                    {
                        // ?? event_flags |= POGO_DMA_IN;  
                    }
                }
            } // _icor->IIPR (pogo)

            if (icor_pending_intr)
            {
                DBG_ERR("[%d]%s() core/pogo idma unhandled interrupt %p\n",gettid(),
                                       __func__, (void *)icor_pending_intr);
                _icor->IICR = icor_pending_intr; 
            }

            // mark pending idma/pogo/udma 'handled'
            top_pending_intr &= ~JPEG_TOP_IRQ_PEND_POGO_IDMA_IRQ_PENDING_MASK; // clear

        } // pending idma/pogo

        if (top_pending_intr & JPEG_TOP_IRQ_PEND_JPEG_ODMA_IRQ_PENDING_MASK)
        {
            DBG_INFO("[%d]     JPEG_TOP_IRQ_PEND_JPEG_ODMA_IRQ_PENDING_MASK\n",gettid());
            if (ocor_pending_intr & JPEG_ODMA_CORE_OIPR_EOI_MASK)
            {
                DBG_INFO("[%d]     JPEG_ODMA_CORE_OIPR_EOI_MASK\n",gettid());
                _ocor->OICR = JPEG_ODMA_CORE_OIPR_EOI_MASK; 
                ocor_pending_intr &= ~JPEG_ODMA_CORE_OIPR_EOI_MASK; // ack
                event_flags |= JPEG_ISR_EOI; 
                odma_intr = true;
            }

            // if transfer-end, set appropriate flag
            if (odma_pending_intr & UDMA_UIPR_DESC_MASK)
            {
                DBG_INFO("[%d]     oUDMA_UIPR_DESC_MASK\n",gettid());
                _odma->UICR = UDMA_UIPR_DESC_MASK; // ack
                odma_pending_intr &= ~UDMA_UIPR_DESC_MASK; // clear
                event_flags |= JPEG_DMA_OUT; 
                dev->core_info.jpeg_odma_oblr = _ocor->OBLR; // output buffer length
                odma_intr = true;
            }

            // mark this pend 'handled'
            top_pending_intr &= ~JPEG_TOP_IRQ_PEND_JPEG_ODMA_IRQ_PENDING_MASK; // clear this pend

            // any residual jpeg/odma activity?
            if (odma_pending_intr)
            {
                DBG_ERR("[%d]%s() jpeg/odma residual pending io %#x\n",gettid(), __func__, _odma->UIPR);
                _odma->UICR = odma_pending_intr; 
            }

            // pending odma/jpeg/udma 'handled'
            top_pending_intr &= ~JPEG_TOP_IRQ_PEND_JPEG_ODMA_IRQ_PENDING_MASK; // clear

        } // pending odma/jpeg

        // these are compress interrupts, should not see them now
        if (top_pending_intr & (JPEG_TOP_IRQ_PEND_JPEG_IDMA_IRQ_PENDING_MASK |
                                JPEG_TOP_IRQ_PEND_POGO_ODMA_IRQ_PENDING_MASK) )
        {
            DBG_ERR("[%d]%s() missed pending io %d\n",gettid(), __func__, top_pending_intr);
            top_pending_intr &= ~JPEG_TOP_IRQ_PEND_JPEG_IDMA_IRQ_PENDING_MASK; // clear
            top_pending_intr &= ~JPEG_TOP_IRQ_PEND_POGO_ODMA_IRQ_PENDING_MASK; // clear
            idma_pending_intr = 0;
            odma_pending_intr = 0;
            _idma->UICR    = 0xffffffff;
            event_flags |= JPEG_ENCODE_ERROR;
            idma_intr = true;
            odma_intr = true;
            _odma->UICR    = 0xffffffff;
        }
    } // pending_io

    jpeg_event_t jpeg_event;
    jpeg_event.event_flags = event_flags;
    jpeg_event.dev_info_ptr = dev;

    // if an output interrupt occurred send the interrupt info to the driver
    if(odma_intr && jpeg_event.event_flags > 0)
    {
        DBG_INFO("[%d]  send omsg 0x%08x len:%d mqd:%d\n",gettid(), 
                               event_flags, sizeof(jpeg_event_t), dev->omqd.mqd);
        mq_send(dev->omqd.mqd, (char *)&jpeg_event, sizeof(jpeg_event_t), 0);
    }
    // if an input interrupt occurred send the interrupt info to the driver
    if(idma_intr && jpeg_event.event_flags > 0)
    {
        DBG_INFO("[%d]  send imsg 0x%08x len:%d mqd:%d\n",gettid(), 
                               event_flags, sizeof(jpeg_event_t), dev->imqd.mqd);
        mq_send(dev->imqd.mqd, (char *)&jpeg_event, sizeof(jpeg_event_t), 0);
    }

    pthread_mutex_unlock(&dev->isr_mutex);
}

// the ISR for the decompression
static void _jhwd_dev_isr(int32_t interrupt_count, void *isr_context)
{
    uint32_t event_flags=0;
    bool idma_intr=false, odma_intr=false;
    struct jpeg_dev_info_s *dev = (struct jpeg_dev_info_s *)isr_context;

    pthread_mutex_lock(&dev->isr_mutex);

    JPEG_CORE_REGS_t*      _core = (JPEG_CORE_REGS_t*)      dev->reg.core;
    JPEG_TOP_REGS_t*       _top  = (JPEG_TOP_REGS_t*)       dev->reg.top;
    JPEG_IDMA_UDMA_REGS_t* _idma = (JPEG_IDMA_UDMA_REGS_t*) dev->reg.jpeg_idma_udma;
    POGO_ODMA_UDMA_REGS_t* _odma = (POGO_ODMA_UDMA_REGS_t*) dev->reg.pogo_odma_udma;
    POGO_ODMA_CORE_REGS_t* _ocor = (POGO_ODMA_CORE_REGS_t*) dev->reg.pogo_odma_core;

    uint32_t core_pending_intr = _core->IRQ_PEND;
    uint32_t top_pending_intr  = _top->IRQ_PEND;
    uint32_t ocor_pending_intr = _ocor->OIPR;
    uint32_t idma_pending_intr = _idma->UIPR;
    uint32_t odma_pending_intr = _odma->UIPR;
    uint32_t idma_burstlen_intr = _idma->UBLR;

    DBG_INFO("[%d]    _idma->UIPR 0x%08x &_idma->UCR 0x%08x\n",gettid(), (unsigned int)&_idma->UIPR, (unsigned int)&_idma->UCR);
    DBG_INFO("[%d]->>-%s cnt:%d\n",gettid(), __func__, interrupt_count);
    DBG_INFO("[%d]    Core :0x%08x Top :0x%08x\n",gettid(), core_pending_intr, top_pending_intr);
    DBG_INFO("[%d]    Ocore:0x%08x Idma:0x%08x Odma:0x%08x\n",gettid(), 
                           ocor_pending_intr, idma_pending_intr, odma_pending_intr);
    DBG_INFO("[%d]    idma USR:0x%08x odma USR:0x%08x\n",gettid(), _idma->USR, _odma->USR);

    if(core_pending_intr != 0)
    {
        if(core_pending_intr & JPEG_CORE_IRQ_EN_DECODE_DONE_IRQ_EN_MASK) 
        {
            core_pending_intr &= ~JPEG_CORE_IRQ_EN_DECODE_DONE_IRQ_EN_MASK;
            _core->IRQ_ACK = JPEG_CORE_IRQ_EN_DECODE_DONE_IRQ_EN_MASK;
            event_flags |= JPEG_DECODED;   
            idma_intr = true;
            odma_intr = true;
        }
        uint32_t huffman_intr = JPEG_CORE_IRQ_EN_HUFF_UNSTUFF_EOI_IRQ_EN_MASK |
                   JPEG_CORE_IRQ_EN_HUFF_UNSTUFF_NON_EOI_IRQ_EN_MASK |
                   JPEG_CORE_IRQ_EN_HUFF_DONE_IRQ_EN_MASK;

        if(top_pending_intr & huffman_intr)
        {
            core_pending_intr &= ~huffman_intr;
            _core->IRQ_ACK = huffman_intr;
            uint32_t huffman_err = JPEG_CORE_IRQ_EN_HUFF_RESTART_ERROR_IRQ_EN_MASK |
                             JPEG_CORE_IRQ_EN_HUFF_EOI_ERROR_IRQ_EN_MASK |
                             JPEG_CORE_IRQ_EN_HUFF_NO_CODE_ERROR_IRQ_EN_MASK |
                             JPEG_CORE_IRQ_EN_HUFF_MULT_CODE_ERROR_IRQ_EN_MASK;
            if(core_pending_intr & huffman_err)
            {
                core_pending_intr &= ~huffman_err;
                _core->IRQ_ACK = huffman_err;
                event_flags |= JPEG_DECODE_ERROR;   
                idma_intr = true;
                odma_intr = true;
            }
        }
    }

    if(top_pending_intr)
    {
        // handle any pending pogo/idma interrupts
        if (top_pending_intr & JPEG_TOP_IRQ_PEND_JPEG_IDMA_IRQ_PENDING_MASK)
        {
            DBG_INFO("[%d]     JPEG_TOP_IRQ_PEND_JPEG_IDMA_IRQ_PENDING_MASK\n",gettid());
            idma_intr = true;
            if (idma_pending_intr)
            {
                if (idma_pending_intr & UDMA_UIPR_OWN_MASK)  // OWN interrupt
                {
                    DBG_INFO("[%d]     iUDMA_UIPR_OWN_MASK\n",gettid());
                    _idma->UICR = UDMA_UIPR_OWN_MASK; // ack
                    idma_pending_intr &= ~UDMA_UIPR_OWN_MASK; 
                    event_flags |= POGO_DMA_HW_OWN;
                }

                if (idma_pending_intr & UDMA_UIPR_DESC_MASK) // Descriptor complete
                {
                    DBG_INFO("[%d]     iUDMA_UIPR_DESC_MASK\n",gettid());
                    _idma->UICR = UDMA_UIPR_DESC_MASK; // ack
                    idma_pending_intr &= ~UDMA_UIPR_DESC_MASK; // clr
                    event_flags |= JPEG_DMA_IN;
                    if (idma_burstlen_intr & UDMA_UBLR_BURSTLAST_MASK) // last link in the dma chain?
                    {
                        // ?? event_flags |= POGO_DMA_IN; // post idma chain done
                    }
                }

                if (idma_pending_intr & UDMA_UIPR_CLEARCOMPLETE_MASK) // 4 clear complete
                {
                    // ignore
                    DBG_INFO("[%d]     iUDMA_UIPR_CLEARCOMPLETE_MASK\n",gettid());
                    _idma->UICR = UDMA_UIPR_CLEARCOMPLETE_MASK; // ack
                    idma_pending_intr &= ~UDMA_UIPR_CLEARCOMPLETE_MASK; // clr
                }

                // bresp (transitional, ignored)
                if (idma_pending_intr & UDMA_UIPR_BRESP_MASK) // 8 bresp
                {
                    // ignore
                    DBG_INFO("[%d]     iUDMA_UIPR_BRESP_MASK\n",gettid());
                    _idma->UICR = UDMA_UIPR_BRESP_MASK; // ack
                    idma_pending_intr &= ~UDMA_UIPR_BRESP_MASK; // clr
                }

                // rresp (transitional, ignored)
                if (idma_pending_intr & UDMA_UIPR_RRESP_MASK) // 16 rresp
                {
                    // ignore
                    DBG_INFO("[%d]     iUDMA_UIPR_RRESP_MASK\n",gettid());
                    _idma->UICR = UDMA_UIPR_RRESP_MASK; // ack
                    idma_pending_intr &= ~UDMA_UIPR_RRESP_MASK; // clr
                }

                // out-or-range err
                if (idma_pending_intr & UDMA_UIPR_OUTOFRANGE_MASK) // 32
                {
                    DBG_INFO("[%d]     iUDMA_UIPR_OUTOFRANGE_MASK\n",gettid());
                    _idma->UICR = UDMA_UIPR_OUTOFRANGE_MASK; // ack
                    idma_pending_intr &= ~UDMA_UIPR_OUTOFRANGE_MASK; // clr
                    event_flags |= POGO_DMA_ERR_RANGE;
                    event_flags |= POGO_DMA_ERR_IN;   
                }

            } // _idma->UIPR (udma)

            // any unhandled idma/pogo remaining?
            if (idma_pending_intr)
            {
                DBG_ERR("[%d]%s() idma/pogo unhandled interrupt %p\n",gettid(), 
                                      __func__, (void *)idma_pending_intr);
                _idma->UICR = idma_pending_intr; 
            }

            // mark pending idma/pogo/udma 'handled'
            top_pending_intr &= ~JPEG_TOP_IRQ_PEND_JPEG_IDMA_IRQ_PENDING_MASK; // clear

        } // idma/jpeg

        if (top_pending_intr & JPEG_TOP_IRQ_PEND_POGO_ODMA_IRQ_PENDING_MASK)
        {
            DBG_INFO("[%d]     JPEG_TOP_IRQ_PEND_POGO_ODMA_IRQ_PENDING_MASK\n",gettid());
            if (ocor_pending_intr & JPEG_ODMA_CORE_OIPR_EOI_MASK)
            {
                DBG_INFO("[%d]     JPEG_ODMA_CORE_OIPR_EOI_MASK\n",gettid());
                _ocor->OICR = JPEG_ODMA_CORE_OIPR_EOI_MASK; 
                ocor_pending_intr &= ~JPEG_ODMA_CORE_OIPR_EOI_MASK; // ack
                event_flags |= JPEG_ISR_EOI; 
                odma_intr = true;
            }

            // if transfer-end, set appropriate flag
            if (odma_pending_intr & UDMA_UIPR_DESC_MASK)
            {
                DBG_INFO("[%d]     oUDMA_UIPR_DESC_MASK\n",gettid());
                _odma->UICR = UDMA_UIPR_DESC_MASK; // ack
                odma_pending_intr &= ~UDMA_UIPR_DESC_MASK; // clear
                event_flags |= JPEG_DMA_OUT; 
                odma_intr = true;
                //dev->core_info.JPEG_ODMA_OBLR = _ocor->OBLR; // output buffer length
            }

            // any residual jpeg/odma activity?
            if (odma_pending_intr)
            {
                DBG_ERR("[%d]%s() jpeg/odma residual pending io %#x\n",gettid(), __func__, _odma->UIPR);
                _odma->UICR = odma_pending_intr; 
            }

            // pending odma/jpeg/udma 'handled'
            top_pending_intr &= ~JPEG_TOP_IRQ_PEND_POGO_ODMA_IRQ_PENDING_MASK; // clear

        } // pending odma/pogo

        // these are decompress interrupts, should not see them now
        if (top_pending_intr & (JPEG_TOP_IRQ_PEND_JPEG_ODMA_IRQ_PENDING_MASK |
                                JPEG_TOP_IRQ_PEND_POGO_IDMA_IRQ_PENDING_MASK) )
        {
            DBG_ERR("[%d]%s() missed pending io %d\n",gettid(), __func__, top_pending_intr);
            top_pending_intr &= ~JPEG_TOP_IRQ_PEND_JPEG_ODMA_IRQ_PENDING_MASK; // clear
            top_pending_intr &= ~JPEG_TOP_IRQ_PEND_POGO_IDMA_IRQ_PENDING_MASK; // clear
            idma_pending_intr = 0;
            odma_pending_intr = 0;
            _idma->UICR    = 0xffffffff;
            _odma->UICR    = 0xffffffff;
            event_flags |= JPEG_DECODE_ERROR;
            idma_intr = true;
            odma_intr = true;
        }
    } // pending_io

    jpeg_event_t jpeg_event;
    jpeg_event.event_flags = event_flags;
    jpeg_event.dev_info_ptr = dev;

    // if an output interrupt occurred send the interrupt info to the driver
    if(odma_intr && jpeg_event.event_flags > 0)
    {
        DBG_INFO("[%d]  send omsg 0x%08x len:%d mqd:%d\n",gettid(), 
                               event_flags, sizeof(jpeg_event_t), dev->omqd.mqd);
        mq_send(dev->omqd.mqd, (char *)&jpeg_event, sizeof(jpeg_event_t), 0);
    }
    // if an input interrupt occurred send the interrupt info to the driver
    if(idma_intr && jpeg_event.event_flags > 0)
    {
        DBG_INFO("[%d]  send imsg 0x%08x len:%d mqd:%d\n",gettid(), 
                               event_flags, sizeof(jpeg_event_t), dev->imqd.mqd);
        mq_send(dev->imqd.mqd, (char *)&jpeg_event, sizeof(jpeg_event_t), 0);
    }

    pthread_mutex_unlock(&dev->isr_mutex);
}


// map the hardware register addresses from physical space to virtual user space
// intialize the ISR mutex
// attach device interrupts to an interrupt number
int jhw_attach_to_device(struct jpeg_dev_info_s * dev)
{
    DBG_INFO("[%d]%s()\n",gettid(), __func__);

    if(dev == NULL || dev->uio_info.uio_fd == -1) return FAIL;
    
    // maps dev# AND mode-specific register assignments[../
    _jhw_iomap_registers(dev);

    if(pthread_mutex_init(&dev->isr_mutex, NULL) != OK) return FAIL;

    // attach uio isr
    if(dev->type == COMPRESSOR)
    {
        DBG_INFO("[%d]Attach to compressor\n",gettid());
        if(_jhw_int_attach( dev, INTNUM_JPEG0, _jhwc_dev_isr) != OK) return FAIL;
    }
    else
    {
        DBG_INFO("[%d]Attach to decompressor\n",gettid());
        if(_jhw_int_attach( dev, INTNUM_JPEG0, _jhwd_dev_isr) != OK) return FAIL;
    }

    DBG_INFO("[%d]uio isr attached and enabled\n",gettid());
    DBG_INFO("[%d]%s() dev %p %s fd(%d) uio(%p)\n",gettid(), 
          __func__, dev, dev->uio_info.uio_dev_name, dev->uio_info.uio_fd, dev->uio_info.uio);
    return OK;
}

// enable the device interrupts
int jhw_enable_device_interrupts(struct jpeg_dev_info_s * dev)
{
    return _jhw_int_enable( dev );
}

// destroy the ISR mutex
// disable and detach from the interrupt
int jhw_detach_from_device(struct jpeg_dev_info_s * dev)
{
    DBG_INFO("[%d]%s()\n",gettid(), __func__);

    if (dev == NULL) return FAIL;

    pthread_mutex_destroy(&dev->isr_mutex);

    _jhw_int_disable(dev);
    _jhw_int_detach(dev);

    return OK;
}

// open the first available core
jpeghw_error_type_t jhw_open_available_core(struct jpeg_dev_info_s * dev, bool wait)
{
    if (dev == NULL) return e_JPEGHW_ERR_INVALID_PARAMETERS;

    uio_lib_init();

    DBG_INFO("[%d]%s()\n",gettid(), __func__);
    return _jhw_get_available_uio_dev(dev, wait);
}

// close the core identified by the dev->uio_info sturcture
jpeghw_error_type_t jhw_close_core(struct jpeg_dev_info_s * dev)
{
    if (dev == NULL) return e_JPEGHW_ERR_INVALID_PARAMETERS;

    DBG_INFO("[%d]%s()\n",gettid(), __func__);

    return _jhw_close_uio_dev(dev);
}
