/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014-2015, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/



#include "jhw_dev.h"
#include "jhw_uio.h"
#include "jhw_mem_test.h"
#include "jpeghw_lib.h"
#include "uio_lib_api.h"

static bool g_jhw_initialized = false;
static FILE *g_jhw_image_power = NULL;

inline struct BigBuffer_s * JHW_FREE_BIG_BUFFER(struct BigBuffer_s *big_buffer)
{
  if(big_buffer != NULL)
  {
    (big_buffer)->freeFunc(big_buffer);
    DEC_G_ALLOCATED();
    DBG_INFO("[%d]   Free big_buffer=%p\n",gettid(), big_buffer);
  }
  return NULL;
}

inline struct BigBuffer_s *JHW_FREE_MAPPED_BIG_BUFFER(struct BigBuffer_s *big_buffer) 
{
  if(big_buffer != NULL)
  {
    DEC_G_SMAPPED();
    dma_buffer_unmap_single(big_buffer, DMA_FROM_DEVICE);
    return JHW_FREE_BIG_BUFFER(big_buffer);
  }

  return NULL;
}

mqd_t jhw_open_message_queue(struct jpeg_dev_info_s * dev, char *msgq_name_fmt, char* msgq_name)
{
    mqd_t mqd = 0;

    sprintf(msgq_name, msgq_name_fmt, dev->device_number);

    DBG_INFO("[%d]msg queue:%s\n",gettid(), msgq_name);
    mqd = mq_open(msgq_name, O_RDWR);

    struct mq_attr attr;

    if(mqd == -1 && errno == ENOENT)
    {
        attr.mq_msgsize = sizeof(jpeg_event_t);
        attr.mq_maxmsg = 10;

        mqd = mq_open(msgq_name, O_CREAT | O_RDWR, S_IRUSR | S_IWUSR, &attr);
        if(mqd == -1) 
        {
           DBG_ERR("[%d]mqd=%d err:(%d)%s\n",gettid(), mqd, errno, strerror(errno));
           return -1;
        }
    }
 
    // clean old msgs from queue
    do
    {
        char ev[20];
        mq_getattr(mqd, &attr);
        //DBG_INFO("----msgs in queue: %d\n", attr.mq_curmsgs);
        if(attr.mq_curmsgs > 0)
        {
        	int rv = -1;
            while((rv = posix_wait_for_message(mqd, (char *)&ev,
            		attr.mq_msgsize, (5*USEC_PER_SECOND))) == ETIMEDOUT)
            {
            	DBG_WARN("[%d]%s(): WARNING - posix_wait_for_message timed out\n",gettid(), __func__);
            	continue;
            }

            //DBG_INFO("msgs len %d errno:%d\n", len, errno);
            if(rv != 0)  // this can cause an infinite loop so get out of the loop
            {
                DBG_ERR("[%d]msg queue is messed up errno:%s\n",gettid(), strerror(errno));
                mq_close(mqd);
                return -1;
            }
        } 

    } while(attr.mq_curmsgs > 1);

    DBG_INFO("[%d]mqd=%d open\n",gettid(), mqd);
    return mqd;
}

jhw_dma_desc_t **jhw_new_dma_descriptors(int number_dma_descriptors)
{
    INC_G_ALLOCATED();
    jhw_dma_desc_t **dma_list = NULL;

    dma_list = (jhw_dma_desc_t **)MEM_MALLOC(sizeof(jhw_dma_desc_t*) * number_dma_descriptors);
    if(dma_list == NULL)
    {
        DBG_ERR("[%d]Failed to allocate memory for dma descriptor list\n",gettid());
        return NULL;
    }

    int idx=0;
    for(idx=0; idx<number_dma_descriptors; idx++)
    {
        void *new_hwaddr;
         INC_G_ALLOCATED();
        dma_list[idx] = (jhw_dma_desc_t *)MEM_MALLOC_UNCACHED(&new_hwaddr,
                                                 sizeof(jhw_dma_desc_t), e_32_byte);
        if(dma_list[idx] == NULL)
        {
            DBG_ERR("[%d]Failed to allocate memory for idma descriptor\n",gettid());

            int j;
            for (j=0;j<idx;j++)
            {
                DEC_G_ALLOCATED();
                MEM_FREE_UNCACHED(dma_list[j]->hwaddr, dma_list[j]);
            }
            DEC_G_ALLOCATED();
            MEM_FREE_AND_NULL(dma_list);

            return NULL;
        }

        dma_list[idx]->hwaddr = new_hwaddr;
        dma_list[idx]->source_addr = 0;
        dma_list[idx]->control_reg = 0;

        DBG_INFO("[%d]   add desc:%p hwaddr:%p to dma_list[%d]\n",gettid(), dma_list[idx], new_hwaddr, idx);
        if(idx > 0)
        {
            dma_list[idx-1]->next_desc_addr = (uintptr_t)new_hwaddr;
        }
    }

    if(idx > 0)
    {
        // set end of list next ptr to the first desc in the list
        dma_list[idx-1]->next_desc_addr = (uintptr_t)dma_list[0]->hwaddr;
    }

    return dma_list;
}

void jhw_destroy_dma_descriptors(jhw_dma_desc_t **dma_list, int number_dma_descriptors)
{
    if(dma_list == NULL)
    {
        DBG_ERR("[%d]Invalid dma descriptor list\n",gettid());
        return;
    }

    // destroy the list of descriptors
    int idx=0;
    for(idx=0; idx<number_dma_descriptors; idx++)
    {
        void *hwaddr = dma_list[idx]->hwaddr;
        DEC_G_ALLOCATED();
        MEM_FREE_UNCACHED(hwaddr, dma_list[idx]);
    }

    DEC_G_ALLOCATED();
    MEM_FREE_AND_NULL(dma_list);
}

jpeghw_error_type_t jhw_device_reset (struct jpeghw_common_struct * jhwinfo)
{
    DBG_INFO("[%d]%s()\n",gettid(), __func__);
    JHW_VALIDATE(jhwinfo,0);

    // perform any require device hardware reset here
    return e_JPEGHW_SUCCESS;
}

uint32_t jhw_get_max_decompress_lines (struct jpeghw_common_struct * jhwinfo)
{
    DBG_INFO("[%d]%s()\n",gettid(), __func__);
    JHW_VALIDATE(jhwinfo,0);
    struct jpeg_dev_info_s * dev = JHW_GET_DEV(jhwinfo);

    return dev->max_number_deccompressed_lines;
}

jpeghw_error_type_t jhw_set_max_decompress_lines (struct jpeghw_common_struct * jhwinfo, uint32_t max_lines)
{
    DBG_INFO("[%d]%s()\n",gettid(), __func__);
    JHW_VALIDATE(jhwinfo,e_JPEGHW_ERR_INVALID_PARAMETERS);
    struct jpeg_dev_info_s * dev = JHW_GET_DEV(jhwinfo);

    dev->max_number_deccompressed_lines = max_lines;

    return e_JPEGHW_SUCCESS;
}

void jhw_initialize()
{
	if (g_jhw_initialized == false)
	{
		DBG_INFO("[%d]%s()\n",gettid(), __func__);
		INIT_MEM_STATS();

		if (g_jhw_image_power == NULL)
		{
			g_jhw_image_power = fopen("/dev/imagepower", "r"); // blocking call that will enable the power island if it isn’t already on.
		}
		else
		{
			DBG_ERR("[%d]%s() - image power island is already on!\n",gettid(), __func__);
		}

		jhw_iomap_registers();
		uio_lib_init();

		g_jhw_initialized = true;
	}
	else
	{
		DBG_ERR("[%d]%s() - already initialized!\n",gettid(), __func__);
	}
}

bool jhw_is_initialized()
{
    return g_jhw_initialized;
}

void jhw_terminate()
{
	if (g_jhw_initialized == true)
	{
		DBG_INFO("[%d]%s()\n",gettid(), __func__);
		jhw_iounmap_registers();

		if (g_jhw_image_power != NULL)
		{
			fclose(g_jhw_image_power); // blocking call that will disable the power island if your the last user.
			g_jhw_image_power = NULL;
		}
		else
		{
			DBG_ERR("[%d]%s() - image power island is already off!\n",gettid(), __func__);
		}

		DSP_MEM_STATS();
		TERM_MEM_STATS();
		g_jhw_initialized = false;
	}
	else
	{
		DBG_ERR("[%d]%s() - not initialized!\n",gettid(), __func__);
	}
}

jpeghw_error_type_t jhw_get_default_quant_table(uint8_t quality, uint8_t *table, uint32_t *size)
{
	int index = 0;

	if (quality > 80) index++; // -> 90
	if (quality > 90) index++; // -> 95
	if (quality > 95) index++; // -> 98
	if (quality > 98) index++; // -> 99

	if (size)
	{
		if  (*size > sizeof(jhw_quantization_table[0]))
		{
			*size = sizeof(jhw_quantization_table[0]);
		}
	}

	if (table && size)
	{
		memcpy(table, &jhw_quantization_table[index][0], *size);
	}

    return e_JPEGHW_SUCCESS;
}

jpeghw_error_type_t jhw_get_default_huff_table(int table_index, bool ac, uint8_t *bits, uint32_t *bits_size, uint8_t *val, uint32_t *val_size)
{
	if (table_index > 1)
	{
		DBG_ERR("%s() ERROR - invalid table_index parameter, received %d but expected 0 or 1!\n",
				__func__, table_index);

		return e_JPEGHW_ERR_INVALID_PARAMETERS;
	}

	if (bits_size)
	{
		uint32_t size = (ac ? sizeof(jhw_ac_bits[table_index]) : sizeof(jhw_dc_bits[table_index]));

		if (*bits_size > size)
		{
			*bits_size = size;
		}
	}

	if (bits && bits_size)
	{
		if (ac)
		{
			memcpy(bits, jhw_ac_bits[table_index], *bits_size);
		}
		else
		{
			memcpy(bits, jhw_dc_bits[table_index], *bits_size);
		}
	}

	if (val_size)
	{
		uint32_t size = (ac ? sizeof(jhw_ac_val[table_index]) : sizeof(jhw_dc_val[table_index]));

		if (*val_size >size)
		{
			*val_size = size;
		}
	}

	if (val && val_size)
	{
		if (ac)
		{
			memcpy(val, jhw_ac_val[table_index], *val_size);
		}
		else
		{
			memcpy(val, jhw_dc_val[table_index], *val_size);
		}
	}

    return e_JPEGHW_SUCCESS;
}


void show_core_regs(struct jpeg_dev_info_s * dev)
{
#if DEBUG
    JPEG_CORE_REGS_t* r = (JPEG_CORE_REGS_t*)dev->reg.core;
    DBG_INFO("[%d] Core Regs:\n",gettid());
    DBG_INFO("  IRQ_EN    :0x%08x\n", r->IRQ_EN);
    DBG_INFO("  IRQ_PEND  :0x%08x\n", r->IRQ_PEND);
    DBG_INFO("  IRQ_ACK   :0x%08x\n", r->IRQ_ACK);
    DBG_INFO("  CR        :0x%08x\n", r->CR);
    DBG_INFO("  IMAGE     :0x%08x\n", r->IMAGE);
    DBG_INFO("  DIM       :0x%08x\n", r->DIM);
    DBG_INFO("  PAD       :0x%08x\n", r->PAD);
    DBG_INFO("  PAD_VAL   :0x%08x\n", r->PAD_VAL);
    DBG_INFO("  SUB       :0x%08x\n", r->SUB);
    DBG_INFO("  HUF_EN_DC :0x%08x\n", r->HUFF_CODE_EN_DC);
    DBG_INFO("  HUF_EN_AC :0x%08x\n", r->HUFF_CODE_EN_AC);
    DBG_INFO("  RESTART   :0x%08x\n", r->RESTART);
    DBG_INFO("  QUANT     :0x%08x\n", r->QUANT);
    DBG_INFO("  STATUS0   :0x%08x\n", r->STATUS0);
    DBG_INFO("  STATUS1   :0x%08x\n", r->STATUS1);
    DBG_INFO("  STATUS2   :0x%08x\n", r->STATUS2);
    DBG_INFO("  STATUS3   :0x%08x\n", r->STATUS3);
    DBG_INFO("  STATUS4   :0x%08x\n", r->STATUS4);
    DBG_INFO("  STATUS5   :0x%08x\n", r->STATUS5);
    DBG_INFO("  STATUS6   :0x%08x\n", r->STATUS6);
    DBG_INFO("  STATUS7   :0x%08x\n", r->STATUS7);
    DBG_INFO("  STATUS8   :0x%08x\n", r->STATUS8);
    DBG_INFO("  STATUS9   :0x%08x\n", r->STATUS9);
#endif
}

void show_udma_regs(UDMA_REGS_t* r, bool idma)
{
#if DEBUG
	if(idma) DBG_INFO("[%d] iUDMA Regs:\n",gettid());
    else DBG_INFO("[%d] oUDMA Regs:\n",gettid());
    DBG_INFO("  UCR  :0x%08x\n", r->UCR);
    DBG_INFO("  USR  :0x%08x\n", r->USR);
    DBG_INFO("  UPR  :0x%08x\n", r->UPR);
    DBG_INFO("  UIER :0x%08x\n", r->UIER);
    DBG_INFO("  UIPR :0x%08x\n", r->UIPR);
    DBG_INFO("  UICR :0x%08x\n", r->UICR);
    DBG_INFO("  UIFR :0x%08x\n", r->UIFR);
#if 0
    DBG_INFO("  UBAR :0x%08x\n", r->UBAR);
    DBG_INFO("  UBLR :0x%08x\n", r->UBLR);
    DBG_INFO("  UBRR :0x%08x\n", r->UBRR);
#endif
#endif
}
