/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014-2016, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/


 
#ifndef JHW_DEV_H
#define JHW_DEV_H

#include <stdbool.h>

#include "jhw_api.h"
#include "jhw_asic.h"
#include "jhw_dbg.h"
#include "jhw_dma.h"
#include "jhw_mem_test.h"
#include "pthread.h"
#include "dma_buffer.h"
#include "error_types.h"

// These defines are used to set the DMA descriptor CR value
#define STOP_ON_FINISH           1
#define  INT_ON_FINISH           2
#define  EOI_ON_FINISH       (1<<8) /* generate eoi on desc done */
#define   READY_HW_OWN      (1<<24)

// These defines are the message queue names used to communicate
// between ISRs and main threads.  The %d is used to identify
// the uio device number the queues are associated with
#define IDMA_C_MSGQ_NAME "/jhwc_imsgq_%d"
#define ODMA_C_MSGQ_NAME "/jhwc_omsgq_%d"
#define IDMA_D_MSGQ_NAME "/jhwd_imsgq_%d"
#define ODMA_D_MSGQ_NAME "/jhwd_omsgq_%d"

// default image settings
#define JPEG_DEFAULT_QUALITY           95

// packed pixel data format, data size/shape constraints
#define POGO_FMT_MONO               0
#define DEPOGO_FMT_24               1
#define POGO_FMT_24     DEPOGO_FMT_24
#define POGO_FMT_RGBx               2
#define POGO_FMT_xRGB               3
#define POGO_FMT_PLANAR             4

// identifies the current jhw instance (a compressor or decompressor)
#define COMPRESSOR                  0
#define DECOMPRESSOR                1

// max number of IDMA descriptors created for compression.
// increasing this number may increase performance but only if the 
// input buffer size is large enough to fill all of the descriptors
#define MAX_IDMA_DESCRIPTORS 10
// this is used as a max line count when creating ODMA descriptors for
// decompression.  If more lines than this are requested from the decompressor
// an error will occur.
#define MAX_DECOMPRESSED_LINES 10 * 16
// used to verify device structure integrity
#define DEV_COOKIE 0xDaC00C1E

// default scanline timeout in seconds
#define DEFAULT_SCANLINE_TIMEOUT 10

// structure validation macros
#define __JHW_VALIDATE(jhw_info,error_retval) if(jhw_info == NULL || \
  jhw_info->common.jpeghw_context == NULL || \
  ((jpeghw_dev_ptr)jhw_info->common.jpeghw_context)->jhw_dev == NULL || \
  ((struct jpeg_dev_info_s *)((jpeghw_dev_ptr)jhw_info->common.jpeghw_context)->jhw_dev)->cookie != DEV_COOKIE) \
       { DBG_ERR("JPEG lib structure is invalid\n"); \
       jpeghw_error(&jhw_info->common, error_retval); \
       return error_retval; }

#define JHWC_VALIDATE(jhw_info) __JHW_VALIDATE(jhw_info,e_JPEGHW_ERR_INVALID_PARAMETERS)
#define JHWD_VALIDATE(jhw_info) __JHW_VALIDATE(jhw_info,e_JPEGHW_ERR_INVALID_PARAMETERS)

#define JHW_VALIDATE(jhw_info,error_retval) if(jhw_info == NULL || \
  jhw_info->jpeghw_context == NULL || \
  ((jpeghw_dev_ptr)jhw_info->jpeghw_context)->jhw_dev == NULL || \
  ((struct jpeg_dev_info_s *)((jpeghw_dev_ptr)jhw_info->jpeghw_context)->jhw_dev)->cookie != DEV_COOKIE) \
       { DBG_ERR("JPEG common lib structure is invalid\n"); \
       jpeghw_error(jhw_info, error_retval); \
       return error_retval; }

// convenience macros for extracting dev pointers from info structures
#define JHW_GET_DEV(jhwinfo) (struct jpeg_dev_info_s *)((jpeghw_dev_ptr)jhwinfo->jpeghw_context)->jhw_dev
#define JPEGHW_GET_DEV(jhwinfo) (jpeghw_dev_ptr)(jhwinfo->jpeghw_context)

#define JHWC_GET_DEV(jhwcinfo) (struct jpeg_dev_info_s *)((jpeghw_dev_ptr)jhwcinfo->common.jpeghw_context)->jhw_dev
#define JPEGHWC_GET_DEV(jhwcinfo) (jpeghw_dev_ptr)(jhwcinfo->common.jpeghw_context)

#define JHWD_GET_DEV(jhwdinfo) (struct jpeg_dev_info_s *)((jpeghw_dev_ptr)jhwdinfo->common.jpeghw_context)->jhw_dev
#define JPEGHWD_GET_DEV(jhwdinfo) (jpeghw_dev_ptr)(jhwdinfo->common.jpeghw_context)

// convenience inline functions for freeing big buffers
inline struct BigBuffer_s *JHW_FREE_BIG_BUFFER(struct BigBuffer_s *big_buffer);
inline struct BigBuffer_s *JHW_FREE_MAPPED_BIG_BUFFER(struct BigBuffer_s *big_buffer);

// prototype for the ISR callback
typedef void (*jhw_isr_cb_t)(int32_t interrupt_count, void* dev_handle); 

// jpeg asic register template
struct jpeg_dev_regs_s
{
	uintptr_t core;
	uintptr_t top;
	uintptr_t csc;
	uintptr_t pogo;
	uintptr_t depogo;
	uintptr_t pogo_idma_udma;
	uintptr_t pogo_idma_core;
	uintptr_t jpeg_idma_udma;
	uintptr_t jpeg_idma_core;
	uintptr_t pogo_odma_udma;
	uintptr_t pogo_odma_core;
	uintptr_t jpeg_odma_udma;
	uintptr_t jpeg_odma_core;

    void* core_base;
    void* top_base;
    void* csc_base;
    void* pogo_base;
    void* depogo_base;
    void* pogo_idma_udma_base;
    void* pogo_idma_core_base;
    void* jpeg_idma_udma_base;
    void* jpeg_idma_core_base;
    void* pogo_odma_udma_base;
    void* pogo_odma_core_base;
    void* jpeg_odma_udma_base;
    void* jpeg_odma_core_base;
};
typedef struct jpeg_dev_regs_s *jpeg_dev_regs_ptr;

// device core information contained in a structure
struct device_core_info_s
{
    uint32_t      ecs_offset; // buffer offset past header
    uint32_t      h0_m1; //   subsampling
    uint32_t      v0_m1; //   subsampling
    uint32_t      h1_m1; //   subsampling
    uint32_t      v1_m1; //   subsampling
    uint32_t      h2_m1; //   subsampling
    uint32_t      v2_m1; //   subsampling
    uint32_t      h3_m1; //   subsampling
    uint32_t      v3_m1; //   subsampling
    uint32_t      num_mcu_m1;
    uint32_t      dc_code_en_0;
    uint32_t      dc_code_en_1;
    uint32_t      ac_code_en_0;
    uint32_t      ac_code_en_1;
    uint32_t      restart_enable;
    uint32_t      restart_interval_m1;
    uint32_t      dequant_q_map_0;
    uint32_t      dequant_q_map_1;
    uint32_t      dequant_q_map_2;
    uint32_t      dequant_q_map_3;
    uint32_t      huff_dc_table_sel;
    uint32_t      huff_ac_table_sel;
    uint32_t      huff_fsymbol_array [64];
    uint32_t      huff_lsymbol_array [64];
    uint32_t      huff_ptr_array [64];
    uint32_t      huff_sym_array [384];
    uint32_t      huff_code_array [384];
    uint8_t       dequant_q_array [256];
    uint32_t      pad; // pad end of line
    uint32_t      pad_value; // value to use for padding
    uint32_t      jpeg_odma_oblr; // output buffer length
};
typedef struct device_core_info_s *device_core_info_ptr;

// information appropriate for a given big buffer
struct big_buffer_info_s
{
    struct BigBuffer_s *too_small_big_buffer; // buffers too small for commpression
    struct BigBuffer_s *old_big_buffer;   // old buffers containing remainder
    struct BigBuffer_s *older_big_buffer; // older buffers pending deletion
    struct BigBuffer_s *split_buffer; // buffer used instead of using split descriptors
    uint32_t buf_hwaddr;
    uint32_t remainder_lines;  // number of lines contained in the remainder
    uint32_t remainder_offset; // offset into big_buffer where remainder begins
};
typedef struct big_buffer_info_s big_buffer_info_t;

// uio device information
struct uio_info_s
{
    void* uio;               // uio dev handle
    char  uio_dev_name[16];  // uio device name
    int   uio_fd;            // os-assigned file-desc
};
typedef struct uio_info_s uio_info_t;

struct msg_queue_s
{
    mqd_t mqd;       // handle to opened message gueue
    char name[40];   // name of opened message queue
};
typedef struct msg_queue_s msg_queue_t;

// the main low-level device structure used to contain all necessary values
// to perform thread-safe compressions and decompressions
struct jpeg_dev_info_s
{
    uint32_t               cookie; // makes sure this structure is valid
    uint32_t               byte_width; // number of bytes per row in buffer
    uint32_t               pixel_width; // number of pixels per row in buffer
    uint32_t               height; // number of pixels per row in buffer
    int                    device_number; // which jpeg block is in use
    int                    type; // is this a compressor or decompressor
    int                    quality;
    uio_info_t             uio_info;
    msg_queue_t            imqd;
    msg_queue_t            omqd;
    jhw_dma_desc_t         **idma_desc_list;
    jhw_dma_desc_t         **odma_desc_list;
    uint32_t               dma_list_idx;
    jhw_dma_desc_t *       idma_desc;
    jhw_dma_desc_t *       odma_desc;
    pthread_t              odma_thd_id;
    pthread_t              idma_thd_id;
    pthread_mutex_t        isr_mutex;
    uint32_t               line_counter;
    big_buffer_info_t      bb_info;
    bool                   encode_complete;
    bool                   decode_complete;
    bool                   decode_trim;

    // the number of descriptors created in a circular linked list
    int                    number_idma_descriptors;
    int                    number_odma_descriptors;
    int                    max_number_deccompressed_lines;

    struct jpeg_dev_regs_s reg; // base addr + dma registers
    struct device_core_info_s core_info;
};

// structure used to contain event-flags used with the message queues
struct jpeg_event_s
{
    uint32_t event_flags;
    struct jpeg_dev_info_s *dev_info_ptr;
};
typedef struct jpeg_event_s jpeg_event_t, *jpeg_event_ptr;

// table externs
extern const uint8_t jhw_quantization_table[5][256];
extern const uint8_t jhw_dc_bits[2][16];
extern const uint8_t jhw_dc_val[2][12];
extern const uint8_t jhw_ac_bits[2][16];
extern const uint8_t jhw_ac_val[2][162];


// creates/opens access to a message queue identified by msgq_name
mqd_t jhw_open_message_queue(struct jpeg_dev_info_s * dev, char *msgq_name_fmt, char* msgq_name);

// create an array of descriptors
jhw_dma_desc_t **jhw_new_dma_descriptors(int number_dma_descriptors);

// destroy the array of descriptors created via jhw_new_dma_descriptors
void jhw_destroy_dma_descriptors(jhw_dma_desc_t **dma_list, int number_dma_descriptors);

// debug routines for displaying register values
void show_core_regs(struct jpeg_dev_info_s * dev);
void show_udma_regs(UDMA_REGS_t* r, bool idma);

bool jhw_is_initialized();

#endif 
