/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014-2015, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/


 
#ifndef JHW_API_H
#define JHW_API_H

#include <stdbool.h>

#include "jpeghw_api.h"
#include "jpeghw_lib.h"
#include "dma_buffer.h"

// when fill is necessary (e.g. image is not mcu aligned) this is the value used
#define JHW_FILL_BYTE_VALUE  0xff  // fill with white

/**
 * 
 * \brief return compression configuration flags
 * 
 * @param none 
 * 
 * @return configuration flags
 *     JPEGHW_CONFIG_FLAG_MCU_WIDTH_ALIGNMENT == 1 implies
 *     the hardware cannot pad width to achive mcu alignment so
 *     the client is responsible for alignment
 *
 *     JPEGHW_CONFIG_FLAG_MCU_HEIGHT_ALIGNMENT == 1 implies
 *     the hardware cannot handle strips that are not mcu aligned so
 *     the client is responsible for alignment
 */
uint32_t jhwc_get_config_flags();

/**
 *
 * \brief return decompression configuration flags
 * 
 * @param none 
 * 
 * @return configuration flags
 *     JPEGHW_CONFIG_FLAG_MCU_WIDTH_ALIGNMENT == 1 implies
 *     the hardware cannot trim mcu aligned width to achive true
 *     image width so the client is responsible for trimming
 *
 *     JPEGHW_CONFIG_FLAG_MCU_HEIGHT_ALIGNMENT == 1 implies
 *     the hardware cannot trim end-of-image strips that are mcu
 *     aligned so the client is responsible for trimming
 */
uint32_t jhwd_get_config_flags();

/**
 * 
 * \brief allow user to perform a hardware reset of the jpeg 
 *        core
 * 
 * @param struct jpeghw_compress_struct *: pointer to a jpeghw_common 
 * 
 * @return jpeghw-error-type (defined in jpeghw_api.h)
 *  
 */
jpeghw_error_type_t jhw_device_reset (struct jpeghw_common_struct * jhwinfo);

/**
 * 
 * \brief sets/gets the maximum allowed lines in a big_buffer 
 *        given to the driver for containing output from the
 *        decompressor (RGB raw data).  For sets, this value
 *        MUST be a multiple of the MCU height or the call will
 *        fail.
 * 
 * @param struct jpeghw_compress_struct *: pointer to a jpeghw_common 
 * 
 * @return jpeghw-error-type (defined in jpeghw_api.h)
 *  
 */
uint32_t jhw_get_max_decompress_lines (struct jpeghw_common_struct * info);
jpeghw_error_type_t jhw_set_max_decompress_lines (struct jpeghw_common_struct * jhwinfo, uint32_t max_lines);

/**
 * 
 * \brief opens a jpeg compressor/decompressor.  Attaches this 
 *        instance of the compressor/decompressor to the next
 *        available uio device (each uio device is associated
 *        with a physical JPEG hardware core). If no available
 *        uio device can be found one of two scenarios occurs:
 *    1.  if JPEGHW_CONFIG_FLAG_BLOCK_JPEG_CORE_AVAIL bit is
 *        set in jpeghw_common structure config_flags the
 *        software will block waiting for the next available
 *        core.
 *    2.  if JPEGHW_CONFIG_FLAG_BLOCK_JPEG_CORE_AVAIL bit is
 *        clear in jpeghw_common structure config_flags the call
 *        will return an error.
 * 
 * @param struct jpeghw_compress_struct *: pointer to a jpeghw_common 
 *      			     structure
 * 
 * @return jpeghw-error-type (defined in jpeghw_api.h)
 *  
 */
jpeghw_error_type_t jhw_open_compress(struct jpeghw_compress_struct * jhwcinfo);
jpeghw_error_type_t jhw_open_decompress(struct jpeghw_decompress_struct * jhwdinfo);

/**
 * 
 * \brief initializes all software structures and hardware core 
 *        and DMA registers that will be needed during
 *        compression/decompression.
 * 
 * @param struct jpeghw_compress_struct *: pointer to a jpeghw_[de]compress 
 *      			     structure
 * 
 * @return jpeghw-error-type (defined in jpeghw_api.h)
 *  
 */
jpeghw_error_type_t jhw_start_compress(struct jpeghw_compress_struct * jhwcinfo);
jpeghw_error_type_t jhw_start_decompress(struct jpeghw_decompress_struct * jhwdinfo);

/**
 * 
 * \brief frees all allocated memory, unmaps all previously 
 *        JPEGHW_CONFIG_FLAG_BLOCK_JPEG_CORE_AVAIL mapped
 *        registers and memory used by the core and DMA
 *        channels. resets the JPEG core and associated IDMA and
 *        IDMA channels.  Hangs waiting for any unfinished data
 *        to be flushed.
 * 
 * @param struct jpeghw_compress_struct *: pointer to a jpeghw_[de]compress 
 *      			     structure
 * 
 * @return jpeghw-error-type (defined in jpeghw_api.h)
 *  
 */
jpeghw_error_type_t jhw_finish_compress(struct jpeghw_compress_struct * jhwcinfo);
jpeghw_error_type_t jhw_finish_decompress(struct jpeghw_decompress_struct * jhwdinfo);

/**
 * 
 * \brief detaches from and closes access to uio device opened 
 *        in the open call above.  closes all
 *        interprocess/interthread communication channels.
 * 
 * @param struct jpeghw_compress_struct *: pointer to a jpeghw_[de]compress 
 *      			     structure
 * 
 * @return jpeghw-error-type (defined in jpeghw_api.h)
 *  
 */
jpeghw_error_type_t jhw_close_compress(struct jpeghw_compress_struct * jhwcinfo);
jpeghw_error_type_t jhw_close_decompress(struct jpeghw_decompress_struct * jhwdinfo);

/**
 * 
 * \brief frees all allocated memory, unmaps all previously 
 *        mapped registers and memory used by the core and DMA
 *        channels. resets the JPEG core and associated IDMA and
 *        IDMA channels.  Does not wait for any unused or
 *        unflushed data.
 * 
 * @param struct jpeghw_compress_struct *: pointer to a jpeghw_[de]compress 
 *      			     structure
 * 
 * @return jpeghw-error-type (defined in jpeghw_api.h)
 *  
 */
jpeghw_error_type_t jhw_abort_compress(struct jpeghw_compress_struct * jhwcinfo);
jpeghw_error_type_t jhw_abort_decompress(struct jpeghw_decompress_struct * jhwdinfo);

/**
 * 
 * \brief during compress raw RGB data is written to the driver 
 *        and the compressed output data is returned via a
 *        call-back assigned in the cmgr structure.  This call
 *        is the write that delivers data to the input DMA
 *        channel. 
 * 
 * @param struct jpeghw_compress_struct *: pointer to a jpeghw_compress 
 *      			     structure
 * 
 * @param Big_buffer struct ptr: pointer to a BigBuffer that 
 *      	     contains the data to be presented to the
 *      	     input DMA descriptors prior to compression.
 *  
 * @param bool last_strip: set to true if this BigBuffer 
 *             contains the last data for the given image
 *             compression.
 *  
 * @return number of scanlines processed
 *  
 */
uint32_t jhw_write_buffer(struct jpeghw_compress_struct * jhwcinfo,
      struct BigBuffer_s *big_buffer, uint32_t num_lines, bool last_strip);

/**
 * 
 * \brief during decompress raw RGB data is read from the driver
 *        and the compressed output data is retrieved via a
 *        call-back assigned in the dmgr structure.  This call
 *        is the read that retrieves data from the output DMA
 *        channel.
 * 
 * @param struct jpeghw_compress_struct *: pointer to a jpeghw_decompress 
 *      			     structure
 * 
 * @param Big_buffer struct ptr: pointer to a BigBuffer that 
 *      	     will accept the data retrieved from the
 *      	     output DMA following decompression.
 *  
 * @param uint32_t byes_read: contains the number of bytes read
 *      	   from the output DMA and contained withing the
 *      	   above Big_buffer.
 *  
 * @param bool last_strip: returned true if this BigBuffer 
 *             contains the last data for the given image
 *             decompression.
 *  
 * @return jpeghw-error-type (defined in jpeghw_api.h)
 *  
 */
jpeghw_error_type_t jhw_read_buffer(struct jpeghw_decompress_struct * jhwdinfo,
      struct BigBuffer_s *big_buffer, uint32_t *bytes_read, bool *last_buffer);

/**
 * 
 * \brief initializes the JHW mutexes and memory mapping of the 
 *        hardware registers.  This must be called before the
 *        jhw_openi_... calls above or they will fail.
 *  
 */
void jhw_initialize();

/**
 * 
 * \brief destroys the JHW mutexes and unmaps the hardware
 *        registers memory.  This must be called before
 *        terminating the application or system memory leaks can
 *        occur. 
 *  
 */
void jhw_terminate();


/**
 *
 * \brief return default Quantization table values
 *
 */
jpeghw_error_type_t jhw_get_default_quant_table(uint8_t quality, uint8_t *table, uint32_t *size);

/**
 *
 * \brief return default Huffman table values
 *
 */

jpeghw_error_type_t jhw_get_default_huff_table(int table_index, bool ac, uint8_t *bits, uint32_t *bits_size, uint8_t *val, uint32_t *val_size);

#endif 
