/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014-2015, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/



#include <stdint.h>
#include <stdbool.h>
#include "jpeg_page_adaptor.h"
#include "jpeg_strip_adaptor.h"

#include "dma_buffer.h"


void jpeg_recv_page( struct jpeg_page_adaptor *jpa, void *blob )
{
    if ( jpa && jpa->recv_page )
        (jpa->recv_page)( jpa, blob );
    DBG_ASSERT(jpa && jpa->recv_page);
}

static void jpeg_recv_page_fn( struct jpeg_page_adaptor *jpa, void *blob )
{
    if (jpa) {
        jpa->strip_in_cnt = 0;
        jpa->strip_out_cnt = 0;
        jpa->strip_in_height = 0;
        jpa->strip_out_height = 0;

        if (jpa->adopt_from_page)
        	jpa->adopt_from_page( jpa, blob );

        if (jpa->jpeg_send_page)
        	jpa->jpeg_send_page( jpa, blob ); /// client can output info about starting a page to next stage
    }
}

static void add_strip_fn( struct jpeg_page_adaptor *page, uint32_t height, bool end_of_image )
{
    if ( page  ) {
	page->strip_in_cnt ++;
	page->strip_in_height += height;
	if (end_of_image) 
            page->out_page_height = page->strip_in_height;
    }
}

bool jpa_from_page_at_output_eoi( struct jpeg_page_adaptor *page, uint32_t output_strip_height )
{
    bool ret = false;

    page->strip_out_cnt ++;
    page->strip_out_height += output_strip_height;
    ret = page->out_page_height && (page->strip_out_height >= page->strip_in_height);

    return ret;
}

void jpa_fixup_jpg_page( struct jpeg_page_adaptor *jpa )
{
    if ( jpa ) {
        if ( jpa->fixup_jpg_page_length ) {
            jpa->fixup_jpg_page_length( jpa );
        }
    }
}

void jpa_from_page_close( struct jpeg_page_adaptor *jpa )
{
    if ( jpa ) {
        if ( jpa->page_close ) {
            jpa->page_close( jpa ); // do derived class page close
        }
    }
}

struct jpeg_page_adaptor *construct_jpeg_page_adaptor( 
    struct jpeg_page_adaptor *reuse_if_not_null )
{
    struct jpeg_page_adaptor *jpa = 0;

    if (reuse_if_not_null) 
	jpa = reuse_if_not_null;
    else
	jpa = (struct jpeg_page_adaptor *) MEM_MALLOC( sizeof( struct jpeg_page_adaptor ) );
    if (!jpa) 
	return 0;
    memset( jpa, 0, sizeof( struct jpeg_page_adaptor ) ); // NULL all the function pointers need override.

    jpa->recv_page = jpeg_recv_page_fn;  ///< base class should do the right thing.
    jpa->add_strip = add_strip_fn;      ///< base class should do the right thing.

    return (struct jpeg_page_adaptor *) jpa;
}

//eof jpeg_page_adaptor.c
