/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014-2016, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/



#include <stdint.h>
#include <stdbool.h>
#include <string.h>
#include <stdio.h>
#include "lassert.h"
#include "logger.h"

/* linux logger inplemenation */

#define DBG_PRFX "LOGGER: "
#define LOGGER_MODULE_MASK DEBUG_LOGGER_MODULE_SYSTEM | LOGGER_SUBMODULE_BIT( 9 ) 

/* \brief global variable used to implement DBG_PRINTF macros */
uint32_t modules_dbg_flags[32][8] = { {0} };

// pointers to static module names. 
const char * modules_dbg_submodule_names[32][32-5] = { {0} };
/* lsptodo no memory log yet.
static int dbg_slog_memory_file_cnt = 0;
static char dbg_slogmem_files[20][33] = {{0}};
*/

static const char* debug_logger_module_idx_names[32] = 
{
    "PRINT",
    "CNTRL_PANEL",
    "USB_DEVICE", 
    "NVRAM", 
    "SYSTEM", 
    "NETWORK",
    "GPIO",
    "SCAN",
    "ENGINE",
    "DPRINTF",
    "DEVICES",
    "VIDEO",
    "JBIG",
    "HTTP",
    "FILESYSTEM",
    "GENERATORS",
    "CONSUMABLES",
    "OEM",
    "OEM2",
    "OEM3",
    "OEM4",
    0
};


int logger_submodule_register( int module_index, int submodule_index, const char * module_name )
{
    int i = 0;

    XASSERT( module_index >= 0 && module_index < 32, module_index );
    XASSERT( submodule_index >= 0 && submodule_index < 27, submodule_index );
    XASSERT( debug_logger_module_idx_names[module_index] != 0, module_index  );

    if ( modules_dbg_submodule_names[module_index][submodule_index] == 0 ) 
    {
        modules_dbg_submodule_names[module_index][submodule_index] = module_name;
    } 
    else 
    {
        for ( i = 0; i < 32 && modules_dbg_submodule_names[module_index][i] != 0; i++ )
            ;
        return -i;
    }
    return i;
}

// convert macro to register submodules
// usage: copy the line from the C file to this file to register the submodule by name.
#define SUBMODULE( module, submodule_name, submodule_id ) \
    logger_submodule_register( module, submodule_id, submodule_name )




void logger_common_register(void)
{
    SUBMODULE( DEBUG_LOGGER_MODULE_PRINT,      "main", 0 );
    {
	SUBMODULE( DEBUG_LOGGER_MODULE_PRINT, "parser", 7 ); // parser/common
        SUBMODULE( DEBUG_LOGGER_MODULE_PRINT, "printstat", 8 ); // printstat 
        SUBMODULE( DEBUG_LOGGER_MODULE_PRINT, "pjl", 9 ); // PJL 
        SUBMODULE( DEBUG_LOGGER_MODULE_PRINT, "passthru", 10 ); // pass through parser
        SUBMODULE( DEBUG_LOGGER_MODULE_PRINT, "printtools", 11 ); 
        SUBMODULE( DEBUG_LOGGER_MODULE_PRINT, "JM", 12 ); // JobMgr
        SUBMODULE( DEBUG_LOGGER_MODULE_PRINT, "PM", 13 ); // PrintMgr
        SUBMODULE( DEBUG_LOGGER_MODULE_PRINT, "EM", 14 ); // ??
        SUBMODULE( DEBUG_LOGGER_MODULE_PRINT, "consumable", 15 );
        SUBMODULE( DEBUG_LOGGER_MODULE_PRINT, "motors", 16 ); // move to devices?
        SUBMODULE( DEBUG_LOGGER_MODULE_PRINT, "zj", 17 ); // zj parser
        SUBMODULE( DEBUG_LOGGER_MODULE_PRINT, "pip", 18 ); // print image pipe
        SUBMODULE( DEBUG_LOGGER_MODULE_PRINT, "printmode", 19 );
        SUBMODULE( DEBUG_LOGGER_MODULE_PRINT, "eng", 20 );
	SUBMODULE( DEBUG_LOGGER_MODULE_PRINT, "zxdrawer", 21 );
	SUBMODULE( DEBUG_LOGGER_MODULE_PRINT, "ggs", 22 ); // PDL parser
	SUBMODULE( DEBUG_LOGGER_MODULE_PRINT, "urf", 23 ); // urf parser
	SUBMODULE( DEBUG_LOGGER_MODULE_PRINT, "vcp", 24 ); // virtual control panel debug only.
        SUBMODULE( DEBUG_LOGGER_MODULE_PRINT, "pip_odma", 25 ); // print image pipe

    }

    SUBMODULE( DEBUG_LOGGER_MODULE_CNTRL_PANEL, "main", 0 ); // add kinoma sub modules?
    SUBMODULE( DEBUG_LOGGER_MODULE_USB_DEVICE,  "main", 0 ); // move to devices? 
    SUBMODULE( DEBUG_LOGGER_MODULE_NVRAM,       "main", 0 ); // move to devices?

    SUBMODULE( DEBUG_LOGGER_MODULE_SYSTEM,      "main", 0 );
    {
        SUBMODULE( DEBUG_LOGGER_MODULE_SYSTEM, "dma_alloc", 6 );
        SUBMODULE( DEBUG_LOGGER_MODULE_SYSTEM, "audio", 7 );
	SUBMODULE( DEBUG_LOGGER_MODULE_SYSTEM, "logger", 9 );
	SUBMODULE( DEBUG_LOGGER_MODULE_SYSTEM, "spi", 11 ); // move to devices?
        SUBMODULE( DEBUG_LOGGER_MODULE_SYSTEM, "sjm", 12 );
        SUBMODULE( DEBUG_LOGGER_MODULE_SYSTEM, "rm", 13 );
        SUBMODULE( DEBUG_LOGGER_MODULE_SYSTEM, "cm", 14 ); 
	SUBMODULE( DEBUG_LOGGER_MODULE_SYSTEM, "mlim", 24 );
	SUBMODULE( DEBUG_LOGGER_MODULE_SYSTEM, "mem", 25 );
	SUBMODULE( DEBUG_LOGGER_MODULE_SYSTEM, "statusmgr", 26 ); // isn't this dead?
    }

    SUBMODULE( DEBUG_LOGGER_MODULE_CONSUMABLES, "main", 0 );


    SUBMODULE( DEBUG_LOGGER_MODULE_GPIO,        "main", 0 ); // move to devices?
    SUBMODULE( DEBUG_LOGGER_MODULE_SCAN,        "main", 0 ); // sub modules?
    SUBMODULE( DEBUG_LOGGER_MODULE_ENGINE,      "main", 0 ); // move to video?

    SUBMODULE( DEBUG_LOGGER_MODULE_DPRINTF,     "main", 0 ); // shouldn't this die?

    SUBMODULE( DEBUG_LOGGER_MODULE_DEVICES,     "main", 0 );
    {
        SUBMODULE( DEBUG_LOGGER_MODULE_DEVICES,     "gpio_led", 1 );
        SUBMODULE( DEBUG_LOGGER_MODULE_DEVICES,     "gpiologger", 2 );
        SUBMODULE( DEBUG_LOGGER_MODULE_DEVICES,     "dec_adc", 3 );
        SUBMODULE( DEBUG_LOGGER_MODULE_DEVICES,     "dec_laser", 4 );
        SUBMODULE( DEBUG_LOGGER_MODULE_DEVICES,     "dec_fuser", 5 );
        SUBMODULE( DEBUG_LOGGER_MODULE_DEVICES,     "dec_sensor", 6 );
        SUBMODULE( DEBUG_LOGGER_MODULE_DEVICES,     "dcmotor", 7 );
        SUBMODULE( DEBUG_LOGGER_MODULE_DEVICES,     "stepper", 8 );
        SUBMODULE( DEBUG_LOGGER_MODULE_DEVICES,     "pwm", 9 );
        SUBMODULE( DEBUG_LOGGER_MODULE_DEVICES,     "delay", 10 );
        SUBMODULE( DEBUG_LOGGER_MODULE_DEVICES,     "eth_mac", 11 );
        SUBMODULE( DEBUG_LOGGER_MODULE_DEVICES,     "eth_phy", 12 );
        SUBMODULE( DEBUG_LOGGER_MODULE_DEVICES,     "cdma", 13 );
        SUBMODULE( DEBUG_LOGGER_MODULE_DEVICES,     "jbig", 14 );
        SUBMODULE( DEBUG_LOGGER_MODULE_DEVICES,     "uio", 15 );
        SUBMODULE( DEBUG_LOGGER_MODULE_DEVICES,     "jpegi", 16 );
        SUBMODULE( DEBUG_LOGGER_MODULE_DEVICES,     "jpego", 17 );
        SUBMODULE( DEBUG_LOGGER_MODULE_DEVICES,     "jpegc", 18 );
        SUBMODULE( DEBUG_LOGGER_MODULE_DEVICES,     "jpegd", 19 );
        SUBMODULE( DEBUG_LOGGER_MODULE_DEVICES,     "jpegg", 20 );
    }

    SUBMODULE( DEBUG_LOGGER_MODULE_VIDEO,       "main", 0 );
    {
        SUBMODULE( DEBUG_LOGGER_MODULE_VIDEO,       "video", 1 );
        SUBMODULE( DEBUG_LOGGER_MODULE_VIDEO,       "video_laser", 2 );
    }
    SUBMODULE( DEBUG_LOGGER_MODULE_JBIG,        "main", 0 ); // move to devices/jbig


    logger_submodule_register( DEBUG_LOGGER_MODULE_NETWORK, 0, "main" );
    {
        logger_submodule_register( DEBUG_LOGGER_MODULE_NETWORK, 1,  "link" );
        logger_submodule_register( DEBUG_LOGGER_MODULE_NETWORK, 2,  "net_io" );
        logger_submodule_register( DEBUG_LOGGER_MODULE_NETWORK, 3,  "raw_io" );
        logger_submodule_register( DEBUG_LOGGER_MODULE_NETWORK, 4,  "print" );
        logger_submodule_register( DEBUG_LOGGER_MODULE_NETWORK, 5,  "scan" );
        logger_submodule_register( DEBUG_LOGGER_MODULE_NETWORK, 6,  "sm_job" );
        logger_submodule_register( DEBUG_LOGGER_MODULE_NETWORK, 7,  "ipp" );
        logger_submodule_register( DEBUG_LOGGER_MODULE_NETWORK, 8,  "gcpp" );
        logger_submodule_register( DEBUG_LOGGER_MODULE_NETWORK, 9,  "wsd" );
        logger_submodule_register( DEBUG_LOGGER_MODULE_NETWORK, 10, "telnet" );
        logger_submodule_register( DEBUG_LOGGER_MODULE_NETWORK, 11,  "iface" );
    }



    SUBMODULE( DEBUG_LOGGER_MODULE_HTTP,        "main", 0 ); 
    {
        SUBMODULE( DEBUG_LOGGER_MODULE_HTTP,        "http", 26 );
        SUBMODULE( DEBUG_LOGGER_MODULE_HTTP,        "methods", 25 );
        SUBMODULE( DEBUG_LOGGER_MODULE_HTTP,        "ssi", 24 );
        SUBMODULE( DEBUG_LOGGER_MODULE_HTTP,        "oid", 23 );
        SUBMODULE( DEBUG_LOGGER_MODULE_HTTP,        "usb", 22 );
        SUBMODULE( DEBUG_LOGGER_MODULE_HTTP,        "net", 21 );
    }
    SUBMODULE( DEBUG_LOGGER_MODULE_FILESYSTEM,  "main", 0 ); // 

    SUBMODULE( DEBUG_LOGGER_MODULE_GENERATORS,  "main", 0 ); // move to video? 
    {
        SUBMODULE( DEBUG_LOGGER_MODULE_GENERATORS,  "gen_comp", 1 );
        SUBMODULE( DEBUG_LOGGER_MODULE_GENERATORS,  "gen_decomp", 2 );
        SUBMODULE( DEBUG_LOGGER_MODULE_GENERATORS,  "gen_mip", 3 );
        SUBMODULE( DEBUG_LOGGER_MODULE_GENERATORS,  "gen_pip", 4 );
        SUBMODULE( DEBUG_LOGGER_MODULE_GENERATORS,  "gen_nup", 5 );
    }
    SUBMODULE( DEBUG_LOGGER_MODULE_OEM1,  "main", 0 );
    SUBMODULE( DEBUG_LOGGER_MODULE_OEM2,  "main", 0 );
    SUBMODULE( DEBUG_LOGGER_MODULE_OEM3,  "main", 0 );
    SUBMODULE( DEBUG_LOGGER_MODULE_OEM4,  "main", 0 );


}


void __attribute__((weak)) logger_oem_register()
{

}

static const char * debug_logger_level_names[] =
{
    "LOG_CRIT",
    "LOG_ERR",
    "LOG_WARNING",     
    "LOG_NOTICE",      
    "LOG_INFO",        
    "LOG_DEBUG",       
    "LOG_DEBUG_M",     
    "LOG_DEBUG_H",
    0
};     

#ifndef cmd_printf
#define cmd_printf printf
#endif
#ifndef CMD_USAGE_ERROR
#define CMD_OK 0
#define CMD_USAGE_ERROR -2
#endif

void logger_clear_module(uint32_t module)
{
    int l;

    // ASSERT(module < 32);
    module &= 0x1f; 

    for (l=0; l < 8; l++ )
    {
        modules_dbg_flags[module][l] = 0; 
    }
}

void logger_enable_module_mask(uint32_t module, uint32_t submod_level)
{
    uint32_t l;
    uint32_t level = submod_level & 0x00000007;
    uint32_t mask = submod_level & ~0x0000001f;

    for (l=0; l <= level; l++ )
    {
        modules_dbg_flags[module][l] |= mask; 
    }
    for (l=level+1; l < 8; l++ )
    {
        modules_dbg_flags[module][l] &= ~mask;
    }
} 


void logger_enable_module_level(uint32_t module, uint32_t level, uint32_t mask)
{
    uint32_t l;

    // force inputs to be legal
    level  &= 0x00000007;
    module &= 0x0000001f;
    mask &= 0xffffffe0;  

    if ( mask )
    {
        for (l=0; l <= level; l++ )
        {
            modules_dbg_flags[module][l] |= mask; 
        }
        l = level + 1;
    } 
    else
    {
        l = level ;  // zero input clears that level, greater than zero sets mask.
        mask = 0xffffffff;
    }

    for ( ; l < 8; l++ )
    {
        modules_dbg_flags[module][l] &= ~mask; 
    }
}


static int bad_level( int l, const char * name )
{
    int i;
  
    cmd_printf("level %s is invalid : ", name );
    for ( i = 0; i < 8 && debug_logger_level_names[i] != 0; i++ ) 
    {
        cmd_printf("%s ", debug_logger_level_names[i]);
    }
    cmd_printf("\n");

    return CMD_USAGE_ERROR;
}

static int bad_module( int m, const char * name )
{
    int i;
    cmd_printf("module %s is invalid : ", name );
    for (i=0; i < 32 && debug_logger_module_idx_names[i] != 0; i++) 
    {
       cmd_printf("%s ", debug_logger_module_idx_names[i]);
    }
    cmd_printf("\n");
    return CMD_USAGE_ERROR;
}

static int bad_submodule( int m, int s, const char * name )
{
    int i;

    cmd_printf("%s submodule [%s] is invalid : ", debug_logger_module_idx_names[m], name );
    for (i=0; i < 32-5; i++) 
    {
        if ( modules_dbg_submodule_names[m][i] != 0 ) 
	{
	  cmd_printf("%s ", modules_dbg_submodule_names[m][i]);
	}
    }
    cmd_printf("\n");

    return CMD_USAGE_ERROR;
}

static int logger_find_level( const char * name )
{
    int i;

    if ( ( 0 == strncmp( "off", name, strlen("off") ) ) ) 
       return 999;

    for ( i = 0; i < 8 && debug_logger_level_names[i] != 0; i++ ) 
    {
        if ( ( 0 == strncasecmp( debug_logger_level_names[i], name, strlen(name) ) ) ) 
	{
	    return i;
	}
    }
    return -i;
}

static int logger_find_module( const char * name )
{
    int i;
    for ( i = 0; i < 32 && debug_logger_module_idx_names[i] != 0; i++ ) 
    {
        DBG_PRINTF_NOTICE("find %s %s %d %d %d %d \n", debug_logger_module_idx_names[i], name, i, 
			  strlen(debug_logger_module_idx_names[i]), strlen(name),
			  strncmp( debug_logger_module_idx_names[i], name, strlen(name) ) );
  
	if ( (0 == strncasecmp( debug_logger_module_idx_names[i], name, strlen(name) ) ) ) 
        {
	    return i;
        }
    }
    return -i;
}



static int logger_find_submodule( int module_index, const char * name )
{
    int i;

    if (0 == name)
       return -1000;

    for ( i = 0; i < 27; i++ ) 
    {
        if ( modules_dbg_submodule_names[module_index][i] 
	     && ( 0 == strncasecmp( modules_dbg_submodule_names[module_index][i], name, strlen(name) ) ) ) 
        {
            return i;
        }
    }
    return -i;
}



int logger_set_3( const char * arg1, const char * arg2, const char * arg3 )
{
    const char *module = arg1;
    const char *submodule = arg2;
    const char *level = arg3;
    
    int m = logger_find_module(module);
    int s = logger_find_submodule(m, submodule);
    int l = logger_find_level(level);
  
    DBG_PRINTF_NOTICE("%s: %s %s %s %d %d %d \n", __FUNCTION__, module, submodule, level, m, s, l ); 

    if ( l < 0 )  { // try changing order of arguments
	    module = arg3;
	    submodule = arg2;
	    level = arg1;

	    m = logger_find_module(module);
	    s = logger_find_submodule(m, submodule);
	    l = logger_find_level(level);
    }
    

    if ( m < 0 ) 
        return bad_module( m, module );
    else if ( s < 0 ) 
        return bad_submodule( m, s, submodule );
    else if ( l < 0 ) 
        return bad_level( l, level );
    else if ( l == 999 ) 
    { 
        for ( l = 0 ; l < 8; l++ ) 
	{
	    modules_dbg_flags[m][l] &= ~LOGGER_SUBMODULE_BIT( s ) ;
	}
    }
    else 
    {
        logger_enable_module_level( m, l, LOGGER_SUBMODULE_BIT( s ) );
    }
    return CMD_OK;
}


int logger_set_2( const char * arg1, const char * arg2 )
{
    const char *module = arg1;
    const char *level = arg2;

    int m = logger_find_module(module);
    int l = logger_find_level(level);
  
    if ( l < 0 )  { // try changing order of arguments
	    module = arg2;
	    level = arg1;
	    m = logger_find_module(module);
	    l = logger_find_level(level);
    }


    if ( m < 0 ) 
        return bad_module( m, module );
    else if ( l < 0 ) 
        return bad_level( l, level );
    else if ( l == 999 ) 
        logger_clear_module(m);
    else 
        logger_enable_module_mask(m, DEBUG_MODULE_MASK | l);

    return CMD_OK;
}

int logger_set_1( const char * level )
{
    int l = logger_find_level( level );
    int i;

    if ( l < 0 ) 
    {
        return bad_level( l, level );
    } 
    else 
    {
	for (i=0; i<32; i++) 
	{
	    if ( l == 999 )
	        logger_clear_module(i);
	    else
	        logger_enable_module_mask(i, DEBUG_MODULE_MASK | l ); 
	}
    }
    return CMD_OK;
}


int logger_init_from_file( char *filename )
{
    FILE *fp = 0;
    char *saveptr;
    char *p1;
    char *p2;
    char *p3;
#define linemax 256
    char line[linemax];
    const char delim[] = " \t,:\n";

    if (!filename) 
	return -1;
    fp = fopen(filename, "r");
    if (!fp) 
	return -1;

    while ((fgets(line, linemax, fp))) {
	printf(line);
	if ((p1 = strtok_r( line, delim, &saveptr ))) {
	    if ((p2 = strtok_r( 0, delim, &saveptr ))) { 	
		if ((p3 = strtok_r( 0, delim, &saveptr ))) {
		    logger_set_3( p1, p2, p3 );
		} else {
		    logger_set_2( p1, p2 );
		}
	    } else {
		logger_set_1( p1 );
	    }
	}
    }
    return 0;
#undef linemax 
}

void logger_init()
{
    logger_common_register();
    logger_oem_register();

    // read in logger config from file?
    if ( 0 != logger_init_from_file( "/home/root/logger_config_file" ) ) {
        // not found default logging settings:
	logger_set_1( "LOG_ERR" ); // default
    }
}
