/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2012-2014, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/


#ifndef DMA_BUFFER_H
#define DMA_BUFFER_H
/** 
 * \file dma_buffer.h
 * 
 * \brief BigBuffer compatible routines for doing dma on linux and threadx.
 *
 *  
 */

#include "memAPI.h"
#include "BigBuffer.h"

#ifndef __linux

#define dma_buffer_malloc BigBuffer_Malloc
#define dma_buffer_free BigBuffer_Free
#define dma_buffer_adopt BigBuffer_adopt
#define dma_buffer_realloc BigBuffer_realloc
#define BigBuffer_convert_to_dma_buffer( bb ) bb


// todo needs to return bb->data
#define dma_buffer_map_single( bb, direction ) \
  ( direction == DMA_FROM_DEVICE ? \
    cpu_dcache_invalidate_region(bb->data, bb->datalen) : \
    cpu_dcache_writeback_region(bb->data, bb->datalen) )
#define dma_buffer_unmap_single( bb, direction ) \
  ( direction == DMA_FROM_DEVICE ? \
    cpu_dcache_invalidate_region(bb->data, bb->datalen) : \
    (void) ) 

#else // linux

#include "dma_alloc_api.h"

struct BigBuffer_s *dma_buffer_malloc( mlimiter_t *limiter, int32_t len );
struct BigBuffer_s *dma_buffer_realloc( struct BigBuffer_s *bigBuffer, int32_t len );
struct BigBuffer_s *dma_buffer_free( struct BigBuffer_s *bigBuffer );

/// dma_buffer_malloc or BigBuffer_convert_to_dma_buffer( BigBuffer_Malloc )
struct BigBuffer_s *BigBuffer_convert_to_dma_buffer( struct BigBuffer_s *big_buffer );

struct BigBuffer_s *PAddr_convert_to_dma_buffer( void *paddr, void *vaddr, int32_t len);

/// adopt a foreign memory block and convert to a dma buffer, 
/// free the original iff a copy is made
struct BigBuffer_s *dma_buffer_adopt( void *data, int32_t len );

/// convert a foreign memory block into a dma buffer via a copy.
struct BigBuffer_s *dma_buffer_copy_from( void *data, int32_t len );


/// must be called pre dma, returns the HW address to program the hw dma,
/// flush/invalidates based on the direction of the dma.
void * dma_buffer_map_single( struct BigBuffer_s *bigBuffer, int direction ); 

/// must be called post dma 
/// flush/invalidates based on the direction of the dma.
void * dma_buffer_unmap_single( struct BigBuffer_s *bigBuffer, int direction ); 

/// get the data pointer mapped to application space to allow modification.
void *dma_buffer_mmap_forcpu( struct BigBuffer_s *bigBuffer );

/// free the mapping when done.
void *dma_buffer_unmmap_forcpu( struct BigBuffer_s *bigBuffer );

/// copys information from kernel to create a BigBuffer 
/// used from a copy_to_user from kernel space to user space
/// a new object is created via a copy of data
struct BigBuffer_s * dma_buffer_from_kernel( struct dma_alloc_s *kernel_copy_to_user_dma_alloc );

/// do limiter tracking from inside of dma_buffer_from_kernel() ONLY please.
///
void dma_alloc_track_add(struct dma_alloc_s *dma_alloc);

/// get a dma descriptor that is uncached, size is limited.
// defined in memAPI.h
//void *memMallocPhysical( void **hw_address, size_t size, int align, char *file, int line );
//void *memFreePhysical( void *hw_address, void *v_address );
#endif // linux 
#endif
