/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2012-2016, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/


#ifndef DMA_ALLOC_API_H
#define DMA_ALLOC_API_H

#include <stdint.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <linux/types.h>

#ifdef __cplusplus
extern "C" {
#endif

/// dma_alloc_api violates the kernel's memory space by allowing:
/// user space allocations in the kernel.
/// MUST be root to use as /dev/mem and /dev/dma_alloc require root permission.
///
/// mmap interface allows application rw of the memory at the cost
///   of using up extra page table space.  Don't keep it mapped forever as this
///   effectively doubles the "ram" usage of a block while it's mapped.
///
///
/// vmalloc and sg lists to come.

#define DMA_ALLOC_DEVICE_FILENAME "/dev/dma_alloc"

enum dma_alloc_operation_e {
	dma_operation_alloc,            ///< kmalloc
	dma_operation_free,             ///< kfree
	dma_operation_map,              ///< dma_map_single
	dma_operation_unmap,            ///< dma_unmap_single
	dma_operation_sync_for_cpu,     ///< dma_sync_single_for_cpu
	dma_operation_sync_for_device,  ///< dma_sync_single_for_device
	dma_operation_write_to_kernel,  ///< memcpy to kernel buffer
	dma_operation_read_from_kernel, ///< memcpy from kernel buffer
	dma_operation_mmap_for_cpu,     ///< mmap kernel buffer into application space
        dma_operation_pool_alloc,       ///< allocate dma descriptor
        dma_operation_pool_free,        ///< free dma descriptor
	dma_operation_alloc_coherent,   ///< allocate uncached memory
        dma_operation_free_coherent,    ///< free uncached memory
	dma_operation_alloc_no_retry, ///< kmalloc, no use dropcache in failed to kmalloc
	dma_operation_map_from_premap, ///< do nothing
    dma_operation_cache_flush_with_v_addr, ///< cache flush with v_addr
    dma_operation_cache_invalidate_with_v_addr ///< cache invalidate with v_addr
};

/// include kernel header file here...
#ifndef DMA_TO_DEVICE
#define DMA_TO_DEVICE 1
#define DMA_FROM_DEVICE 2
#define DMA_BIDIRECTIONAL 0
#endif

#ifndef dma_app_addr_t
#ifdef CONFIG_ARCH_DMA_ADDR_T_64BIT
typedef uint64_t dma_app_addr_t;
#else
typedef uint32_t dma_app_addr_t;
#endif
#endif

/// needs to match drivers max, only doing one pool at the moment.
/// #define MAX_DMA_DESCRIPTOR_SIZE 64

struct dma_alloc_s
{
    enum dma_alloc_operation_e operation;  ///< write the command then read result of operation.
    int lines;
    size_t len; ///< length in bytes of the dma buffer
    char *kv_addr;  ///< kernel virtual address
    dma_app_addr_t hw_addr; ///< physical dma'able hw address
    void *v_addr;   ///< user virtual address
    int direction;  ///< DMA_TO_DEVICE DMA_FROM_DEVICE DMA_BIDIRECTIONAL
    void  (*kfree_function)(struct dma_alloc_s *bb);
    void *kv_irb_adaptor;
    int is_use_premap;
    struct dma_alloc_s * next;
} __attribute__ ((packed));


/// kmalloc a dma'able block of ram. Cached, aligned, and contiguous.
/// Probablility of failure on alloc is a function of size 128K might be a good max, in the megabyte range will rarely ever work.
///
///
struct dma_alloc_s *dma_alloc( size_t len );

/// same as dma_alloc but reuses an existing dma_alloc_ structure.
///
struct dma_alloc_s *dma_alloc_reuse( struct dma_alloc_s *reuse, size_t len );


/// free the dma buffer in the dma_alloc_t structure
/// Does not free the user space dma_alloc_t structure.
///
/// free( dma_free( dma_alloc ) ); // to free both.
/// Do not use memFree. TODO: do we need to use memMALLOC to allocate/free here?
struct dma_alloc_s *dma_free( struct dma_alloc_s *dma_alloc );

struct dma_alloc_s *dma_alloc_from_paddr( void* paddr, void* vaddr, int32_t len);


/// get the hardware physical address in hw_addr, flush/invalidate depending on
/// direction.
///
struct dma_alloc_s *dma_alloc_map_single( struct dma_alloc_s *dma_alloc, int direction );

/// call when hw is done with the transfer to flush/invalidate for cpu usage
/// normally called from the completion isr
///
struct dma_alloc_s *dma_alloc_unmap_single( struct dma_alloc_s *dma_alloc, int direction );

/// get the user space virtual address in v_addr by mapping the hw address into the
/// process space of the application.
///
struct dma_alloc_s *dma_alloc_mmap_forcpu( struct dma_alloc_s *dma_alloc );

/// unmap the memory from process space, do this prior to freeing
///
///
struct dma_alloc_s *dma_alloc_unmmap_forcpu( struct dma_alloc_s *dma_alloc );

/// allocate a coherent block of uncached memory for use as a dma descriptor.
/// don't try large allocations it will fail this is for descriptors. ie < 4k.
///
struct dma_alloc_s *dma_alloc_pool_alloc( struct dma_alloc_s *dma_alloc, size_t size );

/// free a dma descriptor allocated with dma_alloc_pool_alloc()
///
///
struct dma_alloc_s *dma_alloc_pool_free( struct dma_alloc_s *dma_alloc );

/// write as in copy data
///
///
struct dma_alloc_s *dma_alloc_write( struct dma_alloc_s *dma_alloc, const void *buf, size_t count );

/// read as in copy data
///
///
struct dma_alloc_s *dma_alloc_read( struct dma_alloc_s *dma_alloc, void *buf, size_t count );


/// allocate uncached memory
///
///
struct dma_alloc_s *dma_alloc_alloc_coherent( size_t len );

/// free uncached memory
///
///
struct dma_alloc_s *dma_alloc_free_coherent( struct dma_alloc_s *dma_alloc );

#ifdef __cplusplus
}
#endif

#endif // xxx_H
