/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2012-2016, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/



#include <linux/version.h>
#include <linux/module.h>
#include <linux/types.h>

#include <linux/fs.h>
#include <linux/init.h>
#include <linux/kernel.h>

#include <linux/rwsem.h>
#include <linux/mutex.h>

#include <linux/uaccess.h>
#include <linux/interrupt.h>

#include <linux/slab.h>
#include <linux/dma-mapping.h>
#include <linux/dmapool.h>

#include <linux/cdev.h>
#include <linux/device.h>
#include <linux/platform_device.h>
#include <linux/of_platform.h>
#include <linux/of.h>

#include <asm/string.h>   /* for memset */
#include <asm/io.h>       /* for inl outl io access */

#include "premap.h"
#include "premapmm.h"

/******************* F O R W A R D  D E C L A R A T I O N S *******************/

static int dma_alloc_open(struct inode *inode, struct file *filep);
static int dma_alloc_release(struct inode *inode, struct file *filep);
static int dma_alloc_read(struct file *filep, char __user *buf, size_t len, loff_t *ptr);
static int dma_alloc_write(struct file *filep, const char __user *buf, size_t len, loff_t *ptr);
static int dma_alloc_platform_probe(struct platform_device *pdev);
static int dma_alloc_platform_remove(struct platform_device *pdev);

#if 1 /* RICOH retry kmalloc */
extern int sgw_refrain_fcache(int );
#define REFRAIN_FCACHE_RATE 108 /* 8%(force) */
#endif

/******************** D R I V E R  D E C L A R A T I O N S ********************/


enum dma_alloc_operation_e {
	dma_operation_alloc,
	dma_operation_free,
	dma_operation_map,
	dma_operation_unmap,
	dma_operation_sync_for_cpu,
	dma_operation_sync_for_device,
	dma_operation_write_to_kernel,
	dma_operation_read_from_kernel,
	dma_operation_mmap_for_cpu,
	dma_operation_pool_alloc,
	dma_operation_pool_free,
	dma_operation_alloc_coherent,
	dma_operation_free_coherent,
	dma_operation_alloc_no_retry,
	dma_operation_map_from_premap,
	dma_operation_cache_flush_with_v_addr,
	dma_operation_cache_invalidate_with_v_addr
};

#define MAX_DMA_DESCRIPTOR_SIZE 64

/* fileptr -> private data */
struct dma_alloc_s {
	enum dma_alloc_operation_e operation;
	int lines;
	size_t len;
	char *kv_addr;
	dma_addr_t hw_addr;
	void *v_addr;
	int direction;
	void  (*kfree_function)(struct dma_alloc_s *bb);
	void *kv_irb_adaptor;
	int is_use_premap;
    struct dma_alloc_s * next;
} __attribute__ ((packed));


struct device * this_dev = NULL;

struct dma_alloc_dev {
	int Minor;
};

static const struct file_operations dma_alloc_fops = {
	.owner          = THIS_MODULE,
	.open           = dma_alloc_open,
	.release        = dma_alloc_release,
	.read           = dma_alloc_read,
	.write          = dma_alloc_write,
};


struct dma_alloc_drvr {
	int Major;
	int Minor;
	struct cdev Cdev;
	dev_t Dev;
	struct class *Class;
	struct platform_device *pdev; /* = dma_alloc_dev */
	struct dma_alloc_dev *dma_alloc_dev;
	struct dma_pool *pool;
};

static struct dma_alloc_drvr dma_alloc_drvr = {
/*
  .fops = {
  .owner          = THIS_MODULE,
  .open           = dma_alloc_open,
  .release        = dma_alloc_release,
  .read           = dma_alloc_read,
  .write          = dma_alloc_write,
  },
*/
};

#define DRVNAME "dma_alloc"

static u64 default_dmamask = DMA_BIT_MASK(31);


static struct platform_driver dma_alloc_platform_driver = {
	.driver = {
		.owner = THIS_MODULE,
		.name =  DRVNAME,
	},
	.probe = dma_alloc_platform_probe,
	.remove = dma_alloc_platform_remove,
	/* .suspend = dma_alloc_suspend, */
	/* .resume = dma_alloc_resume, */
};

void  dma_alloc_device_release(struct device *dev)
{
    printk("%s\n",__func__);
}

static struct platform_device dmaalloc_device = {
	.name		= DRVNAME,
	.id		= -1,
    .dev	= {
        .dma_mask		= &default_dmamask,
        .coherent_dma_mask	= DMA_BIT_MASK(31),
        .release = dma_alloc_device_release,

    },
};


/*** ***************** Direct Kernel API *******************
Allocate an uncached memory block.
returns virtual address to uncached memory and a physical address for hardware direct access
note the CPU virtual address and the Hardware physical address may be different on some
platforms such as linux.
*/
void *memMallocPhysical(
	void **physicalAddress, /**< return physical hardware address */
	uint32_t size, /**< number of bytes to allocate */
	uint32_t align,  /**< alignment in power of 2; ignored on Linux */
	const char *file, const int line)  /**< Debug; currently ignored on Linux */
{
	BUG_ON(size > MAX_DMA_DESCRIPTOR_SIZE); /* lets find the largest size we need. */
	BUG_ON(!dma_alloc_drvr.pool);

	return dma_pool_alloc(dma_alloc_drvr.pool, GFP_DMA | GFP_KERNEL,
			      (dma_addr_t *)physicalAddress);
}
EXPORT_SYMBOL(memMallocPhysical);

void *memFreePhysical(void *hw_addr, void *v_addr)
{
	dma_pool_free(dma_alloc_drvr.pool, v_addr, (dma_addr_t)hw_addr);
	return 0;
}
EXPORT_SYMBOL(memFreePhysical);

void *dma_buffer_map_single(struct dma_alloc_s *dma_alloc, int direction)
{
	dma_alloc->direction = direction;
	dma_alloc->hw_addr = dma_map_single(this_dev,
					    dma_alloc->kv_addr,
					    dma_alloc->len,
					    direction);
	return (void *)dma_alloc->hw_addr;
}
EXPORT_SYMBOL(dma_buffer_map_single);

void dma_buffer_unmap_single(struct dma_alloc_s *dma_alloc)
{
	dma_unmap_single(this_dev,
			 dma_alloc->hw_addr,
			 dma_alloc->len,
			 dma_alloc->direction);
	dma_alloc->hw_addr = 0;
}
EXPORT_SYMBOL(dma_buffer_unmap_single);

struct dma_alloc_s *dma_free(struct dma_alloc_s *dma_alloc)
{
	if (dma_alloc->kfree_function)
		dma_alloc->kfree_function(dma_alloc);
	else
		kfree(dma_alloc->kv_addr);
	kfree(dma_alloc);
	return 0;
}
EXPORT_SYMBOL(dma_free);

struct dma_alloc_s *dma_alloc(size_t len)
{
	int ret;

	struct dma_alloc_s *dma_alloc_block = kmalloc(sizeof(struct dma_alloc_s), GFP_KERNEL);

	if (dma_alloc_block) {
		dma_alloc_block->kv_addr = kmalloc(len, GFP_DMA | GFP_KERNEL );

#if 1 /* RICOH retry kmalloc */
		while (dma_alloc_block->kv_addr == 0) {
			printk(KERN_ERR " %s: refrain_fcache %d(size:%d)\n", __FUNCTION__, REFRAIN_FCACHE_RATE, dma_alloc_block->len);
			ret = sgw_refrain_fcache(REFRAIN_FCACHE_RATE);

			/* retry kmalloc */
			dma_alloc_block->kv_addr = kmalloc(len, GFP_DMA | GFP_KERNEL );
			printk(KERN_ERR " refrain_fcache ret:%d res:%p\n", ret, dma_alloc_block->kv_addr);
		}
#endif /* RICOH retry kmalloc */

		if (dma_alloc_block->kv_addr) {
			dma_alloc_block->len = len;
			dma_alloc_block->hw_addr = 0;
			dma_alloc_block->v_addr =  0;
			dma_alloc_block->direction = 0;
			dma_alloc_block->kfree_function = 0;
			dma_alloc_block->lines = 0;
			dma_alloc_block->kv_irb_adaptor = 0;
		} else {
			kfree(dma_alloc_block);
			dma_alloc_block = 0;
		}
	}
	return dma_alloc_block;
}
EXPORT_SYMBOL(dma_alloc);

struct dma_alloc_s *dma_alloc_adopt_irb_lines(void *kv_data, size_t len, int lines,
					      void (*kfree_function)(struct dma_alloc_s *),
					      void *kv_irb_adaptor
	)
{
	struct dma_alloc_s *dma_alloc_block = kmalloc(sizeof(struct dma_alloc_s), GFP_KERNEL);

	if (dma_alloc_block) {
		dma_alloc_block->kv_addr = kv_data;
		if (dma_alloc_block->kv_addr) {
			dma_alloc_block->len = len;
			dma_alloc_block->hw_addr = 0;
			dma_alloc_block->v_addr = 0;
			dma_alloc_block->direction = 0;
			dma_alloc_block->kfree_function = kfree_function;
			dma_alloc_block->lines = lines;
			dma_alloc_block->kv_irb_adaptor = kv_irb_adaptor;
		} else {
			kfree(dma_alloc_block);
			dma_alloc_block = 0;
		}
	}
	return dma_alloc_block;
}
EXPORT_SYMBOL(dma_alloc_adopt_irb_lines);

/******************** User level Char Dev Interface ********************/
static int dma_alloc_open(struct inode *inode, struct file *filep)
{
	int retval = -EBUSY;
	struct dma_alloc_s *dma_alloc_block;

	if (!filep->private_data) {
		filep->private_data = kmalloc(sizeof(struct dma_alloc_s), GFP_KERNEL);
		BUG_ON(filep->f_op != &dma_alloc_fops);
		retval = 0;
		((struct dma_alloc_s *) filep->private_data)->operation = -1;
	}
	return retval;
}

/* close */
static int dma_alloc_release(struct inode *inode, struct file *filep)
{
	kfree(filep->private_data);
	filep->private_data = 0;
	return 0;
}

static int dma_alloc_read(struct file *filep, char __user *buf, size_t len, loff_t *ptr)
{
	int retval = len;
	struct dma_alloc_s *dma_alloc_block = filep->private_data;

	if (0 == dma_alloc_block || -1 == dma_alloc_block->operation)
		retval = 0;
	else if (dma_alloc_block->operation == dma_operation_read_from_kernel) {
		if (copy_to_user(buf, dma_alloc_block->kv_addr, len))
			retval = -EFAULT;
	} else if (copy_to_user(buf, dma_alloc_block, sizeof(struct dma_alloc_s)))
		retval = -EFAULT;
	dma_alloc_block->operation = -1; /* write then read once. */

	return retval;
}

static int dma_alloc_write(struct file *filep, const char __user *buf, size_t len, loff_t *ptr)
{
	struct dma_alloc_s *dma_alloc_block;
	int retval = 0;
	int ret;
	void  *addr;

	dma_alloc_block = (struct dma_alloc_s *) filep->private_data;

	BUG_ON(!dma_alloc_block); /* write without open? */

	if (dma_alloc_block->operation == dma_operation_write_to_kernel) {
		/* must be copying data into an existing kernel buffer
		 * todo copy in blocks?
		 * use offset to add to existing. */
		if (len <= dma_alloc_block->len) {
			if (copy_from_user(dma_alloc_block->kv_addr, buf, len)) {
				retval = -EFAULT;
				return retval;
			}
			dma_alloc_block->operation = -1;
			return len;
		}
	}
	if (copy_from_user(dma_alloc_block, buf, sizeof(struct dma_alloc_s))) {
		retval = -EFAULT;
		return retval;
	}
	retval = sizeof(struct dma_alloc_s);

	switch (dma_alloc_block->operation) {
	case dma_operation_alloc:
//		dma_alloc_block->kv_addr = kmalloc(dma_alloc_block->len, GFP_DMA | GFP_KERNEL );
		pr_debug("");
		addr = premap_allocate_mem(dma_alloc_block->len);
		dma_alloc_block->kv_addr = ioremap(addr, dma_alloc_block->len);

#if 0 /* RICOH retry kmalloc */
		while (dma_alloc_block->kv_addr == 0) {
			printk(KERN_ERR " %s: refrain_fcache %d(size:%d)\n", __FUNCTION__, REFRAIN_FCACHE_RATE, dma_alloc_block->len);
			ret = sgw_refrain_fcache(REFRAIN_FCACHE_RATE);

			/* retry kmalloc */
			dma_alloc_block->kv_addr = kmalloc(dma_alloc_block->len, GFP_DMA | GFP_KERNEL );
			printk(KERN_ERR " refrain_fcache ret:%d ptr:%p\n", ret, dma_alloc_block->kv_addr);
		}
#endif /* RICOH retry kmalloc */
//		dma_alloc_block->hw_addr = 0; /* must lock first. */
		dma_alloc_block->hw_addr = addr; // must lock first.
		dma_alloc_block->v_addr =  0; /* must map from hw_addr. */
		dma_alloc_block->kfree_function = 0;
		break;

	case dma_operation_free:
/*
		if (dma_alloc_block->kfree_function)
			dma_alloc_block->kfree_function(dma_alloc_block);
		else
			kfree(dma_alloc_block->kv_addr);
*/
		premap_free_mem(dma_alloc_block->hw_addr);
		iounmap(dma_alloc_block->kv_addr);
		dma_alloc_block->kv_addr = 0;
		break;

	case dma_operation_map:
//		dma_alloc_block->hw_addr = dma_map_single(this_dev,
//							  dma_alloc_block->kv_addr,
//							  dma_alloc_block->len,
//							  dma_alloc_block->direction);
		cache_flush_range((unsigned char*)dma_alloc_block->kv_addr, dma_alloc_block->len);
		break;

    case dma_operation_unmap:
//		dma_unmap_single(this_dev,
//				 dma_alloc_block->hw_addr,
//				 dma_alloc_block->len,
//				 dma_alloc_block->direction);
		/* dma_alloc_block->hw_addr = 0; */
		cache_invalidate_range((unsigned char*)dma_alloc_block->kv_addr, dma_alloc_block->len);
		break;

	case dma_operation_sync_for_cpu:
		dma_sync_single_for_cpu(this_dev,
					dma_alloc_block->hw_addr,
					dma_alloc_block->len,
					dma_alloc_block->direction);
		break;

	case dma_operation_sync_for_device:
		dma_sync_single_for_device(this_dev,
					   dma_alloc_block->hw_addr,
					   dma_alloc_block->len,
					   dma_alloc_block->direction);
		break;

	case dma_operation_pool_alloc:
		dma_alloc_block->kv_addr =
			memMallocPhysical((void **)&dma_alloc_block->hw_addr,
					  dma_alloc_block->len, 32, 0, 0);
		pr_debug("size %d hw %x kv %p\n",
			 dma_alloc_block->len, dma_alloc_block->hw_addr, dma_alloc_block->kv_addr);
		break;

	case dma_operation_pool_free:
		memFreePhysical((void *)dma_alloc_block->hw_addr, (void *)dma_alloc_block->v_addr);
		break;

	case dma_operation_alloc_coherent:
		dma_alloc_block->kv_addr =
			dma_alloc_coherent(this_dev, dma_alloc_block->len,
					   &dma_alloc_block->hw_addr, GFP_DMA | GFP_KERNEL);
		break;

	case dma_operation_free_coherent:
		dma_free_coherent(this_dev, dma_alloc_block->len,
				  dma_alloc_block->kv_addr, dma_alloc_block->hw_addr);
		dma_alloc_block->len = 0;
		dma_alloc_block->kv_addr = 0;
		dma_alloc_block->hw_addr = 0;
		dma_alloc_block->v_addr = 0;
		break;

	case dma_operation_write_to_kernel:
		break;

	case dma_operation_read_from_kernel:
		if (dma_alloc_block->kv_addr)
			;/* dma */
		break;
	case dma_operation_alloc_no_retry :
#if 0 /* RICOH */
		dma_alloc_block->kv_addr = kmalloc(dma_alloc_block->len, GFP_DMA | GFP_KERNEL );
		if (!dma_alloc_block->kv_addr) {
			printk(KERN_ERR " %s: cannot allocate memory, and no retry (size:%d)\n", __FUNCTION__,  dma_alloc_block->len);
		}
		dma_alloc_block->hw_addr = 0xdeadbeef; // must lock first.
#else /* RICOH */
		pr_debug("");
		addr = premap_allocate_mem(dma_alloc_block->len);
		dma_alloc_block->kv_addr = ioremap(addr, dma_alloc_block->len);
		dma_alloc_block->hw_addr = addr; // must lock first.
#endif /* RICOH */
		dma_alloc_block->v_addr = (void*)0xdeadbeef; // must map from hw_addr.
		dma_alloc_block->kfree_function = 0;
		break;
	case dma_operation_cache_flush_with_v_addr:
		cache_flush_range((unsigned char*)dma_alloc_block->v_addr, dma_alloc_block->len);
		break;
	case dma_operation_cache_invalidate_with_v_addr:
		cache_invalidate_range((unsigned char*)dma_alloc_block->v_addr, dma_alloc_block->len);
		break;

	default:
		retval = -EFAULT;
	}

	return retval;
}

static int dma_alloc_platform_probe(struct platform_device *pdev)
{
	struct dma_alloc_dev *dev;

    this_dev = &pdev->dev;

	dev = kzalloc(sizeof(struct dma_alloc_dev), GFP_KERNEL);
	dma_alloc_drvr.pdev = pdev;
	dma_alloc_drvr.dma_alloc_dev = dev;
	platform_set_drvdata(pdev, dev);

	dma_alloc_drvr.pool =
		dma_pool_create("dma_alloc_device", &pdev->dev, MAX_DMA_DESCRIPTOR_SIZE, 32, 0);
	BUG_ON(!dma_alloc_drvr.pool);
	return 0;
}

static int dma_alloc_platform_remove(struct platform_device *pdev)
{
    /* remove dev handle and class */
    cdev_del(&dma_alloc_drvr.Cdev);
    device_destroy(dma_alloc_drvr.Class, MKDEV(dma_alloc_drvr.Major, dma_alloc_drvr.Minor));
    class_destroy(dma_alloc_drvr.Class);

    /* reclaim any drv/dev-specific allocation */
    if (pdev) {
        struct dma_alloc_dev *dev = platform_get_drvdata(pdev);
        /* remove dma_pool  todo: */
        kfree(dev);
        platform_set_drvdata(pdev, NULL);
    }
    return 0;
}


static int __init dma_alloc_init(void)
{
	int err;
    struct device_node *np;



    platform_set_drvdata(&dmaalloc_device, NULL);
    err = platform_device_register(&dmaalloc_device);

    if (!err)
    {
        this_dev = &dmaalloc_device.dev;
    }

	dma_alloc_drvr.Minor = 1;
	dma_alloc_drvr.Class = class_create(THIS_MODULE, "dma_alloc");
	if (IS_ERR(dma_alloc_drvr.Class)) {
		pr_err("can't register dma_alloc character driver CLASS\n");
		return PTR_ERR(dma_alloc_drvr.Class);
	}

	err = alloc_chrdev_region(&dma_alloc_drvr.Dev, 0, 1, "dma_alloc devices");
	if (err) {
		class_destroy(dma_alloc_drvr.Class);
		pr_err("can't register dma_alloc character driver %d\n", err);
		return err;
	}


	dma_alloc_drvr.Major = MAJOR(dma_alloc_drvr.Dev);
	device_create(dma_alloc_drvr.Class, NULL, MKDEV(dma_alloc_drvr.Major, dma_alloc_drvr.Minor),
		      NULL, "dma_alloc");

    cdev_init(&dma_alloc_drvr.Cdev, &dma_alloc_fops);
	dma_alloc_drvr.Cdev.owner = THIS_MODULE;
	dma_alloc_drvr.Cdev.ops   = &dma_alloc_fops;

	if (cdev_add(&dma_alloc_drvr.Cdev, MKDEV(dma_alloc_drvr.Major, 1), 1) < 0) {
		pr_err("can't add jbig device driver\n");
		return(-1);
	}

	err = platform_driver_register(&dma_alloc_platform_driver);
	if (err)
		goto exit;
	/* sysfs ... */


	pr_err("dma_alloc driver init exit\n");
	return 0;
exit:
	pr_err("dma_alloc driver init failure\n");
	return err;
}



static void __exit dma_alloc_exit(void)
{
    platform_device_unregister( &dmaalloc_device );

    /* cleanup drv */
	platform_driver_unregister(&dma_alloc_platform_driver);
	pr_err("%s done", __func__);
}

module_init(dma_alloc_init)
module_exit(dma_alloc_exit)


MODULE_AUTHOR("Copyright (c) 2012 - 2015 Marvell International Ltd.");
MODULE_DESCRIPTION("Marvell export dma interface to application space");
MODULE_VERSION("1.0");
MODULE_LICENSE("Dual MPL/GPL");
