/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014-2016, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/




#ifndef dma_buffer_h
#define dma_buffer_h

#ifndef __KERNEL__
#error "This kernel api is for the linux kernel modules only"
#endif 

/// in the kernel BigBuffer and struct dma_alloc_s are identical only app space sees a difference.
/// the define aliases incorrectly but this allows kernel code to be threadx compatible.
/// 
#define BigBuffer_s dma_alloc_s   ///< not really for threadx compatibility
#define mlimiter_s dma_alloc_s    ///< always pass 0 in the kernel.

#define dma_alloc_t struct dma_alloc_s 

struct dma_alloc_s
{
    int operation;  ///< write the command then read result of operation.
    int lines; ///< for irb attachment
    size_t len; ///< length in bytes of the dma buffer
    char *kv_addr;  ///< kernel virtual address
    size_t hw_addr; ///< physical dma'able hw address
    void *v_addr;   ///< user virtual address
    size_t direction;  ///< DMA_TO_DEVICE DMA_FROM_DEVICE DMA_BIDIRECTIONAL
    void *(*kfree_function)(struct BigBuffer_s *bigbuffer); ///< if NULL kfree else call function, return to cache etc.
    void *kv_irb_adaptor; ///< for irb attachment
} __attribute__ ((packed));

/// These 3 functions are really private to the kernel's dmaalloc.c / module and shouldn't be used by 
/// clients.  BigBuffer is intended to be a app space owned allocation, 
/// struct BigBuffer_s *dma_buffer_malloc( struct mlimiter_s *limiter, int32_t len );
/// struct BigBuffer_s *dma_buffer_realloc( struct BigBuffer_s *bigBuffer, int32_t len );
/// struct BigBuffer_s *dma_buffer_free( struct BigBuffer_s *bigBuffer );

/// kernel level allocation 
/// intent is to allow common code between threadx and linux this effectively does a interface wrapped kmalloc
struct dma_alloc_s *dma_alloc( size_t len );

/// kernel level free
/// intent is to allow common code between threadx and linux this effectively does a interface wrapped kfree
struct dma_alloc_s *dma_free( struct dma_alloc_s *dma_alloc );

/// a few more functions that really only make sense in application space:
/// dma_buffer_malloc or BigBuffer_convert_to_dma_buffer( BigBuffer_Malloc )
/// struct BigBuffer_s *BigBuffer_convert_to_dma_buffer( struct BigBuffer_s *big_buffer );

/// adopt a foreign memory block and convert to a dma buffer, 
/// free the original iff a copy is made
/// struct BigBuffer_s *dma_buffer_adopt( void *data, int32_t len );

/// convert a foreign memory block into a dma buffer via a copy.
/// struct BigBuffer_s *dma_buffer_copy_from( void *data, int32_t len );


/// must be called pre dma, returns the HW address to program the hw dma,
/// flush/invalidates based on the direction of the dma.
/// equates to dma_map_single( kv_addr, direction );
void * dma_buffer_map_single( struct BigBuffer_s *bigBuffer, int direction ); 

/// must be called post dma 
/// flush/invalidates based on the direction of the dma.
/// equates to dma_unmap_single( hw_addr, direction );
void dma_buffer_unmap_single( struct BigBuffer_s *bigBuffer, int direction );

/// kernel space get the pointer really just a compatibility function with threadx 
#define dma_buffer_mmap_forcpu( bigBuffer ) bigBuffer->kv_addr

/// do nothing compatibility function with threadx.
#define dma_buffer_unmmap_forcpu( bigBuffer ) 0

/// get a dma descriptor that is uncached, size is limited.
// defined in memAPI.h
// void *memMallocPhysical( void **hw_address, size_t size, int align, char *file, int line );
// void *memFreePhysical( void *hw_address, void *v_address );


// 
struct dma_alloc_s *dma_alloc_adopt_irb_lines( void *kv_data, //< kernel virtual address
					       size_t len,  //< byte length
					       int lines,  //< is informational for irb use
                                               void (*kfree_function)(struct dma_alloc_s *), //< can be null 
                                               void *kv_irb_adaptor //< can be null 
    );

#endif
