/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2010-2014, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/



/** 
 * \file jbig_api.h
 * 
 * \brief JBIG driver.
 * 
 * This driver provides functionality to execute JBIG encoding and decoding.This 
 * file contains all driver api functions and data types. 
 *
 * Note: Call jbig_init() at system initialization before using any driver 
 * functions. 
 *
 * To perform a JBIG decode:
 * - Open with jbig_open().
 * - Set the decode configuration parameters with jbig_set_config(). 
 * - Execute the transfer with JbigTransfer().
 * - Close JBIG driver with jbig_close().
 *
 **/

#ifndef _JBIG_H
#define _JBIG_H

#include <stdint.h>
#include <stdbool.h>

#include "jbig.h"
#include <dmaalloc-app/error_types.h>
#include <dmaalloc-app/BigBuffer.h>
#define BigBuffer_t struct BigBuffer_s


#ifdef __cplusplus
extern "C" {
#endif


/** 
 * \brief definitions for event handling
 * 
 * This defines the bitmasks used in the JBIG event callback
 **/ 
#define JBIG_IDMA_COMPLETE_FLAG   0x10000000  /// input (read) DMA is complete
#define JBIG_ODMA_COMPLETE_FLAG   0x20000000  /// output (write) DMA is complete
#define JBIG_CONTROL_FLAG         0x40000000  /// jbig control event; typically encode/decode complete
#define JBIG_STOP_FLAG            0x80000000  /// unsupported
#define JBIG_EVENT_FLAG_MASK      0xF0000000
#define JBIG_EVENT_STATUS_MASK    0x0FFFFFFF
#define JBIG_ODMA_LINES_LEFT_MASK 0x0FFFFFFF  /// Lines remaining in output DMA mask


/** 
 * \brief Transfer width in bits.
 * 
 * This defines the size of the word transfer.
 **/ 
typedef enum
{
    JBIG_8_BITS,            //!< 8 bit.
    JBIG_16_BITS,           //!< 16 bit.
    JBIG_32_BITS,           //!< 32 bit.
} jbig_xfer_width_t;

/** 
 * \brief Burst transfer size.
 * 
 * This defines throughput performance of the device.  Consult hardware 
 * specifications when selecting a size.  If the transfer width is less than 32 
 * bits or the device is non-bursting, then the size must be #CDMA_BURST_1. 
 * 
 * Typically a transfer to/from memory can use the maximum size.
 **/
typedef enum
{
    JBIG_BURST_1,           //!< Size 1.
    JBIG_BURST_2,           //!< Size 2.
    JBIG_BURST_4,           //!< Size 4.
    JBIG_BURST_8            //!< Size 8.
} jbig_burst_size_t;


typedef enum
{
	ATMOV_DISABLED,
	ATMOV_ENABLED
} jbig_atmov_state_t;

typedef enum
{
	JBIG_BPP_1,		//!< 1 bit per pixel.
	JBIG_BPP_2,		//!< 2 bit per pixel.
	JBIG_BPP_4		//!< 4 bit per pixel.
} jbig_bpp_t;


/**
 * \brief JBIG Codec/Decode
 *
 * This defines which JBIG block will be used, CODEC or DECODE. The CODEC block
 * supports both decode and encode and will DMA results to memory only. The
 * DECODE block will decode only but will allow the DMA to write directly to
 * the Davinci block.
 *
 **/
typedef enum
{
	JBIG_CODEC_BLOCK,
	JBIG_DECODE_BLOCK
} jbig_block_t;

/**
 * \brief JBIG Operation
 *
 * This defines whether the JBIG block will be used to encode or decode 
 * using the JBIG compression algorithm.
 *
 **/
typedef enum
{
	JBIG_ENCODE,
	JBIG_DECODE
} jbig_operation_t;

/**
 * \brief Line Template
 *
 * This defines which line template to use during JBIG
 * decode/encode.
 *
 * \warning Currently only 2 Line template should be used
 **/
typedef enum
{
	JBIG_LINE_TEMPLATE_DISABLED,
	JBIG_2_LINE_TEMPLATE,
	JBIG_3_LINE_TEMPLATE
} jbig_line_template_t;

/**
 * \brief Typical Line Detection ( Prediction )
 *
 * This defines whether typical line detection ( prediction ) will be enabled 
 * for JBIG decoding/encoding. 
 *
 **/
typedef enum
{
	JBIG_TPBON_DISABLED,
	JBIG_TPBON_ENABLED
} jbig_tpd_on_state_t;

/**
 * \brief Bypass Endian
 *
 * This field determines how the the 32 bit value from the read DMA
 * is broken into 16 bit words in bypass mode. The 16 bit words are
 * then passed to another unpack which ALWAYS unpacks in big endian
 * format.
 *
 **/
typedef enum
{
	JBIG_LITTLE_ENDIAN,
	JBIG_BIG_ENDIAN
} jbig_bypass_endian_t;

/**
 * \brief Bypass JBIG Decompression
 *
 * This defines whether the JBIG core will decompress incoming data or simply 
 * pass it onto the CBI output regardless of the state of the enable CBI 
 * setting. 
 *
 **/
typedef enum
{
	JBIG_BYPASS_CORE_DISABLED,  //!< Data Passes through JBIG core and is decompressed
	JBIG_BYPASS_CORE_ENABLED	//!< Data passes through JBIG core and is NOT decompressed
} jbig_bypass_core_t;

/**
 * \brief Enable CBI
 *
 * This defines whether the JBIG will go to DMA or to CBI. 
 *
 **/
typedef enum
{
	JBIG_CBI_DISABLED,  //!< Output of JBIG goes to DMA 1
	JBIG_CBI_ENABLED	//!< Output of JBIG goes to the CBI
} jbig_cbi_state_t;

/** 
 * \brief JBIG configuration.
 **/
typedef struct
{
    jbig_block_t            Block;          //!< Which JBIG block will be used CODEC or DECODE
    jbig_operation_t        Operation;       //!< Signifies whether JBIG will encode or decode incoming data
    jbig_burst_size_t       SrcBurst;       //!< Source burst transfer size.
    jbig_bpp_t              JbigBpp;        //!< Bits Per Pixel used to calculate data types
    jbig_atmov_state_t      AtmovState;
    jbig_line_template_t    LineTemplate;
    jbig_tpd_on_state_t     TpbonState;
    jbig_bypass_endian_t    Endian;         //!< determines in bypass mode how 32 bit value is handled
    jbig_bypass_core_t      BypassCore;
    jbig_cbi_state_t        CBI;
    uint8_t                 ATMOVvalue;
    uint32_t                Xd;
    uint32_t                Yd;
    uint32_t                Ld;
    uint32_t                LineWidth;      //!< line width in bytes
    uint32_t                AddExtraLines;
    uint32_t                CheckForWhite;
} jbig_config_t; 

/**
 *
 * \brief JBIG handle type
 *
 */
typedef struct jbig_handle_s jbig_handle_t;

/**
 * \brief JBIG callback function type
 *
 * \param user_data User defined data.
 * \param int_status JBIG interrupt status.
 * 
 **/
typedef void (*jbig_callback_t)(jbig_handle_t* handle, void* context, uint32_t status);

/**
 * \brief JBIG DMA callback function type. Used for both IDMA 
 *        and ODMA.
 *  
 * \param handle pointer to jbig handle on which the ODMA event 
 *        occurred.
 * \param context Pointer to context data supplied by user when 
 *        the buffer was submitted to the DMA.
 * 
 **/
typedef void (*jbig_dma_callback_t)(jbig_handle_t *handle,  void *context);


/**
 * \brief JBIG DMA callback function type. Used for ODMA when 
 *        exact byte count is required.
 *  
 * \param handle pointer to jbig handle on which the ODMA event 
 *        occurred.
 * \param bytes written during to this ODMA descriptor.
 * \param context Pointer to context data supplied by user when 
 *        the buffer was submitted to the DMA.
 * 
 **/
typedef void (*jbig_dma_callback_with_byte_count_t)(jbig_handle_t *handle, uint32_t bytes_written, void *context);


/** 
 * \brief Initialize JBIG driver.
 *
 * This function will initialize the driver. This must be called before using any other driver function.
 **/
void jbig_init( void );
 

/**
 * \brief Open the given JBIG block.
 * 
 * \param handle - Return pointer handle of opened decoder.
 *  
 * \param block - signify JBIG_DECODE or JBIG_CODEC
 * 
 * \return OK if successful, FAIL if not.
 * 
 * This function will attempt to open the JBIG hardware block given in the block
 * parameter. After the jbig block is opened it will be reserved until 
 * jbig_close() is called. This call will BLOCK and not return if a JBIG 
 * hardware block is not available! 
 **/
error_type_t jbig_open( jbig_handle_t **handle, jbig_block_t block );

/**
 * \brief Close the JBIG block.
 * 
 * \param handle Return pointer handle of opened decoder.
 * 
 * \return OK if successful, FAIL if not.
 * 
 * This function will attempt to open the JBIG block given in the block parameter.
 * After the jbig block is opened it will be reserved
 * until jbig_close() is called.
 **/
error_type_t jbig_close( jbig_handle_t *handle );

/**
 * \brief Set the JBIG event callback
 * 
 * \param handle handle to jbig channel 
 * \param jbig_callback JBIG callback function pointer 
 * 
 * \return OK if successful, FAIL if not.
 * 
 **/
error_type_t jbig_set_event_callback( jbig_handle_t *handle, jbig_callback_t jbig_callback, void* context);

/**
 * \brief Set the IDMA callback
 * 
 * \param handle handle to jbig channel 
 * \param idma_callback IDMA callback function pointer 
 * 
 * \return OK if successful, FAIL if not.
 * 
 **/
error_type_t jbig_set_idma_callback( jbig_handle_t *handle, jbig_dma_callback_t idma_callback);

/**
 * \brief Set the ODMA callback
 * 
 * \param handle handle to jbig channel 
 * \param idma_callback ODMA callback function pointer 
 * 
 * \return OK if successful, FAIL if not.
 * 
 **/
error_type_t jbig_set_odma_callback( jbig_handle_t *handle, jbig_dma_callback_t odma_callback);

/**
 * \brief Set the ODMA callback with byte count
 * 
 * \param handle handle to jbig channel 
 * \param idma_callback ODMA with byte count callback function 
 *        pointer
 * 
 * \return OK if successful, FAIL if not.
 * 
 **/
error_type_t jbig_set_odma_byte_count_callback( jbig_handle_t *handle, jbig_dma_callback_with_byte_count_t odma_callback);

/** 
 * \brief Set the channel configuration.
 * 
 * \param handle The handle of the channel to configure.
 * \param config The address of the configuration.
 * 
 * \return OK if successful, FAIL if not.
 * 
 * This function will set the configuration for an opened channel.
 * 
 * \warning Do not set the channel configuration while waiting
 * for a CdmaTransfer() call to complete.
 **/

error_type_t jbig_set_config(jbig_handle_t *handle, const jbig_config_t *config);

/** 
 * \brief Get the channel configuration
 * 
 * \param handle The handle of the channel.
 * \param config Return configuration to the address of this location.
 * 
 * \return OK if successful, FAIL if not.
 * 
 * This function will get the configuration for an opened channel.
 **/
error_type_t jbig_get_config(jbig_handle_t *handle, jbig_config_t *config);


/**
 * \brief Blocking API to encode a strip of data.  The strip will be encoded as a single 
 * jbig image.  On successful return the encode is complete.  The resulting number of bytes 
 * is contained in the bytes_output output parameter.  Note that this routine will reset 
 * the callback routines to internal versions and modify the config information. Thus if the block 
 * is being used in both blocking and asynchronous mode the callbacks and config will need to be reset 
 * after calling this API. 
 *  
 * If the dest_buffer is not large enough to contain the output data the operation will fail.  It is 
 * recommended that a buffer equal to the size of src_buf be provided for the output data and then 
 * resized as needed. 
 * 
 * @param handle 
 * @param src_buf - BigBuffer_t containing the input data; will be converted to a dma buffer 
 * @param dest_buf - BigBuffer_t output data buffer; will be converted to a dma buffer.
 * @param Xd - represents the padded data width, in bits, of each line in the image; must be a multiple of 8
 * @param Yd - represents the length in lines of the image.
 * @param L0 - JBIG lines per stripe; Typically the same as Yd.  Required to be identical to Yd if the image 
 * will be sent on the video driver. 
 * @param option_flags - JBIG option flags 
 * @param bytes_output - number of data bytes output by the encode operation.  This will be the number of output 
 * bytes contained in the dest_buf.  May be used to resize the dest_buf. 
 * 
 * @return error_type_t OK on success
 */
error_type_t jbig_encode_strip( jbig_handle_t *handle, 
                                BigBuffer_t* src_buf, 
                                BigBuffer_t* dest_buf, 
                                uint32_t Xd, 
                                uint32_t Yd, 
                                uint32_t L0, 
                                uint8_t option_flags, 
                                uint32_t* bytes_output );

/**
 * \brief Blocking API to decode a strip of data.  The strip to be decoded must be a single 
 * jbig image.  On successful return the decode is complete.  The resulting number of bytes 
 * is contained in the bytes_output output parameter.  Note that this routine will reset 
 * the callback routines to internal versions and modify the config information. Thus if the block 
 * is being used in both blocking and asynchronous mode the callbacks and config will need to be reset 
 * after calling this API. 
 *  
 * If the dest_buffer is not large enough to contain the output data the operation will fail.
 * 
 * @param handle 
 * @param src_buf - BigBuffer_t containing the input data; will be converted to a dma buffer 
 * @param dest_buf - BigBuffer_t output data buffer; will be converted to a dma buffer. 
 * @param src_buf_len_bytes - length in bytes of the data in the src_buf 
 * @param Xd - represents the padded data width, in bits, of each line in the image; must be a multiple of 8
 * @param Yd - represents the length in lines of the image.
 * @param L0 - JBIG lines per stripe; Typically the same as Yd.  Required to be identical to Yd if the image 
 * will be sent on the video driver. 
 * @param option_flags - JBIG option flags 
 * @param bytes_output - number of data bytes output by the encode operation.  This will be the number of output 
 * bytes contained in the dest_buf.  May be used to resize the dest_buf. 
 * 
 * @return error_type_t OK on success
 */
error_type_t jbig_decode_strip( jbig_handle_t *handle, 
                                BigBuffer_t* src_buf, 
                                uint32_t src_buf_len_bytes, 
                                BigBuffer_t* dest_buf, 
                                uint32_t Xd, 
                                uint32_t Yd, 
                                uint32_t L0, 
                                uint8_t option_flags, 
                                uint32_t* bytes_output );
 

/**
 * \brief Setup the decode parameters for JBIG host based print data.
 * 
 * @param handle 
 * @param Xd - represents the padded data width, in bits, of each line in the video buffer
 * @param Yd 
 * @param L0 - JBIG lines per stripe
 * @param option_flags - JBIG option flags
 * 
 * @return error_type_t 
 */
error_type_t jbig_setup_decode_parms( jbig_handle_t *handle,
                                      bool enable_cbi,
                                      uint32_t Xd,  
                                      uint32_t Yd,
                                      uint32_t L0, 
                                      uint8_t option_flags ); 

/**
 * \brief Add a buffer to JBIG input
 *
 * \param handle The handle of the decoder/encoder.
 *
 * \return OK if successful, FAIL if not.
 *
 * This function will attempt to add an additional buffer
 * to the current JBIG input process.
 *
 **/
error_type_t jbig_add_input_buffer( jbig_handle_t * handle,
                                    BigBuffer_t* buffer,
                                    uint32_t length,
                                    void * userData );

/**
 * \brief Add a buffer to JBIG output
 *
 * \param handle The handle of the decoder/encoder.
 *
 * \return OK if successful, FAIL if not.
 *
 * This function will attempt to add an additional buffer
 * to receive the results of the current JBIG output process.
 *
 **/
error_type_t jbig_add_output_buffer( jbig_handle_t * handle,
                                     BigBuffer_t* buffer,
                                     uint32_t length,
                                     void * userData );

/**
 * \brief start a JBIG process
 *
 * \param handle The handle of the decoder/encoder.
 *
 * \return OK if successful, FAIL if not.
 *
 * This function will attempt to start the JBIG
 * process associated with the given handle.
 *
 **/
error_type_t jbig_start_transaction( jbig_handle_t * handle );

/**
 * \brief stop a JBIG process
 *
 * \param handle The handle of the decoder/encoder.
 *
 * \return OK if successful, FAIL if not.
 *
 * This function will stope the JBIG
 * process associated with the given handle.
 *
 **/
error_type_t jbig_stop_transaction( jbig_handle_t * handle );

/**
 * 
 * \brief Read the number of bytes processed in the last transaction.
 * 
 * @param handle 
 * @param bytes_output 
 * 
 * @return error_type_t 
 */
error_type_t jbig_read_bytes_output_last_transaction( jbig_handle_t *handle, 
                                                      uint32_t *bytes_output );

void jbig_set_output_dma_line_width( jbig_handle_t *handle,
                                     uint32_t line_width_in_bytes );

// functions to configue the DECODE::JCTL register
void jbig_enable_tpbon( jbig_handle_t * handle );
void jbig_disable_tpbon( jbig_handle_t * handle );
void jbig_enable_atmov( jbig_handle_t * handle );
void jbig_disable_atmov( jbig_handle_t * handle );
void jbig_set_atmov_value( jbig_handle_t * handle, uint8_t value);
void jbig_clear_atmov_value( jbig_handle_t * handle );
void jbig_enable_3_line( jbig_handle_t * handle );
void jbig_disable_3_line( jbig_handle_t * handle );
void jbig_enable_bpend( jbig_handle_t * handle );
void jbig_disable_bpend( jbig_handle_t * handle );
void jbig_enable_bypass( jbig_handle_t * handle );
void jbig_disable_bypass( jbig_handle_t * handle );
void jbig_enable_cbi( jbig_handle_t * handle );
void jbig_disable_cbi( jbig_handle_t * handle );
void jbig_set_x_dimension( jbig_handle_t * handle, uint16_t xDimension );
void jbig_set_y_dimension( jbig_handle_t * handle, uint16_t yDimension );
void jbig_set_stripe_length( jbig_handle_t * handle, uint16_t stripLength);
void jbig_set_decode( jbig_handle_t * handle );
void jbig_set_encode( jbig_handle_t * handle );
void jbig_enable_add_line(jbig_handle_t *handle);
void jbig_disable_add_line(jbig_handle_t *handle);
void jbig_enable_check_for_white_strip(jbig_handle_t *handle);
void jbig_disable_check_for_white_strip(jbig_handle_t *handle);

error_type_t jbig_started( jbig_handle_t * handle, uint8_t * started );


/** EXAMPLE DISABLED /example jbig_example.c
 * This is an example of a JBIG transfer.
 */

#ifdef __cplusplus
}
#endif
#endif

