/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2008-2016, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/



 // Note: The JBENABLE register must be manipulated directly, and NOT with the
 // REG_SET_BITS, REG_WRITE, and REG_CLEAR_BITS macros. This is because the soft
 // reset bit operates incorrectly - when a "1" is written to reset it, the
 // register will read a "0". And to clear it by writing a "0" yeilds a "1" when
 // the register is read.
#include <posix_ostools.h>
#include <logger.h>
#include <lassert.h>
#include <memAPI.h>

#include "jbigUtils.h"
//#include "jbig_test.h"
#include <dma_buffer.h>


#ifdef HAVE_POWER_MGR
#include "pwr_mgr_api.h"
#endif

#ifdef ASIC_gemstone

#if ASIC_REV < ASIC_REV_Z1
#error "Unsupported gemstone rev"
#endif

#elif defined(ASIC_SS1000)
// all revs of SS1000 supported
#elif defined(ASIC_granite)
// all revs of granite supported
#elif defined(ASIC_granite2)
// all revs of granite supported
#else
#error "Unsupported ASIC "
#endif


/* -------------------------------------------------------------------------- */
/*                               D E F I N E S                                */
/* -------------------------------------------------------------------------- */

#define DBG_PRFX "JBIG: "
#define LOGGER_MODULE_MASK DEBUG_LOGGER_MODULE_DEVICES | LOGGER_SUBMODULE_BIT( 14 )

#define DBG_SLOG_ENTRY DBG_MEMLOG(LOG_DEBUG_H,"%s ENTRY\n", __func__);
#define DBG_SLOG_EXIT DBG_MEMLOG(LOG_DEBUG_H,"%s EXIT\n", __func__);
#undef DBG_PRINTF_CRIT
#define DBG_PRINTF_CRIT printf
#define HAVE_DBG_PRINTF

// Enable all interrupts from UDMA
#define JBIG_ODMA_UDMA_INT_ENABLE_MASK 0x3F
#define JBIG_ODMA_CORE_INT_ENABLE_MASK 0x3
 

/* -------------------------------------------------------------------------- */
/*                    F O R W A R D  D E C L A R A T I O N S                  */
/* -------------------------------------------------------------------------- */
 
#ifdef __linux__
static void isrJbigIDMA( int32_t interrupt_count, void* context );
static void isrJbigODMA( int32_t interrupt_count, void* context );
static void isrJbigControl( int32_t interrupt_count, void* context );
#else
static void isrJbigIDMA( uint32_t intPos );
static void isrJbigODMA( uint32_t intPos );
static void isrJbigControl( uint32_t intPos );
#endif

static void jbig_delete_pad_strip( BigBuffer_t* src_buf );

static void enableJBIGInterrupts(jbig_handle_t * inHandle );
static void disableJBIGInterrupts(jbig_handle_t * inHandle );

static void jbigFreeDescriptors( );

static uint8_t isJbigStarted( jbig_handle_t * inHandle );

// Jbig block threads. There are indivi
static void* jbigThread( void* Temp);

#ifdef HAVE_POWER_MGR
static void jbig_pwr_init(void);
#endif

static inline uint32_t jbig_calc_strip_buffer_size(jbig_handle_t * inHandle);

/* -------------------------------------------------------------------------- */
/*                         L O C A L  V A R I A B L E S                       */
/* -------------------------------------------------------------------------- */

jbig_handle_t jbig_handles[NUM_JBIG_BLOCKS];

// ThreadX thread object
static pthread_t jbigMessageThread = 0; /* for SIMVA JT No.277 omo */

#define NUM_JBIG_MESSAGES_PER_BLOCK 25

// Stack
#define JBIG_STACK_SIZE  POSIX_MIN_STACK_SIZE
static __attribute__ ((aligned (8))) unsigned char JBIGStack[JBIG_STACK_SIZE];


// Message queue
static mqd_t jbig_msg_queue=-1;


#if __aarch64__ == 1
    #define HI32BIT(val) (uint32_t)(val >> 32)
    #define LO32BIT(val) (uint32_t)(val & 0xffffffff)
    #define U32_TO_PTR(val) ({uintptr_t val64 = 0; val64 |= val; val64;})
#else
    #define HI32BIT(val) (uint32_t)val
    #define LO32BIT(val) (uint32_t)val
    #define U32_TO_PTR(val) (uintptr_t)val
#endif

/**
 * jbig_block_init - modularize jbig initialization, to allow pwrmgr to reinit
 * hardware. 
 * 
 * @author  (10/13/2011)
 * 
 * @param jbig_block_interface 
 */
void jbig_block_init(const jbig_block_config_t* jbig_block_interface)
{
    DBG_MEMLOG(LOG_INFO,"JBIG BLOCK INIT FROM POWER MANAGER!!!\n");
    //jbig_block_interface->jbig_regs->JCTL = 0;
    jbig_block_interface->jbig_idma_regs->int_cl = 0x1FF;
    jbig_block_interface->jbig_odma_regs->UICR = JBIG_ODMA_UDMA_INT_ENABLE_MASK;
    jbig_block_interface->jbig_idma_regs->int_en = 0x1FF /*JBIG_DMA_INT_MASK*/;
    jbig_block_interface->jbig_odma_regs->UIER = JBIG_ODMA_UDMA_INT_ENABLE_MASK;

    jbig_block_interface->jbig_odma_core_regs->OICR = JBIG_ODMA_CORE_INT_ENABLE_MASK;
    jbig_block_interface->jbig_odma_core_regs->OIER = JBIG_ODMA_CORE_INT_ENABLE_MASK;

    // Configure the EOI length update for the output DMA so that the DMA will
    // finish cleanly on a partial output buffer when EOI occurs.
    jbig_block_interface->jbig_odma_core_regs->OCR = 1;

    // Soft reset the JBIG core
    jbig_block_interface->jbig_regs->JBIG_EN = 0;
    jbig_block_interface->jbig_regs->JBIG_EN = JBIG_CODEC_0_CORE_JBIG_EN_S_MASK;
    jbig_block_interface->jbig_regs->JBIG_EN = 0;
}


void jbig_init_rtos(const jbig_block_config_t * jbig_plat_config)
{
    uint32_t block_count;

    for(block_count = 0; block_count < NUM_JBIG_BLOCKS; block_count++)
    {
        memset(&jbig_handles[block_count], 0, sizeof(jbig_handle_t));

        // attach the low level block config to the handle
        jbig_handles[block_count].jbig_block_interface = &jbig_plat_config[block_count];

        jbig_block_init(jbig_handles[block_count].jbig_block_interface);

        uio_int_attach( jbig_handles[block_count].jbig_block_interface->dev_jbig, 0, isrJbigControl, &jbig_handles[block_count]);
        uio_int_attach( jbig_handles[block_count].jbig_block_interface->dev_odma, 0, isrJbigODMA, &jbig_handles[block_count]);
        uio_int_attach( jbig_handles[block_count].jbig_block_interface->dev_idma, 0, isrJbigIDMA, &jbig_handles[block_count]);

        posix_mutex_init(&jbig_handles[block_count].lock);
        sem_init(&jbig_handles[block_count].open_sem, 0, 1);
        sem_init(&jbig_handles[block_count].completion_sem, 0, 0);
    }

    // Create the message queue first, then the thread.
    if(jbig_msg_queue==-1)
    {
        posix_create_message_queue(&jbig_msg_queue, "/jbig_queue", (NUM_JBIG_MESSAGES_PER_BLOCK*NUM_JBIG_BLOCKS), sizeof(JBIG_MESSAGE));
        printf("[%s]LINE:%d create msgq :%d\n", __FUNCTION__, __LINE__, jbig_msg_queue);
    }

    if( jbigMessageThread == 0 ) /* for SIMVA JT No.277 omo */
    {
        posix_create_thread(&jbigMessageThread, jbigThread, 0, "jbig_thread", JBIGStack, JBIG_STACK_SIZE, POSIX_THR_PRI_IMAGE);
        printf("[%s]LINE:%d create thread pid:%x\n", __FUNCTION__, __LINE__, jbigMessageThread);
    }

#ifdef HAVE_POWER_MGR
    jbig_pwr_init();
#endif

}

static inline void jbig_increment_pending_messages(jbig_handle_t* handle)
{
    pthread_mutex_lock(&handle->lock);
    handle->numPendingMessages++;
    pthread_mutex_unlock(&handle->lock);
}

static inline void jbig_decrement_pending_messages(jbig_handle_t* handle)
{
    pthread_mutex_lock(&handle->lock);
    if (handle->numPendingMessages > 0)
    {
        handle->numPendingMessages--;
    }
    pthread_mutex_unlock(&handle->lock);
}

static void jbigSendMessage( jbig_handle_t * handle, JBIGMSG msgType, jbig_descriptor_t * descriptor )
{
    JBIG_MESSAGE jbigMsg;
    // We use an event flag here so that in the event that jbig close is
    // called in an interrupt context, which it is from video. We can still
    // free memory.    
    
    jbigMsg.msgType = msgType;
    jbigMsg.handle = handle;
    jbigMsg.descriptor = descriptor;

    jbig_increment_pending_messages(handle);
    posix_message_send(jbig_msg_queue, (char*)&jbigMsg, sizeof(JBIG_MESSAGE),MQ_DEFAULT_PRIORITY, 0);
}

/**
 * jbigAddDescriptorToList
 *
 * \brief 
 * 
 * \param 
 * 
 * \return 
 *
 *
 **/

void jbigAddDescriptorToList( tJBIGDescriptorList * inList, jbig_descriptor_t * inDescriptor)
{
    ASSERT( inList );
    ASSERT( inDescriptor );
    
    DBG_MEMLOG(LOG_DEBUG_H,"JBIG: Add Pending Descriptor %#x list %#x head %#x tail %#x\n",inDescriptor, inList, inList->ListHead, inList->ListTail);
    
    if( inDescriptor && inList )
    {
        inDescriptor->next_desc_phys_addr = 0;
        inDescriptor->next_desc_virt_addr = NULL;

        if( inList->ListHead == NULL )
        {
            inList->ListHead = inDescriptor;
            inList->ListTail = inDescriptor;
        }
        else
        {
            jbig_descriptor_t * tmpDescriptor = inList->ListTail;
            
            tmpDescriptor->next_desc_virt_addr = inDescriptor;
            tmpDescriptor->next_desc_phys_addr = LO32BIT( (uintptr_t)inDescriptor->hw_addr);
            
            inList->ListTail = inDescriptor;
            DBG_MEMLOG(LOG_DEBUG_H,"JBIG: previous Descriptor next %#x\n",tmpDescriptor->next_desc_virt_addr);                                    
        }    
    }
}

/**
 * jbigGetDescriptorListHead
 *
 * \brief 
 * 
 * \param 
 * 
 * \return 
 *
 *
 **/

static inline jbig_descriptor_t * jbigGetDescriptorListHead( tJBIGDescriptorList inList )
{
    return inList.ListHead;
}

/**
 * jbigGetDescriptorListTail
 *
 * \brief 
 * 
 * \param 
 * 
 * \return 
 *
 *
 **/

static inline jbig_descriptor_t * jbigGetDescriptorListTail( tJBIGDescriptorList inList )
{
    return inList.ListTail;
}

static inline void jbigTransferDescriptorList(tJBIGDescriptorList* destList, tJBIGDescriptorList* srcList)
{
    destList->ListHead = srcList->ListHead;
    destList->ListTail = srcList->ListTail;
                    
    srcList->ListHead = NULL;
    srcList->ListTail = NULL;
}

/**
 * jbigRemoveDescriptorFromListHead
 *
 * \brief 
 * 
 * \param 
 * 
 * \return 
 *
 *
 **/

jbig_descriptor_t * jbigRemoveDescriptorFromListHead( tJBIGDescriptorList * inList )
{
    ASSERT(inList);
    jbig_descriptor_t * returnDescriptor = inList->ListHead;
    
    if( inList->ListHead )
    {
        inList->ListHead = inList->ListHead->next_desc_virt_addr;
        if( inList->ListHead == NULL )
        {
            inList->ListTail = NULL;
        }
        returnDescriptor->next_desc_virt_addr = NULL;
        returnDescriptor->next_desc_phys_addr = 0;
    }
    
    
    DBG_MEMLOG(LOG_DEBUG_H,"JBIG: Remove Descriptor %#x from list %#x\n",returnDescriptor, inList);
    
    return  returnDescriptor;
}

/**
 * @brief Search the virtual address space list and return the descriptor with the matching physical address.
 * @param inList 
 * @param desc_phys_addr 
 * 
 * @return tJBIGDMADescriptor* 
 */
jbig_descriptor_t * jbig_get_desc_vaddr_from_list(tJBIGDescriptorList * inList, jbig_descriptor_t * desc_phys_addr)
{
    jbig_descriptor_t *list_iter = inList->ListHead;
    while(list_iter != NULL)
    {
        if(list_iter->hw_addr == desc_phys_addr)
        {
            return list_iter;
        }
        list_iter = list_iter->next_desc_virt_addr;
    }
    return NULL;
}

/**
 * jbigAddDMABuffer
 *
 * \brief 
 * 
 * \param 
 * 
 * \return 
 *
 *
 **/

static error_type_t jbig_add_DMA_buffer_IDMA( jbig_handle_t * inHandle, 
                                              tJBIGDescriptorList* inList, 
                                              BigBuffer_t* buf, uint32_t size, 
                                              void * context, 
                                              uint32_t dma_direction )
{
    error_type_t returnValue = FAIL;
    void *phys_addr=0;

    // If the specified size is 0, simply return fail.
    if (size == 0)
    {
        return returnValue;
    }
    
    if( inHandle && inHandle->inUse)
    {
        jbig_descriptor_t * p_JBIG_Descriptor = NULL;

        p_JBIG_Descriptor = (jbig_descriptor_t *)MEM_MALLOC_UNCACHED(&phys_addr, sizeof(jbig_descriptor_t),e_32_byte);                            
        if( p_JBIG_Descriptor )
        {
            DBG_MEMLOG(LOG_DEBUG_M,"JBIG: add descriptor %#x addr %#x size %d context %#x\n", p_JBIG_Descriptor, buf, size, context);
            p_JBIG_Descriptor->hw_addr = phys_addr;



            // Set up JBIG read DMA. DMA starts when transfer count register
            // is programmed.
            BigBuffer_convert_to_dma_buffer(buf);

            uint8_t * inputData = (uint8_t *)dma_buffer_mmap_forcpu(buf);


            if ( inputData)
            {
                inHandle->last_input_byte = inputData[size-1];
            }

            dma_buffer_unmmap_forcpu(buf);

            // capture the last byte of the input data to compare to "pad strip" later to make sure the pad strip is unique by
            // at least one pixel.

            p_JBIG_Descriptor->idma.dataBuffer = LO32BIT( (uintptr_t)dma_buffer_map_single(buf, dma_direction));
            p_JBIG_Descriptor->idma.length = size;
            p_JBIG_Descriptor->idma.control = 0;
            p_JBIG_Descriptor->context = context;


            REG_SET_BITS(&p_JBIG_Descriptor->idma.control, // JBIG_DESCRIPTOR_END_OF_IMAGE |
                                                      JBIG_DESCRIPTOR_INTERRUPT_ON_COMPLETION |
                                                      JBIG_DESCRIPTOR_OWNERSHIP |
                                                      JBIG_DESCRIPTOR_STOP);

            pthread_mutex_lock(&inHandle->lock);
            jbig_descriptor_t * lastJBIGDescriptor = jbigGetDescriptorListTail( *inList );
            
            if( lastJBIGDescriptor != NULL)
            {
                if( (lastJBIGDescriptor->idma.control & JBIG_DESCRIPTOR_OWNERSHIP ) == 0 )
                {
                    DBG_MEMLOG(LOG_WARNING,"JBIG: last descriptor ownership bit is NULL\n" );
                }

                REG_CLEAR_BITS( &lastJBIGDescriptor->idma.control, JBIG_DESCRIPTOR_END_OF_IMAGE | JBIG_DESCRIPTOR_STOP | JBIG_DESCRIPTOR_INTERRUPT_ON_COMPLETION);
            }
            else
            {
                DBG_MEMLOG(LOG_DEBUG_H,"JBIG: chained!!!! last descriptor is NULL\n" );
    
                REG_SET_BITS( &p_JBIG_Descriptor->idma.control, JBIG_DESCRIPTOR_START_OF_IMAGE);                       
            }
            
            inHandle->dirty = 1; // indicates that we may need to clean up the descriptor list later
            jbigAddDescriptorToList( inList, p_JBIG_Descriptor);
            pthread_mutex_unlock(&inHandle->lock);
        
            returnValue = OK;

        }
        else
        {
           DBG_MEMLOG(LOG_DEBUG_H,"JBIG: Descriptor MALLOC failed!!!\n");
        }

    }

    return ( returnValue );
}

static error_type_t jbig_add_odma_to_descriptor_list(jbig_handle_t * inHandle, tJBIGDescriptorList* inList, uint32_t buffer_ptr, uint32_t size, void * context )
{
    error_type_t returnValue = FAIL;
    jbig_descriptor_t * p_JBIG_Descriptor = NULL;
    void *phys_addr=0;


    p_JBIG_Descriptor = (jbig_descriptor_t *)MEM_MALLOC_UNCACHED(&phys_addr, sizeof(jbig_descriptor_t),e_32_byte);                            
    if( p_JBIG_Descriptor )
    {
        p_JBIG_Descriptor->hw_addr = phys_addr;


        p_JBIG_Descriptor->odma.dataBuffer = buffer_ptr;
        p_JBIG_Descriptor->odma.length = size;
        p_JBIG_Descriptor->odma.control = 0;
        p_JBIG_Descriptor->context = context;

        REG_SET_BITS(&p_JBIG_Descriptor->odma.control, JBIG_ODMA_DESCRIPTOR_OWNERSHIP |
                                                  JBIG_ODMA_DESC_INT_ON_COMPLETION |
                                                  JBIG_ODMA_DESC_STOP_ON_COMPLETION );

        pthread_mutex_lock(&inHandle->lock);
        jbig_descriptor_t * lastJBIGDescriptor = jbigGetDescriptorListTail( *inList );
        
        if( lastJBIGDescriptor != NULL)
        {
            if( (lastJBIGDescriptor->odma.control & JBIG_ODMA_DESCRIPTOR_OWNERSHIP ) == 0 )
            {
                DBG_MEMLOG(LOG_WARNING,"JBIG: last descriptor ownership bit is NULL\n" );
            }

            // Clear the "stop on completion" and "interrupt on completion" bits
            // from the control word. The descriptor chain is configured, stop IF
            // there are no more descriptors in the chain!
                REG_CLEAR_BITS(&lastJBIGDescriptor->odma.control,
                                JBIG_ODMA_DESC_STOP_ON_COMPLETION | JBIG_ODMA_DESC_INT_ON_COMPLETION );
            }
        else
        {
            DBG_MEMLOG(LOG_DEBUG_H,"JBIG: chained!!!! last descriptor is NULL\n" );
        }
        
        inHandle->dirty = 1; // indicates that we may need to clean up the descriptor list later
        jbigAddDescriptorToList( inList, p_JBIG_Descriptor);
        pthread_mutex_unlock(&inHandle->lock);
    
        returnValue = OK;
    }
    else
    {
       DBG_MEMLOG(LOG_DEBUG_H,"JBIG: Descriptor MALLOC failed!!!\n");
    }

    return returnValue;
}

static inline uint32_t jbig_calc_strip_buffer_size(jbig_handle_t * inHandle)
{
    return ( inHandle->config.LineWidth * inHandle->config.Ld );
}


static error_type_t jbig_add_odma_with_context( jbig_handle_t * inHandle, 
                 tJBIGDescriptorList* inList, 
                 uint32_t buffer_ptr, 
                 uint32_t buffer_size, 
                 void * context, 
                 uint32_t last_byte )
{
    error_type_t returnValue = FAIL;

    // If this is the last strip for the incoming buffer then add it to the list with the context so it will 
    // notify the user that their buffer is complete.
    if (buffer_ptr + buffer_size == last_byte)
    {
        returnValue = jbig_add_odma_to_descriptor_list(inHandle, inList, buffer_ptr, buffer_size, context);
    }
    else
    {
        // else leave the context NULL on the interim strip so that the user is not notified of it's completion
        // which would cause confusion since we are artificially adding strips that the user application does not
        // know about.
        returnValue = jbig_add_odma_to_descriptor_list(inHandle, inList, buffer_ptr, buffer_size, NULL);
    }



    return returnValue;

}

static error_type_t pad_odma_buffer(   jbig_handle_t * inHandle, 
                    tJBIGDescriptorList* inList, 
                    BigBuffer_t* buf, uint32_t buffer_size, 
                    void * context, 
                    uint32_t dma_direction  )
{
    int32_t remainder = buffer_size;
    int32_t bytes_added = 0;
    uint32_t buffer_ptr = 0;
    uint32_t last_byte = 0;
    error_type_t returnValue = FAIL;

    
    if (inHandle->bytes_until_pad < 0)
    {
        // initialize the bytes_until_pad counter to the size of a jbig strip
        inHandle->bytes_until_pad = jbig_calc_strip_buffer_size( inHandle );
    }

    buffer_ptr = LO32BIT( (uintptr_t)dma_buffer_map_single(buf, dma_direction) );

    last_byte = buffer_ptr + buffer_size;


    do
    {
        if (inHandle->bytes_until_pad <= remainder)
        {
            returnValue = jbig_add_odma_with_context( inHandle, inList, buffer_ptr, inHandle->bytes_until_pad, context, last_byte );
            buffer_ptr += inHandle->bytes_until_pad;

            returnValue = jbig_add_pad_write_strip( inHandle, inList );
            bytes_added += inHandle->bytes_until_pad;
            remainder -= inHandle->bytes_until_pad;
            inHandle->bytes_until_pad = jbig_calc_strip_buffer_size( inHandle );
        }
        else
        {

            returnValue = jbig_add_odma_with_context( inHandle, inList, buffer_ptr, remainder, context, last_byte  );
            buffer_ptr += remainder;
            inHandle->bytes_until_pad -= remainder;
            bytes_added += remainder;
        }

    } while( bytes_added < buffer_size );

    return returnValue;
}

static error_type_t jbig_add_DMA_buffer_ODMA( jbig_handle_t * inHandle, 
                                      tJBIGDescriptorList* inList, 
                                      BigBuffer_t* buf, uint32_t size, 
                                      void * context, 
                                      uint32_t dma_direction )
{
    error_type_t returnValue = FAIL;
    uint32_t buffer_size = size;
    uint32_t buffer_ptr = 0;

    // If the specified size is 0, simply return fail.
    if (size == 0)
    {
        return returnValue;
    }
    
    if( inHandle && inHandle->inUse)
    {

        BigBuffer_convert_to_dma_buffer(buf);

        if (( inHandle->config.Operation == JBIG_DECODE ) && ( inHandle->config.AddExtraLines ))
        {
            return ( pad_odma_buffer( inHandle, inList, buf, buffer_size, context, dma_direction ) );
        }


        buffer_ptr = LO32BIT( (uintptr_t)dma_buffer_map_single(buf, dma_direction));

        returnValue = jbig_add_odma_to_descriptor_list(inHandle, inList, buffer_ptr, buffer_size, context);


    }
    return ( returnValue );
}


/**
 * jbigStartWriteDMAIfPossible
 *
 * \brief 
 * 
 * \param 
 * 
 * \return 
 *
 *
 **/

error_type_t jbigStartWriteDMAIfPossible( jbig_handle_t * inHandle )
{
    
    error_type_t returnValue = FAIL;
    uint8_t dmaBusy = 0;
    JBIG_CODEC_ODMA_REGS_t * p_JBIG_ODMA_regs = NULL;
    jbig_descriptor_t * p_JBIG_Descriptor = NULL;
    
    DBG_SLOG_ENTRY
    
    if( inHandle && inHandle->inUse)
    { 
        p_JBIG_ODMA_regs = inHandle->jbig_block_interface->jbig_odma_regs;
        
        dmaBusy = p_JBIG_ODMA_regs->USR & JBIG_CODEC_0_ODMA_JBIG_ODMA_UDMA_USR_BUSY_MASK ;

        pthread_mutex_lock(&inHandle->lock);
        if( inHandle->pendingWriteDescriptorList.ListHead && !inHandle->currentWriteDescriptorList.ListHead )
        {
            if( isJbigStarted( inHandle ) )
            {
                // This is the first descriptor in the write DMA chain. Perform any 
                // initialization for this transaction.
                DBG_PRINTF_INFO("JBIG: Add Write buffer Start DMA now  %d!!!!!!!\n",isJbigStarted( inHandle ));

                inHandle->odma_complete = false;
                        
                // Set up the ODMA output data width in words (beats)
                // p_JBIG_ODMA_regs->UCR = JBIG_CODEC_0_ODMA_JBIG_ODMA_UDMA_UCR_BEATS_REPLACE_VAL( p_JBIG_ODMA_regs->UCR, 1);
    
                // transfer the pending descriptor list to the current descriptor list
                jbigTransferDescriptorList(&inHandle->currentWriteDescriptorList, &inHandle->pendingWriteDescriptorList);
        
                p_JBIG_Descriptor = jbigGetDescriptorListHead( inHandle->currentWriteDescriptorList );
        
                dmaBusy = p_JBIG_ODMA_regs->USR & JBIG_CODEC_0_ODMA_JBIG_ODMA_UDMA_USR_BUSY_MASK; 
                
                DBG_MEMLOG(LOG_DEBUG_H,"JBIG: Write desc_write %#x status %#x  \n", p_JBIG_Descriptor, p_JBIG_ODMA_regs->status );
                
                p_JBIG_ODMA_regs->UCR = JBIG_CODEC_0_ODMA_JBIG_ODMA_UDMA_UCR_ENABLE_REPLACE_VAL( p_JBIG_ODMA_regs->UCR, 1 );
                
                p_JBIG_ODMA_regs->UDR = LO32BIT((uintptr_t)p_JBIG_Descriptor->hw_addr);
        
                returnValue = OK;
            }   
            else
            {
                inHandle->started = 2;
                DBG_PRINTF_INFO("JBIG: Can't Start Write DMA now %#x %#x \n",isJbigStarted( inHandle ), dmaBusy );
            }
        }
        pthread_mutex_unlock(&inHandle->lock);
    }

    DBG_SLOG_EXIT
        
    return returnValue;
}

/**
 * jbigStartReadDMAIfPossble
 *
 * \brief 
 * 
 * \param 
 * 
 * \return 
 *
 *
 **/

error_type_t jbigStartReadDMAIfPossible( jbig_handle_t * inHandle )
{
    error_type_t returnValue = FAIL;
    uint8_t dmaBusy = 0;
    JBIG_CODEC_IDMA_REGS_t * p_JBIG_IDMA_regs = NULL;
    jbig_descriptor_t * p_JBIG_Descriptor = NULL;

    if (inHandle && inHandle->inUse )
    {
        p_JBIG_IDMA_regs = inHandle->jbig_block_interface->jbig_idma_regs;

        dmaBusy = p_JBIG_IDMA_regs->status & JBIG_CODEC_IDMA_STATUS_DMA_BUSY_MASK;

        pthread_mutex_lock(&inHandle->lock);
        if ( inHandle->pendingReadDescriptorList.ListHead && !inHandle->currentReadDescriptorList.ListHead )
        {
            if ( isJbigStarted( inHandle ) )
            {
                DBG_PRINTF_INFO("JBIG: Add Read buffer Start DMA now  %d!!!!!!!\n",isJbigStarted( inHandle ));

                // enable the DMA
                p_JBIG_IDMA_regs->cfg = JBIG_CODEC_IDMA_CFG_ENABLE_REPLACE_VAL( p_JBIG_IDMA_regs->cfg, 1 );

                p_JBIG_IDMA_regs->cfg = JBIG_CODEC_IDMA_CFG_REPLICATE_REPLACE_VAL(p_JBIG_IDMA_regs->cfg, 0 );

                // set up the output Data Width to 32 bits
                p_JBIG_IDMA_regs->cfg = JBIG_CODEC_IDMA_CFG_OUT_WIDTH_REPLACE_VAL( p_JBIG_IDMA_regs->cfg, 5 );

                // set the DMA burst length 4 words
                p_JBIG_IDMA_regs->cfg = JBIG_CODEC_IDMA_CFG_BURST_LEN_REPLACE_VAL( p_JBIG_IDMA_regs->cfg, 0 );

                // The JBIG line width must be set for proper operation when the
                // output of the decoder is connected directly to other hardware
                // via the CBI interface. Linewidth is used by the pipeline for
                // determining end of line and end of image for blocks that need
                // to know. Note that since JBIG assumes all pixels are 1 bit,
                // but the pipeline can unpack the data in 1 2 or 4 bit chunks,
                // this line width value might be different than the JBIG
                // register value JX. 
                
                jbigTransferDescriptorList(&inHandle->currentReadDescriptorList, &inHandle->pendingReadDescriptorList);

                p_JBIG_Descriptor = jbigGetDescriptorListHead( inHandle->currentReadDescriptorList );
                DBG_PRINTF_INFO("%s desc: %p data 0x%x, len: %d\n", __func__, p_JBIG_Descriptor, p_JBIG_Descriptor->idma.dataBuffer, p_JBIG_Descriptor->idma.length);

                dmaBusy = p_JBIG_IDMA_regs->status & JBIG_CODEC_IDMA_STATUS_DMA_BUSY_MASK;

                DBG_MEMLOG(LOG_DEBUG_H,"JBIG: Read desc_write %#x status %#x  \n", p_JBIG_Descriptor, p_JBIG_IDMA_regs->status );

                p_JBIG_IDMA_regs->desc_write = LO32BIT((uintptr_t)p_JBIG_Descriptor->hw_addr);

                returnValue = OK;
            }
            else
            {
                inHandle->started = 2;
                DBG_PRINTF_INFO("JBIG: Can't Start READ DMA now %#x %#x \n",isJbigStarted( inHandle ), dmaBusy );
            }
        }
        else
        {
            DBG_PRINTF_INFO("JBIG: Can't Start READ DMA now no pending list pend %p current %p \n", inHandle->pendingReadDescriptorList.ListHead, inHandle->currentReadDescriptorList.ListHead );
        }
        pthread_mutex_unlock(&inHandle->lock);
    }
    else
    {
        DBG_PRINTF_INFO("JBIG: Can't Start READ DMA handle not valid\n" );
    }

    return returnValue;
}


/**
 * disableJBIGInterrupts
 *
 * \brief 
 * 
 * \param 
 * 
 * \return 
 *
 *
 **/

static void disableJBIGInterrupts( jbig_handle_t * inHandle )
{
    if( inHandle && inHandle->inUse)
    {
        uio_int_disable( inHandle->jbig_block_interface->dev_jbig );
        uio_int_disable( inHandle->jbig_block_interface->dev_odma );
        uio_int_disable( inHandle->jbig_block_interface->dev_idma );
    }
}

/**
 * enableJBIGInterrupts
 *
 * \brief 
 * 
 * \param 
 * 
 * \return 
 *
 *
 **/

static void enableJBIGInterrupts( jbig_handle_t * inHandle)
{
    if( inHandle && inHandle->inUse)
    {
        uio_int_enable( inHandle->jbig_block_interface->dev_jbig );
        uio_int_enable( inHandle->jbig_block_interface->dev_odma );
        uio_int_enable( inHandle->jbig_block_interface->dev_idma );
    }
}


/**
 * \brief  Return the number of elements in an os queue
 *
 */

static uint32_t jbig_get_queue_pending( mqd_t *queue_ptr ) 
{
    struct mq_attr attr;

    mq_getattr(*queue_ptr, &attr);
    return attr.mq_curmsgs;
}


void jbig_update_lines_Remaining( jbig_handle_t * inHandle, uint32_t bytesProcessed)
{
    jbig_config_t tempConfig;
    jbig_get_config( inHandle, &tempConfig);
    uint32_t linesProcessed = bytesProcessed / tempConfig.LineWidth;
    inHandle->linesRemaining -= linesProcessed;
}

static void jbig_delete_pad_strip( BigBuffer_t* src_buf )
{
    DBG_PRINTF_INFO("%s\n",__func__ );

    if (src_buf)
    {
        BigBuffer_Free(src_buf);
    }
}

error_type_t  jbig_add_pad_read_strip( jbig_handle_t *handle )
{
    mlimiter_t* limiter = mlimiter_by_name("print");
    uint32_t bytes_per_line = handle->config.Xd/8;
    uint32_t pad_buffer_size = bytes_per_line;
    BigBuffer_t* pad_buffer = NULL;
    uint8_t * outData;
    error_type_t return_result = FAIL;

    DBG_PRINTF_INFO("%s %p bytes per line %d\n",__func__, handle, bytes_per_line );

    // In order to work around the counter overflow bug in the JBIG block
    // we are adding a line of unique pixels to the bottom of the strip
    pad_buffer = dma_buffer_malloc(limiter, pad_buffer_size );

    if (pad_buffer)
    {
        outData = (uint8_t *)dma_buffer_mmap_forcpu(pad_buffer);


        memset(outData, 0, pad_buffer_size);

        if (handle->last_input_byte == 0x01 )
        {

                outData[pad_buffer_size-1] = 0x2;
        }
        else
        {
            outData[pad_buffer_size-1] = 0x1;
        }

        dma_buffer_unmmap_forcpu(pad_buffer);

        return_result = jbigAddReadDMABuffer( handle, pad_buffer, pad_buffer_size, jbig_delete_pad_strip );

        ASSERT(return_result == OK);
    }

    return return_result;
}


error_type_t  jbig_add_pad_write_strip( jbig_handle_t *handle, tJBIGDescriptorList* inList  )
{
    mlimiter_t* limiter = mlimiter_by_name("print");
    uint32_t bytes_per_line = handle->config.Xd/8;
    uint32_t pad_buffer_size = bytes_per_line;
    BigBuffer_t* pad_buffer;
    uint8_t * outData;
    error_type_t return_result = FAIL;
    jbig_dma_context_t* dma_context = NULL;

    //DBG_PRINTF_INFO("%s bytes per line %d\n",__func__, bytes_per_line );

    // In order to work around the counter overflow bug in the JBIG block
    // we are adding a line of unique pixels to the bottom of the strip
    pad_buffer = dma_buffer_malloc(limiter, pad_buffer_size );

    if ( pad_buffer )
    {
        outData = (uint8_t *)dma_buffer_mmap_forcpu(pad_buffer);


        memset(outData, 0, pad_buffer_size);

        DBG_MEMLOG(LOG_DEBUG_H, "%s user data 0x%#x, buf 0x%#x size = %d\n", __func__, userData, buf, size);
        dma_context = (jbig_dma_context_t*)MEM_MALLOC(sizeof(jbig_dma_context_t));
        ASSERT(dma_context);
        if (dma_context)
        {
            dma_context->len = pad_buffer_size;
            dma_context->buf = pad_buffer;
            dma_context->user_callback_data = jbig_delete_pad_strip;

        }


        //BigBuffer_convert_to_dma_buffer( pad_buffer );

        dma_buffer_unmmap_forcpu(pad_buffer);

        uint32_t buffer_ptr = LO32BIT( (uintptr_t)dma_buffer_map_single( pad_buffer, DMA_FROM_DEVICE ));

        return_result = jbig_add_odma_to_descriptor_list(handle, inList, buffer_ptr, pad_buffer_size, dma_context);

    }

    return return_result;
}


static void jbig_add_pad_strip_update_config( jbig_handle_t *handle )
{
    uint32_t num_strips = 0;
    uint32_t adjusted_Yd = 0;
    if (handle)
    {
        num_strips = handle->config.Yd/handle->config.Ld;
        if ( handle->config.Yd % handle->config.Ld )
        {
            num_strips += 1;
        }
        adjusted_Yd = handle->config.Yd;
        adjusted_Yd += num_strips;
        handle->config.Ld += 1; // Add one more line for pad strip
        handle->config.Yd = adjusted_Yd;
    }
}

/**
 * jbigMessageHandler
 *
 * \brief 
 * 
 * \param 
 * 
 * \return 
 *
 *
 **/
uint32_t jbig_buffers_completed = 0;

static void jbigMessageHandler(JBIG_MESSAGE message)
{
    tJBIGDescriptorList jbigCompletionList;
    jbig_descriptor_t *firstJBIGDescriptor = NULL;
    jbig_handle_t *currentHandle = NULL;
    jbig_dma_context_t *dma_context = NULL;

    currentHandle = message.handle;

    if (!currentHandle)
    {
        DBG_MEMLOG(LOG_WARNING, "JBIG: message without handle %#x\n", message.descriptor);
        return;
    }

    switch (message.msgType)
    {
        case JBIG_MSG_IDMA_COMPLETE:
            {
                DBG_PRINTF(LOG_DEBUG_M, "JBIG: JBIG_MSG_IDMA_COMPLETE\n");
                if (!currentHandle->inUse)
                {
                    DBG_MEMLOG(LOG_WARNING, "JBIG: Handle closed before messages processed!!!\n");
                    return;
                }

                jbig_decrement_pending_messages(currentHandle);

                pthread_mutex_lock(&currentHandle->lock);
                jbigTransferDescriptorList(&jbigCompletionList, &currentHandle->currentReadDescriptorList);
                pthread_mutex_unlock(&currentHandle->lock);

                firstJBIGDescriptor = jbigRemoveDescriptorFromListHead(&jbigCompletionList);

                //DBG_MEMLOG(LOG_DEBUG_H,"JBIG: IDMA descriptor %#x %#x\n", firstJBIGDescriptor, message.descriptor );

                while (firstJBIGDescriptor)
                {
                    ASSERT((firstJBIGDescriptor->idma.control & JBIG_DESCRIPTOR_OWNERSHIP) == 0);

                    dma_context = (jbig_dma_context_t *)firstJBIGDescriptor->context;
                    dma_buffer_unmap_single(dma_context->buf, DMA_TO_DEVICE);

                    if (dma_context->user_callback_data == jbig_delete_pad_strip)
                    {
                        jbig_delete_pad_strip(dma_context->buf);
                    }
                    else if (currentHandle->idma_callback)
                    {
                        currentHandle->idma_callback(currentHandle, dma_context->user_callback_data);
                    }

                    MEM_FREE_AND_NULL(dma_context);
                    MEM_FREE_UNCACHED(firstJBIGDescriptor->hw_addr, firstJBIGDescriptor);
                    firstJBIGDescriptor = jbigRemoveDescriptorFromListHead(&jbigCompletionList);
                }

                if (jbigStartReadDMAIfPossible(currentHandle) == OK)
                {
                    DBG_MEMLOG(LOG_DEBUG_M, "JBIG: IDMA start another list\n");
                }
            }
            break;


        case JBIG_MSG_ODMA_EOI:

            // If we get here and there is still input descriptors in the list we need to flush
            // them.
            pthread_mutex_lock(&currentHandle->lock);
            jbigTransferDescriptorList(&jbigCompletionList, &currentHandle->currentReadDescriptorList);
            pthread_mutex_unlock(&currentHandle->lock);

            firstJBIGDescriptor = jbigRemoveDescriptorFromListHead(&jbigCompletionList);

            while (firstJBIGDescriptor)
            {
                DBG_PRINTF_INFO("JBIG: Freeing incomplete IDMA descriptor \n" );
                dma_context = (jbig_dma_context_t *)firstJBIGDescriptor->context;
                if (dma_context)
                {
                    dma_buffer_unmap_single(dma_context->buf, DMA_TO_DEVICE);

                    if (dma_context->user_callback_data == jbig_delete_pad_strip)
                    {
                        jbig_delete_pad_strip(dma_context->buf);
                    }
                    else if (currentHandle->idma_callback)
                    {
                        currentHandle->idma_callback(currentHandle, dma_context->user_callback_data);
                    }

                    MEM_FREE_AND_NULL(dma_context);
                }
                MEM_FREE_UNCACHED(firstJBIGDescriptor->hw_addr, firstJBIGDescriptor);
                firstJBIGDescriptor = jbigRemoveDescriptorFromListHead(&jbigCompletionList);
            }

        case JBIG_MSG_ODMA_COMPLETE:
            {
                DBG_PRINTF(LOG_DEBUG_M, "JBIG: %s\n", (message.msgType == JBIG_MSG_ODMA_EOI) ? "JBIG_MSG_ODMA_EOI" : "JBIG_MSG_ODMA_COMPLETE");
                if (!currentHandle->inUse)
                {
                    DBG_MEMLOG(LOG_WARNING, "JBIG: Handle closed before messages processed!!!\n");
                    return;
                }

                jbig_decrement_pending_messages(currentHandle);

                pthread_mutex_lock(&currentHandle->lock);
                jbigTransferDescriptorList(&jbigCompletionList, &currentHandle->currentWriteDescriptorList);
                //DBG_MEMLOG(LOG_DEBUG_M,"JBIG: ODMA removed %#x\n",firstJBIGDescriptor);
                pthread_mutex_unlock(&currentHandle->lock);

                firstJBIGDescriptor = jbigRemoveDescriptorFromListHead(&jbigCompletionList);

                while (firstJBIGDescriptor != NULL)
                {
                    static uint32_t byte_count = 0;
                    jbig_buffers_completed++;

                    XASSERT((firstJBIGDescriptor->odma.control & JBIG_ODMA_DESCRIPTOR_OWNERSHIP) == 0, firstJBIGDescriptor->odma.control);

                    if ((firstJBIGDescriptor->odma.control & JBIG_ODMA_DESCRIPTOR_OWNERSHIP) == 0)
                    {
                        uint32_t dma_count = firstJBIGDescriptor->odma.length;

                        if ((currentHandle->config.Operation == JBIG_DECODE)&&(currentHandle->config.AddExtraLines))
                        {

                            if ((firstJBIGDescriptor->odma.control & JBIG_ODMA_DESCRIPTOR_COMPLETE) )
                            {
                                
                                int32_t pad_size = currentHandle->config.LineWidth;
                                dma_count = firstJBIGDescriptor->odma.length - pad_size;

                                //printf("last ODMA buffer pad size %d dma_count: %d\n", pad_size, dma_count);
                            }
                        }

                        byte_count += dma_count;
                        currentHandle->bytes_output += dma_count;

                    }

                    dma_context = (jbig_dma_context_t *)firstJBIGDescriptor->context;
                    if (dma_context)
                    {
                        if (dma_context->user_callback_data == jbig_delete_pad_strip)
                        {
                            
                            // we have to correct the byte count for pad strip data that has come in at the end of the last
                            // ODMA buffer.
                            if ((firstJBIGDescriptor->odma.control & JBIG_ODMA_DESCRIPTOR_OWNERSHIP) == 0)
                            {
                                if ((firstJBIGDescriptor->odma.control & JBIG_ODMA_DESCRIPTOR_COMPLETE) == 0 )
                                {
                                    byte_count -= firstJBIGDescriptor->odma.length;
                                    currentHandle->bytes_output -= firstJBIGDescriptor->odma.length;
                                }
                            }
                            

                            jbig_delete_pad_strip(dma_context->buf);

                        }
                        else
                        {
                            dma_buffer_unmap_single(dma_context->buf, DMA_FROM_DEVICE);

                            if (currentHandle->odma_callback)
                            {
                                DBG_PRINTF(LOG_DEBUG_M, "    ODMA callback - data %p, big buffer data %p\n",
                                    dma_context->user_callback_data,
                                    ((BigBuffer_t*)(dma_context->user_callback_data+2))->data );
                            
                                currentHandle->odma_callback(currentHandle,  dma_context->user_callback_data);
                            }
                            if (currentHandle->odma_byte_count_callback)
                            {
                                DBG_PRINTF(LOG_DEBUG_M,"    ODMA callback - data %p, big buffer data %p\n",
                                        dma_context->user_callback_data,
                                        ((BigBuffer_t*)(dma_context->user_callback_data+2))->data );
                                
                                currentHandle->odma_byte_count_callback(currentHandle, byte_count, dma_context->user_callback_data);
                            }
                            byte_count = 0;
                        }
                        MEM_FREE_AND_NULL(dma_context);
                        
                    }

                    MEM_FREE_UNCACHED(firstJBIGDescriptor->hw_addr, firstJBIGDescriptor);

                   firstJBIGDescriptor = jbigRemoveDescriptorFromListHead(&jbigCompletionList);
                }

                currentHandle->odma_complete = true;
                if (jbigStartWriteDMAIfPossible(currentHandle) == OK)
                {
                    DBG_MEMLOG(LOG_DEBUG_H, "JBIG: ODMA  EOI start another list\n");
                }

                if (currentHandle->complete && currentHandle->jbigCallback)
                {
                    currentHandle->jbigCallback(currentHandle, currentHandle->jbig_callback_context, currentHandle->control_status);
                    currentHandle->complete = false;
                }
            }
            break;

        case JBIG_MSG_OPERATION_COMPLETE:
            {
                DBG_PRINTF(LOG_DEBUG_M, "JBIG: JBIG_MSG_OPERATION_COMPLETE\n");
                if (!currentHandle->inUse)
                {
                    DBG_MEMLOG(LOG_WARNING, "JBIG: Handle closed before messages processed!!!\n");
                    return;
                }

                jbig_decrement_pending_messages(currentHandle);

                // we used to always call callback directly here.  Now if odma is not complete we set flag so that 
                // completion isn't notified until after all ODMA callbacks are done.  This gives somewhat nicer behavior
                // in that the caller will not get further DMA notifications after jbig completion is signaled.
                if (currentHandle->odma_complete && currentHandle->jbigCallback)
                {
                    currentHandle->jbigCallback(currentHandle, currentHandle->jbig_callback_context, currentHandle->control_status);
                }
                else
                {
                    currentHandle->complete = true;
                }
            }
            break;

        case JBIG_MSG_FREE_MEMORY:
            {
                jbig_decrement_pending_messages(currentHandle);
                DBG_MEMLOG(LOG_DEBUG_M, "JBIG: JBIG_MSG_FREE_MEMORY\n");
                jbigFreeDescriptors();
            }
            break;

        default:
            {
                DPRINTF(DBG_SOFT | DBG_OUTPUT, ("Jbig: Unknown Message %d received\n", message.msgType));
                break;
            }
    }
}


/**
 * jbigThread
 *
 * \brief 
 * 
 * \param 
 * 
 * \return 
 *
 *
 **/
static void* jbigThread( void* Temp)
{
    JBIG_MESSAGE message;
    int result;
    uint32_t numMessagesRemaining = 0;
    uint32_t jbigMaxMessage = 0;
    
    while(1)
    {
        result = posix_wait_for_message(jbig_msg_queue, (char*)&message, sizeof(JBIG_MESSAGE), POSIX_WAIT_FOREVER);
        XASSERT(result == 0, result);

        numMessagesRemaining = NUM_JBIG_MESSAGES_PER_BLOCK - jbig_get_queue_pending( &jbig_msg_queue );
        if( numMessagesRemaining && numMessagesRemaining < 2)
        {
            DBG_PRINTF_ERR("JBIG: LOW ON MESSAGES %d %d \n",numMessagesRemaining, jbigMaxMessage);
        }
        if( jbigMaxMessage < numMessagesRemaining )
        {
            jbigMaxMessage = numMessagesRemaining;
        }

        jbigMessageHandler( message ); 
    }
    return 0;
}


/**
 * jbigInitDescriptors
 *
 * \brief 
 * 
 * \param 
 * 
 * \return 
 *
 *
 **/

void jbigInitDescriptors( jbig_handle_t *handle )
{
    if( handle )
    {
        handle->pendingReadDescriptorList.ListHead = NULL;
        handle->pendingReadDescriptorList.ListTail = NULL;
        handle->currentReadDescriptorList.ListHead = NULL;
        handle->currentReadDescriptorList.ListTail = NULL;
        
        handle->pendingWriteDescriptorList.ListHead = NULL;
        handle->pendingWriteDescriptorList.ListTail = NULL;
        handle->currentWriteDescriptorList.ListHead = NULL;
        handle->currentWriteDescriptorList.ListTail = NULL;
    }
}


/**
 * 
 *  This function will find the first available jbig handle in the
 *  jbig handle array.
 *
 **/

int16_t JbigGetNextAvailableHandle( jbig_block_t block )
{
    uint16_t i;
    int16_t available_handle = -1;


    for ( i = 0; i < NUM_JBIG_BLOCKS; i++)
    {
        // if dirty flag is set the memory has not been cleaned up yet
        // on that handle so we don't want to use it.
    // also make sure block matches requested type
            if( jbig_handles[i].jbig_block_interface->jbig_block_type == block && !jbig_handles[i].inUse && !jbig_handles[i].dirty )
            {
                    available_handle = i;
                    break;
            }
    }

    if (available_handle < 0)
    {
        // didn't find an available handle; return one of the inuse handles; caller will block until available
        DBG_PRINTF_ERR("JBIG: WARNING: all blocks in use.  Will block for available. Potential DEADLOCK.\n");
        for (i = 0; i < NUM_JBIG_BLOCKS; i++)
        {
            if (jbig_handles[i].jbig_block_interface->jbig_block_type == block)
            {
                available_handle = i;
                break;
            }
        }
    }
    return available_handle; 
}

/**
 * jbigAssignHandle 
 *
 * \brief 
 * 
 * \param 
 * 
 * \return 
 *
 *
 **/

jbig_handle_t * jbigAssignHandle( jbig_block_t block )
{
    int16_t handle_index;
    jbig_handle_t * handle = NULL;
    int err;
    
    handle_index = JbigGetNextAvailableHandle(block);
    if( handle_index >= 0 )
    {
        err = sem_wait(&jbig_handles[handle_index].open_sem);
        ASSERT(err == 0);
        if (err == 0)
        {
            handle = &jbig_handles[handle_index];
            handle->inUse = 1;
            handle->lastError = 0;
            handle->numPendingMessages = 0;
            handle->maxNumMessages = NUM_JBIG_MESSAGES_PER_BLOCK;
            memset((void *)&handle->config, 0x0, sizeof(jbig_config_t));
            handle->config.Block = block;
            handle->bytes_until_pad = -1;
            handle->pendingReadDescriptorList.ListHead = NULL;
            handle->pendingReadDescriptorList.ListTail = NULL;
            handle->currentReadDescriptorList.ListHead = NULL;
            handle->currentReadDescriptorList.ListTail = NULL;
        }
    }
    else
    {
        XASSERT(handle_index >= 0, handle_index );
    }
    
    return handle;
}


/**
 * isrJbigControl
 *
 * \brief 
 * 
 * \param 
 * 
 * \return 
 *
 *
 **/
#ifdef __linux__
static void isrJbigControl( int32_t interrupt_count, void *context )
#else
static void isrJbigControl( uint32_t context )
#endif
{
    JBIG_REGS_t * p_JBIG_control_regs = NULL;
    JBIG_CODEC_ODMA_REGS_t * p_JBIG_ODMA_regs = NULL;
    uint32_t errorStatus;
    uint32_t odma_status;
    uint32_t status = 0;
    jbig_handle_t * currentHandle = (jbig_handle_t*)context;

    ASSERT(currentHandle != NULL);
    
    p_JBIG_control_regs = currentHandle->jbig_block_interface->jbig_regs;
    p_JBIG_ODMA_regs = currentHandle->jbig_block_interface->jbig_odma_regs;

    status = p_JBIG_control_regs->JSTAT;
    odma_status = p_JBIG_ODMA_regs->USR;
    errorStatus = status & JBIG_CODEC_JSTAT_E_MASK;

    status = p_JBIG_control_regs->JSTAT;
    errorStatus = status & JBIG_CODEC_JSTAT_E_MASK;

    DBG_PRINTF_INFO("JBIG: isrJbigControl currentHandle = %p status = %#x\n", currentHandle, status);

    DBG_PRINTF(LOG_DEBUG_M,"JBIG: isrJbigControl currentHandle = %p status = %#x, block_type = %s\n", currentHandle, status,
               (currentHandle->jbig_block_interface->jbig_block_type == JBIG_CODEC_BLOCK) ? "CODEC" : "DECODE");

    /*  This isr gets hit only when the decoder signals that it's done,
     *  successfully or not.
     */

    if(status & JBIG_CODEC_JSTAT_E_MASK)
    {
       DBG_MEMLOG(LOG_WARNING, "  isrJbigControl: decoder stopped with an error: 0x%#x\n", status, 0 );
    }

    if( status & JBIG_CODEC_JSTAT_DD_MASK ) // Decode is done.
    {
        DBG_MEMLOG(LOG_DEBUG_M,"  isrJbigControl: decoder completed successfully.\n" );
        /* we need to clear the done bit */
        REG_CLEAR_BITS(&p_JBIG_control_regs->JSTAT, JBIG_CODEC_JSTAT_DD_MASK );
    }

    if( status & JBIG_CODEC_JSTAT_ED_MASK ) // Encode is done.
    {
        DBG_MEMLOG(LOG_DEBUG_M,"  isrJbigControl: encoder completed successfully.\n" );
        /* we need to clear the done bit */
        REG_CLEAR_BITS(&p_JBIG_control_regs->JSTAT, JBIG_CODEC_JSTAT_ED_MASK );

    }

    if( status & JBIG_CODEC_JSTAT_I_MASK) // if these bits are set then a Write or Read DMA has completed and is pending
    {
        if( (status & JBIG_CODEC_JSTAT_I_MASK) == 0x4000 )
        {
           DBG_MEMLOG(LOG_DEBUG_M,"  isrJbigControl: Write DMA Complete.\n" );
        }
        if( (status & JBIG_CODEC_JSTAT_I_MASK) == 0x8000 )
        {
           DBG_MEMLOG(LOG_DEBUG_M,"  isrJbigControl: Read DMA Complete.\n" );
        }
    }

    if( status & JBIG_CODEC_JSTAT_D_MASK ) // Decode/Encode is done.
    {
        DBG_PRINTF(LOG_DEBUG_M,"  isrJbigControl: JBIG DONE.\n" );

        /* we need to clear the done bit */
        REG_CLEAR_BITS(&p_JBIG_control_regs->JSTAT, JBIG_CODEC_JSTAT_D_MASK );

        // Check on the ODMA to see if it has completed. It is expected that the 
        // ODMA will be done at this point, either the data buffer is full or 
        // the EOI event has occurred. If either the data buffer empty or command
        // buffer empty flags are not set, then the ODMA is NOT complete.
        if ( (JBIG_CODEC_0_ODMA_JBIG_ODMA_UDMA_USR_DATABUFEMPTY_MASK != (odma_status & JBIG_CODEC_0_ODMA_JBIG_ODMA_UDMA_USR_DATABUFEMPTY_MASK)) &&
             (JBIG_DECODE_0_ODMA_JBIG_ODMA_UDMA_USR_CMDBUFEMPTY_MASK != (odma_status & JBIG_DECODE_0_ODMA_JBIG_ODMA_UDMA_USR_CMDBUFEMPTY_MASK)) )
        {
            DBG_PRINTF(LOG_ERR,"  isrJbigControl: JBIG ODMA NOT DONE!\n" );
            
        }
        
        currentHandle->control_status = errorStatus;

        jbigSendMessage( currentHandle, JBIG_MSG_OPERATION_COMPLETE, NULL );
    }
}

/**
 * isrJbigIDMA
 *
 * \brief 
 * 
 * \param 
 * 
 * \return 
 *
 *
 **/
#ifdef __linux__
static void isrJbigIDMA( int32_t interrupt_count, void *context )
#else
static void isrJbigIDMA( uint32_t context )
#endif
{

    jbig_descriptor_t * pDescriptor;
    jbig_handle_t * currentHandle;
    uint32_t result;
    JBIG_CODEC_IDMA_REGS_t * p_JBIG_IDMA_regs = NULL;
    JBIG_REGS_t * p_JBIG_control_regs = NULL;
    JBIG_CODEC_ODMA_REGS_t * p_JBIG_ODMA_regs = NULL;
    uint32_t status = 0;
    uint32_t odma_status = 0;

    uint32_t pendingInterrupts;

    currentHandle = (jbig_handle_t*)context;
    ASSERT(currentHandle != NULL);
    
    p_JBIG_IDMA_regs = currentHandle->jbig_block_interface->jbig_idma_regs;
    p_JBIG_control_regs = currentHandle->jbig_block_interface->jbig_regs;
    p_JBIG_ODMA_regs = currentHandle->jbig_block_interface->jbig_odma_regs;

    status = p_JBIG_control_regs->JSTAT;
    odma_status = p_JBIG_ODMA_regs->USR;

    DBG_PRINTF(LOG_DEBUG_M,  "%s idma status 0x%x, control status 0x%x, odma status 0x%x, pending int 0x%x\n", __func__, p_JBIG_IDMA_regs->status, status, odma_status, p_JBIG_IDMA_regs->int_st);

    result = p_JBIG_IDMA_regs->status;

    pendingInterrupts = p_JBIG_IDMA_regs->int_st;

    DBG_PRINTF_INFO("JBIG: isrJbigIDMA currentHandle = %p pending %#x control stat = %#x idma stat = %#x odma status = %#x\n", currentHandle, pendingInterrupts, status, result, odma_status );

    // Error Ownership bit was not set on descriptor
    if( pendingInterrupts & JBIG_CODEC_IDMA_INT_ST_WHO_INT_ST_MASK)
    {
        DBG_MEMLOG(LOG_WARNING,"isrJbigIDMA: Ownership Error!\n" );
    }

    // Finished a descriptor
    if( pendingInterrupts & JBIG_CODEC_IDMA_INT_ST_FIN_INT_ST_MASK)
    {
        
        pDescriptor = NULL;

        // This register points to the current descriptor and not necessarily the one that caused the interrupt. 
        // So relying on this to be the descriptor to take action on is in some cases not accurate.
        // We will set the descriptor pointer to null and let the message handle sort out which descriptors are done
        // based on their status bits.
        
        //pDescriptor = (jbig_descriptor_t *)U32_TO_PTR(p_JBIG_IDMA_regs->desc_read);
        //pDescriptor = jbig_get_desc_vaddr_from_list(&currentHandle->currentReadDescriptorList, pDescriptor);

        DBG_PRINTF_INFO("    desc_read = %p, xfer_addr = %#x, xfer_len = %d, status = %#x\n", pDescriptor, (uint32_t)p_JBIG_IDMA_regs->xfer_addr, (uint32_t)p_JBIG_IDMA_regs->xfer_length, (uint32_t)p_JBIG_IDMA_regs->status);
        if(currentHandle)
        {
            DBG_PRINTF_INFO("    read chain list head = %p\n", currentHandle->currentReadDescriptorList.ListHead);
        }
        DBG_MEMLOG(LOG_DEBUG_M,"            desc_read = %#x\n", pDescriptor);
        DBG_MEMLOG(LOG_DEBUG_M,"            xfr_addr = %#x\n",p_JBIG_IDMA_regs->xfer_addr);
        DBG_MEMLOG(LOG_DEBUG_M,"            xfr_length = %#x\n",p_JBIG_IDMA_regs->xfer_length);
        DBG_MEMLOG(LOG_DEBUG_M,"            status = %#x\n",p_JBIG_IDMA_regs->status);

        jbigSendMessage( currentHandle, JBIG_MSG_IDMA_COMPLETE, pDescriptor );
    }

    // End of Image
    if( pendingInterrupts & JBIG_CODEC_IDMA_INT_ST_EOI_INT_ST_MASK)
    {
        DBG_MEMLOG(LOG_DEBUG_M,"             End Of Image\n" );
    }

    // Not sure what RST is at this time
    if( pendingInterrupts & JBIG_CODEC_IDMA_INT_ST_RST_INT_ST_MASK)
    {

    }

    REG_SET_BITS(&p_JBIG_IDMA_regs->int_cl, pendingInterrupts);
}

/**
 * isrJbigODMA
 *
 * \brief 
 * 
 * \param 
 * 
 * \return 
 *
 *
 **/
static uint8_t waitForODMAResetBit = 0;
#ifdef __linux__
static void isrJbigODMA( int32_t interrupt_count, void *context )
#else
static void isrJbigODMA( uint32_t context )
#endif
{
    uint32_t pendingInterrupts = 0;
    uint32_t pending_odma_core_ints = 0;

    JBIG_CODEC_ODMA_REGS_t * p_JBIG_ODMA_regs = NULL;
    JBIG_CODEC_ODMA_CORE_REGS_t *p_JBIG_ODMA_core_regs = NULL;
    jbig_descriptor_t * pDescriptor = NULL;
    jbig_handle_t * currentHandle;

    currentHandle = (jbig_handle_t*)context;
    ASSERT(currentHandle != NULL);

    p_JBIG_ODMA_regs = currentHandle->jbig_block_interface->jbig_odma_regs;
    p_JBIG_ODMA_core_regs = currentHandle->jbig_block_interface->jbig_odma_core_regs;

    pendingInterrupts = p_JBIG_ODMA_regs->UIPR;
    pending_odma_core_ints = p_JBIG_ODMA_core_regs->OIPR;

    // This register points to the current descriptor and not necessarily the one that caused the interrupt. 
    // So relying on this to be the descriptor to take action on is in some cases not accurate.
    // We will set the descriptor pointer to null and let the message handle sort out which descriptors are done
    // based on their status bits.

    pDescriptor = NULL;

    //pDescriptor = (jbig_descriptor_t *)U32_TO_PTR(p_JBIG_ODMA_regs->UDR);
    //pDescriptor = jbig_get_desc_vaddr_from_list(&currentHandle->currentWriteDescriptorList, pDescriptor);
    //ASSERT(pDescriptor);

    DBG_PRINTF(LOG_DEBUG_M,"JBIG: isrJbigODMA: Pending Int %#x status = %#x currentHandle = %p\n", pendingInterrupts, p_JBIG_ODMA_regs->USR, currentHandle);

    DBG_PRINTF(LOG_DEBUG_M,"JBIG: isrJbigODMA: Pending Int %#x status = %#x\n", pendingInterrupts, p_JBIG_ODMA_regs->USR );
    DBG_MEMLOG(LOG_DEBUG_M,"                 currentHandle = %p\n", currentHandle );

    // the following values are for current descriptor not necessarily the one that
    // caused the interrupt to fire.
    DBG_MEMLOG(LOG_DEBUG_M,"                 xfr_addr = %#x\n",p_JBIG_ODMA_regs->xfer_addr );
    DBG_MEMLOG(LOG_DEBUG_M,"                 xfr_length = %#x\n",p_JBIG_ODMA_regs->xfer_length );
    DBG_MEMLOG(LOG_DEBUG_M,"                 status = %#x\n",p_JBIG_ODMA_regs->status );
    //DBG_PRINTF(LOG_DEBUG_M,"                 desc_read = %#x\n", LO32BIT((uintptr_t)pDescriptor));

    if( pDescriptor)
    {
        DBG_PRINTF(LOG_DEBUG_M,"                 desc_next = %p\n", pDescriptor->next_desc_virt_addr);
        if(pDescriptor->next_desc_virt_addr)
        {
            DBG_PRINTF(LOG_DEBUG_M,"                 desc_next_control = %#x\n", pDescriptor->next_desc_virt_addr->odma.control);
        }
    }

    // Filter through the UDMA interrupts
        
    // Error Ownership bit was not set on descriptor
    if ( JBIG_CODEC_0_ODMA_JBIG_ODMA_UDMA_UIER_OWN_MASK != (pendingInterrupts & JBIG_CODEC_0_ODMA_JBIG_ODMA_UDMA_UIER_OWN_MASK) )
    {
        DBG_MEMLOG(LOG_DEBUG_H,"                 xfr_addr = %#x\n",p_JBIG_ODMA_regs->xfer_addr );
        DBG_MEMLOG(LOG_DEBUG_H,"                 xfr_length = %#x\n",p_JBIG_ODMA_regs->xfer_length );
        DBG_MEMLOG(LOG_DEBUG_H,"                 status = %#x\n",p_JBIG_ODMA_regs->status );
        //DBG_MEMLOG(LOG_DEBUG_H,"                 desc_read = %#x\n", pDescriptor);
        //DBG_MEMLOG(LOG_DEBUG_H,"                 desc_ctrl = %#x\n", pDescriptor->control);
    }

    if ( JBIG_CODEC_0_ODMA_JBIG_ODMA_UDMA_UIER_CLEARCOMPLETE_MASK & pendingInterrupts )
    {
        DBG_PRINTF( LOG_ERR, "%s UNEXPECTED INTERRUPT 0x%x\n", __func__, pendingInterrupts );
    }

    if ( JBIG_DECODE_0_ODMA_JBIG_ODMA_UDMA_UIER_BRESP_MASK & pendingInterrupts )
    {
          DBG_PRINTF( LOG_ERR, "%s UNEXPECTED INTERRUPT 0x%x\n", __func__, pendingInterrupts );
    }

    if ( JBIG_DECODE_0_ODMA_JBIG_ODMA_UDMA_UIER_RRESP_MASK & pendingInterrupts )
    {
          DBG_PRINTF( LOG_ERR, "%s UNEXPECTED INTERRUPT 0x%x\n", __func__, pendingInterrupts );
    }

    if ( JBIG_DECODE_0_ODMA_JBIG_ODMA_UDMA_UIER_OUTOFRANGE_MASK & pendingInterrupts )
    {
          DBG_PRINTF( LOG_ERR, "%s UNEXPECTED INTERRUPT 0x%x\n", __func__, pendingInterrupts );
    }

    // Finished a descriptor
    // End of Image

    // Two conditions for processing the completion of a descriptor:
    //      If the EOI interrupt occurs. This occurs when a JBIG encode is done. 
    //           The write buffer will most likely not be full.
    //      If the DESC interrupt occurs (requires interrupt on completion bit be set 
    //           in the descriptor control word).
    if ( (pending_odma_core_ints & JBIG_CODEC_0_ODMA_JBIG_ODMA_CORE_OIER_EOI_MASK) ||
         (pendingInterrupts & JBIG_DECODE_0_ODMA_JBIG_ODMA_UDMA_UIER_DESC_MASK) )
    {
        DBG_PRINTF_DEBUG_M("%s EOI or DESCRIPTOR complete occurred!\n", __func__);
        //ASSERT(pDescriptor != NULL);

        jbigSendMessage( currentHandle, JBIG_MSG_ODMA_EOI, pDescriptor );
    }
    
    // Acknowledge the interrupts
    REG_SET_BITS(&p_JBIG_ODMA_regs->UICR, pendingInterrupts);
    REG_SET_BITS(&p_JBIG_ODMA_core_regs->OICR, pending_odma_core_ints);

}


/**
 * jbigResetHardware
 *
 * \brief 
 * 
 * \param 
 * 
 * \return 
 *
 *
 **/

void jbigResetHardware( jbig_handle_t * handle )
{
    JBIG_REGS_t * p_JBIG_regs = NULL;
    
    ASSERT(handle != NULL);
    p_JBIG_regs = handle->jbig_block_interface->jbig_regs;

    DBG_MEMLOG(LOG_DEBUG_M,"%s core status = %#x\n",__func__, p_JBIG_regs->JSTAT);

    jbigResetWriteDMA( handle );
    jbigResetReadDMA( handle );

    p_JBIG_regs->JBIG_EN = JBIG_CODEC_JBIG_EN_E_REPLACE_VAL(p_JBIG_regs->JBIG_EN, 0); // disable block

    p_JBIG_regs->JBIG_EN = JBIG_CODEC_JBIG_EN_S_REPLACE_VAL(p_JBIG_regs->JBIG_EN, 1); // reset block
    p_JBIG_regs->JBIG_EN = JBIG_CODEC_JBIG_EN_S_REPLACE_VAL(p_JBIG_regs->JBIG_EN, 0);

    p_JBIG_regs->JBIG_EN = JBIG_CODEC_JBIG_EN_E_REPLACE_VAL(p_JBIG_regs->JBIG_EN, 0); // disable block
}

/**
 * jbigFreeMemory
 *
 * \brief 
 * 
 * \param 
 * 
 * \return 
 *
 *
 **/

void jbigFreeMemory( jbig_handle_t * handle )
{
    jbigSendMessage( handle, JBIG_MSG_FREE_MEMORY, NULL );
}

/**
 * jbigFreeDescriptors
 *
 * \brief 
 * 
 * \param 
 * 
 * \return 
 *
 *
 **/

static void jbigFreeDescriptors( void )
{
    int16_t handleIndex = 0;
    jbig_handle_t * handle;
    
    DBG_SLOG_ENTRY

    for( handleIndex = 0; handleIndex < NUM_JBIG_BLOCKS; handleIndex++ )
    {
        handle = &jbig_handles[handleIndex];

        jbigFreeHandleDescriptors(handle);
    }
    DBG_SLOG_EXIT
}

/**
 * jbigFreeHandleDescriptors
 *
 * \brief 
 * 
 * \param 
 * 
 * \return 
 *
 *
 **/

void jbigFreeHandleDescriptors( jbig_handle_t * handle )
{
    jbig_descriptor_t * tmpDescriptor = NULL;
    
    DBG_MEMLOG(LOG_DEBUG_H,"JBIG: Free Descriptors for handle %#x state = %d\n",handle, handle->started);

    if( !handle->started && handle->dirty )
    {        
        while( jbigGetDescriptorListHead( handle->pendingReadDescriptorList ) )
        {
            
            DBG_MEMLOG(LOG_DEBUG_H,"JBIG: Free Pending Read Descriptor %#x\n",jbigGetDescriptorListHead( handle->pendingReadDescriptorList ));
            tmpDescriptor = jbigRemoveDescriptorFromListHead( &handle->pendingReadDescriptorList );

			jbig_dma_context_t* dma_context = (jbig_dma_context_t *)tmpDescriptor->context;
			if (dma_context) {
				dma_buffer_unmap_single(dma_context->buf, DMA_TO_DEVICE);
				MEM_FREE_AND_NULL(dma_context);
			}
            MEM_FREE_UNCACHED( tmpDescriptor->hw_addr, tmpDescriptor );
        }
        
        while(jbigGetDescriptorListHead( handle->currentReadDescriptorList ))
        {
            
            DBG_MEMLOG(LOG_DEBUG_H,"JBIG: Free Current Read Descriptor %#x\n",jbigGetDescriptorListHead( handle->currentReadDescriptorList ));
            tmpDescriptor = jbigRemoveDescriptorFromListHead( &handle->currentReadDescriptorList );
			jbig_dma_context_t* dma_context = (jbig_dma_context_t *)tmpDescriptor->context;
			if (dma_context) {
				dma_buffer_unmap_single(dma_context->buf, DMA_TO_DEVICE);
				MEM_FREE_AND_NULL(dma_context);
			}
            MEM_FREE_UNCACHED( tmpDescriptor->hw_addr, tmpDescriptor );
        }            
        
        while( jbigGetDescriptorListHead( handle->pendingWriteDescriptorList ) )
        {
            DBG_MEMLOG(LOG_DEBUG_H,"JBIG: Free Pending Write Descriptor %#x\n",jbigGetDescriptorListHead( handle->pendingWriteDescriptorList ));
            tmpDescriptor = jbigRemoveDescriptorFromListHead( &handle->pendingWriteDescriptorList );
			jbig_dma_context_t* dma_context = (jbig_dma_context_t *)tmpDescriptor->context;
			if (dma_context) {
				dma_buffer_unmap_single(dma_context->buf, DMA_TO_DEVICE);
				MEM_FREE_AND_NULL(dma_context);
			}
            MEM_FREE_UNCACHED( tmpDescriptor->hw_addr, tmpDescriptor );
        }
        
        while( jbigGetDescriptorListHead( handle->currentWriteDescriptorList ))
        {
            DBG_MEMLOG(LOG_DEBUG_H,"JBIG: Free Current Write Descriptor %#x\n",jbigGetDescriptorListHead( handle->currentWriteDescriptorList ));
            tmpDescriptor = jbigRemoveDescriptorFromListHead( &handle->currentWriteDescriptorList );
			jbig_dma_context_t* dma_context = (jbig_dma_context_t *)tmpDescriptor->context;
			if (dma_context) {
				dma_buffer_unmap_single(dma_context->buf, DMA_TO_DEVICE);
				MEM_FREE_AND_NULL(dma_context);
			}
            MEM_FREE_UNCACHED( tmpDescriptor->hw_addr, tmpDescriptor );
        } 
                    
        handle->dirty = 0;
    }

}


/**
 * jbigStart
 *
 * \brief 
 * 
 * \param 
 * 
 * \return 
 *
 *
 **/

void jbigStart( jbig_handle_t * inHandle )
{    
    JBIG_REGS_t * p_JBIG_regs = NULL;
    
    if( inHandle )
    {
        inHandle->linesRemaining = inHandle->config.Yd;
        enableJBIGInterrupts( inHandle );
            
        p_JBIG_regs = (JBIG_REGS_t *) inHandle->jbig_block_interface->jbig_regs;
    
        if( inHandle->config.Operation == JBIG_DECODE )
        {
            DBG_MEMLOG(LOG_DEBUG_M,"JBIG: Start DECODE\n");
            p_JBIG_regs->JCTL = JBIG_CODEC_JCTL_E_REPLACE_VAL(p_JBIG_regs->JCTL, 0);
            p_JBIG_regs->JCTL = JBIG_CODEC_JCTL_D_REPLACE_VAL(p_JBIG_regs->JCTL, 1);
        }
        else
        {
            DBG_MEMLOG(LOG_DEBUG_M,"JBIG: Start ENCODE\n");
            p_JBIG_regs->JCTL = JBIG_CODEC_JCTL_E_REPLACE_VAL(p_JBIG_regs->JCTL, 1);
            p_JBIG_regs->JCTL = JBIG_CODEC_JCTL_D_REPLACE_VAL(p_JBIG_regs->JCTL, 0);
        }
        
        p_JBIG_regs->JBIG_EN = JBIG_CODEC_JBIG_EN_E_REPLACE_VAL( p_JBIG_regs->JBIG_EN, 1 ); // enable the jbig block        
    }
}

/**
 * jbigStop
 *
 * \brief 
 * 
 * \param 
 * 
 * \return 
 *
 *
 **/
 
void jbigStop( jbig_handle_t * inHandle )
{
    JBIG_REGS_t * p_JBIG_regs = NULL;
    JBIG_CODEC_ODMA_REGS_t * p_JBIG_ODMA_regs = NULL;
    JBIG_CODEC_IDMA_REGS_t * p_JBIG_IDMA_regs = NULL;
    
    DBG_SLOG_ENTRY

    if( inHandle )
    {
        inHandle->jbigCallback = NULL;

        disableJBIGInterrupts( inHandle );

        p_JBIG_regs = (JBIG_REGS_t *) inHandle->jbig_block_interface->jbig_regs;
        p_JBIG_IDMA_regs = inHandle->jbig_block_interface->jbig_idma_regs;
        p_JBIG_ODMA_regs = inHandle->jbig_block_interface->jbig_odma_regs;
        
        DBG_MEMLOG(LOG_DEBUG_M,"%s core status = %#x\n",__func__, p_JBIG_regs->JSTAT);
        p_JBIG_IDMA_regs->cfg = JBIG_CODEC_IDMA_CFG_ENABLE_REPLACE_VAL( p_JBIG_IDMA_regs->cfg, 0 );
        p_JBIG_ODMA_regs->UCR = JBIG_CODEC_0_ODMA_JBIG_ODMA_UDMA_UCR_ENABLE_REPLACE_VAL( p_JBIG_ODMA_regs->UCR, 0 ); // Put ODMA into soft reset
        
        p_JBIG_regs->JBIG_EN = JBIG_CODEC_JBIG_EN_E_REPLACE_VAL( p_JBIG_regs->JBIG_EN, 0 ); // disable the jbig block
                                                                                     
        p_JBIG_regs->JBIG_EN = JBIG_CODEC_JBIG_EN_S_REPLACE_VAL( p_JBIG_regs->JBIG_EN, 1 ); // reset the block
        p_JBIG_regs->JBIG_EN = JBIG_CODEC_JBIG_EN_S_REPLACE_VAL( p_JBIG_regs->JBIG_EN, 0 );

        p_JBIG_regs->JSTAT = 0;

        p_JBIG_regs->JBIG_EN = JBIG_CODEC_JBIG_EN_E_REPLACE_VAL( p_JBIG_regs->JBIG_EN, 0 ); // disable the jbig block
    }

    DBG_SLOG_EXIT
}



/**
 * jbigResetWriteDMA
 *
 * \brief 
 * 
 * \param 
 * 
 * \return 
 *
 *
 **/

void jbigResetWriteDMA( jbig_handle_t * handle )
{
    JBIG_CODEC_ODMA_REGS_t* p_JBIG_ODMA_regs;
    JBIG_CODEC_ODMA_CORE_REGS_t* p_JBIG_ODMA_core_regs;
    uint32_t loopCount = 0;
    DBG_MEMLOG(LOG_DEBUG,"jbig: jbigResetWriteDMA\n");

    DBG_SLOG_ENTRY
    ASSERT(handle != NULL);

    waitForODMAResetBit = 0;

    p_JBIG_ODMA_regs= handle->jbig_block_interface->jbig_odma_regs;
    p_JBIG_ODMA_core_regs = handle->jbig_block_interface->jbig_odma_core_regs;

    DBG_MEMLOG(LOG_DEBUG_M,"jbigODMA: status = %#x\n", p_JBIG_ODMA_regs->status);

    // Wait for data to clear before resetting

    // It is not uncommon for the dma busy flag to be set on write dma if the 
    // buffer length given is larger than decompresse/compressed data
    // provided by the IDMA.
    if ( 0 != (p_JBIG_ODMA_regs->USR & JBIG_CODEC_0_ODMA_JBIG_ODMA_UDMA_USR_BUSY_MASK))
    {
        DBG_PRINTF(LOG_DEBUG_M,  "%s JBIG ODMA BUSY!!\n", __func__);
    }

    if (0 != (p_JBIG_ODMA_regs->USR & JBIG_CODEC_0_ODMA_JBIG_ODMA_UDMA_USR_DMAPAUSED_MASK)) 
    {
        DBG_PRINTF(LOG_DEBUG_M,  "%s JBIG ODMA PAUSED!!\n", __func__);
    }

    if ( (0 == (p_JBIG_ODMA_regs->USR & JBIG_CODEC_0_ODMA_JBIG_ODMA_UDMA_USR_DATABUFEMPTY_MASK)) ||
         (0 == (p_JBIG_ODMA_regs->USR & JBIG_CODEC_0_ODMA_JBIG_ODMA_UDMA_USR_CMDBUFEMPTY_MASK)) )
    {
        DBG_PRINTF(LOG_DEBUG_M,  "%s JBIG ODMA command buffer or packer not empty!!", __func__);
    }

    // Acknowledge any pending interrupts
    p_JBIG_ODMA_regs->UICR = JBIG_ODMA_UDMA_INT_ENABLE_MASK;
    p_JBIG_ODMA_core_regs->OICR = JBIG_ODMA_CORE_INT_ENABLE_MASK;

    while (JBIG_CODEC_0_ODMA_JBIG_ODMA_UDMA_USR_CMDBUFEMPTY_MASK != (p_JBIG_ODMA_regs->USR & JBIG_CODEC_0_ODMA_JBIG_ODMA_UDMA_USR_CMDBUFEMPTY_MASK) && ( loopCount++ <1000) )
    {
        DBG_MEMLOG(LOG_DEBUG_H,"jbigODMA: status = %#x\n", p_JBIG_ODMA_regs->status);
    }

    // Perform soft reset on the ODMA hardware - disable and re-enable
    p_JBIG_ODMA_regs->UCR = JBIG_CODEC_0_ODMA_JBIG_ODMA_UDMA_UCR_ENABLE_REPLACE_VAL( p_JBIG_ODMA_regs->UCR, 1 ); 
    p_JBIG_ODMA_regs->UCR = JBIG_CODEC_0_ODMA_JBIG_ODMA_UDMA_UCR_ENABLE_REPLACE_VAL( p_JBIG_ODMA_regs->UCR, 0 ); 

     // Acknowledge any pending interrupts
    p_JBIG_ODMA_regs->UICR = JBIG_ODMA_UDMA_INT_ENABLE_MASK;
    p_JBIG_ODMA_core_regs->OICR = JBIG_ODMA_CORE_INT_ENABLE_MASK;

    DBG_MEMLOG(LOG_DEBUG_M,"jbigODMA: after reset status = %#x\n", p_JBIG_ODMA_regs->status);

    DBG_MEMLOG(LOG_DEBUG_M,"jbigODMA: status = %#x\n", p_JBIG_ODMA_regs->status);

    DBG_SLOG_EXIT
}


/**
 * jbigResetReadDMA
 *
 * \brief 
 * 
 * \param 
 * 
 * \return 
 *
 *
 **/

void jbigResetReadDMA( jbig_handle_t * handle )
{
    JBIG_CODEC_IDMA_REGS_t* p_JBIG_IDMA_regs;
    uint32_t loopCount = 0;

    DBG_SLOG_ENTRY

    ASSERT(handle != NULL);

    p_JBIG_IDMA_regs = handle->jbig_block_interface->jbig_idma_regs;

    DBG_MEMLOG(LOG_DEBUG_M,"jbigIDMA: cfg = %#x \n\t int_en %#x \n\tint_pend %#x \n\tint_ack %#x\n\tstatus = %#x\n", p_JBIG_IDMA_regs->cfg, p_JBIG_IDMA_regs->int_en,
                                                                                                   p_JBIG_IDMA_regs->int_st,p_JBIG_IDMA_regs->int_cl,
                                                                                                   p_JBIG_IDMA_regs->status);


    while(1 == (p_JBIG_IDMA_regs->status & JBIG_CODEC_IDMA_STATUS_DMA_BUSY_MASK) &&( loopCount++ <1000) );


    p_JBIG_IDMA_regs->reset = JBIG_CODEC_IDMA_RESET_SOFT_RESET_REPLACE_VAL(p_JBIG_IDMA_regs->reset, 1);
    p_JBIG_IDMA_regs->reset = JBIG_CODEC_IDMA_RESET_SOFT_RESET_REPLACE_VAL(p_JBIG_IDMA_regs->reset, 0);


    p_JBIG_IDMA_regs->int_cl = 0xFF;

    p_JBIG_IDMA_regs->cfg = JBIG_CODEC_IDMA_CFG_ENABLE_REPLACE_VAL(p_JBIG_IDMA_regs->cfg, 0);

    DBG_SLOG_EXIT
}

/**
 * isJbigStarted
 *
 * \brief 
 * 
 * \param 
 * 
 * \return 
 *
 *
 **/

static uint8_t isJbigStarted( jbig_handle_t * inHandle )
{
    uint8_t started = 0;
    error_type_t startederror = jbig_started(inHandle, &started );

    if ( startederror == OK )
    {
        return started;
    }
    else
        return 0;
}



/**
 * jbigAddReadDMABuffer
 * 
 * \brief 
 * 
 * \param 
 * 
 * \return 
 *
 *
 **/

error_type_t jbigAddReadDMABuffer( jbig_handle_t * inHandle, BigBuffer_t* buf, uint32_t size, void * userData )
{

    error_type_t returnValue = FAIL;
    jbig_dma_context_t* dma_context = NULL;
    
    if( inHandle && inHandle->inUse )
    {
        if( inHandle->numPendingMessages < inHandle->maxNumMessages )
        {
            dma_context = (jbig_dma_context_t*)MEM_MALLOC(sizeof(jbig_dma_context_t));
            ASSERT(dma_context);
            if (dma_context)
            {
                dma_context->buf = buf;;
                dma_context->len = size;
                dma_context->user_callback_data = userData;
                returnValue = jbig_add_DMA_buffer_IDMA( inHandle, &inHandle->pendingReadDescriptorList, buf, size, dma_context, DMA_TO_DEVICE);
            }
            else {
                printf("JBIG:%s dma_context MALLOC failed!!!\n", __func__);
            }
        }
        else
        {
            DBG_MEMLOG(LOG_WARNING,"JBIG: too many buffers !!!!! %d\n",inHandle->numPendingMessages);
        }

        if( jbigStartReadDMAIfPossible( inHandle ) == OK )
        {
            DBG_MEMLOG(LOG_DEBUG_H,"JBIG: Read DMA Start Now\n");
        }
    
    }
    
    return ( returnValue );
}


/**
 * jbigStartReadDMA
 *
 * \brief 
 * 
 * \param 
 * 
 * \return 
 *
 *
 **/

void jbigStartReadDMA( jbig_handle_t * inHandle )
{

    if( inHandle )
    {
        DBG_MEMLOG(LOG_INFO,"JBIG: start chained Read DMA\n");

        jbigStartReadDMAIfPossible( inHandle );
    }
}

/**
 * jbigStartWriteDMA
 *
 * \brief 
 * 
 * \param 
 * 
 * \return 
 *
 *
 **/

void jbigStartWriteDMA( jbig_handle_t * inHandle )
{
    if( inHandle )
    {
        // enable the DMA
        DBG_MEMLOG(LOG_INFO,"JBIG: start chained Write DMA\n");

        inHandle->bytes_until_pad = -1;

        jbigStartWriteDMAIfPossible( inHandle );
    }
}

/**
 * jbigAddWriteDMABuffer
 *
 * \brief 
 * 
 * \param 
 * 
 * \return 
 *
 *
 **/

error_type_t jbigAddWriteDMABuffer( jbig_handle_t * inHandle, BigBuffer_t* buf, uint32_t size, void * userData )
{
    error_type_t returnValue = FAIL;
    jbig_dma_context_t* dma_context = NULL;
        
    DBG_SLOG_ENTRY

    if( inHandle )
    {
        if( inHandle->numPendingMessages < inHandle->maxNumMessages )
        {
            DBG_MEMLOG(LOG_DEBUG_H, "%s user data 0x%#x, buf 0x%#x size = %d\n", __func__, userData, buf, size);
            dma_context = (jbig_dma_context_t*)MEM_MALLOC(sizeof(jbig_dma_context_t));
            ASSERT(dma_context);
            if (dma_context)
            {
                dma_context->len = size;
                dma_context->buf = buf;
                dma_context->user_callback_data = userData;
                
                returnValue = jbig_add_DMA_buffer_ODMA( inHandle, &inHandle->pendingWriteDescriptorList, buf, size, dma_context, DMA_FROM_DEVICE);
            }
            else {
                printf("JBIG:%s dma_context MALLOC failed!!!\n", __func__);
            }
        }
        else
        {
            DBG_MEMLOG(LOG_WARNING,"JBIG: too many write buffers !!!!! %d %d\n", inHandle->numPendingMessages, inHandle->maxNumMessages);
        }

        if( jbigStartWriteDMAIfPossible( inHandle ) == OK )
        {
            DBG_MEMLOG(LOG_INFO,"JBIG: Write DMA Start Now\n");
        }        

    }
    
    DBG_SLOG_EXIT

    return ( returnValue );
}
 

/** 
 * jbigCommitConfig
 *
 * \brief Commit current JBIG configuration to the JBIG hardware block
 * 
 * \param handle The handle of the current JBIG session
 * 
 * \return OK if successful, FAIL if not.
 * 
 * This function will set the appropriate configuration bits based on
 * the configuration variables store in the the given sessions 
 * JBIG_CONFIG structure
 **/

error_type_t jbigCommitConfig(jbig_handle_t *handle)
{
    JBIG_REGS_t * p_JBIG_regs = NULL;
    error_type_t return_result = FAIL;

    ASSERT(handle != NULL);
    if( handle )
    {
        // get the register interfaces
        p_JBIG_regs = handle->jbig_block_interface->jbig_regs;

        DBG_MEMLOG(LOG_DEBUG_M,"%s core status = %#x\n",__func__, p_JBIG_regs->JSTAT);

        // set TPBON
        p_JBIG_regs->JCTL = JBIG_CODEC_JCTL_T_REPLACE_VAL( p_JBIG_regs->JCTL, handle->config.TpbonState);

        //setup ATMOV
        p_JBIG_regs->JCTL = JBIG_CODEC_JCTL_A_REPLACE_VAL( p_JBIG_regs->JCTL, handle->config.AtmovState);
        if( handle->config.AtmovState == ATMOV_ENABLED )
        {
            p_JBIG_regs->JCTL = JBIG_CODEC_JCTL_M_REPLACE_VAL( p_JBIG_regs->JCTL, handle->config.ATMOVvalue);
        }
        else
        {
            p_JBIG_regs->JCTL = JBIG_CODEC_JCTL_M_REPLACE_VAL( p_JBIG_regs->JCTL, 0);
        }        

        // sanity check
        ASSERT(handle->config.Xd != 0);
        ASSERT(handle->config.Yd != 0);
        ASSERT(handle->config.Ld != 0);
        ASSERT(handle->config.LineWidth != 0);

        if (handle->config.AddExtraLines)
        {
            jbig_add_pad_strip_update_config(handle);
        }

        // Set image dimensions
        p_JBIG_regs->JX = handle->config.Xd;
        p_JBIG_regs->JY = handle->config.Yd;
        p_JBIG_regs->JSL = handle->config.Ld;

        // set 3Line
        p_JBIG_regs->JCTL = JBIG_CODEC_JCTL_L_REPLACE_VAL( p_JBIG_regs->JCTL, handle->config.LineTemplate);

        // set BPEND
        p_JBIG_regs->JCTL = JBIG_CODEC_JCTL_BE_REPLACE_VAL( p_JBIG_regs->JCTL, handle->config.Endian);

        // set BYPASS CORE
        p_JBIG_regs->JCTL = JBIG_CODEC_JCTL_B_REPLACE_VAL( p_JBIG_regs->JCTL, handle->config.BypassCore);

        // Enable CBI
        p_JBIG_regs->JCTL = JBIG_CODEC_JCTL_C_REPLACE_VAL( p_JBIG_regs->JCTL, handle->config.CBI);

        return_result = OK;
    }
    return return_result;
}


void jbig_dump_control_regs(jbig_handle_t *handle)
{
#ifdef HAVE_DBG_PRINTF
    JBIG_REGS_t * jbig_regs = handle->jbig_block_interface->jbig_regs;
    DBG_PRINTF_CRIT("JBIG Control Reg Dump:\n");
    DBG_PRINTF_CRIT("\tJCTL    = %#x\n", jbig_regs->JCTL);
    DBG_PRINTF_CRIT("\tJSTAT   = %#x\n", jbig_regs->JSTAT);
    DBG_PRINTF_CRIT("\tJX      = %#x\n", jbig_regs->JX);
    DBG_PRINTF_CRIT("\tJY      = %#x\n", jbig_regs->JY);
    DBG_PRINTF_CRIT("\tJBIG_EN = %#x\n", jbig_regs->JBIG_EN);
    DBG_PRINTF_CRIT("\tJSL     = %#x\n", jbig_regs->JSL);
#endif
}

void jbig_dump_odma_regs(jbig_handle_t *handle)
{
#ifdef HAVE_DBG_PRINTF
    JBIG_CODEC_ODMA_REGS_t* odma_regs = handle->jbig_block_interface->jbig_odma_regs;
    JBIG_CODEC_ODMA_CORE_REGS_t* odma_core_regs = handle->jbig_block_interface->jbig_odma_core_regs;
    DBG_PRINTF_CRIT("JBIG ODMA Reg Dump:\n");
    DBG_PRINTF_CRIT("\tcfg     = %#x\n", odma_regs->UCR);
    DBG_PRINTF_CRIT("\tstatus  = %#x\n", odma_regs->USR);
    DBG_PRINTF_CRIT("\tparameter = %#x\n", odma_regs->UPR);
    DBG_PRINTF_CRIT("\tint_en = %#x\n", odma_regs->UIER);
    DBG_PRINTF_CRIT("\tint_pend = %#x\n", odma_regs->UIPR);
    DBG_PRINTF_CRIT("\tdesc_write = %#x\n", odma_regs->UDR);
    DBG_PRINTF_CRIT("\tburst address = %#x\n", odma_regs->UBAR);
    DBG_PRINTF_CRIT("\tburst_length = %#x\n", odma_regs->UBLR);
    DBG_PRINTF_CRIT("\txfer bytes remaining = %#x\n", odma_regs->UBRR);
    DBG_PRINTF_CRIT("\tcore cfg          %#x\n", odma_core_regs->OCR);
    DBG_PRINTF_CRIT("\tcore int enable   %#x\n", odma_core_regs->OIER);
    DBG_PRINTF_CRIT("\tcore int pending  %#x\n", odma_core_regs->OIPR);
    DBG_PRINTF_CRIT("\tcore buffer len   %#x\n", odma_core_regs->OBLR);
#endif
}

void jbig_dump_idma_regs(jbig_handle_t *handle)
{
#ifdef HAVE_DBG_PRINTF
    JBIG_CODEC_IDMA_REGS_t* idma_regs = handle->jbig_block_interface->jbig_idma_regs;
    DBG_PRINTF_CRIT("JBIG IDMA Reg Dump:\n");
    DBG_PRINTF_CRIT("\tcfg = %#x\n", idma_regs->cfg);
    DBG_PRINTF_CRIT("\tstatus = %#x\n", idma_regs->status);
    DBG_PRINTF_CRIT("\tline_width = %#x\n", idma_regs->line_width);
    DBG_PRINTF_CRIT("\tint_en = %#x\n", idma_regs->int_en);
    DBG_PRINTF_CRIT("\tint_pend = %#x\n", idma_regs->int_st);
    DBG_PRINTF_CRIT("\tint_ack = %#x\n", idma_regs->int_cl);
    DBG_PRINTF_CRIT("\tdesc_write = %#x\n", idma_regs->desc_write);
    DBG_PRINTF_CRIT("\tdesc_read = %#x\n", idma_regs->desc_read);
    DBG_PRINTF_CRIT("\txfer_length = %#x\n", idma_regs->xfer_length);
    DBG_PRINTF_CRIT("\txfer_addr = %#x\n", idma_regs->xfer_addr);
    DBG_PRINTF_CRIT("\tctrl_word = %#x\n", idma_regs->ctrl_word);
    DBG_PRINTF_CRIT("\treset = %#x\n", idma_regs->reset);
#endif
}

/**
 * \brief Debug routine to dump all jbig registers for the blocks associated with the given handle
 */
void jbig_dump_all_regs(jbig_handle_t * handle)
{
    jbig_dump_control_regs(handle);
    jbig_dump_odma_regs(handle);
    jbig_dump_idma_regs(handle);   
}

#ifdef HAVE_POWER_MGR
#define PWR_MGR_MY_NAME  "jbig"

static uint32_t pwr_mgr_id;

static void pwr_on(void)
{
    uint32_t block_count;

    for(block_count = 0; block_count < NUM_JBIG_BLOCKS; block_count++)
    {
        jbig_block_init(jbig_handles[block_count].jbig_block_interface);
    }
}
static void pwr_off(void)
{
}
static bool jbig_power_save(pwr_mgr_level_t level, pwr_mgr_cmd_t cmd)
{
    static pwr_mgr_level_t pwr_level = pwr_mgr_on_e;
    switch(level)
    {
    case pwr_mgr_on_e:
        if (pwr_level == pwr_mgr_off_e)
        {
            pwr_on();
        }
        pwr_mgr_powered_up(pwr_mgr_id);
        pwr_level = level;
        break;
    case pwr_mgr_reduced_power_e:
    case pwr_mgr_lowest_power_e:
        ASSERT(pwr_level == pwr_mgr_on_e);
        pwr_mgr_powered_down(pwr_mgr_id);
        pwr_level = level;
        break;
    case pwr_mgr_off_e:
        ASSERT(pwr_level == pwr_mgr_on_e);
        pwr_off();
        pwr_mgr_powered_down(pwr_mgr_id);
        pwr_level = level;
        break;
    default:
        break;
    }
    return true;
}


////////////////////////////////////////////////////////////////////////////////
//
// jbig_pwr_init
//
//    Inputs:  
//    Returns: 
//
////////////////////////////////////////////////////////////////////////////////
void jbig_pwr_init(void)
{
    DPRINTF(DBG_SOFT|DBG_OUTPUT,("%s()\r\n",__FUNCTION__));

    pwr_mgr_id = pwr_mgr_reg_callback(PWR_MGR_MY_NAME, jbig_power_save, PWRMGR_JBIG_PRIORITY);

    // temp set pwr level here till unity get AOAO schema mgr
    pwr_mgr_set_module_pwr_level(PWR_MGR_MY_NAME, PWRMGR_JBIG_LEVEL);   

    return;
} 
#endif

