#!/usr/bin/env python

# Link multiple ThreadX executable images into a ROM-able format.
#
# This script tries to be python2/python3 compatible.
#
# Input is a JSON file. See thxlink.json for an example.
#   {
#    "thxuniversal" : [
#        { "filename" : "file1.thx",  
#          "chip_id"  : 42,
#          "chip_rev" : 1,
#          "flags"    : 0
#        },
#        {
#        }
#       ]
#   }
#
# 20-Nov-2014
#
# History:
#   1.0 First version. 20-Nov-2014
#   1.1 TRDX_entries image_offset_address and size must be 32-bits
#       Add usage() and better error checking.
#

# support python2 but be as python3 as possible
from __future__ import print_function
from __future__ import unicode_literals

import sys
import os
import struct
import json
import logging
import argparse
import filecmp

version = (1,1)

debug = False

# 
# **  All fields are stored Little Endian **
#

#typedef struct TRDX_entries_s
#{
#   uint16_t chip_id; // Matches to CIU::CHIP_ID [15:0]
#   uint8_t chip_rev; // Matches to CIU::CHIP_ID [23:16]
#   uint8_t flags;
#   uint32_t CRC32_val;
#   uint32_t image_offset_address;
#   uint32_t size;
#} TRDX_entries_t;
entry_fmt = "<HBBLLL"

#typedef struct TRDX_multi_package_s
#{
#   uint32_t magic_key;
#   uint32_t entries_count;
#   TRDX_entries_t entries[];
#} TRDX_multi_package_t;
package_fmt = "<LL"

TRDX_MAGIC_KEY  = 0x54524458 # // "TRDX" in ASCII

# all files in the output file will start on this boundary
sector_size = 512

class LinkerError(Exception):
    pass

class FileError(LinkerError):
    pass

class ParseError(LinkerError):
    pass

def load_linker_script(linker_script_name):
    logging.debug("loading linker json script \"{0}\"".format(linker_script_name))
    with open(linker_script_name,"r") as infile:
        script = json.load(infile)
    return script
    
def load_file( infilename ) : 
    logging.debug("loading file \"{0}\"".format(infilename))
    with open(infilename,"rb") as infile :
        buf = infile.read()
    logging.debug("read bytes={0} from file=\"{1}\"".format(len(buf),infilename))
    return buf

def calc_cksum( buf ) :
    # TODO
    return 0

def sanity( linker_script ) : 
    # TODO  verify filename is a string, chip_id is an integer, etc, etc, etc.

    vkey = "thxuniversal_v1"
    if vkey not in linker_script : 
        errmsg = "Unable to find required top-level field \"{0}\" in linker script.".format(vkey)
        raise ParseError(errmsg)

    file_entries = linker_script[vkey]

    required_fields = ( "filename", "chip_id", "chip_rev", "flags" )
    required_types = ( type(""), type(0), type(0), (type(0),type("")) )

    for idx,thx in enumerate(file_entries) : 
        for f,t in zip(required_fields,required_types) : 
            if f not in thx : 
                errmsg = "record number {0} is missing required field \"{1}\"".format(idx,f)
                raise ParseError(errmsg)

            logging.debug("idx={0} f={1} {2}".format(idx,f,type(thx[f])))
            if not isinstance( thx[f], t ):
                typename = str(t).replace("class ","")
                errmsg = "Wrong type for record number {0} field \"{1}\". Need {2}".format( idx,f,typename)
                raise ParseError(errmsg)
        
        # Secret helper hack -- flags can be either a decimal integer or a
        # string. JSON doesn't support hex constants (0x). So allow users to
        # put in a hex value as a string, e.g., "0x01"
        if isinstance(thx["flags"],type("")):
            # convert to integer
            if thx["flags"].startswith("0x") or thx["flags"].startswith("0X") :
                num = int(thx["flags"],16)
            else:
                # try decimal
                num = int(thx["flags"])
            thx["flags"] = num

        # flags must be an 8-bit number
        if thx["flags"]<0 or thx["flags"]>255:
            errmsg = "Flags value={0} for filename={1} record number {2} is out of 8-bit range.".format(
                        thx["flags"],thx["filename"],idx)
            raise ParseError(errmsg)

def write_entry( outfile, thx_entry ) : 
    logging.debug("writing file=\"{0}\"".format(thx_entry["filename"]))
    # each file entry must be on a 512-byte sector boundary
    pos = outfile.tell()

    if pos%sector_size != 0 : 
        # round up to a 512 byte boundary
        new_pos = (pos + (sector_size-1)) & ~(sector_size-1)
        logging.debug("aligning file offset from pos={0} to pos={1}".format(pos,new_pos))
        pos = new_pos

    assert pos%sector_size==0, pos
    outfile.seek(pos,os.SEEK_SET)
    outfile.write(thx_entry["data"])

    # keep the file position so can write it into the table of contents
    thx_entry["offset_bytes"] = pos

def write_header( outfile, all_files ) : 
    outfile.seek(0,os.SEEK_SET)

    outfile.write(struct.pack(str(package_fmt),TRDX_MAGIC_KEY,len(all_files)))

    for thx in all_files: 
        buf = struct.pack(str(entry_fmt),thx["chip_id"],thx["chip_rev"],thx["flags"],
                            thx["cksum"], thx["offset_bytes"], thx["size_bytes"] )
        outfile.write(buf)

        logging.debug("{6} chip_id={0} chip_rev={1} flags={2:#04x} cksum={3} offset_bytes={4} size_bytes={5}".format(
                thx["chip_id"],thx["chip_rev"],thx["flags"],
                thx["cksum"], thx["offset_bytes"], thx["size_bytes"],
                thx["filename"] ) )

def run_linker( linker_script_name, output_file_name ) :
    try : 
        linker_script = load_linker_script(linker_script_name)
    except IOError as err:
        errmsg = "Unable to open linker script \"{0}\": {1}".format(
                    linker_script_name, os.strerror(err.errno))
        raise FileError(errmsg)
    except ValueError as err:
        # json parser throws this if not a valid JSON doc
        errmsg = "Unable to parse json linker script \"{0}\": {1}".format(
                    linker_script_name, err)
        raise ParseError(errmsg)

    # typecheck the linker script fields
    sanity( linker_script )

    file_entries = linker_script["thxuniversal_v1"]

    # load all the files before starting the output file
    # (verifies all files exist and are readable)
    for thx_file in file_entries : 
        thx_file["data"] = load_file(thx_file["filename"])
        thx_file["cksum"] = calc_cksum(thx_file["data"])
        thx_file["size_bytes"] = len(thx_file["data"])

    try : 
        outfile = open(output_file_name, "wb")
    except IOError as err:
        errmsg = "Unable to open output file \"{0}\": {1}".format(
                        output_file_name, os.strerror(err.errno))
        raise FileError(errmsg)

    # skip past the header space
    outfile.seek(512,os.SEEK_SET)

    thx_previous = thx_file
    thx_previous["filename"] = "empty"
    thx_previous["offset_bytes"] = 0

    for thx_file in file_entries :
        logging.debug("current file=\"{0}\"".format(thx_file["filename"]))
        logging.debug("previous file=\"{0}\"".format(thx_previous["filename"]))
        #if thx_file["filename"] == thx_previous["filename"] :
        if thx_previous["filename"] != "empty" and filecmp.cmp(thx_file["filename"],thx_previous["filename"]):
            # current file is same as previous, so use previous offset
            thx_file["offset_bytes"] = thx_previous["offset_bytes"]
            logging.debug("File duplicate, using previous offset=\"{0}\"".format(thx_previous["offset_bytes"]))
        else:
            # current file is new, write it into image
            write_entry(outfile,thx_file)
        thx_previous["filename"] = thx_file["filename"]
        thx_previous["offset_bytes"] = thx_file["offset_bytes"]

    outfile.seek(0,os.SEEK_SET)
    write_header(outfile,file_entries)

    logging.debug("finished writing num_files={0} to output file \"{1}\"".format(len(file_entries),output_file_name))
    outfile.close()
    
def parse_args():
    def version_string():
        return "v{0}.{1}".format(*version)
    parser = argparse.ArgumentParser(description="ThreadX linker to MRVL Universal Binary {0}".format(version_string()))
    parser.add_argument("--debug","-d",
                        action='store_true',
                        help="enable debug messages")
    parser.add_argument("--version","-v",
                        action='version',
                        version=version_string() )
    parser.add_argument("jsonscript",help="linker script in JSON format")
    parser.add_argument("outfile",help="output file name")

    args = parser.parse_args()

    return args

if __name__=='__main__':
    args = parse_args()

    if args.debug : 
        logging.basicConfig(format='%(levelname)s:%(message)s', level=logging.DEBUG)
    else:
        logging.basicConfig(format='%(levelname)s:%(message)s', level=logging.INFO)

    try : 
        run_linker( args.jsonscript, args.outfile )
    except LinkerError as err:
        logging.error("{0}".format(err))
        sys.exit(1)

