/*
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2007-2014, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
*/

//#define DEBUG

#define __APP_C_DEFINES
#include <stdint.h>
#include "assert.h"
#include "error_types.h"
#include "bootCode.h"
#include "uart.h"
#include "utils.h"
#include "regAddrs.h"
#include "minPrintf.h"
#include "hex_dump.h"
#include "code_images.h"
#include "board_types.h"
#include "rom_types.h"

#include "apb_top_regheaders.h"
#include "cpu_api.h"
#include "avs_api.h"
#include "ddr_api.h"
#include "pincfg_api.h"
#include "pmu_api.h"
#include "pll_api.h"
#include "flash_api.h"
#include "ID_utils.h"
#include "tpm_api.h"
#include "tpm_spec.h"

/*  RICOH add : Charm  */
#include "hex_dump.h"
#include "sha1.h"

#define checkpoint(c)		dbg_uart_out(c)
//#define SERIAL_BAUD_RATE	(38400)
#define SERIAL_BAUD_RATE	(115200) /* Working Now!! */

#define VEND_UNIQUE_ID_OFFSET   (0x04)
#define TIMH_BASE_ADDR_FIELD    (0xD100003C)
#define USB_DWNLD_PASS2         (0x50617332)    // Ascii for "Pas2".  This must match TIM OEM UniqueID!

#define ROM_BASE_ADDR		(0xFFE00000)	/* 0xFFE0_0000--0xFFE1_EFFF */
#define ROM_BASE_SIZE		(0x1F000)	/* 124KB */
#define ROM_BUILD_FIELD     (0xFFE00034)
#define ROM_MAJOR_FIELD     (0xFFE00027)
#define ROM_MINOR_FIELD     (0xFFE00026)
#define ROM_SUBa_FIELD      (0xFFE00025)
#define ROM_SUBb_FIELD      (0xFFE00024)
#ifdef CONFIG_TPM	/* RICOH change */
static struct tpm_device *tpm_open(void);
static error_type_t tpm_close(struct tpm_device *tpm);
static error_type_t tpm_save_digest(struct tpm_device *tpm, int pcr_index,
	const char *name, const struct image_info *img);
#endif	/* CONFIG_TPM */

extern const char BUILD_INFO[];

// FIXME Kill this workaround in Rev B when it can die after the HW fix is proven.
#define MPUSIZE_2GB                 0x1E
#define MPUTYPE_NOCACHE             0x308
extern void setMpuRegion (uint32_t region, uint32_t address, uint32_t size, uint32_t subregionMask, uint32_t type);
// FIXME Kill this workaround in Rev B when it can die after the HW fix is proven.

static uint32_t g_SDMMC_BusSpeedMhz = 0;
static uint32_t g_ProcSpeedMhz =    0;
static uint32_t g_corepll_freq = 0;
static int g_memorysize = 0;

typedef void (*ROM_restart_entry_func_t) (void);

#define start_checkpoints() checkpoint('\n');checkpoint('\n');checkpoint('\r')
#define end_checkpoints()   start_checkpoints()

uint32_t hwGetProcSpeed(void)
{
	return g_ProcSpeedMhz;
}


uint32_t hwGetSDMMCBusSpeed(void)
{
	return g_SDMMC_BusSpeedMhz;
}


static void init_board_status (void)
{
	if (is_FPGA())
	{
		g_SDMMC_BusSpeedMhz = 52;
		g_ProcSpeedMhz = 52;
	} else
	{
		g_SDMMC_BusSpeedMhz = 25;
		g_ProcSpeedMhz = 25;
	}
}


static void log_eROM_board_version ( board_types_e board_type, rom_types_e rom_type )
{
	char *major_ptr     = (char *)    ROM_MAJOR_FIELD;
	char *minor_ptr     = (char *)    ROM_MINOR_FIELD;
	char *subA_ptr      = (char *)    ROM_SUBa_FIELD;
	char *subB_ptr      = (char *)    ROM_SUBb_FIELD;
	uint32_t *build_ptr = (uint32_t*) ROM_BUILD_FIELD;
	char *RevName[] = { [CHIP_REVA] = "A", [CHIP_REVB] = "B", [CHIP_REVB1] ="B1",
						[CHIP_REVC] = "C", [CHIP_REVD] = "D", [CHIP_GE2_REVB] = "B" };
	int chiprev = 0;

	chiprev = get_chip_rev();
	if ((chiprev != CHIP_REVA) && (chiprev != CHIP_REVB) && (chiprev != CHIP_REVB1) &&
	    (chiprev != CHIP_REVC) && (chiprev != CHIP_GE2_REVB)) {
		chiprev = CHIP_REVD;
	}

	minPrintf("\n\n%s %s-Rev%s %c.%c.%c%c-%xh %d %d %d\n%s\n",
			(is_Si())   ?"Si":"FPGA",
			(is_Gr2())  ?"Gr2":"Ge2",
			((is_Gr2_RevC1()) ? "C1" : RevName[chiprev]),
			*major_ptr, *minor_ptr, *subA_ptr, *subB_ptr,
			*build_ptr, rom_type, board_type, g_ProcSpeedMhz, BUILD_INFO);
}


static void fixupMpuRegion(void)
{
	if (is_Gr2() && is_RevA())
	{
		// Re-enable execute from DDR (BR turns it off to workaround a Gr2 RevA HW issue)
		setMpuRegion(0,0x00000000,MPUSIZE_2GB, 0,MPUTYPE_NOCACHE); // DDR
	}
}


static void cache_init()
{
	bool bIcache = true;
	bool bDcache = true;

	if (bIcache || bDcache)
	{
		if (bIcache)
		{
			cpu_enable_icache();     // automatically invalidates all icache
			checkpoint('i');
		}
		if (bDcache)
		{
			cpu_enable_dcache();     // automatically invalidates all dcache
			checkpoint('d');
		}
	}
}

pll_freq_t get_core_pll(void)
{
    /**
     * GPIOG[5]    IOPAD197:0xd401e34c/0x00019843
     *             PDRG    :0xd4019604/0x00000000
     *             PLRG    :0xd4019600/NA
     */
    volatile unsigned long *iopad_197 = (unsigned long *)0xd401e34c;
    volatile unsigned long *gpio_pdrg = (unsigned long *)0xd4019604;
    volatile unsigned long *gpio_plrg = (unsigned long *)0xd4019600;
    pll_freq_t pll_freq = eCORE_PLL_800MHZ;

    /* set our configuration */
    *iopad_197 = 0x00018040;
    *gpio_pdrg = 0x00000000;/* set all dir to in */
    Delay(5000);

    pll_freq = eCORE_PLL_2400MHZ;
    minPrintf("\nCore-PLL=%dMHz\n", (pll_freq == eCORE_PLL_800MHZ) ? 800 : 1200);
	g_corepll_freq = pll_freq;

    return pll_freq;
}

static void initialize_state(rom_types_e rom_type, board_types_e board_type, bool USB_download)
{
	init_board_status();

	dbg_uart_init( board_type, SERIAL_BAUD_RATE );

	assert( invalid_Brd != board_type );
	assert( invalid_ROM != rom_type );

	initPinConfig(board_type);      // Setup the Multi Function Pins

	log_eROM_board_version(board_type, rom_type);

	start_checkpoints();
	checkpoint('u');

	cache_init();
	checkpoint('c');

	PMU_enable(ePMU_DEVICE_I2C); // Turn on I2C so AVS can talk to regulators
	initialize_AVS();
	checkpoint('a');

	if (! USB_download)
	{
		flash_callback_api_init(rom_type, hwGetSDMMCBusSpeed());      // Read FLASH interface
		checkpoint('f');
	}

	setupPMUDividers(board_type);
	checkpoint('v');

	// Turn on clocks to non-primary UARTs as a courtesy
	PMU_enable(ePMU_DEVICE_UARTS);

	initialize_PLL(board_type, get_core_pll());
	g_ProcSpeedMhz = 400;
	g_SDMMC_BusSpeedMhz = ((is_NOT_Gr2_RevA()) ? 200 : 100);
    /* RICOH changed setup SYSPLL at g2-loader */
    initialize_PLL(board_type, eSYS_PLL_1000MHZ);

    /* RICOH changed setup SCANPLL at g2-loader */
    initialize_PLL(board_type, eSCAN_PLL_1500MHZ);

	cpu_init();                     // cycle counter
	checkpoint('e');
}


static void deadlock(void)
{
	while(1) {
#ifdef DEBUG
		minPrintf("\r-\r");
		minPrintf("\r\\\r");
		minPrintf("\r|\r");
		minPrintf("\r/\r");
		minPrintf("\r-\r");
		minPrintf("\r\\\r");
		minPrintf("\r|\r");
		minPrintf("\r/\r");
#else
		asm volatile
			(
			 "wfi;"             // Wait For Int
			 :::                // no reg is clobbered
			);
#endif
	}
}

/**************************** >> begin NVRAM sharing */
static inline void __attribute__((always_inline)) cpu_spin_delay_100clk(void)
{
    register uint32_t startClockCount;
    register uint32_t endClockCount;
    register uint32_t currentClockCount;

    asm volatile
    (
        "mrc p15, 0, %0, c9, c13, 0;" // Load cycle counter into "count"
        :"=r" (startClockCount)                 // output
    );

    // to delay n microseconds, we just need to wait n*clockspeed cycles since
    // the processor is clocked in Mhz.
    endClockCount = startClockCount + 100; //wait 100clock in 100/400us (4MHz)

    // deal w/ rollover
    if(startClockCount < endClockCount)
    {
        while(1) {
            asm volatile
            (
                "mrc p15, 0, %0, c9, c13, 0;" // Load cycle counter into "count"
                :"=r" (currentClockCount)                 // output
            );
            if ( (endClockCount < currentClockCount) || (currentClockCount < startClockCount) ) break;    // spin for delay
        }
    }
    else
    {
        while(1) {
            asm volatile
            (
                "mrc p15, 0, %0, c9, c13, 0;" // Load cycle counter into "count"
                :"=r" (currentClockCount)                 // output
            );
            if ( (currentClockCount < startClockCount) && (endClockCount < currentClockCount) ) break;    // spin for delay
        }
    }
}

static inline void __attribute__((always_inline)) cpu_spin_delay_25clk(void)
{
    register uint32_t startClockCount;
    register uint32_t endClockCount;
    register uint32_t currentClockCount;

    asm volatile
    (
        "mrc p15, 0, %0, c9, c13, 0;" // Load cycle counter into "count"
        :"=r" (startClockCount)                 // output
    );

    // to delay n microseconds, we just need to wait n*clockspeed cycles since
    // the processor is clocked in Mhz.
    endClockCount = startClockCount + 25; //wait 25clock in 25/400us (400MHz)

    // deal w/ rollover
    if(startClockCount < endClockCount)
    {
        while(1) {
            asm volatile
            (
                "mrc p15, 0, %0, c9, c13, 0;" // Load cycle counter into "count"
                :"=r" (currentClockCount)                 // output
            );
            if ( (endClockCount < currentClockCount) || (currentClockCount < startClockCount) ) break;    // spin for delay
        }
    }
    else
    {
        while(1) {
            asm volatile
            (
                "mrc p15, 0, %0, c9, c13, 0;" // Load cycle counter into "count"
                :"=r" (currentClockCount)                 // output
            );
            if ( (currentClockCount < startClockCount) && (endClockCount < currentClockCount) ) break;    // spin for delay
        }
    }
}

#define SCK_PORT 10
#define DO_PORT 11
#define DI_PORT 12
#define CS_PORT 15

#define H_GPIOA10_SCK (  *((volatile unsigned long *)0xD4019008) = 0x00000400)
#define L_GPIOA10_SCK (  *((volatile unsigned long *)0xD401900C) = 0x00000400)
#define H_GPIOA11_DO (  *((volatile unsigned long *)0xD4019008) = 0x00000800)
#define L_GPIOA11_DO (  *((volatile unsigned long *)0xD401900C) = 0x00000800)
#define H_GPIOA15_CS (  *((volatile unsigned long *)0xD4019008) = 0x00008000)
#define L_GPIOA15_CS (  *((volatile unsigned long *)0xD401900C) = 0x00008000)

#define getGPIOA12_DI ( *((volatile unsigned long *)0xD4019000) & 0x00001000 )

#define CLOCKTIM1 cpu_spin_delay_100clk();
#define CLOCKTIM2 cpu_spin_delay_100clk();
#define CLOCKTIM3 cpu_spin_delay_25clk();

static void readEEPROM4096(unsigned long in_addr, unsigned char *buf, int size)
{
    unsigned char cmd;
    unsigned long addr;
    unsigned char ret;
    unsigned int rettmp;
    int i;
    int copycnt;
	unsigned int addr_width = 24;
    unsigned int bit_mask = 0x800000;
    register uint32_t startClockCount;
    register uint32_t endClockCount;
    register uint32_t currentClockCount;

    if (512 == g_memorysize) {
        if(eCORE_PLL_800MHZ == g_corepll_freq) {
            addr_width = 16;
            bit_mask   = 0x8000;
            minPrintf("nv_addr_width=%d for chimay_p \n", addr_width);
        }
    }
	
    //initial cs 1
    H_GPIOA15_CS; //setGPIOA(CS_PORT, 1);       //cs(A15) initial 1
    CLOCKTIM1;      //wait 100us

    //initial do 0
    L_GPIOA11_DO;// setGPIOA(DO_PORT, 0);       //write do(A11) initial 0
    CLOCKTIM1;      //wait 100us

    for(copycnt=0; copycnt<size; copycnt++) {

        //initial sck 0
        L_GPIOA10_SCK;      //sck(A10) initial 0
        CLOCKTIM2;      //wait 100us

        //cs 0
        L_GPIOA15_CS; //setGPIOA(CS_PORT, 0);       //cs(A15) 0
        CLOCKTIM2;      //wait 100us

        //initilize cmd
        cmd = 0x03;

        //send cmd 0x03
        for(i=0; i<8; i++) {
            {
                asm volatile
                (
                    "mrc p15, 0, %0, c9, c13, 0;" // Load cycle counter into "count"
                    :"=r" (startClockCount)                 // output
                );
            }
    //      setGPIOA(DO_PORT, cmd & 0x80);      //write bit(A11)
            if ((cmd & (0x80>>i)) == 0) {
                *((volatile unsigned long *)0xD401900C) = (1 << DO_PORT);
            } else {
                *((volatile unsigned long *)0xD4019008) = (1 << DO_PORT);
            }
            {
                endClockCount=startClockCount+25;

                // deal w/ rollover
                if(startClockCount < endClockCount)
                {
                    while(1) {
                        asm volatile
                        (
                            "mrc p15, 0, %0, c9, c13, 0;" // Load cycle counter into "count"
                            :"=r" (currentClockCount)                 // output
                        );
                        if ( (endClockCount < currentClockCount) || (currentClockCount < startClockCount) ) break;    // spin for delay
                    }
                }
                else
                {
                    while(1) {
                        asm volatile
                        (
                            "mrc p15, 0, %0, c9, c13, 0;" // Load cycle counter into "count"
                            :"=r" (currentClockCount)                 // output
                        );
                        if ( (currentClockCount < startClockCount) && (endClockCount < currentClockCount) ) break;    // spin for delay
                    }
                }
            }
            //fetch bit0( initial do)
            H_GPIOA10_SCK;      //sck(A10) 1
            CLOCKTIM3;      //wait 100us

            //set bit1
            L_GPIOA10_SCK;      //sck(A10) 0
        }

        addr = in_addr + copycnt;
        //send addr 16bit
        for(i=0; i<addr_width; i++) {
            {
                asm volatile
                (
                    "mrc p15, 0, %0, c9, c13, 0;" // Load cycle counter into "count"
                    :"=r" (startClockCount)                 // output
                );
            }
    //      setGPIOA(DO_PORT, addr & 0x8000);       //write bit(A11)
            if ((addr & (bit_mask>>i)) == 0) {
                *((volatile unsigned long *)0xD401900C) = (1 << DO_PORT);
            } else {
                *((volatile unsigned long *)0xD4019008) = (1 << DO_PORT);
            }
            {
                endClockCount=startClockCount+25;

                // deal w/ rollover
                if(startClockCount < endClockCount)
                {
                    while(1) {
                        asm volatile
                        (
                            "mrc p15, 0, %0, c9, c13, 0;" // Load cycle counter into "count"
                            :"=r" (currentClockCount)                 // output
                        );
                        if ( (endClockCount < currentClockCount) || (currentClockCount < startClockCount) ) break;    // spin for delay
                    }
                }
                else
                {
                    while(1) {
                        asm volatile
                        (
                            "mrc p15, 0, %0, c9, c13, 0;" // Load cycle counter into "count"
                            :"=r" (currentClockCount)                 // output
                        );
                        if ( (currentClockCount < startClockCount) && (endClockCount < currentClockCount) ) break;    // spin for delay
                    }
                }
            }

            //fetch bit0( initial do)
            H_GPIOA10_SCK;      //sck(A10) 1
            CLOCKTIM3;      //wait 100us

            //set bit1
            L_GPIOA10_SCK;      //sck(A10) 0
        }

        L_GPIOA11_DO; //setGPIOA(DO_PORT, 0);       //reset do

        ret=0;
        for(i=0; i< 8; i++) {
            CLOCKTIM3;      //wait 100us
            H_GPIOA10_SCK;      //sck(A10) 1

            {
                asm volatile
                (
                    "mrc p15, 0, %0, c9, c13, 0;" // Load cycle counter into "count"
                    :"=r" (startClockCount)                 // output
                );
            }
            ret = ret << 1;
            rettmp = getGPIOA12_DI;
            if (rettmp != 0) {
                ret = ret | 0x01;
            }
            {
                endClockCount=startClockCount+25;

                // deal w/ rollover
                if(startClockCount < endClockCount)
                {
                    while(1) {
                        asm volatile
                        (
                            "mrc p15, 0, %0, c9, c13, 0;" // Load cycle counter into "count"
                            :"=r" (currentClockCount)                 // output
                        );
                        if ( (endClockCount < currentClockCount) || (currentClockCount < startClockCount) ) break;    // spin for delay
                    }
                }
                else
                {
                    while(1) {
                        asm volatile
                        (
                            "mrc p15, 0, %0, c9, c13, 0;" // Load cycle counter into "count"
                            :"=r" (currentClockCount)                 // output
                        );
                        if ( (currentClockCount < startClockCount) && (endClockCount < currentClockCount) ) break;    // spin for delay
                    }
                }
            }

            L_GPIOA10_SCK;      //sck(A10) 0
        }

        //finalize
        L_GPIOA10_SCK;      //sck(A10) 0
        CLOCKTIM2;      //wait 100us

        H_GPIOA15_CS; //setGPIOA(CS_PORT, 1);       //cs(A15) 1
        CLOCKTIM2;      //wait 100us

        *(buf+copycnt)=ret;
    }

    return;
}

#if 0
static char getcha(char in)
{
    char ret;

    if ( (0 <= in) && (in <= 0x9) ) {
        ret='0'+in;
    } else if ( (0xa <= in) && (in <= 0xf) ) {
        ret='a'+in-0xa;
    } else {
        ret='?';
    }
    return ret;
}

static void fillzerochar2(char in, char *buf) {
    char chtmpin;
    char chtmpout;

    chtmpin=(in>>4)&0xf;
    chtmpout = getcha(chtmpin);
    *(buf+0)=chtmpout;

    chtmpin=(in>>0)&0xf;
    chtmpout = getcha(chtmpin);
    *(buf+1)=chtmpout;

    *(buf+2)='\0';
}

static void fillzero2char4(unsigned short in, char *buf) {
    char chtmpin;
    char chtmpout;

    *(buf+0)='0';
    *(buf+1)='0';

    chtmpin=(in>>12)&0xf;
    chtmpout = getcha(chtmpin);
    *(buf+2)=chtmpout;

    chtmpin=(in>> 8)&0xf;
    chtmpout = getcha(chtmpin);
    *(buf+3)=chtmpout;

    chtmpin=(in>> 4)&0xf;
    chtmpout = getcha(chtmpin);
    *(buf+4)=chtmpout;

    chtmpin=(in>> 0)&0xf;
    chtmpout = getcha(chtmpin);
    *(buf+5)=chtmpout;

    *(buf+6)='\0';
}

#define EEPROMBUFSIZE 4096
static unsigned char eeprombuf[EEPROMBUFSIZE];


static int gDbgSaiCnt1;
static int gDbgSaiCnt2;
#endif
extern int get_memory_size(void);

static void load_eeprom_data(void)
{
#if 0
    int i,j;
    char strtmpsai[8];
#endif
    unsigned char *addr;

#if 0
    minPrintf("\n");
    minPrintf("sai read eeprom50 %s %s\n", __DATE__, __TIME__);
    for(gDbgSaiCnt1=3; gDbgSaiCnt1>0; gDbgSaiCnt1--) {
        minPrintf("sai read eeprom cntdown %d\n", gDbgSaiCnt1);
        for(gDbgSaiCnt2=0; gDbgSaiCnt2<400*10000; gDbgSaiCnt2++) {
        }
    }
#endif
    g_memorysize = get_memory_size();
	
    {
        volatile unsigned long *regaddr;
        regaddr = (unsigned long *)0xD401E028; *regaddr = 0x000110C0; /* IO_PAD10 */
        regaddr = (unsigned long *)0xD401E02C; *regaddr = 0x000110C0; /* IO_PAD11 */
        regaddr = (unsigned long *)0xD401E030; *regaddr = 0x000110C0; /* IO_PAD12 */
        regaddr = (unsigned long *)0xD401E03C; *regaddr = 0x000150C0; /* IO_PAD15 */

        regaddr = (unsigned long *)0xD4019008; *regaddr = 0x00008000; /* set */
        regaddr = (unsigned long *)0xD401900C; *regaddr = 0x00000c04; /* clear */
        regaddr = (unsigned long *)0xD4019004; *regaddr = 0x00008c04; /* dir */
    }

    switch (g_memorysize) {
    case 512:
        addr = (unsigned char *)0x1fa00800;
        if (eCORE_PLL_2400MHZ == g_corepll_freq) {
minPrintf("don't readnv g2loader 1fa00800\n");
        } else {
minPrintf("readnv g2loader 1fa00800\n");
            readEEPROM4096(0xf000, addr, 4096);
        }
        break;
    case 1024:
        addr = (unsigned char *)0x3ef00800;
        if (eCORE_PLL_2400MHZ == g_corepll_freq) {
minPrintf("don't readnv g2loader 3ef00800\n");
        } else {
minPrintf("readnv g2loader 3ef00800\n");
            readEEPROM4096(0x04b0, addr, 4096);
        }
        break;
    case 2048:
        if (eCORE_PLL_2400MHZ == g_corepll_freq) {
minPrintf("readnv g2loader 7ef00560\n");
            addr = (unsigned char *)0x7ef00560;
            readEEPROM4096(0x04b0, addr, 1024);
        } else {
minPrintf("readnv g2loader 7ef00800\n");
            addr = (unsigned char *)0x7ef00800;
            readEEPROM4096(0x04b0, addr, 4096);
        }
        break;
    default:
        assert(0);
        break;
    }

	
#if 0
    for(i=0; i<1024; i++, addr++) {
        if ((i % 16) == 0) {
            fillzero2char4(i, strtmpsai);
            minPrintf("%s", strtmpsai);
        }
        fillzerochar2(*(addr), strtmpsai);
        minPrintf(" %s", strtmpsai);

        if ((i % 16) == 15) {
            minPrintf("\n");
        }
    }
#endif
}
/* << end NVRAM sharing ******************************/

/* Define main entry point.  */

/* FUNCTION NAME: main_entry */

/**
 * \brief This is the main routine.  When we enter here we are using a stack
 * that is located in lcm.  There is not any memory initialized and the PLLs may
 * not be running.   The first thing we do is look for a configuration table,
 * and initialize the memory controller & PLLs with the appropriate settings.
 * We assume the memory will start at 0.
 *
 * Once this is done, we load R4 image & AP bootcode to continue with the rest of
 * boot up functions.
 *
 * \retval int !! NOT expected to return !!
 *
 **/

int main_entry()
{
	bool valid_code_images = true;
	bool USB_download = false;
	rom_types_e rom_type = check_rom_type();
	board_types_e board_type = check_board_type();
#ifdef CONFIG_TPM	/* RICOH change */
	static const struct image_info ROMinfo = {
		.dest_addr = ROM_BASE_ADDR,
		.load_size = ROM_BASE_SIZE,
	};
	struct image_info G2info, APinfo, R4info;
	extern void *entry, *__bss_end__; /* See "MiniLoader.ld" script */
#define AP_loaded_code_launch_addr	(APinfo.dest_addr)
#define R4_loaded_code_launch_addr	(R4info.dest_addr)
#else	/* CONFIG_TPM */
	uint32_t AP_loaded_code_launch_addr, R4_loaded_code_launch_addr;
#endif	/* CONFIG_TPM */
	ROM_restart_entry_func_t ROM_restart = (ROM_restart_entry_func_t)0xFFFF0000;
	uint32_t *TIMH                = *((uint32_t **) TIMH_BASE_ADDR_FIELD);
	uint32_t TIMH_uniqueID        = TIMH[VEND_UNIQUE_ID_OFFSET];
	uint8_t  load_media_id        = get_media_source_id();

#ifdef CONFIG_TPM	/* RICOH change */
	G2info.dest_addr = (uint32_t)&entry;
	G2info.load_size = (uint32_t)&__bss_end__ - (uint32_t)&entry;
#endif	/* CONFIG_TPM */
	if ((load_media_id == 0x1E) || (load_media_id == 0x1F))
		USB_download = true;

	initialize_state(rom_type, board_type, USB_download);// set up cpu, clocks(mhz & timebase) + UART
	checkpoint('s');  // i Initialized

	if (USB_download)
	{
		dbg_printf("\nTIMH_uniqueID: %08x\n",TIMH_uniqueID);
		if (USB_DWNLD_PASS2 == TIMH_uniqueID)
		{
#ifdef CONFIG_TPM	/* RICOH change */
			locate_USB_code_images(&APinfo, &R4info);
#else	/* CONFIG_TPM */
			locate_USB_code_images(&AP_loaded_code_launch_addr, &R4_loaded_code_launch_addr);
#endif	/* CONFIG_TPM */

			fixupMpuRegion();

			// x Execute loaded images, DOES NOT RETURN!!
			launch_code_images(rom_type, board_type, AP_loaded_code_launch_addr, R4_loaded_code_launch_addr);
		}
		else
		{
			initialize_DDR(board_type);
			checkpoint('r');  // d Initialized DDR

			fixupMpuRegion();

			minPrintf("\nRe-Entering eROM Code\n");
			ROM_restart();
		}

		// Should never get here
		minPrintf("eROM Re-Entry or Code Image Launch failed!  Deadlocking\n");
		deadlock();
	}
	else
	{
		initialize_DDR(board_type);
		checkpoint('r');  // d Initialized DDR

		fixupMpuRegion();
	}

	if (false == recovery_boot())
	{
#ifdef CONFIG_TPM	/* RICOH change */
		if (STATUS_OK != load_code_images(rom_type, &APinfo, &R4info))
#else	/* CONFIG_TPM */
		if (STATUS_OK != load_code_images(rom_type, &AP_loaded_code_launch_addr, &R4_loaded_code_launch_addr))
#endif	/* CONFIG_TPM */
		{
			minPrintf("\nCode Image loading failed\n");
			assert(0);
		}
		else
		{
			checkpoint('l');  // l Loaded
			cpu_disable_interrupts();
		}

	} else
	{
#ifdef CONFIG_TPM	/* RICOH change */
		if (STATUS_OK != check_code_images(&APinfo, &R4info))
#else	/* CONFIG_TPM */
		if (STATUS_OK != check_code_images(&AP_loaded_code_launch_addr, &R4_loaded_code_launch_addr))
#endif	/* CONFIG_TPM */
		{
			minPrintf("\nRecovery Code Image checking failed\n");
			assert(0);
		}
	}

	load_eeprom_data();
#ifdef CONFIG_TPM	/* RICOH change */
{
	struct tpm_device *tpm = tpm_open();
	tpm_save_digest(tpm, 0, "BootROM", &ROMinfo);
	/* FastBoot: PCRs of Loader/U-Boot/R4firm, be set later.(initrd) */
	tpm_close(tpm);
}
#endif	/* CONFIG_TPM */

	if (valid_code_images)
	{
		/*  RICOH add : Charm  */
		hex_dump((void *)AP_loaded_code_launch_addr, 0x40);

		// x Execute loaded images, DOES NOT RETURN!!
		launch_code_images(rom_type, board_type, AP_loaded_code_launch_addr, R4_loaded_code_launch_addr);
		checkpoint('x');
	} else
	{
		minPrintf("Code Image validation failed\n");
		assert(0);
	}

	// Should never get here
	minPrintf("Code Image Launch failed!  Deadlocking\n");
	deadlock();

	// Should never return
	return -1;
}

#ifdef CONFIG_TPM	/* RICOH change */
#ifdef CONFIG_TPM_PHYSICAL_INIT
#define TPM_CAP_FLAG                   ((uint32_t)0x00000004)
#define TPM_CAP_FLAG_PERMANENT         ((uint32_t)0x00000108)
#define TPM_CAP_FLAG_VOLATILE          ((uint32_t)0x00000109)

static int tpm_get_flag(struct tpm_device* tpm, uint32_t* nvflag, uint32_t* vflag) {
	const uint32_t perm_size = 22;
	const uint32_t vola_size = 7;
	uint8_t buf[256];
	uint8_t i;

	if (tpm_get_capability(tpm, TPM_CAP_FLAG, TPM_CAP_FLAG_PERMANENT,
			       buf, perm_size) != 0) {
		return -1;
	}

	*nvflag = 0;
	for (i = 2; i < perm_size; ++i) {
		*nvflag |= (buf[i] & 0x01) << (i - 2);
	}

	if (tpm_get_capability(tpm, TPM_CAP_FLAG, TPM_CAP_FLAG_VOLATILE,
			       buf, vola_size) != 0) {
		return -1;
	}

	*vflag = 0;
	for (i = 2; i < vola_size; ++i) {
		*vflag |= (buf[i] & 0x01) << (i - 2);
	}

	return 0;
}

#define TPM_PF_DISABLE                      ((uint32_t)0x00000001)
#define TPM_PF_OWNERSHIP                    ((uint32_t)0x00000002)
#define TPM_PF_DEACTIVATED                  ((uint32_t)0x00000003)
#define TPM_PF_PHYSICALPRESENCEHWENABLE     ((uint32_t)0x00000008)
#define TPM_PF_PHYSICALPRESENCECMDENABLE    ((uint32_t)0x00000009)

#define TPM_SF_DEACTIVATED             ((uint32_t)0x00000001)
#define TPM_SF_PHYSICALPRESENCE        ((uint32_t)0x00000003)

#define TSS_TPM_PF_DISABLE_BIT                      (1 << (TPM_PF_DISABLE - 1))
#define TSS_TPM_PF_OWNERSHIP_BIT                    (1 << (TPM_PF_OWNERSHIP - 1))
#define TSS_TPM_PF_DEACTIVATED_BIT                  (1 << (TPM_PF_DEACTIVATED - 1))
#define TSS_TPM_PF_PHYSICALPRESENCEHWENABLE_BIT     (1 << (TPM_PF_PHYSICALPRESENCEHWENABLE - 1))
#define TSS_TPM_PF_PHYSICALPRESENCECMDENABLE_BIT    (1 << (TPM_PF_PHYSICALPRESENCECMDENABLE - 1))

#define TSS_TPM_SF_DEACTIVATED_BIT          (1 << (TPM_SF_DEACTIVATED - 1))
#define TSS_TPM_SF_PHYSICALPRESENCE_BIT     (1 << (TPM_SF_PHYSICALPRESENCE - 1))

enum tpm_phy_init_t {
	TPM_PHY_INIT_T_OK = 0,
	TPM_PHY_INIT_T_NG,
	TPM_PHY_INIT_T_NEED_RESET
};

static enum tpm_phy_init_t tpm_physical_init(struct tpm_device *tpm)
{
	int ret;
	uint32_t nvflag, vflag;

	if (tpm_get_flag(tpm, &nvflag, &vflag) < 0) {
		minPrintf("ERROR: failed to do tpm_get_flag.\n");
		return TPM_PHY_INIT_T_NG;
	}

	if (((nvflag & TSS_TPM_PF_DEACTIVATED_BIT) == 0) &&
	    ((vflag & TSS_TPM_SF_DEACTIVATED_BIT) == 0) &&
	    ((nvflag & TSS_TPM_PF_PHYSICALPRESENCECMDENABLE_BIT) != 0) &&
	    ((nvflag & TSS_TPM_PF_OWNERSHIP_BIT) != 0) &&
	    ((nvflag & TSS_TPM_PF_DISABLE_BIT) == 0)) {
		/* already initialized. */
		return TPM_PHY_INIT_T_OK;
	}

	minPrintf("INFO: physical initializing TPM device.\n");
	if ((nvflag & TSS_TPM_PF_PHYSICALPRESENCEHWENABLE_BIT) == 0) {
		ret = tpm_tsc_physical_presence(tpm, TPM_PHYSICAL_PRESENCE_HW_ENABLE);
		if (ret) {
			minPrintf("ERROR: failed to TSC_PhysicalPresence(HW) %d\n", ret);
			return TPM_PHY_INIT_T_NG;
		}
	}

	if ((nvflag & TSS_TPM_PF_PHYSICALPRESENCECMDENABLE_BIT) == 0) {
		ret = tpm_tsc_physical_presence(tpm, TPM_PHYSICAL_PRESENCE_CMD_ENABLE);
		if (ret) {
			minPrintf("ERROR: failed to TSC_PhysicalPresence(CMD) %d\n", ret);
			return TPM_PHY_INIT_T_NG;
		}
	}

	ret = tpm_physical_enable(tpm);
	if (ret) {
		minPrintf("ERROR: failed to TPM_PhysicalEnable %d\n", ret);
		return TPM_PHY_INIT_T_NG;
	}

	if (((nvflag & TSS_TPM_PF_DEACTIVATED_BIT) != 0) ||
	    ((vflag & TSS_TPM_SF_DEACTIVATED_BIT) != 0)) {
		ret = tpm_physical_set_deactivated(tpm, 0);
		if (ret) {
			minPrintf("ERROR: failed to TPM_PhysicalSetDeacticated %d\n", ret);
			return TPM_PHY_INIT_T_NG;
		}
		return TPM_PHY_INIT_T_NEED_RESET;
	}

	return TPM_PHY_INIT_T_OK;
}
#endif /* CONFIG_TPM_PHYSICAL_INIT */

static struct tpm_device *tpm_open(void)
{
	struct tpm_device *tpm = NULL;
	int spi_bus = 2;	/* default: SPi3 */
	int ret;
#ifdef CONFIG_TPM_PHYSICAL_INIT
	const int pcr_index = 0;
	uint8_t digest[TPM_PCR_DIGEST_LENGTH];
	int physical_init = 0;
again:
#endif /* CONFIG_TPM_PHYSICAL_INIT */

	/* unreset the TPM device */
	tpm_device_init();

	/* probe the TPM device (enable APMU too) */
	tpm = tpm_alloc_device(TPM_DEVICE_SPI, spi_bus, 0);
	if (tpm == NULL && spi_bus != 1) {
		spi_bus = 1;	/* alternate: SPi2 */
		minPrintf("INFO: TPM device not found at SPi3,"
			" scannig SPi2 instead.\n");
		tpm = tpm_alloc_device(TPM_DEVICE_SPI, spi_bus, 0);
	}
	if (tpm == NULL) {
		minPrintf("INFO: TPM device not found, disable TrustedBoot.\n");
		return NULL;
	}

	ret = tpm_init(tpm);
	if (ret) {
		minPrintf("ERROR: failed to TPM_Init: %d\n", ret);
		goto tpm_error;
	}
	ret = tpm_startup(tpm, TPM_ST_CLEAR);
	if (ret) {
		minPrintf("ERROR: failed to TPM_Startup: %d\n", ret);
		goto tpm_error;
	}
	ret = tpm_self_test_full(tpm);
	if (ret) {
		minPrintf("ERROR: failed to TPM_SefTestFull: %d\n", ret);
		goto tpm_error;
	}
#if 0
	ret = tpm_continue_self_test(tpm);
	if (ret) {
		minPrintf("ERROR: failed to TPM_ContinueSefTest: %d\n", ret);
		goto tpm_error;
	}
#endif

#ifdef CONFIG_TPM_PHYSICAL_INIT
	if ((tpm_physical_init(tpm) == TPM_PHY_INIT_T_NEED_RESET) &&
	    !physical_init) {
		physical_init++;
		/* reset the TPM device */
		tpm_free_device(tpm, 0);
		goto again;
	}

	ret = tpm_pcr_read(tpm, pcr_index, digest, sizeof(digest));
	if (ret > 0) {
		minPrintf("ERROR: failed to read PCR %d\n", ret);
	}
#endif /* CONFIG_TPM_PHYSICAL_INIT */
	return tpm;

tpm_error:
	tpm_free_device(tpm, 0);
	return NULL;
}

static error_type_t tpm_close(struct tpm_device *tpm)
{
	if (!tpm) return FAIL;
	tpm_free_device(tpm, 1);
	return STATUS_OK;
}

static error_type_t tpm_save_digest(struct tpm_device *tpm, int pcr_index,
	const char *name, const struct image_info *img)
{
	SHA1Context cxt;
	uint8_t digest[TPM_PCR_DIGEST_LENGTH];
	uint8_t result[TPM_PCR_DIGEST_LENGTH];
#ifdef TPM_DEBUG
	uint8_t buffer[TPM_PCR_DIGEST_LENGTH];
#endif /* TPM_DEBUG */
	error_type_t ret = STATUS_OK;

	if (!tpm) {
		return FAIL;
	}

	minPrintf("\nDEBUG: SHA-1 %s: 0x%08x + 0x%08x:\n",
		name, img->dest_addr, img->load_size);
	SHA1Reset(&cxt);
	SHA1Input(&cxt, (const void *)img->dest_addr, img->load_size);
	SHA1Result(&cxt, digest);
	hex_dump(digest, sizeof(digest));

	ret = tpm_extend(tpm, pcr_index, digest, result);
	if (ret) {
		minPrintf("ERROR: PCR%d: failed to TPM_Extend: %d\n",
			pcr_index, ret);
		return FAIL;
	} else {
		minPrintf("DEBUG: PCR%d: response:\n", pcr_index);
		hex_dump(result, sizeof(result));
	}

#ifdef TPM_DEBUG
	ret = tpm_pcr_read(tpm, pcr_index, buffer, sizeof(buffer));
	if (ret) {
		minPrintf("ERROR: PCR%d: failed to TPM_PCRRead: %d\n",
			pcr_index, ret);
		ret = FAIL;
	} else {
		minPrintf("DEBUG: PCR%d:\n", pcr_index);
		hex_dump(buffer, sizeof(buffer));
	}
#endif /* TPM_DEBUG */

	return ret;
}
#endif	/* CONFIG_TPM */

/* Used by vim and some versions of vi: set tabstop=4 shiftwidth=4: */
