/*
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2007-2014, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
*/

//#define DEBUG

#define __APP_C_DEFINES
#include <stdint.h>
#include <stdbool.h>
#include "string.h"

#include "error_types.h"
#include "board_types.h"
#include "rom_types.h"
#include "regAddrs.h"
#include "minPrintf.h"
#include "hex_dump.h"
#include "code_images.h"
#include "R4_multi_package.h"

#include "APMU_regheaders.h"
#include "ciu_regheaders.h"

#include "cpu_api.h"
#include "flash_api.h"
#include "crypto_api.h"
#include "pmu_api.h"
#include "ID_utils.h"
#include "utils.h"
#include "bootCode.h"

#define SECTOR_SIZE             (512)
#define TIMH_BASE_ADDR_FIELD    (0xD100003C)
#define TIM_3_5_00			    (0x30500)	    // Support for Encrypted Boot


inline void force_syncronization(void)
{
	cpu_dcache_writeback_all();
	asm volatile
	(
	    "DSB ;"
	    "ISB ;"
	    :::
	);
}

static error_type_t find_code_image(char *image_name, void **image_info, uint32_t *flash_addr, uint32_t *dest_addr, uint32_t *load_size)
{
	error_type_t ret_val = FAIL;
	rom_types_e rom_type = check_rom_type();
	uint32_t *TIMH       = *((uint32_t **) (((eROM_RevA == rom_type)||(eROM_RevA_bSPI == rom_type)) ? TIMH_BASE_ADDR_FIELD-4 : TIMH_BASE_ADDR_FIELD));
	uint32_t *i;
	uint32_t image_id = SWAP32(*((uint32_t *)image_name));
	uint32_t image_id_offset_size = 24;

	if (TIM_3_5_00 == TIMH[0]) // Check TIM version
	    image_id_offset_size = 27;

	dbg_printf("Looking for  ImageID: %08x \n",image_id);
	i = TIMH+14-image_id_offset_size;
	do {
		i += image_id_offset_size;
		dbg_printf("i: %08x ImageID: %08x \n",(uint32_t)i, *i);
		if ( *i == image_id )
		{
			ret_val = STATUS_OK;
	        *image_info = i;
			*flash_addr = *(i+2);
			*dest_addr = *(i+3);
			*load_size = *(i+4);
			dbg_printf("Found our Image @ i: %08x ImageID: %08x \n",(uint32_t)i, *i);
			break;
		}
	} while ( *(i+1) != 0xFFFFFFFF );

	return ret_val;
}


static void load_wfi_loop(uint32_t code_launch_addr, uint32_t size)
{
	uint32_t 			*code_ptr = (uint32_t*)code_launch_addr;

	memset(code_ptr, 0, size);

// Setup reset vector on AP complex to launch to the loaded code.
	*code_ptr++ = 0xea000006;       // Reset Vector - branch to 0x20
	*code_ptr++ = 0xeafffffe;       // branch to pc-0x8
	*code_ptr++ = 0xeafffffe;       // branch to pc-0x8
	*code_ptr++ = 0xeafffffe;       // branch to pc-0x8
	*code_ptr++ = 0xeafffffe;       // branch to pc-0x8
	*code_ptr++ = 0xeafffffe;       // branch to pc-0x8
	*code_ptr++ = 0xeafffffe;       // branch to pc-0x8
	*code_ptr++ = 0xeafffffe;       // branch to pc-0x8
	*code_ptr++ = 0xe1a00000;       // nop
	*code_ptr++ = 0xe1a00000;       // nop
	*code_ptr++ = 0xe1a00000;       // nop
	*code_ptr++ = 0xe320f003;       // WFI
	*code_ptr++ = 0xe1a00000;       // nop
	*code_ptr++ = 0xeafffffa;       // branch to pc-0x20
}


static void load_ap_reset_vector(uint32_t AP_loaded_code_launch_addr)
{
	uint32_t 			*code_ptr = 0x0;
	CIU_REGS_t *ciu_regs = (CIU_REGS_t*) get_CIU_base();

	ciu_regs->A53_CORE_0_CFG_CTL &= ~(AARCH32_MODE | VINITHI_SET_LOW_VEC); // AArch32 mode, VINITHI=low vectors

// Setup reset vector on AP complex to launch to the loaded code.
	*code_ptr++ = 0xea000006;       // Reset Vector - branch to 0x20
	*code_ptr++ = 0xeafffffe;       // branch to pc-0x8
	*code_ptr++ = 0xeafffffe;       // branch to pc-0x8
	*code_ptr++ = 0xeafffffe;       // branch to pc-0x8
	*code_ptr++ = 0xeafffffe;       // branch to pc-0x8
	*code_ptr++ = 0xeafffffe;       // branch to pc-0x8
	*code_ptr++ = 0xeafffffe;       // branch to pc-0x8
	*code_ptr++ = 0xeafffffe;       // branch to pc-0x8
	*code_ptr++ = 0xe1a00000;       // nop
	*code_ptr++ = 0xe1a00000;       // nop
	*code_ptr++ = 0xe1a00000;       // nop
	*code_ptr++ = 0xe1a00000;       // nop
	*code_ptr++ = 0xe1a00000;       // nop

// Swap the next two lines commenting to either continue or loop in place till debug attach
	*code_ptr++ = 0xe1a00000;       // nop
	//*code_ptr++ = 0xeafffffa;       // branch to pc-0x20

	*code_ptr++ = 0xe59f0000;       // ldr r0, [pc,#8]
	*code_ptr++ = 0xe12fff10;       // bx r0
	*code_ptr++ = AP_loaded_code_launch_addr;  // Load address of AP code loaded from FLASH.
}

static void load_ap_aarch64_launch_addr(uint64_t AP_loaded_code_launch_addr)
{
	CIU_REGS_t *ciu_regs = (CIU_REGS_t*) get_CIU_base();

	ciu_regs->A53_CORE_0_CFG_CTL |= AARCH32_MODE; //AArch64 mode

	ciu_regs->RVBARADDR0L = AP_loaded_code_launch_addr >> 2;
	ciu_regs->RVBARADDR0H = AP_loaded_code_launch_addr >> (32 + 2);
}

typedef void (*lp_core_entry_func_t) (void);

static void launch_lp_core( uint32_t R4_loaded_code_launch_addr )
{
	lp_core_entry_func_t lp_core_entry_func = (lp_core_entry_func_t) R4_loaded_code_launch_addr;

	dbg_printf("Launch R4 core @ 0x%08x\n",R4_loaded_code_launch_addr);

	lp_core_entry_func();
}

static void launch_ap_core0( rom_types_e rom_type, board_types_e board_type )
{
	if (is_Gr2() && is_RevA())
	{
	    // Rev A has polarity of DISTPWRUP backwards, so fix the unused cores
	    PMU_disable (ePMU_DEVICE_CORE3);   // Turn off AP CPU core 3
	    PMU_disable (ePMU_DEVICE_CORE2);   // Turn off AP CPU core 2
	    PMU_disable (ePMU_DEVICE_CORE1);   // Turn off AP CPU core 1
	}
	else if (is_Ge2())
	{
		PMU_disable (ePMU_DEVICE_CORE1);   // Turn off AP CPU core 1
	}

	PMU_enable (ePMU_DEVICE_CORE0);   // Turn on AP CPU core 0
}


bool recovery_boot(void)
{
	// TODO

	// Likely method for this area is to check a fixed well know address in SQU or LCM memory that isn't
	// going to get stomped on by the BootROM or MiniLoader for a "flag" that tells us we are in a recovery
	// boot.  A likely flag value method would be to use the address that the recovery needs to launch from.
	// The OS early stages of a normal boot would initialize the flag address to NULL, so this function would
	// just check against NULL.

	// return (NULL != *well_know_flag_address);
	return false;
}


#ifdef CONFIG_TPM	/* RICOH change */
static error_type_t handle_R4_mp_load(rom_types_e rom_type, struct image_info *R4info)
#else	/* CONFIG_TPM */
static error_type_t handle_R4_mp_load(rom_types_e rom_type, uint32_t *R4_loaded_code_launch_addr )
#endif	/* CONFIG_TPM */
{
	error_type_t ret_val = FAIL;
	void     *image_info;
	uint32_t flash_addr;
	uint32_t dest_addr;
	uint32_t load_size;
	uint32_t *uImage_dest_addr; /* omo */
	uint32_t SDMMCBusSpeed = hwGetSDMMCBusSpeed();
	R4_multi_package_t *R4_mp;
    char     *image_name=NULL;

    flash_addr = 0x00200000;
    dest_addr  = 0x00100000;
    load_size  = 0x00000004;
    uImage_dest_addr = 0x00400000; /* referred at bootselect.c@u-boot */
    ret_val = read_flash(rom_type, flash_addr, dest_addr, load_size, SDMMCBusSpeed);
    if(*(unsigned char *)dest_addr==1)
    {
        image_name = "R4B2";
        *(unsigned char *)uImage_dest_addr = (unsigned char)1; /* referred at bootselect.c@u-boot */
    }
    else
    {
        image_name = "TRDX";
        *(unsigned char *)uImage_dest_addr = (unsigned char)0; /* referred at bootselect.c@u-boot */
    }
    minPrintf("\n*dest_addr = R4:%d LNX:%d, image_name = %s\n", *(unsigned char *)dest_addr, *(unsigned char *)uImage_dest_addr, image_name);

    if (STATUS_OK == (ret_val = find_code_image(image_name, &image_info, &flash_addr, &dest_addr, &load_size)))
	{
	    dbg_printf("\nLoading 'THDX' @ flash_addr: %08x dest_addr: %08x load_size: %08x\n",flash_addr,dest_addr,load_size);
#ifdef CONFIG_TPM	/* RICOH change */
	    R4info->flash_addr = flash_addr;
	    R4info->dest_addr = dest_addr;
	    R4info->load_size = load_size;
#else	/* CONFIG_TPM */
	    *R4_loaded_code_launch_addr = dest_addr;
#endif	/* CONFIG_TPM */
	    R4_mp = (R4_multi_package_t*) dest_addr;

	    ret_val = read_flash(rom_type, flash_addr, dest_addr, SECTOR_SIZE, SDMMCBusSpeed);

	    dbg_hex_dump((void*)dest_addr,64);
	    dbg_printf("Magic Key: %08xh\n", R4_mp->magic_key);

	    if (R4_MP_MAGIC_KEY != R4_mp->magic_key)
	    {
	        // We have a simple R4 image, just load the whole thing in
	        ret_val = read_flash(rom_type, flash_addr, dest_addr, load_size, SDMMCBusSpeed);

	        if ((STATUS_OK == ret_val) && signed_code_mode() && (false == validate_code_signatures(rom_type, image_info)))
	        {
	            load_wfi_loop(dest_addr, load_size);
	        }
	    }
	    else // We have a multi-part image
	    {
	        uint32_t i, mpi_count = R4_mp->entries_count;
	        uint16_t my_chip_id  = get_chip_id();
	        uint8_t  my_chip_rev = get_chip_rev();
	        uint8_t  my_flags    = (is_Si() ? R4_MP_SI_FLAG : R4_MP_FPGA_FLAG); 

	        // Need to load the whole package to verify the SHA on it.
	        if (FAIL == (ret_val = read_flash(rom_type, flash_addr, dest_addr, load_size, SDMMCBusSpeed)))
	            return ret_val;

	        if (signed_code_mode())
	        {
	            if (false == validate_code_signatures(rom_type, image_info))
	            {
	                load_wfi_loop(dest_addr, load_size);
	                return ret_val;
	            }
	        }

	        for (i=0;i < mpi_count;i++)
	        {
	            uint16_t chip_id     = R4_mp->entries[i].chip_id;
	            uint8_t  chip_rev    = R4_mp->entries[i].chip_rev;
	            uint8_t  flags       = R4_mp->entries[i].flags;
	            uint32_t image_offset_addr = R4_mp->entries[i].image_offset_address;

	            dbg_printf("i: %d myID: %04xh myRev: %02xh myFlags: %02xh\n", i, my_chip_id, my_chip_rev, my_flags);
	            dbg_printf("ID: %04xh Rev: %02xh Flags: %02xh\n",chip_id,chip_rev,flags);
	            if ((my_chip_id == chip_id) &&
	                (my_chip_rev == chip_rev) &&
	                (my_flags & flags))
	            {
	                uint32_t load_size   = R4_mp->entries[i].size;
	                dbg_printf("Match! Moving %08xh bytes from: %08xh to %08xh\n",
				   load_size, dest_addr + image_offset_addr, dest_addr);
			dbg_hex_dump((void*)dest_addr+image_offset_addr,64);
			/* Since our custom bare-metal programming environment
			 * leaves much to be desired, we have no memmove and
			 * must rely on the assumptions about the distance
			 * between the buffers and the memcpy implementation
			 * being compatible. */
			memcpy((void *)dest_addr, (void *)dest_addr + image_offset_addr, load_size);
#ifdef CONFIG_TPM	/* RICOH change */
			/* Override load_size by part specific value */
			R4info->load_size = load_size;
#endif	/* CONFIG_TPM */
	                break;
	            }
	        }
	        if (i == mpi_count)
	        {
	            minPrintf("\n\nERROR!! No matching R4 image found - board is not supported!\n"); 
	            minPrintf("ID: 0x%04x Rev: 0x%02x Flags: 0x%02x\n", my_chip_id, my_chip_rev, my_flags); 
	            assert(i < mpi_count);
	        }
	    }
	    dbg_hex_dump((void*)dest_addr,64);
	}
	return ret_val;
}



#ifdef CONFIG_TPM	/* RICOH change */
error_type_t load_code_images(rom_types_e rom_type,
	struct image_info *APinfo, struct image_info *R4info)
#else	/* CONFIG_TPM */
error_type_t load_code_images( rom_types_e rom_type, uint32_t *AP_loaded_code_launch_addr, uint32_t *R4_loaded_code_launch_addr )
#endif	/* CONFIG_TPM */
{
	error_type_t ret_val = FAIL;
	void     *image_info;
	uint32_t flash_addr;
	uint32_t dest_addr;
	uint32_t load_size;

	if (STATUS_OK == find_code_image("OSLO", &image_info, &flash_addr, &dest_addr, &load_size))
	{
		dbg_printf("\nLoading 'OSLO' @ flash_addr: %08x dest_addr: %08x load_size: %08x\n",flash_addr,dest_addr,load_size);
#ifdef CONFIG_TPM	/* RICOH change */
		APinfo->flash_addr = flash_addr;
		APinfo->dest_addr = dest_addr;
		APinfo->load_size = load_size;
#else	/* CONFIG_TPM */
		*AP_loaded_code_launch_addr = dest_addr;
#endif	/* CONFIG_TPM */
		/* Word-align the destination. The bottom bits contain CPU mode parameters */
		dest_addr &= ~0x3;

		ret_val = read_flash(rom_type, flash_addr, dest_addr, load_size, hwGetSDMMCBusSpeed());
	    dbg_hex_dump((void*)dest_addr,64);

	    if (STATUS_OK == ret_val)
	    {
	        if (signed_code_mode() && (false == validate_code_signatures(rom_type, image_info)))
	        {
	            load_wfi_loop(dest_addr, load_size);
	        }

#ifdef CONFIG_TPM	/* RICOH change */
	        ret_val = handle_R4_mp_load(rom_type, R4info);
#else	/* CONFIG_TPM */
	        ret_val = handle_R4_mp_load(rom_type, R4_loaded_code_launch_addr);
#endif	/* CONFIG_TPM */
	    }
	}

	return ret_val;
}


#ifdef CONFIG_TPM	/* RICOH change */
error_type_t locate_USB_code_images(struct image_info *APinfo, struct image_info *R4info)
#else	/* CONFIG_TPM */
error_type_t locate_USB_code_images( uint32_t *AP_loaded_code_launch_addr, uint32_t *R4_loaded_code_launch_addr )
#endif	/* CONFIG_TPM */
{
	error_type_t ret_val = FAIL;
	void     *image_info;
	uint32_t flash_addr;
	uint32_t dest_addr;
	uint32_t load_size;

	if (STATUS_OK == find_code_image("TZII", &image_info, &flash_addr, &dest_addr, &load_size))
	{
		dbg_printf("\nLocated 'TZII' @ flash_addr: %08x dest_addr: %08x load_size: %08x\n",flash_addr,dest_addr,load_size);
#ifdef CONFIG_TPM	/* RICOH change */
		APinfo->flash_addr = flash_addr;
		APinfo->dest_addr = dest_addr;
		APinfo->load_size = load_size;
#else	/* CONFIG_TPM */
		*AP_loaded_code_launch_addr = dest_addr;
#endif	/* CONFIG_TPM */

		if (STATUS_OK == (ret_val = find_code_image("OBMI", &image_info, &flash_addr, &dest_addr, &load_size)))
		{
			dbg_printf("\nLocated 'OBMI' @ flash_addr: %08x dest_addr: %08x load_size: %08x\n",flash_addr,dest_addr,load_size);
#ifdef CONFIG_TPM	/* RICOH change */
			R4info->flash_addr = flash_addr;
			R4info->dest_addr = dest_addr;
			R4info->load_size = load_size;
#else	/* CONFIG_TPM */
			*R4_loaded_code_launch_addr = dest_addr;
#endif	/* CONFIG_TPM */
		}
	}

	return ret_val;
}


#ifdef CONFIG_TPM	/* RICOH change */
error_type_t check_code_images(struct image_info *APinfo, struct image_info *R4info)
#else	/* CONFIG_TPM */
error_type_t check_code_images(uint32_t *AP_loaded_code_launch_addr, uint32_t *R4_loaded_code_launch_addr)
#endif	/* CONFIG_TPM */
{
	//TODO Need to implement
	return FAIL;
}

static int  check_ap_reset_vector(uint32_t  vector)
{
    volatile int doubt=0;
    volatile uint32_t *code_ptr=0x0;

    doubt |= (*code_ptr++ != 0xea000006);       // Reset Vector - branch to 0x20
    doubt |= (*code_ptr++ != 0xeafffffe);       // branch to pc-0x8
    doubt |= (*code_ptr++ != 0xeafffffe);       // branch to pc-0x8
    doubt |= (*code_ptr++ != 0xeafffffe);       // branch to pc-0x8
    doubt |= (*code_ptr++ != 0xeafffffe);       // branch to pc-0x8
    doubt |= (*code_ptr++ != 0xeafffffe);       // branch to pc-0x8
    doubt |= (*code_ptr++ != 0xeafffffe);       // branch to pc-0x8
    doubt |= (*code_ptr++ != 0xeafffffe);       // branch to pc-0x8
    doubt |= (*code_ptr++ != 0xe1a00000);       // nop
    doubt |= (*code_ptr++ != 0xe1a00000);       // nop
    doubt |= (*code_ptr++ != 0xe1a00000);       // nop
    doubt |= (*code_ptr++ != 0xe1a00000);       // nop
    doubt |= (*code_ptr++ != 0xe1a00000);       // nop

    // Swap the next two lines commenting to either continue or loop in place till debug attach
    doubt |= (*code_ptr++ != 0xe1a00000);       // nop
    doubt |= (*code_ptr++ != 0xe59f0000);       // ldr r0, [pc,#8]
    doubt |= (*code_ptr++ != 0xe12fff10);       // bx r0
    doubt |= (*code_ptr++ != vector);           // Load address of AP code loaded from FLASH.

    return(doubt);
}

#define IOPAD146 (*(volatile unsigned long *)0xd401e268) /* KANSI_MCU   GPIOE[18] */
#define IOPAD148 (*(volatile unsigned long *)0xd401e270) /* RBT_REQ_N   GPIOE[20] */
#define IOPAD152 (*(volatile unsigned long *)0xd401e280) /* SDMODE_OK_N GPIOE[24] */

#define GPIOE_BASE ((volatile unsigned long *)0xd4019400)
#define GIO_CLR(base, no)         *(volatile unsigned long *)(base+0x03)=(1<<(no));force_syncronization()
#define GIO_SET(base, no)         *(volatile unsigned long *)(base+0x02)=(1<<(no));force_syncronization()
#define GIO_OUTPUT(base)  *(volatile unsigned long *)(base+0x01)=(0xffffffff);force_syncronization()

static void  system_reboot(void)
{
    /* set iopad */
    IOPAD146 = 0x0001c040;
    IOPAD148 = 0x0001c040;
    IOPAD152 = 0x0001c040;
    force_syncronization();
    Delay(100);

    /* set direction of all gpioe to out */
    GIO_OUTPUT(GPIOE_BASE);
    Delay(100);

    /**
     * reboot procedure
     * 1. 146 to hi(set PSRE bit18)
     * 2. 152 to hi(set PSRE bit24)
     * 3. 146 to lo(clr PCRE bit18)
     * 4. 152 to lo(clr PCRE bit24)
     * 5. 148 to lo(clr PCRE bit20)
     */

    /* 1. set KANSI_MCU to hi */
    GIO_SET(GPIOE_BASE, 18);
    Delay(100000);

    /* 2. set SDMODE_OK_N to hi */
    GIO_SET(GPIOE_BASE, 24);
    Delay(100000);

    /* 3. set KANSI_MCU to lo */
    GIO_CLR(GPIOE_BASE, 18);
    Delay(100000);

    /* 4. set SDMODE_OK_N to lo */
    GIO_CLR(GPIOE_BASE, 24);
    Delay(100000);

    /* 5. set RBT_REQ_N to lo */
    GIO_CLR(GPIOE_BASE, 20);
    Delay(100000);

    return;
}

void launch_code_images( rom_types_e rom_type, board_types_e board_type, uint32_t AP_loaded_code_launch_addr, uint32_t R4_loaded_code_launch_addr )
{
	int ap_aarch64 = 0;
	cpu_disable_interrupts();
    volatile uint32_t *code_ptr=0x00000000;
    volatile uint32_t *code_ptr_=0x00000000;

	/* The bottom two bits of the launch address == 2 indicates AArch64 mode */
	if ((AP_loaded_code_launch_addr & 0x3) == 2) {
		ap_aarch64 = 1;
		load_ap_aarch64_launch_addr(AP_loaded_code_launch_addr &= ~3UL);
	} else if (0 != AP_loaded_code_launch_addr) {
		load_ap_reset_vector(AP_loaded_code_launch_addr);
	} else {
		minPrintf("Cannot launch AP core at 0!\n");
	}

	force_syncronization();

    if(!ap_aarch64 && check_ap_reset_vector(AP_loaded_code_launch_addr))
    {
        minPrintf("\nCan't set AP reset vector correctly. Go reboot...\n");
        minPrintf(" 0x%08x = 0x%08x(0xEA000006)\n", code_ptr, *code_ptr_++);
        minPrintf(" 0x%08x = 0x%08x(0xEAFFFFFE)\n", code_ptr, *code_ptr_++);
        minPrintf(" 0x%08x = 0x%08x(0xEAFFFFFE)\n", code_ptr, *code_ptr_++);
        minPrintf(" 0x%08x = 0x%08x(0xEAFFFFFE)\n", code_ptr, *code_ptr_++);
        minPrintf(" 0x%08x = 0x%08x(0xEAFFFFFE)\n", code_ptr, *code_ptr_++);
        minPrintf(" 0x%08x = 0x%08x(0xEAFFFFFE)\n", code_ptr, *code_ptr_++);
        minPrintf(" 0x%08x = 0x%08x(0xEAFFFFFE)\n", code_ptr, *code_ptr_++);
        minPrintf(" 0x%08x = 0x%08x(0xEAFFFFFE)\n", code_ptr, *code_ptr_++);
        minPrintf(" 0x%08x = 0x%08x(0xE1A00000)\n", code_ptr, *code_ptr_++);
        minPrintf(" 0x%08x = 0x%08x(0xE1A00000)\n", code_ptr, *code_ptr_++);
        minPrintf(" 0x%08x = 0x%08x(0xE1A00000)\n", code_ptr, *code_ptr_++);
        minPrintf(" 0x%08x = 0x%08x(0xE1A00000)\n", code_ptr, *code_ptr_++);
        minPrintf(" 0x%08x = 0x%08x(0xE1A00000)\n", code_ptr, *code_ptr_++);
        minPrintf(" 0x%08x = 0x%08x(0xE1A00000)\n", code_ptr, *code_ptr_++);
        minPrintf(" 0x%08x = 0x%08x(0xE59F0000)\n", code_ptr, *code_ptr_++);
        minPrintf(" 0x%08x = 0x%08x(0xE12FFF10)\n", code_ptr, *code_ptr_++);
        system_reboot();
    }
	minPrintf("\nLaunch AP Core0 @ 0x%08x\n",AP_loaded_code_launch_addr);

	launch_ap_core0(rom_type, board_type);

	launch_lp_core( R4_loaded_code_launch_addr );
}

/* Used by vim and some versions of vi: set tabstop=4 shiftwidth=4: */
