#include <string.h>
#include "error_types.h"
#define NULL		((void *)0)
#include "tpm_api.h"
#include "tpm_spec.h"
#include "tpm_device.h"

#define	__U16(x)	(((x) >> 8) & 0xff), (((x) >> 0) & 0xff)
#define	__U24(x)	(((x) >> 16) & 0xff), __U16(x)
#define	__U32(x)	(((x) >> 24) & 0xff), __U24(x)
static inline u16 __B16(const u8 *b) {
	return ((u16)b[0] << 8) | ((u16)b[1] << 0);
}
static inline u32 __B32(const u8 *b) {
	return ((u32)b[0] << 24) | ((u32)b[1] << 16)
		| ((u32)b[2] << 8) | ((u32)b[3] << 0);
}

#define	TPM_COMMAND_BUFSIZ	256

// 3.1. TPM_Init
int tpm_init(struct tpm_device *tpm)
{
	/* nothing to do */
	return 0;
}

// 3.2. TPM_Startup
//
int tpm_startup(struct tpm_device *tpm, u16 startup)
{
	const u8 command[12] = {
		__U16(TPM_TAG_RQU_COMMAND),	// [0] tag
		__U32(12),			// [2] paramSize
		__U32(TPM_ORD_Startup),		// [6] ordinal
		__U16(startup),			// [10] startupType
	};
	const u32 cmdlen = 12;

	return tpm_execute(tpm, command, cmdlen, NULL, NULL);
}

int tpm_resume(struct tpm_device *tpm)
{
	return tpm_startup(tpm, TPM_ST_STATE);
}

// 4.1. TPM_SelfTestFull
//
int tpm_self_test_full(struct tpm_device *tpm)
{
	const u8 command[10] = {
		__U16(TPM_TAG_RQU_COMMAND),	// [0] tag
		__U32(10),			// [2] paramSize
		__U32(TPM_ORD_SelfTestFull),	// [6] ordinal
	};
	const u32 cmdlen = 10;

	return tpm_execute(tpm, command, cmdlen, NULL, NULL);
}

// 4.2. TPM_ContinueSelfTest
//
int tpm_continue_self_test(struct tpm_device *tpm)
{
	const u8 command[10] = {
		__U16(TPM_TAG_RQU_COMMAND),	// [0] tag
		__U32(10),			// [2] paramSize
		__U32(TPM_ORD_ContinueSelfTest),// [6] ordinal
	};
	const u32 cmdlen = 10;

	return tpm_execute(tpm, command, cmdlen, NULL, NULL);
}

#ifdef CONFIG_TPM_PHYSICAL_INIT
// 5.3. TPM_PhysicalEnable
int tpm_physical_enable(struct tpm_device *tpm)
{
	const u8 command[10] = {
		__U16(TPM_TAG_RQU_COMMAND),	// [0] tag
		__U32(10),			// [2] paramSize
		__U32(TPM_ORD_PhysicalEnable),	// [6] ordinal
	};
	const int cmdlen = 10;

	return tpm_execute(tpm, command, cmdlen, NULL, NULL);
}

// 5.5. TPM_PhysicalSetDeactivated
//
int tpm_physical_set_deactivated(struct tpm_device *tpm, u8 state)
{
	const u8 command[11] = {
		__U16(TPM_TAG_RQU_COMMAND),	// [0] tag
		__U32(11),			// [2] paramSize
		__U32(TPM_ORD_PhysicalSetDeactivated),	// [6] ordinal
		state,				// [10] state
	};
	const int cmdlen = 11;

	return tpm_execute(tpm, command, cmdlen, NULL, NULL);
}

// 6.6. TSC_PhysicalPresence
//
int tpm_tsc_physical_presence(struct tpm_device *tpm, u16 flags)
{
	const u8 command[12] = {
		__U16(TPM_TAG_RQU_COMMAND),	// [0] tag
		__U32(12),			// [2] paramSize
		__U32(TSC_ORD_PhysicalPresence),// [6] ordinal
		__U16(flags),			// [10] physicalPresence
	};
	const int cmdlen = 12;

	return tpm_execute(tpm, command, cmdlen, NULL, NULL);
}
#endif /* CONFIG_TPM_PHYSICAL_INIT */

// 7.1. TPM_GetCapability
//
int tpm_get_capability(struct tpm_device *tpm, u32 cap_area, u32 sub_cap,
		       void *cap, int count)
{
	const u8 command[22] = {
		__U16(TPM_TAG_RQU_COMMAND),	// [0] tag
		__U32(22),			// [2] paramSize
		__U32(TPM_ORD_GetCapability),	// [6] ordinal
		__U32(cap_area),		// [10] capArea
		__U32(4),			// [14] subCapSize
		__U32(sub_cap),			// [18] subCap
	};
	const u32 cmdlen = 22;
	u8 response[TPM_COMMAND_BUFSIZ];
	const int resp_size_offset = 10;
	const int resp_offset = 14;
	u32 response_length = sizeof(response);
	u32 resp_size;
	int ret;

	ret = tpm_execute(tpm, command, cmdlen, response, &response_length);
	if (ret) {
		return ret;
	}
	resp_size = __B32(response + resp_size_offset);
	if (resp_size > response_length || resp_size > count) {
		return FAIL;
	}
	memcpy(cap, response + resp_offset, resp_size);
	return 0;
}

#ifdef CONFIG_TPM_SHA1_HARDWARE
// 13.1. TPM_SHA1Start
int tpm_sha1_start(struct tpm_device *tpm)
{
	const u8 command[10] = {
		__U16(TPM_TAG_RQU_COMMAND),	// [0] tag
		__U32(10),			// [2] paramSize
		__U32(TPM_ORD_SHA1Start),	// [6] ordinal
	};
	const u32 cmdlen = 10;

	return tpm_execute(tpm, command, cmdlen, NULL, NULL);
}

// 13.2. TPM_SHA1Update
int tpm_sha1_update(struct tpm_device *tpm, const void *data, int count)
{
	u8 command[14+64] = {
		__U16(TPM_TAG_RQU_COMMAND),	// [0] tag
		__U32(14+count),		// [2] paramSize
		__U32(TPM_ORD_SHA1Update),	// [6] ordinal
		__U32(count),			// [10] numBytes
						// [14] BYTE[<64]
	};
	const int data_offset = 14;
	const u32 cmdlen = 14 + count;

	if (!data || !count) {
		return 0;
	}
	if (count > 64) {
		return -1;
	}
	memcpy(command + data_offset, data, count);
	return tpm_execute(tpm, command, cmdlen, NULL, NULL);
}

// 13.3. TPM_SHA1Complete
int tpm_sha1_complete(struct tpm_device *tpm, const void *data, int count,
	void *hash, int size)
{
	u8 command[14+64] = {
		__U16(TPM_TAG_RQU_COMMAND),	// [0] tag
		__U32(14+count),		// [2] paramSize
		__U32(TPM_ORD_SHA1Complete),	// [6] ordinal
		__U32(count),			// [10] hashDataSize
						// [14] BYTE[<64]
	};
	const int data_offset = 14;
	const u32 cmdlen = 14 + count;
	u8 response[TPM_COMMAND_BUFSIZ];
	const int value_offset = 10;
	u32 response_length = sizeof(response);
	int ret;

	if (!data || !count) {
		return 0;
	}
	if (count > 64 || size < TPM_PCR_DIGEST_LENGTH) {
		return -1;
	}
	memcpy(command + data_offset, data, count);
	ret = tpm_execute(tpm, command, cmdlen, response, &response_length);
	if (ret) {
		return ret;
	}
	memcpy(hash, response + value_offset, TPM_PCR_DIGEST_LENGTH);
	return 0;
}

int tpm_sha1_digest(struct tpm_device *tpm, const void *mesg, int count,
	void *hash, int size)
{
	const int chunk = 64;
	int ret;

	if (!count) {
		return 0;
	}
	if (!mesg || !hash || size < TPM_PCR_DIGEST_LENGTH) {
		return -1;
	}

	if ((ret = tpm_sha1_start(tpm)) != 0) {
		return ret;
	}
	while (count > chunk) {
	    if ((ret = tpm_sha1_update(tpm, mesg, chunk)) != 0) {
		    return ret;
	    }
	    mesg += chunk;
	    count -= chunk;
	}
	return tpm_sha1_complete(tpm, mesg, count, hash, size);
}
#endif /* CONFIG_TPM_SHA1_HARDWARE */

// 16.1. TPM_Extend
//
int tpm_extend(struct tpm_device *tpm, u32 index,
		const void *in_digest, void *out_digest)
{
	u8 command[14 + TPM_PCR_DIGEST_LENGTH] = {
		__U16(TPM_TAG_RQU_COMMAND),		// [0] tag
		__U32(34),				// [2] paramSize
		__U32(TPM_ORD_Extend),			// [6] orginal
		__U32(index),				// [10] nvIndex
							// [14] inDigest*
	};
	const u32 cmdlen = 14 + TPM_PCR_DIGEST_LENGTH;
	const int in_digest_offset = 14;
	u8 response[10 + TPM_PCR_DIGEST_LENGTH];
	const int out_digest_offset = 10;
	u32 response_length = sizeof(response);
	int ret;

	memcpy(command + in_digest_offset, in_digest, TPM_PCR_DIGEST_LENGTH);
	ret = tpm_execute(tpm, command, cmdlen, response, &response_length);
	if (ret) {
		return ret;
	}
	memcpy(out_digest, response + out_digest_offset, TPM_PCR_DIGEST_LENGTH);
	return 0;
}

// 16.2. TPM_PCRRead
//
int tpm_pcr_read(struct tpm_device *tpm, u32 index, void *data, int count)
{
	const u8 command[14] = {
		__U16(TPM_TAG_RQU_COMMAND),		// [0] tag
		__U32(14),				// [2] paramSize
		__U32(TPM_ORD_PCRRead),			// [6] ordinal
		__U32(index),				// [10] nvIndex
	};
	const u32 cmdlen = 14;
	u8 response[TPM_COMMAND_BUFSIZ];
	const int out_digest_offset = 10;
	u32 response_length = sizeof(response);
	int ret;

	if (count < TPM_PCR_DIGEST_LENGTH) {
		return FAIL;
	}

	ret = tpm_execute(tpm, command, cmdlen, response, &response_length);
	if (ret) {
		return ret;
	}
	memcpy(data, response + out_digest_offset, TPM_PCR_DIGEST_LENGTH);
	return 0;
}

#ifdef CONFIG_TPM_NV_HARDWARE
// 20.1. TPM_NV_DefineSpace
//
int tpm_nv_define_space(struct tpm_device *tpm, u32 index, u32 perm, u32 size)
{
	const u8 command[101] = {
		__U16(TPM_TAG_RQU_AUTH1_COMMAND),	// [0] tag
		__U32(101),			// [2] paramSize:
		__U32(TPM_ORD_NV_DefineSpace),	// [6] orginal
		// TPM_NV_DATA_PUBLIC::
		__U16(TPM_TAG_NV_DATA_PUBLIC),	// [10] tag
		__U32(index),			// [12] nvIndex
		// TPM_NV_DATA_PUBLIC.TPM_PCR_INFO_SHORT pcrInfoRead
		__U16(3),			// [16] sizeOfSelect
		__U24(0),			// [18] pcrSelect
		0x1f,				// [21] TPM_LOCALITY_SELECTION
		0x00, 0x00, 0x00, 0x00,		// [22] TPM_COMPOSITE_HASH
		0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00,
		// TPM_NV_DATA_PUBLIC.TPM_PCR_INFO_SHORT pcrInfoWrite
		__U16(3),			// [42] sizeOfSelect
		__U24(0),			// [44] pcrSelect
		0x1f,				// [47] TPM_LOCALITY_SELECTION
		0x00, 0x00, 0x00, 0x00,		// [48] TPM_COMPOSITE_HASH
		0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00,
		// TPM_NV_DATA_PUBLIC.TPM_NV_ATTRIBUTES permission
		__U16(TPM_NV_PER_OWNERREAD),	// [68] tag
		__U32(perm),			// [70] attributes
		0x00,				// [74] BOOL bReadSTClear
		0x00,				// [75] BOOL bWriteSTClear
		0x00,				// [76] BOOL bWriteDefine
		__U32(size),			// [77] UINT32 dataSize
		// TPM_ENCAUTH::
		0x00, /* repeats 20 bytes */	// [81] encAuth
	};
	const u32 cmdlen = 101;

	return tpm_execute(tpm, command, cmdlen, NULL, NULL);
}

// 20.2. TPM_NV_WriteValue
//
int tpm_nv_write_value(struct tpm_device *tpm, u32 index, const void *data,
		       u32 length)
{
	u8 command[256] = {
		__U32(TPM_TAG_RQU_AUTH1_COMMAND),	// [0] tag
		__U32(22 + length),			// [2] paramSize
		__U32(TPM_ORD_NV_WriteValue),		// [6] orginal
		__U32(index),				// [10] nvIndex
		__U32(0),				// [14] offset
		__U32(length),				// [18] dataSize
							// [22] data*
	};
	const int data_offset = 22;
	const u32 cmdlen = data_offset + length;
	u8 response[TPM_COMMAND_BUFSIZ];
	u32 response_length = sizeof(response);

	memcpy(command + data_offset, data, length);
	return tpm_execute(tpm, command, cmdlen, response, &response_length);
}

// 20.4. TPM_NV_ReadValue
//
int tpm_nv_read_value(struct tpm_device *tpm, u32 index, void *data, u32 count)
{
	const u8 command[22] = {
		__U16(TPM_TAG_RQU_AUTH1_COMMAND),	// [0] tag
		__U32(22),				// [2] paramSize:
		__U32(TPM_ORD_NV_ReadValue),		// [6] orginal
		__U32(index),				// [10] nvIndex
		__U32(0),				// [14] offset
		__U32(count),				// [18] dataSize
	};
	const u32 cmdlen = 22;
	u8 response[TPM_COMMAND_BUFSIZ] = {
		// [0] tag: TPM_TAG_RES_AUTH1_COMMAND
		// [2] paramSize:
		// [6] returnCode:
		// [10] dataSize
		// [14] data ...
	};
	const int data_size_offset = 10;
	const int data_body_offset = 14;
	u32 response_length = sizeof(response);
	u32 data_size;
	int ret;

	ret = tpm_execute(tpm, command, cmdlen, response, &response_length);
	if (ret) {
		return ret;
	}
	data_size = __B32(response + data_size_offset);
	if (data_size > count) {
		return FAIL;
	}
	memcpy(data, response + data_body_offset, data_size);
	return 0;
}

int tpm_nv_set_locked(struct tpm_device *tpm)
{
	return tpm_nv_define_space(tpm, TPM_NV_INDEX_LOCK, 0, 0);
}

int tpm_nv_set_global_lock(struct tpm_device *tpm)
{
	u32 x;
	return tpm_nv_write_value(tpm, TPM_NV_INDEX0, (uint8_t *)&x, 0);
}
#endif /* CONFIG_TPM_NV_HARDWARE */
