#include "error_types.h"
#include "minPrintf.h"
#include "hex_dump.h"
#include "tpm_device.h"

#define TPM_DEBUG	0

static error_type_t tpm_xfer(struct tpm_device *tpm,
	const void *command, int cmdlen, void *response, int *preslen)
{
	int ret;

#if (TPM_DEBUG>0)
	minPrintf("#DEBUG: TPM Command %d bytes:\n", cmdlen);
	hex_dump(command, cmdlen);
#endif
	if (tpm->xfer) {
		ret = tpm->xfer(tpm, command, cmdlen, response, preslen);
		return ret < 0 ? FAIL : STATUS_OK;
	}

	/* sanity check */
	if (!tpm->send || !tpm->recv) {
		minPrintf("ERROR:%s(): internal error.\n", __func__);
		return FAIL;
	}
	ret = tpm->send(tpm, command, cmdlen);
	if (ret < 0) {
		return FAIL;
	}
	ret = tpm->recv(tpm, response, *preslen);
	if (ret < 0) {
		return FAIL;
	}
#if (TPM_DEBUG>0)
	minPrintf("#DEBUG: TPM Response %d bytes:\n", ret);
	hex_dump(response, ret);
#endif
	*preslen = ret;
	return STATUS_OK;
}

static inline int tpm_result(const void *response, int reslen)
{
	const int retcode_offset = 6;
	const u8 *b = (const u8 *)response + retcode_offset;

	if (reslen < retcode_offset + 4) {
		return FAIL;
	}
	return ((u32)b[0] << 24) | ((u32)b[1] << 16)
		| ((u32)b[2] << 8) | ((u32)b[3] << 0);
}

int tpm_execute(struct tpm_device *tpm, const void *command, u32 cmdlen,
	void *response, u32 *preslen)
{
	u8 buffer[256];
	int reslen = response ? *preslen : sizeof(buffer);
	int ret;

	ret = tpm_xfer(tpm, command, cmdlen,
		response ? response : buffer, &reslen);
	if (ret < 0) {
		return ret;
	}
	if (preslen) {
		*preslen = reslen;
	}
	ret = tpm_result(response ? response : buffer, reslen);
#if (TPM_DEBUG>0)
	minPrintf("#DEBUG: TPM response: length=%d code=%d\n", reslen, ret);
#endif
	return ret;
}
