#include <stdint.h>
#include "error_types.h"
#include "spi_api.h"
#include "pmu_api.h"
#include "cpu_api.h"
#include "minPrintf.h"
#include "tpm_spec.h"
#include "tpm_tis.h"
extern u32 hwGetProcSpeed(void);

#define TPM_ACCESS(x)		(0x0000 | ((x) << 12))
#define TPM_INT_ENABLE(x)	(0x0008 | ((x) << 12))
#define TPM_STS(x)		(0x0018 | ((x) << 12))
#define TPM_DATA_FIFO(x)	(0x0024 | ((x) << 12))
#define TPM_DID_VID(x)		(0x0f00 | ((x) << 12))
#define TPM_RID(x)		(0x0f04 | ((x) << 12))

#define TPM_SPI_MAX_FRAMESIZE	0x40	/* 64bytes */
#define TPM_SPI_MAX_WAITBYTES	16
#define TIS_SPI_MAX_CLOCK	20000000	/* 20MHz */

#define MIN(a,b)	((a) < (b) ? (a) : (b))
static inline u32 __B32(const u8 *b) {
	return ((u32)b[0] << 24) | ((u32)b[1] << 16)
		| ((u32)b[2] << 8) | ((u32)b[3] << 0);
}
static inline u32 le32_to_cpu(const u32 val) {
	return val;
}
static inline u32 cpu_to_le32(const u32 val) {
	return val;
}

struct tpm_tis_spi_device {
	struct tis_device tis;
	struct spi_device *spi;
	int busno, csno;
};
static struct tpm_tis_spi_device tis_spi_dev;

static inline unsigned long cpu_time(u32 base)
{
	u32 cc = cpu_get_ccount();
	return (cc > base) ? (cc - base) : (cc + (UINT32_MAX - base));
}
static inline void mdelay(u32 msec)
{
	cpu_spin_delay(msec * 1000);
}

static int tpm_tis_spi_xfer(struct tpm_tis_spi_device *dev,
	u32 addr, const u8 *out, u8 *in, u16 len)
{
	struct spi_device *spi = dev->spi;
	u8 txb[TPM_SPI_MAX_FRAMESIZE + 4];
	u8 rxb[TPM_SPI_MAX_FRAMESIZE + 4];
	u16 xfer_len;
	int i;
	int ret;

	ret = spi_claim_bus(spi);
	if (ret < 0) {
		minPrintf("#ERROR:%s(): spi_claim_bus() failed: %d\n",
			__func__, ret);
		return ret;
	}

	while (len) {
		xfer_len = MIN(len, TPM_SPI_MAX_FRAMESIZE);
		txb[0] = (in ? 0x80 : 0) | (xfer_len - 1);
		txb[1] = 0xd4;
		txb[2] = (addr >> 8) & 0xff;
		txb[3] = (addr >> 0) & 0xff;
		ret = spi_xfer(spi, txb, rxb, 4, SPI_XFER_BEGIN);
		if (ret < 0) {
			minPrintf("#ERROR:%s(): spi_xfer(cmd) failed: %d\n",
				__func__, ret);
			goto spi_release;
		}

		rxb[0] = rxb[3];
		for (i = 0; !(rxb[0] & 0x1) && i < TPM_SPI_MAX_WAITBYTES; i++) {
			ret = spi_xfer(spi, NULL, rxb, 1, 0);
			if (ret) {
				minPrintf("#ERROR:%s(): spi_xfer(wait) failed: %d\n",
					__func__, ret);
				goto spi_release;
			}
		}
		if (i >= TPM_SPI_MAX_WAITBYTES) {
			minPrintf("#ERROR:%s(): spi_xfer() timeout.\n",
				__func__);
			ret = FAIL;
			goto spi_release;
		}

		/* Read/Write Data */
		ret = spi_xfer(spi, out, in, xfer_len, SPI_XFER_END);
		if (ret < 0) {
			minPrintf("#ERROR:%s(): spi_xfer(data) failed: %d\n",
				__func__, ret);
			goto spi_release;
		}

		len -= xfer_len;
		if (out) {
			out += xfer_len;
		}
		if (in) {
			in += xfer_len;
		}
	}

spi_release:
	/* If an error occurred, release the chip by deasserting the CS */
	if (ret < 0) {
		spi_xfer(spi, NULL, NULL, 0, SPI_XFER_END);
	}
	spi_release_bus(spi);
	return ret;
}

static inline int tpm_tis_spi_read(struct tpm_tis_spi_device *dev,
	u32 addr, u8 *in, u16 len)
{
	return tpm_tis_spi_xfer(dev, addr, NULL, in, len);
}

static inline int tpm_tis_spi_read8(struct tpm_tis_spi_device *dev,
	u32 addr, u8 *result)
{
	return tpm_tis_spi_read(dev, addr, result, 1);
}

static inline int tpm_tis_spi_read32(struct tpm_tis_spi_device *dev,
	u32 addr, u32 *result)
{
	u32 result_le;
	int ret;

	ret = tpm_tis_spi_read(dev, addr, (u8 *)&result_le, 4);
	if (ret == 0) {
		*result = le32_to_cpu(result_le);
	}
	return ret;
}

static inline int tpm_tis_spi_write(struct tpm_tis_spi_device *dev,
	u32 addr, const u8 *out, u16 len)
{
	return tpm_tis_spi_xfer(dev, addr, out, NULL, len);
}

static inline int tpm_tis_spi_write8(struct tpm_tis_spi_device *dev,
	u32 addr, const u8 value)
{
	return tpm_tis_spi_write(dev, addr, &value, 1);
}

static inline int tpm_tis_spi_write32(struct tpm_tis_spi_device *dev,
	u32 addr, const u32 value)
{
	u32 value_le = cpu_to_le32(value);
	return tpm_tis_spi_write(dev, addr, (const u8 *)&value_le, 4);
}

static int tpm_tis_spi_check_locality(struct tpm_tis_spi_device *dev, int loc)
{
	const u8 mask = TPM_ACCESS_ACTIVE_LOCALITY | TPM_ACCESS_VALID;
	u8 buf;
	int ret;

	ret = tpm_tis_spi_read8(dev, TPM_ACCESS(loc), &buf);
	if (ret) {
		return ret;
	}
	if ((buf & mask) == mask) {
		dev->tis.locality = loc;
		return 0;
	}
	return 1;
}

static void tpm_tis_spi_release_locality(struct tpm_tis_spi_device *dev,
	int loc, int force)
{
	const u8 mask = TPM_ACCESS_REQUEST_PENDING | TPM_ACCESS_VALID;
	u8 buf;
	int ret;

	ret = tpm_tis_spi_read8(dev, TPM_ACCESS(loc), &buf);
	if (ret < 0) {
		return;
	}
	if (force || (buf & mask) == mask) {
		buf = TPM_ACCESS_ACTIVE_LOCALITY;
		tpm_tis_spi_write8(dev, TPM_ACCESS(loc), buf);
	}
}

static int tpm_tis_spi_request_locality(struct tpm_tis_spi_device *dev, int loc)
{
	u8 buf = TPM_ACCESS_REQUEST_USE;
	int ret;
	unsigned long start_cc, timeout;

	ret = tpm_tis_spi_check_locality(dev, loc);
	if (ret == 0) {
		return ret;
	} else
	if (ret < 0) {
		minPrintf("#ERROR:%s(): failed to get locality%d: %d\n",
			__func__, loc, ret);
		return ret;
	}

	ret = tpm_tis_spi_write8(dev, TPM_ACCESS(loc), buf);
	if (ret) {
		minPrintf("#ERROR:%s(): failed to write locality%d: %d\n",
			__func__, loc, ret);
		return ret;
	}

	start_cc = cpu_time(0);
	timeout = dev->tis.timeout_a;
	do {
		ret = tpm_tis_spi_check_locality(dev, loc);
		if (ret == 0) {
			return ret;
		} else
		if (ret < 0) {
			minPrintf("#ERROR:%s(): failed to get locality%d: %d\n",
				__func__, loc, ret);
			return ret;
		}
		mdelay(TPM_TIMEOUT_MS);
	} while (cpu_time(start_cc) < timeout);

	minPrintf("#ERROR:%s(): failed to get locality%d: timedout\n",
		__func__, loc);
	return FAIL;
}

static inline int tpm_tis_spi_status(struct tpm_tis_spi_device *dev, u8 *status)
{
	return tpm_tis_spi_read8(dev, TPM_STS(dev->tis.locality), status);
}

static int tpm_tis_spi_wait_for_stat(struct tpm_tis_spi_device *dev,
	u8 mask, unsigned long timeout, u8 *status)
{
	unsigned long start_cc;
	int ret;

	start_cc = cpu_time(0);
	do {
		mdelay(TPM_TIMEOUT_MS);
		ret = tpm_tis_spi_status(dev, status);
		if (ret) {
			return ret;
		}
		if ((*status & mask) == mask) {
			return 0;
		}
	} while (cpu_time(start_cc) < timeout);
	return FAIL;
}

static int tpm_tis_spi_get_burstcount(struct tpm_tis_spi_device *dev)
{
	unsigned long start_cc, timeout;
	u32 burstcount, ret;

	start_cc = cpu_time(0);
	timeout = dev->tis.timeout_d;
	do {
		ret = tpm_tis_spi_read32(dev, TPM_STS(dev->tis.locality),
					 &burstcount);
		if (ret) {
			return ret;
		}
		burstcount = (burstcount >> 8) & 0xFFFF;
		if (burstcount) {
			return burstcount;
		}
		mdelay(TPM_TIMEOUT_MS);
	} while (cpu_time(start_cc) < timeout);
	return FAIL;
}

static int tpm_tis_spi_cancel(struct tpm_tis_spi_device *dev)
{
	const u8 command = TPM_STS_COMMAND_READY;
	return tpm_tis_spi_write8(dev, TPM_STS(dev->tis.locality), command);
}

static int tpm_tis_spi_recv_data(struct tpm_tis_spi_device *dev,
	u8 *buf, int count)
{
	int size = 0, burstcnt, len, ret;
	u8 status;

	while (size < count &&
	       tpm_tis_spi_wait_for_stat(dev,
					 TPM_STS_DATA_AVAIL | TPM_STS_VALID,
					 dev->tis.timeout_c, &status) == 0) {
		burstcnt = tpm_tis_spi_get_burstcount(dev);
		if (burstcnt < 0) {
			minPrintf("#ERROR:%s(): unable to get burstcount: %d\n",
				__func__, burstcnt);
			return burstcnt;
		}
		len = MIN(burstcnt, count - size);
		ret = tpm_tis_spi_read(dev, TPM_DATA_FIFO(dev->tis.locality),
				       buf + size, len);
		if (ret < 0) {
			minPrintf("#ERROR:%s(): failed to read data: %d\n",
				__func__, ret);
			return ret;
		}
		size += len;
	}
	return size;
}

static int tpm_tis_spi_recv(struct tpm_device *tpm, void *buf, int count)
{
	struct tpm_tis_spi_device *dev = (struct tpm_tis_spi_device *)tpm;
	int size, expected;

	if (count < TPM_HEADER_LENGTH) {
		minPrintf("#ERROR:%s(): invalid parameter: count=%d\n",
			__func__, count);
		size = FAIL;
		goto out;
	}
	size = tpm_tis_spi_recv_data(dev, buf, TPM_HEADER_LENGTH);
	if (size < TPM_HEADER_LENGTH) {
		minPrintf("#ERROR:%s(): TPM header too short: size %d\n",
			__func__, size);
		size = FAIL;
		goto out;
	}

	expected = __B32(buf + 2);
	if (expected > count) {
		minPrintf("#ERROR:%s(): response too large: expected %d\n",
			__func__, expected);
		size = FAIL;
		goto out;
	}

	buf += TPM_HEADER_LENGTH;
	expected -= TPM_HEADER_LENGTH;
	size += tpm_tis_spi_recv_data(dev, buf, expected);
	if (size < expected) {
		minPrintf("#ERROR:%s(): unable to read remaining bytes.\n",
			__func__);
		size = FAIL;
		goto out;
	}
out:
	tpm_tis_spi_cancel(dev);
	tpm_tis_spi_release_locality(dev, dev->tis.locality, 0);
	return size;
}

static int tpm_tis_spi_send(struct tpm_device *tpm, const void *buf, int len)
{
	struct tpm_tis_spi_device *dev = (struct tpm_tis_spi_device *)tpm;
	u32 i, size;
	u8 status;
	int burstcnt, ret;
	u8 data;

	if (len > TPM_BUFFER_LENGTH) {
		minPrintf("#ERROR:%s(): invalid parameter: length=%d\n",
			__func__, len);
		return FAIL;
	}
	ret = tpm_tis_spi_request_locality(dev, dev->tis.locality);
	if (ret) {
		minPrintf("#ERROR:%s(): request_locality(%d) failed: %d\n",
			__func__, dev->tis.locality, ret);
		return ret;
	}
	ret = tpm_tis_spi_status(dev, &status);
	if (ret) {
		minPrintf("#ERROR:%s(): unable to get TPM status: %d\n",
			__func__, ret);
		return ret;
	}

	if (!(status & TPM_STS_COMMAND_READY)) {
		ret = tpm_tis_spi_cancel(dev);
		if (ret) {
			minPrintf("#ERROR:%s(): could not cancel previous operation\n",
			    __func__);
			goto out_err;
		}

		ret = tpm_tis_spi_wait_for_stat(dev, TPM_STS_COMMAND_READY,
						dev->tis.timeout_b, &status);
		if (ret < 0 || !(status & TPM_STS_COMMAND_READY)) {
			minPrintf("#ERROR:%s(): status %d after wait for stat returned %d\n",
			    __func__, status, ret);
			goto out_err;
		}
	}

	for (i = 0; i < len - 1;) {
		burstcnt = tpm_tis_spi_get_burstcount(dev);
		if (burstcnt < 0) {
			minPrintf("#ERROR:%s(): unable to get burstcount: %d\n",
				__func__, burstcnt);
			return burstcnt;
		}
		size = MIN(len - i - 1, burstcnt);
		ret = tpm_tis_spi_write(dev, TPM_DATA_FIFO(dev->tis.locality),
					buf + i, size);
		if (ret < 0) {
			minPrintf("#ERROR:%s(): failed to write data: %d\n",
				__func__, ret);
			goto out_err;
		}
		i += size;
	}

	ret = tpm_tis_spi_status(dev, &status);
	if (ret) {
		minPrintf("#ERROR:%s(): unable to get TPM status: %d\n",
			__func__, ret);
		goto out_err;
	}

	if ((status & TPM_STS_DATA_EXPECT) == 0) {
		minPrintf("#ERROR:%s(): invalid TPM status: 0x%x\n",
			__func__, status);
		ret = FAIL;
		goto out_err;
	}

	ret = tpm_tis_spi_write(dev, TPM_DATA_FIFO(dev->tis.locality),
				buf + len - 1, 1);
	if (ret) {
		minPrintf("#ERROR:%s(): failed to write remaining data: %d\n",
			__func__, ret);
		goto out_err;
	}

	ret = tpm_tis_spi_status(dev, &status);
	if (ret) {
		minPrintf("#ERROR:%s(): unable to get TPM status: %d\n",
			__func__, ret);
		goto out_err;
	}

	if ((status & TPM_STS_DATA_EXPECT) != 0) {
		minPrintf("#ERROR:%s(): invalid TPM status: 0x%x\n",
			__func__, status);
		ret = FAIL;
		goto out_err;
	}

	data = TPM_STS_GO;
	ret = tpm_tis_spi_write(dev, TPM_STS(dev->tis.locality), &data, 1);
	if (ret) {
		minPrintf("#ERROR:%s(): unable to write TPM status: %d\n",
			__func__, ret);
		goto out_err;
	}
	return len;

out_err:
	tpm_tis_spi_cancel(dev);
	tpm_tis_spi_release_locality(dev, dev->tis.locality, 0);
	return ret;
}

static int tpm_tis_spi_open(struct tpm_device *tpm)
{
	return 0;
}

static void tpm_tis_spi_close(struct tpm_device *tpm)
{
	return;
}

static void tpm_tis_spi_remove(struct tpm_device *tpm, int keep_pmu)
{
	struct tpm_tis_spi_device *dev = (struct tpm_tis_spi_device *)tpm;
	int bus = dev->busno;

	tpm_tis_spi_release_locality(dev, dev->tis.locality, 1);
	spi_free_slave(dev->spi);
	dev->spi = NULL;
	if (!keep_pmu) {
		PMU_disable(ePMU_DEVICE_SPI1 + bus);
	}
}

static int tpm_tis_wait_for_init(struct tpm_tis_spi_device *dev)
{
	const int loc = dev->tis.locality;
	unsigned long start_cc, timeout;
	u8 status;
	int ret;

	start_cc = cpu_time(0);
	timeout = dev->tis.timeout_b;
	do {
		mdelay(TPM_TIMEOUT_MS);
		ret = tpm_tis_spi_read(dev, TPM_ACCESS(loc), &status, 1);
		if (ret) {
			break;
		}
		if (status & TPM_ACCESS_VALID) {
			return 0;
		}
	} while (cpu_time(start_cc) < timeout);
	return FAIL;
}

struct tpm_device *tpm_tis_spi_probe(int bus, int cs)
{
	struct tpm_tis_spi_device *dev = &tis_spi_dev;
	struct tpm_device *tpm = &dev->tis.tpm;
	struct spi_device *spi;
	const unsigned long cpu_mhz = hwGetProcSpeed();	/* MHz */
	int ret;
	u32 did;
	u8 rid;

	PMU_enable(ePMU_DEVICE_SPI1 + bus);

	spi = spi_alloc_slave(bus, cs, TIS_SPI_MAX_CLOCK, 0);
	if (!spi) {
		minPrintf("#ERROR:%s(): spi_alloc_slave(%d, %d) failed.\n",
			__func__, bus, cs);
		return NULL;
	}
	dev->spi = spi;
	dev->busno = bus;
	dev->csno = cs;

	dev->tis.locality = 0;
	dev->tis.timeout_a = TIS_SHORT_TIMEOUT_MS * cpu_mhz * 1000;
	dev->tis.timeout_b = TIS_LONG_TIMEOUT_MS * cpu_mhz * 1000;
	dev->tis.timeout_c = TIS_SHORT_TIMEOUT_MS * cpu_mhz * 1000;
	dev->tis.timeout_d = TIS_SHORT_TIMEOUT_MS * cpu_mhz * 1000;

	ret = tpm_tis_wait_for_init(dev);
	if (ret) {
		minPrintf("#ERROR:%s(): wait_for_init() failed: %d\n",
			__func__, ret);
		goto spi_free;
	}

	ret = tpm_tis_spi_request_locality(dev, dev->tis.locality);
	if (ret) {
		minPrintf("#ERROR:%s(): request_locality(%d) failed: %d\n",
			__func__, dev->tis.locality, ret);
		goto spi_free;
	}

	ret = tpm_tis_spi_read32(dev, TPM_DID_VID(dev->tis.locality), &did);
	if (ret) {
		minPrintf("#ERROR:%s(): failed to get DID/VID: %d\n",
			__func__, ret);
		goto spi_free;
	}

	ret = tpm_tis_spi_read8(dev, TPM_RID(dev->tis.locality), &rid);
	if (ret) {
		minPrintf("#ERROR:%s(): failed toget RID: %d\n",
			__func__, ret);
		goto spi_free;
	}

	minPrintf("INFO: TPM device found (vid:%04x, did:%04x, rid:%02x).\n",
	    __func__, (did >> 0) & 0xffff, (did >> 16) & 0xffff, rid);

	tpm->remove = tpm_tis_spi_remove;
	tpm->open = tpm_tis_spi_open;
	tpm->close = tpm_tis_spi_close;
	tpm->send = tpm_tis_spi_send;
	tpm->recv = tpm_tis_spi_recv;
	tpm->xfer = NULL;
	return tpm;

spi_free:
	spi_free_slave(dev->spi);
	dev->spi = NULL;
	PMU_disable(ePMU_DEVICE_SPI1 + bus);
	return NULL;
}
