/*
 * SPI driver via Pegmatite SSP.
 */
#include "error_types.h"
#include "board_types.h"
#include "regAddrs.h"
/* redefine AP_APMU_APMU_CLKRSTGEN_BASE for Rev.B SoC */
#undef AP_APMU_APMU_CLKRSTGEN_BASE
#define AP_APMU_APMU_CLKRSTGEN_BASE	((AP_APMU_BASE) + 0x400)
#include "cpu_api.h"
#include "spi_api.h"
#include "minPrintf.h"
#define NULL	((void *)0)

#include "spi-pxa2xx.h"
typedef unsigned char	u8;
typedef unsigned short	u16;
typedef unsigned int	u32;

#define SPI_MAX_BUS	3
#define SPI_MAX_DEVICE	3
#define SSP_REFCLOCK	200000000	/* APBUs Clock = 200MHz */

#define SSP_FLUSH_NUM	0x2000
#define SSP_TIMEOUT_DEF	1000

struct spi_master {
	volatile void *sspbase;
	volatile void *pmuaddr;
	struct {
		struct {
			volatile void *addr;
			u32 ini, val;
		} clk, mosi, miso, cs_n;
	} iopad;
};
static const struct spi_master spi_master_context[SPI_MAX_BUS] = {
#define	_IO(x)	((volatile void *)(x))
/* SPi1 */
{
	.sspbase = _IO(AP_AP_APB_SSP0_BASE),
	.pmuaddr = _IO(AP_APMU_APMU_CLKRSTGEN_BASE + 0x05c),
	.iopad = {
	.clk  = {_IO(AP_AP_APB_PADRING_IO_PAD10_BASE), 0x0001C040, 0x00018042},
	.mosi = {_IO(AP_AP_APB_PADRING_IO_PAD11_BASE), 0x0001C040, 0x00018042},
	.miso = {_IO(AP_AP_APB_PADRING_IO_PAD12_BASE), 0x0001C040, 0x00018042},
	.cs_n = {_IO(AP_AP_APB_PADRING_IO_PAD15_BASE), 0x0001C040, 0x00018042},
	},
},
/* SPi2 */
{
	.sspbase = _IO(AP_AP_APB_SSP1_BASE),
	.pmuaddr = _IO(AP_APMU_APMU_CLKRSTGEN_BASE + 0x064),
	.iopad = {
	.clk  = {_IO(AP_AP_APB_PADRING_IO_PAD146_BASE), 0x0001A040, 0x00018841},
	.miso = {_IO(AP_AP_APB_PADRING_IO_PAD147_BASE), 0x0001A040, 0x00018841},
	.mosi = {_IO(AP_AP_APB_PADRING_IO_PAD148_BASE), 0x0001A040, 0x00018841},
	.cs_n = {_IO(AP_AP_APB_PADRING_IO_PAD149_BASE), 0x0001A040, 0x00018841},
	},
},
/* SPi3 */
{
	.sspbase = _IO(AP_AP_APB_SSP2_BASE),
	.pmuaddr = _IO(AP_APMU_APMU_CLKRSTGEN_BASE + 0x06c),
	.iopad = {
	.clk  = {_IO(AP_AP_APB_PADRING_IO_PAD97_BASE),  0x0001A040, 0x00018044},
	.cs_n = {_IO(AP_AP_APB_PADRING_IO_PAD98_BASE),  0x0001A040, 0x00018044},
	.mosi = {_IO(AP_AP_APB_PADRING_IO_PAD99_BASE),  0x0001A040, 0x00018044},
	.miso = {_IO(AP_AP_APB_PADRING_IO_PAD100_BASE), 0x0001A040, 0x00018044},
	},
},
#undef	_IO
};

struct spi_device {
	const struct spi_master *master;
	u16 busnum, chipnum;
	unsigned long max_clk_rate;
	unsigned int cur_clk_div;
	u32 init_cr0;
	u32 init_cr1;
	u32 intr_cr1;
	u32 clear_sr;
	u32 timeout;
	const void *tx;
	void *rx;
};
static struct spi_device spi_device_context[SPI_MAX_DEVICE];

static inline struct spi_device *spi_alloc_device(int bus, int cs)
{
	struct spi_device *spi;
	int i;

	for (i = 0, spi = spi_device_context; i < SPI_MAX_DEVICE; i++, spi++) {
		if (!spi->master) {
			spi->master = &spi_master_context[bus];
			spi->busnum = bus;
			spi->chipnum = cs;
			return spi;
		}
	}
	return NULL;
}
static inline void spi_free_device(struct spi_device *spi)
{
	spi->master = NULL;
	spi->busnum = spi->chipnum = 0;
}

static inline void __out32(volatile void *addr, const u32 value) {
	*((volatile u32 *)addr) = value;
}
static inline u32 __in32(volatile void *addr) {
	return *((volatile u32 *)addr);
}

static inline void ssp_reg_write(struct spi_device *spi,
	const u32 offset, const u32 value)
{
	__out32((void *)spi->master->sspbase + offset, value);
}
static inline u32 ssp_reg_read(struct spi_device *spi, const u32 offset)
{
	return __in32((void *)spi->master->sspbase + offset);
}

static inline void udelay(unsigned long usec) {
	cpu_spin_delay(usec);
}

#if 1
static inline u32 ssp_reg_dump(void *reg)
{
	u32 val = __in32((volatile void *)reg);
	minPrintf("#DEBUG: reg %08x: %08x\n", reg, val);
	return val;
}
#endif

/* Exported */
/*
 * spi_init() -- system initialize of SPI.
 *
 */
void spi_init(void)
{
	static int init_once = 0;

	if (init_once)
		return;
	init_once = 1;
#if 1
	const struct spi_master *m;
	int bus;
	u32 val;

	for (bus = 0, m = spi_master_context; bus < SPI_MAX_BUS; bus++, m++)
	{
		(void)ssp_reg_dump((void *)m->iopad.clk.addr);
		(void)ssp_reg_dump((void *)m->iopad.mosi.addr);
		(void)ssp_reg_dump((void *)m->iopad.miso.addr);
		(void)ssp_reg_dump((void *)m->iopad.cs_n.addr);
		(void)val;
	}
#else
	/* nothing to do */
#endif
}

static void spi_pin_config(struct spi_device *spi, int enable)
{
	const struct spi_master *m = spi->master;

	__out32(m->iopad.clk.addr, enable ? m->iopad.clk.val : m->iopad.clk.ini);
	__out32(m->iopad.mosi.addr, enable ? m->iopad.mosi.val : m->iopad.mosi.ini);
	__out32(m->iopad.miso.addr, enable ? m->iopad.miso.val : m->iopad.miso.ini);
	__out32(m->iopad.cs_n.addr, enable ? m->iopad.cs_n.val : m->iopad.cs_n.ini);
}

/*
 * ssp_get_clk_max() -- figure out SSP clock rate from APMU register.
 *
 *	Raad the APMU SSPxClkConfig_ClkGenStatus register, then
 *	figure out the frequency of SSPx.
 */
static unsigned long ssp_get_clk_max(struct spi_device *spi)
{
	u32 clkconf = __in32((void *)spi->master->pmuaddr);
	unsigned long hidiv = (clkconf >> 16) & 0xff;
	unsigned long lodiv = (clkconf >>  8) & 0xff;
	unsigned long prediv = (clkconf & 0x4) ? ((clkconf >> 24) & 0xff) : 1;
	return SSP_REFCLOCK / (prediv * (hidiv + lodiv));
}

/*
 * ssp_get_clk_div() -- figure out divisor value for SSP.
 */
static unsigned int ssp_get_clk_div(struct spi_device *spi, unsigned long rate)
{
	unsigned long ssp_clk = spi->max_clk_rate;

	if (rate > ssp_clk)
		rate = ssp_clk;
	return (ssp_clk / rate - 1) & 0xfff;
}

/* Exported */
/*
 * spi_alloc_slave() -- allocate SPI slave structure.
 */
struct spi_device *spi_alloc_slave(int bus, int cs, int max_hz, int mode)
{
	struct spi_device *spi;
	const unsigned int bits = 8;

	if (bus < 0 || bus >= SPI_MAX_BUS)
		return NULL;
	if ((spi = spi_alloc_device(bus, cs)) == NULL)
		return NULL;

	spi->max_clk_rate = ssp_get_clk_max(spi);
	spi->cur_clk_div = ssp_get_clk_div(spi, max_hz);
	spi->init_cr0 = SSCR0_FRF_Motorola
		| SSCR0_DSS_DataSize(bits > 16 ? (bits - 16) : bits)
		| (bits > 16 ? SSCR0_EDSS : 0);
	spi->init_cr1 = SSCR1_RFT_RxThresh(RX_THRESH_DEFAULT)
		| SSCR1_TFT_TxThresh(TX_THRESH_DEFAULT);
	spi->intr_cr1 |= SSCR1_FEN;
	spi->clear_sr = SSSR_ROR | SSSR_TINT;

	return spi;
}

/* Exported */
/*
 * spi_free_slave() -- free SPI slave structure.
 */
void spi_free_slave(struct spi_device *spi)
{
	spi_free_device(spi);
}

static int spi_flush(struct spi_device *spi)
{
	int retry = SSP_FLUSH_NUM;
	do {
		if (--retry < 0)
			break;
		while (ssp_reg_read(spi, SSP_SSSR) & SSSR_RNE)
			(void)ssp_reg_read(spi, SSP_SSDR);
	} while(ssp_reg_read(spi, SSP_SSSR) & SSSR_BSY);
	ssp_reg_write(spi, SSP_SSSR, SSSR_ROR);
	return retry;
}

/* Exported */
/*
 * spi_claim_bus() -- accquire exclusive access to SPI bus.
 */
int spi_claim_bus(struct spi_device *spi)
{
	/* setup values of SSP IOPAD registers */
	spi_pin_config(spi, 1);

	/* setup initial values of SSP registers. */
	ssp_reg_write(spi, SSP_SSCR0, spi->init_cr0 | (2 << 8));
	ssp_reg_write(spi, SSP_SSCR1, spi->init_cr1);
	ssp_reg_write(spi, SSP_SSTO, 0);
	ssp_reg_write(spi, SSP_SSPSP, 0);
	if (spi_flush(spi) < 0)
		return -1;
	return 0;
}

/* Exported */
/*
 * spi_release_bus() -- release exclusive access to SPI bus.
 */
void spi_release_bus(struct spi_device *spi)
{
	u32 val;

	/* restore reset values of SSP registers. */
	ssp_reg_write(spi, SSP_SSCR0, 0);
	ssp_reg_write(spi, SSP_SSCR1, 0);
	ssp_reg_write(spi, SSP_SSTO, 0);
	ssp_reg_write(spi, SSP_SSPSP, 0);
	val = ssp_reg_read(spi, SSP_SSSR);
	ssp_reg_write(spi, SSP_SSSR, val & SSSR_BITS_W1C);

	/* restore reset values of SSP IOPAD registers */
	spi_pin_config(spi, 0);
}

/* Exported */
/*
 * spi_cs_activate() -- assert CS signal of SPI device.
 */
void spi_cs_activate(struct spi_device *spi)
{
	u32 val = ssp_reg_read(spi, SSP_SSCR0);
	ssp_reg_write(spi, SSP_SSCR0, val & ~SSCR0_FCTRL);
}

/* Exported */
/*
 * spi_cs_deactivate() -- negate CS signal of SPI device.
 */
void spi_cs_deactivate(struct spi_device *spi)
{
	u32 val = ssp_reg_read(spi, SSP_SSCR0);
	ssp_reg_write(spi, SSP_SSCR0, val | SSCR0_FCTRL);
}

static int spi_data_write(struct spi_device *spi)
{
	int retry = SSP_FLUSH_NUM;
	while ((ssp_reg_read(spi, SSP_SSSR) & SSSR_TNF) == 0) {
		if (--retry < 0) break;
	}
	if (retry < 0) {
		return -1;
	}

	if (spi->tx) {
		ssp_reg_write(spi, SSP_SSDR, *(u8 *)spi->tx);
		++spi->tx;
	} else {
		ssp_reg_write(spi, SSP_SSDR, 0);
	}
	return 0;
}

static int spi_data_read(struct spi_device *spi)
{
	if (spi->rx != NULL) {
		int retry = SSP_FLUSH_NUM;
		while ((ssp_reg_read(spi, SSP_SSSR) & SSSR_RNE) == 0) {
			if (--retry < 0) break;
			udelay(1);
		}
		if (retry < 0) {
			return -1;
		}
		*(u8 *)spi->rx = ssp_reg_read(spi, SSP_SSDR);
		++spi->rx;
	} else {
		while ((ssp_reg_read(spi, SSP_SSSR) & SSSR_RNE) != 0) {
			(void)ssp_reg_read(spi, SSP_SSDR);
		}
	}
	return 0;
}

/* Exported */
/*
 * spi_xfer() - transfer (send/recv) data bytes to/from SPI device.
 *
 * returns:
 *	number of bytes transferred.
 */
int spi_xfer(struct spi_device *spi, const void *txbuf, void *rxbuf, int bytes,
		unsigned long flags)
{
	int ret = 0;
	int count;

	if (flags & SPI_XFER_BEGIN) {
		ssp_reg_write(spi, SSP_SSTO, SSP_TIMEOUT_DEF);
		ssp_reg_write(spi, SSP_SSCR0, spi->init_cr0 | SSCR0_SSE);
		spi_cs_activate(spi);
		ssp_reg_write(spi, SSP_SSCR1, spi->init_cr1 | spi->intr_cr1);
	}

	spi->tx = txbuf;
	spi->rx = rxbuf;
	while (bytes--) {
		ret = spi_data_write(spi);
		if (ret) break;
		count = SSP_FLUSH_NUM;
		while (ssp_reg_read(spi, SSP_SSSR) & SSSR_BSY) {
			if (--count < 0) {
				ret = -1;
				break;
			}
			udelay(1);
		}
		ret = spi_data_read(spi);
		if (ret) break;
	}

	if (flags & SPI_XFER_END) {
		/* Stop SSP */
		ssp_reg_write(spi, SSP_SSSR, spi->clear_sr);
		ssp_reg_write(spi, SSP_SSCR1, spi->init_cr1);
		ssp_reg_write(spi, SSP_SSTO, 0);
		spi_cs_deactivate(spi);
		ssp_reg_write(spi, SSP_SSCR0, spi->init_cr0 & ~SSCR0_SSE);
	}

	return ret;
}
