/*
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
*/

#include "dwApbUart.h"
//#include "bootCode.h"
#include "utils.h"
#include "regAddrs.h"
#include "apb_top_regheaders.h"
//#include "UART_regheaders.h"

#define NUM_UARTS   (4)
#define UART_REGS_t UART1_REGS_t
#define UART_UART_LSR_TEMT_MASK_SHIFT UART1_UART_LSR_TEMT_MASK_SHIFT

static UART_REGS_t *uart_regs[NUM_UARTS] = {
    (UART_REGS_t *)AP_AP_APB_UART1_BASE,
    (UART_REGS_t *)AP_AP_APB_UART2_BASE,
    (UART_REGS_t *)AP_AP_APB_UART3_BASE,
    (UART_REGS_t *)AP_AP_APB_UART4_BASE
};



/**************************************************
 * Function name   : dwApb_uart_init
 *    returns      : none
 *    arg1         : none
 * Created by      : 
 * Date created    : 28.Aug.06
 * Description     : 
 *      Initializes the Designware ARM Adv Periph Bus UART.
 *
 * Notes           : 
 *      See http://leda-design.com/products/designware/docs/iip/DW_apb_uart/latest/doc/dw_apb_uart_db.pdf
 **************************************************/
/* FUNCTION NAME: dwApb_uart_init */
 
/** 
 * \brief
 * 
 * \retval None
 * 
 * \author 
 * 
 * \date 8/28/2006
 *
 *  Note that busSpd is busSpd * 10.  If your busSpd is 8MHz, pass in 80
 **/
void dwApb_uart_init( int uartNum, int baudrate, int busSpd )
{
    uint32_t divisor;


    // wait until the transmit fifo empty bit is set to avoid clobbering data on
    // re-initialization
//    PollReg(&uart_regs[uartNum]->UART_LSR, UART_LSR_TEMT_MASK,UART_LSR_TEMT_MASK,USEC_100,50,EQUALS); 
    PollReg(&uart_regs[uartNum]->UART_LSR, LSR_TEMT,LSR_TEMT,USEC_100,50,EQUALS); 

    uart_regs[uartNum]->union1.UART_IER = 0x40;  // Clear any Int enables that might be set.

    // note that bus speed is actually bus speed * 10 when passed in, so
    // we need to divide by 10 when we are all done.  This lets us pass
    // in 8.3 MHz as 83, and we get better results

    //divisor = (BUS_FREQ_MHZ * 1000000) / (16 * baudrate); simplified below
    divisor = ((busSpd * 6250) + (baudrate/2)) / baudrate;

    // Setup our Baud rate.
    uart_regs[uartNum]->UART_LCR |= LCR_DLAB; // Enable access to DLL & DLH
    uart_regs[uartNum]->union0.UART_DLL = divisor & 0x00FF;
    uart_regs[uartNum]->union1.UART_DLH = (divisor >> 8) & 0x00FF;
    uart_regs[uartNum]->UART_LCR &= ~LCR_DLAB; // Disable access to DLL & DLH


    uart_regs[uartNum]->UART_LCR = (LCR_DLS_8B); // Set 8bit, No Parity, 1 stop

    uart_regs[uartNum]->union2.UART_FCR = (FCR_RFIFORESET | FCR_XFIFORESET | 
                     FCR_TET_QTR   | FCR_RT_HALF |
                     FCR_FIFOENBL );

     uart_regs[uartNum]->UART_MCR = (MCR_RTS | MCR_DTR);    

}

/**************************************************
 * Function name   : dwApb_uart_out
 *    returns      : none
 *    arg1         : single char to squirt out debug serial port
 * Arguments       : uart number, data to write to serial
 * Created by      : 
 * Date created    : 28.Aug.2006
 * Description     : 
 *      
 *      BLOCKING call to squirt a single character out the serial port.
 *
 * Notes           : 
 **************************************************/
/* FUNCTION NAME: dwApb_uart_out */
 
/** 
 * \brief
 * 
 * \param uartNum, c
 * 
 * \retval None
 * 
 * \author 
 * 
 * \date 8/28/2006
 * 
 **/
#define COUNT_1SEC 1000
void dwApb_uart_out( int uartNum, uint8_t c )
{
//   int i;
   
   
   /* NOTE: blocking write with timeout */
   while (1) //for (i=0;i<COUNT_1SEC;i++)  // 20120512 removed timeout -- simulation PIA
   {
        if (UART_UART_LSR_TEMT_MASK_SHIFT(uart_regs[uartNum]->UART_LSR) != 0x00)
        {
            uart_regs[uartNum]->union0.UART_THR = c;
            return;
        }
   }
}
/**************************************************
 * Function name   : dwApb_uart_puts
 *    returns      : string write to serial port
 * Created by      : 
 * Date created    : 18-Oct-06
 * Description     : 
 *
 *      Write a string of characters to the 
 *      ARM DW APB UART.
 *
 * Notes           : 
 **************************************************/

 
 
/* FUNCTION NAME: dwApb_uart_in */
 
/** 
 * \brief
 * 
 * \retval void
 * 
 * \author 
 * 
 * \date 10/18/2006
 * 
 **/
void dwApb_uart_puts( int uartNum, uint8_t *s )
{
    while( *s != 0 )
        dwApb_uart_out( uartNum, *s++ );
}



/**************************************************
 * Function name   : dwApb_uart_in
 *    returns      : character read from serial port
 * Created by      : 
 * Date created    : 17-Nov-04
 * Description     : 
 *
 *      Read a single character from the ARM PL011 UART.
 *
 * Notes           : 
 **************************************************/

 
 
/* FUNCTION NAME: dwApb_uart_in */
 
/** 
 * \brief
 * 
 * \retval uint8_t
 * 
 * \author 
 * 
 * \date 8/28/2006
 * 
 **/
uint8_t dwApb_uart_in( int uartNum )
{
	while (!(uart_regs[uartNum]->UART_LSR & LSR_DR)) {} // Spin lock until there is data ready.

	return ( (uint8_t) (uart_regs[uartNum]->union0.UART_RBR & 0xFF) );
}

void dwApb_uart_disable_rcvr_int( int uartNum )
{
	// This routine disables the Receiver interrupt bit.
	uart_regs[uartNum]->union1.UART_IER &= ~IER_ERBFI; 
}


void dwApb_uart_enable_rcvr_int( int uartNum )
{
	// This routine enables the Receiver interrupt bit.
	uart_regs[uartNum]->union1.UART_IER |= IER_ERBFI; 
}

#if 0
void dwApb_uart_interrupt( int uartNum )
{
	// This routine services the Receiver interrupt.
	DbgSerialMsg msg;
	char *ptr, *endptr;
    uint16_t c;

    // if this is a character timeout interrupt and we don't have valid data available,
    // just read the receiver buffer register to clear the interrupt & exit
    if (((uart->two.IIR & IIR_IID_MASK) == IIR_IID_CT) && (!(uart->UART_LSR & LSR_DR)))
    {
  //      c = (uint8_t) (uart->zero.RBR & 0xFF);
        return;
    }

  //  if (((uart->two.IIR & IIR_IID_MASK) == IIR_IID_RDA) || ((uart->two.IIR & IIR_IID_MASK) == IIR_IID_CT))
	{
        /* read until the uart says it's empty */
        msg.msg = DBG_SERIAL_MSG_RECEIVE; 
        msg.len = 0;
        ptr = msg.str;
        endptr = &msg.str[MAX_SERIAL_DBG_MSG];

   // 	while (uart->UART_LSR & LSR_DR)
		{
  //  		c = (uint8_t) (uart->zero.RBR & 0xFF);

            /* save the read data */
            msg.len++;
            *ptr++ = c & 0x00ff;

            if( ptr >= endptr ) {
                /* send it to the debug serial task */
                dbg_send_message( &msg );
                /* start over */
                ptr = msg.str;
                msg.len = 0;
            }
		}
		if( msg.len ) {
            dbg_send_message( &msg );
        }
	}
}
#endif

