/*
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
*/

//#define DEBUG

#include <stdint.h>
#include <stdbool.h>

#include "pmu_api.h"
#include "pmu_internal.h"
#include "error_types.h"
#include "board_types.h"
#include "minPrintf.h"
#include "ID_utils.h"
#include "pll_api.h"

#define FPGA_D_REVISION_REG     (0xFD000000)

static bool ap_core_enabled = DISABLE;
static uint32_t cpu_cores_enabled = 0;
extern pll_freq_t get_core_pll(void);
extern int get_memory_size(void);

error_type_t PMU_set_state( pmu_unit_t PMU_unit, bool enable )
{
	ENTRY();
	dbg_printf("%s PMU_unit: %d cpu_cores_enabled: %x ap_core_enabled: %d enable: %d\n",__func__,PMU_unit,cpu_cores_enabled,ap_core_enabled,enable);
	switch (PMU_unit)
	{
		case ePMU_DEVICE_MC5:
			pmu_power_device(IPMU_DEVICE_MC5, enable);
			break;

		case ePMU_DEVICE_SMMU:
			pmu_power_device(PMU_DEVICE_SMMU, enable);
			break;

		case ePMU_DEVICE_SPI1:
			pmu_power_device(APMU_DEVICE_SPI, enable);
			break;
		case ePMU_DEVICE_SPI2:
			pmu_power_device(APMU_DEVICE_SPI2, enable);
			break;
		case ePMU_DEVICE_SPI3:
			pmu_power_device(APMU_DEVICE_SPI3, enable);
			break;

		case ePMU_DEVICE_I2C:
			pmu_power_device(APMU_DEVICE_I2C, enable);
			break;

		case ePMU_DEVICE_UARTS:
			pmu_power_device(APMU_DEVICE_UART, enable);
			break;

		case ePMU_DEVICE_CORE0:
		case ePMU_DEVICE_CORE1:
		case ePMU_DEVICE_CORE2:
		case ePMU_DEVICE_CORE3:
			if ((ENABLE == enable) && (ENABLE != ap_core_enabled))
			{
				apmu_power_island( apmu_pi_device_other, enable);
				apmu_power_island( apmu_pi_device_l2, enable);
				ap_core_enabled = ENABLE;
			}

			switch (PMU_unit)
			{
				case ePMU_DEVICE_CORE1:
					apmu_power_island( apmu_pi_device_core1, enable);
					(enable) ? (cpu_cores_enabled |= 1<<1) : (cpu_cores_enabled &= ~(1<<1));
					break;

				case ePMU_DEVICE_CORE2:
					apmu_power_island( apmu_pi_device_core2, enable);
					(enable) ? (cpu_cores_enabled |= 1<<2) : (cpu_cores_enabled &= ~(1<<2));
					break;

				case ePMU_DEVICE_CORE3:
					apmu_power_island( apmu_pi_device_core3, enable);
					(enable) ? (cpu_cores_enabled |= 1<<3) : (cpu_cores_enabled &= ~(1<<3));
					break;

				case ePMU_DEVICE_CORE0:
				default:
					apmu_power_island( apmu_pi_device_core0, enable);
					(enable) ? (cpu_cores_enabled |= 1<<0) : (cpu_cores_enabled &= ~(1<<0));
			}

			if ((DISABLE == enable) && (ENABLE == ap_core_enabled) && (cpu_cores_enabled == 0))
			{
				apmu_power_island( apmu_pi_device_other, enable);
				apmu_power_island( apmu_pi_device_l2, enable);
				ap_core_enabled = DISABLE;
			}
			break;

		default:
			assert(0);
	}
	dbg_printf("%s cpu_cores_enabled: %x ap_core_enabled: %d\n",__func__,cpu_cores_enabled,ap_core_enabled);
	LEAVE();
	return STATUS_OK;
}



error_type_t PMU_enable( pmu_unit_t PMU_unit )
{
	if (is_FPGA())
	{
		if (ePMU_DEVICE_CORE0 == PMU_unit)
		{
			APMU_CLKRSTGEN_REGS_t *apmu_clk_rst_gen_regs = (APMU_CLKRSTGEN_REGS_t *) AP_APMU_APMU_CLKRSTGEN_BASE;

			if (is_Gr2() && is_RevB())
			{
				volatile uint32_t *fpgaD_ID_reg = (uint32_t*) FPGA_D_REVISION_REG;
				uint32_t fpgaD_ID = (*fpgaD_ID_reg & 0xFF); // FPGA D holds the APMU.

				if (fpgaD_ID >= 0x12 )
				{
						// They moved the APMU_CLK_RST_GEN reg between RevB build #9 & #10
						// Build #10s FPGA D ID reg was 0x12.
						apmu_clk_rst_gen_regs = (APMU_CLKRSTGEN_REGS_t*)((uint32_t)apmu_clk_rst_gen_regs + 0x30);
				}
			}
			else if (is_Ge2())
			{
				apmu_clk_rst_gen_regs = (APMU_CLKRSTGEN_REGS_t*)((uint32_t)apmu_clk_rst_gen_regs + 0x230);
			}

			dbg_printf("apmu_clk_rst_gen_regs base: 0x%08x, APCPUMiscControl: 0x%08x\n",
					(uint32_t) apmu_clk_rst_gen_regs,
					(uint32_t) &(apmu_clk_rst_gen_regs->APCPUMiscControl));

			apmu_clk_rst_gen_regs->APCPUMiscControl |= 0x00010000; // Release L2 from reset
			cpu_spin_delay(100);                                   // Have to wait 25 CPU clock cycles minimum
			apmu_clk_rst_gen_regs->APCPUMiscControl |= 0x00050000; // Release AT from reset
			apmu_clk_rst_gen_regs->APCPUMiscControl |= 0x00050001; // Power up CPU 0
			apmu_clk_rst_gen_regs->APCPUMiscControl |= 0x00050011; // Release CPU 0 from reset
		}

		return STATUS_OK;
	}
	else
	{
		return PMU_set_state( PMU_unit, ENABLE );
	}
}



error_type_t PMU_disable( pmu_unit_t PMU_unit )
{
	if (is_FPGA())
	{
		return STATUS_OK;
	}
	else
	{
		return PMU_set_state( PMU_unit, DISABLE );
	}
}

error_type_t setupPMUDividers( board_types_e board_type )
{
	error_type_t ret;
	pmu_div_config_t div_config_1200[] = R4_400_AP_1200_DDR_400_CONFIG;/* RICOH changed */
	pmu_div_config_t div_config_800[] = R4_400_AP_800_DDR_400_CONFIG;/* RICOH changed */
	pmu_div_config_t div_config_1200_ddr4[] = R4_400_AP_1200_DDR_800_CONFIG;/* RICOH changed */
	pmu_div_config_t div_config_ge2[] = TURN_ON_CONFIG_DDR_600_AP_1000;

	if (is_FPGA())
		return STATUS_OK;

	/* Set default PMU clock dividers */
	if (is_Ge2())
		ret = set_PMU_dividers(div_config_ge2, sizeof(div_config_ge2)/sizeof(pmu_div_config_t));
	else
		if (get_core_pll() == eCORE_PLL_800MHZ) {
			set_PMU_dividers(div_config_800, sizeof(div_config_800)/sizeof(pmu_div_config_t));/* RICOH changed */
		} else {
			if (get_memory_size() == 2048) {
				set_PMU_dividers(div_config_1200_ddr4, sizeof(div_config_1200_ddr4)/sizeof(pmu_div_config_t));/* RICOH changed */
			} else {
				set_PMU_dividers(div_config_1200, sizeof(div_config_1200)/sizeof(pmu_div_config_t));/* RICOH changed */
			}
		}

	return ret;
}

/* Used by vim and some versions of vi: set tabstop=4 shiftwidth=4: */
