/*
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
*/

/**
 *
 * \file lpp_apmu.c
 *
 * \brief This header file contains APMU specific routines for
 *  power up/down the power islands.
 *
 **/

#include <stdint.h>
#include <stdbool.h>
#include <string.h>
#include <assert.h>

#include "pmu_internal.h"
#include "cpu_api.h"
#include "regAddrs.h"
#include "APMU_regheaders.h"
#include "ID_utils.h"

//#include "lpp_apmu.h"
//#include "asic.h"

#define APMU_CLKRSTGEN_REG_ADDR_OF(reg)                             \
    (is_Gr2() && is_RevA()) ?                                       \
    &(((APMU_CLKRSTGEN_REGS_t*)AP_APMU_APMU_CLKRSTGEN_BASE)->reg) : \
    &(((APMU_CLKRSTGEN_REGS_REVB_t*)(AP_APMU_APMU_CLKRSTGEN_BASE + 0x200))->reg)

#define DISTPWRUP_VAL ((!(is_Gr2() && is_RevA())) ? 1 : 0)
#define DISTPWRDWN_VAL ((!(is_Gr2() && is_RevA())) ? 0 : 1)

/*
 * APMU Core power island functions
 * !!note: uses macros for Core0 'APMU_MISC_REG_A53CORE0PWRISLAND..' to configure all cores!!
 */
static bool apmu_core_x_assert_reset(volatile uint32_t *picr, volatile uint32_t *pisr, apmu_pi_device_t device, bool power_on)
{
    uint32_t bit = (1<<device);
    volatile uint32_t *misc_control;

    misc_control = APMU_CLKRSTGEN_REG_ADDR_OF(APCPUMiscControl);

    // clear specific cpu bits
    uint32_t mc = *misc_control;
    uint32_t value =    APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUPOR_MASK_SHIFT(mc) & ~bit;
    mc =                APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUPOR_REPLACE_VAL(mc,value);
    value =             APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPU_MASK_SHIFT(mc) & ~bit;
    *misc_control =     APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPU_REPLACE_VAL(mc, value);

    // wait for bit to clear
    while (APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUPOR_MASK_SHIFT(*misc_control) & bit);

    return true;
}
static bool apmu_core_x_deassert_reset(volatile uint32_t *picr, volatile uint32_t *pisr, apmu_pi_device_t device, bool power_on)
{
    uint32_t bit = (1<<device);
    volatile uint32_t *misc_control;

    // This last item is for CPUs only
    // Set the DbgPwrdUp field to let the debugger know the core is powered up.
    *picr = APMU_MISC_REG_A53CORE0PWRISLAND_PICR_DBGPWRDUP_REPLACE_VAL(*picr,1);

    misc_control = APMU_CLKRSTGEN_REG_ADDR_OF(APCPUMiscControl);

    // set specific cpu bits
    uint32_t mc = *misc_control;
    uint32_t value =    APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUPOR_MASK_SHIFT(mc) | bit;
    mc =                APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUPOR_REPLACE_VAL(mc,value);
    value =             APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPU_MASK_SHIFT(mc) | bit;
    *misc_control =     APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPU_REPLACE_VAL(mc, value);

    // wait for bit to set
    while (!(APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUPOR_MASK_SHIFT(*misc_control) & bit));

    return true;
}
static bool apmu_core_x_bulkpwrup(volatile uint32_t *picr, volatile uint32_t *pisr, apmu_pi_device_t device, bool power_on)
{
    // Set the BULKPWRUP field to 01.
    *picr = APMU_MISC_REG_A53CORE0PWRISLAND_PICR_BULKPWRUP_REPLACE_VAL(*picr,1);
    // Wait at least 20 us (no maximum).
    APMU_DELAY_US(20);
#ifndef EMULATION
    // Read the PISR until the BULKSTATUS field is 01.
    while (APMU_MISC_REG_A53CORE0PWRISLAND_PISR_BULKSTATUS_MASK_SHIFT(*pisr) != 1);
#endif
    // Set the BULKPWRUP field to 11.
    *picr = APMU_MISC_REG_A53CORE0PWRISLAND_PICR_BULKPWRUP_REPLACE_VAL(*picr,3);

    // Set the DISTPWRUP field to DISTPWRUP_VAL.
    *picr = APMU_MISC_REG_A53CORE0PWRISLAND_PICR_DISTPWRUP_REPLACE_VAL(*picr,DISTPWRUP_VAL);

#ifndef EMULATION
    // Read the PISR until the BULKSTATUS field is 11.
    while (APMU_MISC_REG_A53CORE0PWRISLAND_PISR_BULKSTATUS_MASK_SHIFT(*pisr) != 3);
#endif
    return true;
}
static bool apmu_core_x_bulkpwrdown(volatile uint32_t *picr, volatile uint32_t *pisr, apmu_pi_device_t device, bool power_on)
{
    // Set the BULKPWRUP field to 0.
    *picr = APMU_MISC_REG_A53CORE0PWRISLAND_PICR_BULKPWRUP_REPLACE_VAL(*picr,0);

    // Set the DISTPWRUP field to DISTPWRDWN_VAL.
    *picr = APMU_MISC_REG_A53CORE0PWRISLAND_PICR_DISTPWRUP_REPLACE_VAL(*picr,DISTPWRDWN_VAL);

    // Read the PISR until the BULKSTATUS field is 00.
    while (APMU_MISC_REG_A53CORE0PWRISLAND_PISR_BULKSTATUS_MASK_SHIFT(*pisr) != 0);
    return true;
}

static bool apmu_core_x_disable_clocks(volatile uint32_t *picr, volatile uint32_t *pisr, apmu_pi_device_t device, bool power_on)
{
    // No functionality for disable clocks, but we do need to check some bits on a power down!!
    uint32_t bit = (1<<device);
    volatile uint32_t *clk_status = APMU_CLKRSTGEN_REG_ADDR_OF(APCPUClkStatus);

    if (!power_on)
    {
        // Before powering down the island, check the corresponding WFI bit in the APCPUClkStatus.StandByWFI field to ensure the core is in WFI
        if ( !APMU_CLKRSTGEN_APCPUCLKSTATUS_STANDBYWFI_MASK_SHIFT(*clk_status) && bit)
        {
            // TBD .... now what?
            APMU_PRINTF("core %#x not in WFI\r\n",bit);
            //APMU_ASSERT(0);
            //return false;
        }

        // Before powering down the island, check the corresponding bit in the APCPUClkStatus.DbgNoPwrDwn field to ensure the core isn't in the process of being debugged.
        if ( APMU_CLKRSTGEN_APCPUCLKSTATUS_DBGNOPWRDOWN_MASK_SHIFT(*clk_status) && bit)
        {
            // TBD .... now what?
            APMU_ASSERT(0);
            return false;
        }
        // Clear the DbgPwrdUp field to let the debugger know you are powering down the core.
        *picr = APMU_MISC_REG_A53CORE0PWRISLAND_PICR_DBGPWRDUP_REPLACE_VAL(*picr,0);
    }
    return true;
}
static bool apmu_core_x_deisolate(volatile uint32_t *picr, volatile uint32_t *pisr, apmu_pi_device_t device, bool power_on)
{
    // Wait 10 us to ensure stable before deisolating.
    APMU_DELAY_US(10);
    // Set the ISOB field to 1 to de-isolate the island.
    *picr = APMU_MISC_REG_A53CORE0PWRISLAND_PICR_ISOB_REPLACE_VAL(*picr,1);
#ifndef EMULATION
    // Read the PISR until the ISOBSTATUS field is 1.
    while (APMU_MISC_REG_A53CORE0PWRISLAND_PISR_ISOB_MASK_SHIFT(*pisr) != 1);
#endif
    // Wait 1 us
    APMU_DELAY_US(1);
    return true;
}
static bool apmu_core_x_isolate(volatile uint32_t *picr, volatile uint32_t *pisr, apmu_pi_device_t device, bool power_on)
{
    // Clear the ISOB field to 0 to isolate the inteface to the island.
    *picr = APMU_MISC_REG_A53CORE0PWRISLAND_PICR_ISOB_REPLACE_VAL(*picr,0);
    // Read the PISR until the ISOBSTATUS field is 0.
    while (APMU_MISC_REG_A53CORE0PWRISLAND_PISR_ISOB_MASK_SHIFT(*pisr) != 0);
    return true;
}

/*
 * APMU Other/Complex power island functions
 */
static bool apmu_other_assert_reset(volatile uint32_t *picr, volatile uint32_t *pisr, apmu_pi_device_t device, bool power_on)
{
    volatile uint32_t *reg = APMU_CLKRSTGEN_REG_ADDR_OF(DDRClkConfig);
    *reg = APMU_CLKRSTGEN_DDRCLKCONFIG_RSTN_REPLACE_VAL(*reg,0);

#ifndef EMULATION
    volatile uint32_t *misc_control = APMU_CLKRSTGEN_REG_ADDR_OF(APCPUMiscControl);
    *misc_control = APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUAT_REPLACE_VAL(*misc_control, 0);
    // wait for bit to clear
    while (APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUAT_MASK_SHIFT(*misc_control) != 0);
#endif
    return true;
}
static bool apmu_other_deassert_reset(volatile uint32_t *picr, volatile uint32_t *pisr, apmu_pi_device_t device, bool power_on)
{
    volatile uint32_t *reg = APMU_CLKRSTGEN_REG_ADDR_OF(DDRClkConfig);
    *reg = APMU_CLKRSTGEN_DDRCLKCONFIG_RSTN_REPLACE_VAL(*reg,1);

    volatile uint32_t *misc_control = APMU_CLKRSTGEN_REG_ADDR_OF(APCPUMiscControl);
    *misc_control = APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUAT_REPLACE_VAL( *misc_control, 1);
    // wait for bit to set
    while (APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUAT_MASK_SHIFT( *misc_control) != 1);
    return true;
}

static bool apmu_other_gate_clock(bool gate)
{
    volatile uint32_t *reg = APMU_CLKRSTGEN_REG_ADDR_OF(APCPUClkControl);
    volatile uint32_t *status = APMU_CLKRSTGEN_REG_ADDR_OF(APCPUClkStatus);

    APMU_PRINTF("Control reg = 0x%08x and Status reg = 0x%08x \r\n",*reg ,*status);
    // gate = false --> enable clocks, stop gating
    // verilog shows 3 clocks gated as same signal:    ca53_CLKIN, ca53_aclk, ca53_atclk
    *reg = APMU_CLKRSTGEN_APCPUCLKCONTROL_CPUCLKGEN_STOP_REPLACE_VAL(*reg,gate);
    // wait for bit
#ifndef EMULATION
    APMU_PRINTF("Control reg = 0x%08x and Status reg = 0x%08x \r\n",*reg ,*status);
    while (APMU_CLKRSTGEN_APCPUCLKSTATUS_CLKGENIDLE_MASK_SHIFT(*status) != gate);
#endif
    return true;
}
static bool apmu_other_enable_clocks(volatile uint32_t *picr, volatile uint32_t *pisr, apmu_pi_device_t device, bool power_on)
{
    return apmu_other_gate_clock(false);
}
static bool apmu_other_disable_clocks(volatile uint32_t *picr, volatile uint32_t *pisr, apmu_pi_device_t device, bool power_on)
{
    return apmu_other_gate_clock(true);
}
static bool apmu_other_deisolate(volatile uint32_t *picr, volatile uint32_t *pisr, apmu_pi_device_t device, bool power_on)
{
    // Set the ISOB field to 1 to de-isolate the island.
    *picr = APMU_MISC_REG_A53OTHERPWRISLAND_PICR_ISOB_REPLACE_VAL(*picr,1);
#ifndef EMULATION
    // Read the PISR until the ISOBSTATUS field is 1.
    while  (APMU_MISC_REG_A53OTHERPWRISLAND_PISR_ISOB_MASK_SHIFT(*pisr) != 1);
#endif
    return true;
}
static bool apmu_other_isolate(volatile uint32_t *picr, volatile uint32_t *pisr, apmu_pi_device_t device, bool power_on)
{
    // Clear the ISOB field to 0 to isolate the inteface to the island.
    *picr = APMU_MISC_REG_A53OTHERPWRISLAND_PICR_ISOB_REPLACE_VAL(*picr,0);
    // Read the PISR until the ISOBSTATUS field is 0.
    while  (APMU_MISC_REG_A53OTHERPWRISLAND_PISR_ISOB_MASK_SHIFT(*pisr) != 0);
    return true;
}

/*
 * APMU L2 power island functions
 */
static bool apmu_l2_assert_reset(volatile uint32_t *picr, volatile uint32_t *pisr, apmu_pi_device_t device, bool power_on)
{
    volatile uint32_t *misc_control = APMU_CLKRSTGEN_REG_ADDR_OF(APCPUMiscControl);

    uint32_t mc = 0;
    mc =            APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUL2_REPLACE_VAL(*misc_control,0);
    *misc_control = APMU_CLKRSTGEN_APCPUMISCCONTROL_CPUL2RSTDISABLE_REPLACE_VAL(mc, 0);
    // wait for bit to clear
    while          (APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUL2_MASK_SHIFT(*misc_control) != 0);
    return true;
}
static bool apmu_l2_deassert_reset(volatile uint32_t *picr, volatile uint32_t *pisr, apmu_pi_device_t device, bool power_on)
{
    volatile uint32_t *misc_control = APMU_CLKRSTGEN_REG_ADDR_OF(APCPUMiscControl);

    // deassert
    uint32_t mc = 0;
    mc =            APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUL2_REPLACE_VAL(*misc_control,1);
    *misc_control = APMU_CLKRSTGEN_APCPUMISCCONTROL_CPUL2RSTDISABLE_REPLACE_VAL(mc, 0);
    // wait for bit to set
    while          (APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUL2_MASK_SHIFT(*misc_control) != 1);
    return true;
}
static bool apmu_l2_bulkpwrup(volatile uint32_t *picr, volatile uint32_t *pisr, apmu_pi_device_t device, bool power_on)
{
    // Set the BULKPWRUP field to 01.
    *picr = APMU_MISC_REG_A53_L2PWRISLAND_PICR_BULKPWRUP_REPLACE_VAL(*picr,1);
    // Wait at least 20 us (no maximum).
    APMU_DELAY_US(20);
#ifndef EMULATION
    // Read the PISR until the BULKSTATUS field is 01.
    while (APMU_MISC_REG_A53_L2PWRISLAND_PISR_BULKSTATUS_MASK_SHIFT(*pisr) != 1);
#endif
    // Set the BULKPWRUP field to 11.
    *picr = APMU_MISC_REG_A53_L2PWRISLAND_PICR_BULKPWRUP_REPLACE_VAL(*picr,3);
#ifndef EMULATION
    // Read the PISR until the BULKSTATUS field is 11.
    while (APMU_MISC_REG_A53_L2PWRISLAND_PISR_BULKSTATUS_MASK_SHIFT(*pisr) != 3);
#endif
    return true;
}
static bool apmu_l2_bulkpwrdown(volatile uint32_t *picr, volatile uint32_t *pisr, apmu_pi_device_t device, bool power_on)
{
    // Set the BULKPWRUP field to 0.
    *picr = APMU_MISC_REG_A53_L2PWRISLAND_PICR_BULKPWRUP_REPLACE_VAL(*picr,0);
    // Wait at least 20 us (no maximum).
    APMU_DELAY_US(20);
    // Read the PISR until the BULKSTATUS field is 00.
    while (APMU_MISC_REG_A53_L2PWRISLAND_PISR_BULKSTATUS_MASK_SHIFT(*pisr) != 0);
    return true;
}


static bool Assert_L2_PDWN(void)
{
    // Place L2 SRAM into lowpower state
    volatile uint32_t *clk_control = APMU_CLKRSTGEN_REG_ADDR_OF(APCPUClkControl);
    volatile uint32_t *sramctrl = &(((APMU_MISC_REG_REGS_t*)AP_APMU_APMU_MISC_REG_BASE)->A53L2_SRAMPdwn);

    // toggle the dummy clock for what ever reason?
    *clk_control = APMU_CLKRSTGEN_APCPUCLKCONTROL_L2RAMDUMMYCLKSEL_REPLACE_VAL(*clk_control,1);
    *clk_control = APMU_CLKRSTGEN_APCPUCLKCONTROL_L2RAMDUMMYCLK_REPLACE_VAL(*clk_control,0);
    *clk_control = APMU_CLKRSTGEN_APCPUCLKCONTROL_L2RAMDUMMYCLK_REPLACE_VAL(*clk_control,1);
    *clk_control = APMU_CLKRSTGEN_APCPUCLKCONTROL_L2RAMDUMMYCLK_REPLACE_VAL(*clk_control,0);
    *clk_control = APMU_CLKRSTGEN_APCPUCLKCONTROL_L2RAMDUMMYCLKSEL_REPLACE_VAL(*clk_control,0);

    // Set bits to 0 to bring out of lowleakage when powering up. Set to 1 to enter lowleakage when powering down
    *sramctrl = APMU_MISC_REG_A53L2_SRAMPDWN_PDWN_REPLACE_VAL(*sramctrl,1);

    return true;
}


static bool DeAssert_L2_PDWN(void)
{
    // Bring L2 SRAM out of lowpower state
    volatile uint32_t *clk_control = APMU_CLKRSTGEN_REG_ADDR_OF(APCPUClkControl);
    volatile uint32_t *sramctrl = &(((APMU_MISC_REG_REGS_t*)AP_APMU_APMU_MISC_REG_BASE)->A53L2_SRAMPdwn);

    // Set bits to 0 to bring out of lowleakage when powering up. Set to 1 to enter lowleakage when powering down
    *sramctrl = APMU_MISC_REG_A53L2_SRAMPDWN_PDWN_REPLACE_VAL(*sramctrl,0);

    // toggle the dummy clock for what ever reason?
    *clk_control = APMU_CLKRSTGEN_APCPUCLKCONTROL_L2RAMDUMMYCLKSEL_REPLACE_VAL(*clk_control,1);
    *clk_control = APMU_CLKRSTGEN_APCPUCLKCONTROL_L2RAMDUMMYCLK_REPLACE_VAL(*clk_control,0);
    *clk_control = APMU_CLKRSTGEN_APCPUCLKCONTROL_L2RAMDUMMYCLK_REPLACE_VAL(*clk_control,1);
    *clk_control = APMU_CLKRSTGEN_APCPUCLKCONTROL_L2RAMDUMMYCLK_REPLACE_VAL(*clk_control,0);
    *clk_control = APMU_CLKRSTGEN_APCPUCLKCONTROL_L2RAMDUMMYCLK_REPLACE_VAL(*clk_control,1);
    *clk_control = APMU_CLKRSTGEN_APCPUCLKCONTROL_L2RAMDUMMYCLK_REPLACE_VAL(*clk_control,0);
    *clk_control = APMU_CLKRSTGEN_APCPUCLKCONTROL_L2RAMDUMMYCLKSEL_REPLACE_VAL(*clk_control,0);

    APMU_DELAY_US(1);//Only needed 20ns
    return true;
}

static bool apmu_L2_change_sram_power(volatile uint32_t *picr, volatile uint32_t *pisr, apmu_pi_device_t device, bool power_on)
{
    if(power_on)
    {
        // Ensure that clock is off
        apmu_other_gate_clock(true);
        DeAssert_L2_PDWN();
    }else
    {
        // Ensure that clock is off
        apmu_other_gate_clock(true);
        Assert_L2_PDWN();
    }
    return true;
}

static bool apmu_L2_fuse_bits_repair(volatile uint32_t *picr, volatile uint32_t *pisr, apmu_pi_device_t device, bool power_on)
{
    volatile uint32_t *repairCtrl = &(((APMU_MISC_REG_REGS_t*)AP_APMU_APMU_MISC_REG_BASE)->RedunRepairCtl); //RepairDone
    *repairCtrl =  APMU_MISC_REG_REDUNREPAIRCTL_NRST_CA53_AXIM_REPLACE_VAL(*repairCtrl,power_on);
    *repairCtrl =  APMU_MISC_REG_REDUNREPAIRCTL_L2_REDUN_START_REPLACE_VAL(*repairCtrl,power_on);
    *repairCtrl =  APMU_MISC_REG_REDUNREPAIRCTL_NRST_L2_REDUN_REPLACE_VAL(*repairCtrl,power_on);

#ifndef EMULATION
    volatile uint32_t *repairStatus = &(((APMU_MISC_REG_REGS_t*)AP_APMU_APMU_MISC_REG_BASE)->RepairDone);
    while (APMU_MISC_REG_REPAIRDONE_APL2_SRBIST_REPAIRDONE_MASK_SHIFT(*repairStatus) != power_on);
#endif
    return true;
}

/*
 * APMU GPU power island functions
 */
static bool apmu_gpu_assert_reset(volatile uint32_t *picr, volatile uint32_t *pisr, apmu_pi_device_t device, bool power_on)
{
    volatile uint32_t *ap_gpu_reg =  APMU_CLKRSTGEN_REG_ADDR_OF(APBusClk_GPUConfig_ClkGenConfig);
    volatile uint32_t *gpu_bus_reg = APMU_CLKRSTGEN_REG_ADDR_OF(GPUBusClkConfig_ClkGenConfig);
    volatile uint32_t *gpu_reg =     APMU_CLKRSTGEN_REG_ADDR_OF(GPUClkConfig_ClkGenConfig);

    *ap_gpu_reg  = APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_RSTN_REPLACE_VAL(*ap_gpu_reg ,0);
    *gpu_bus_reg = APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RSTN_REPLACE_VAL(*gpu_bus_reg,0);
    *gpu_reg = APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RSTN_REPLACE_VAL(*gpu_reg,0);
    // wait for bit to clear
    while (APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RSTN_MASK_SHIFT(*gpu_reg) != 0);
    return true;
}
static bool apmu_gpu_deassert_reset(volatile uint32_t *picr, volatile uint32_t *pisr, apmu_pi_device_t device, bool power_on)
{
    volatile uint32_t *ap_gpu_reg =  APMU_CLKRSTGEN_REG_ADDR_OF(APBusClk_GPUConfig_ClkGenConfig);
    volatile uint32_t *gpu_bus_reg = APMU_CLKRSTGEN_REG_ADDR_OF(GPUBusClkConfig_ClkGenConfig);
    volatile uint32_t *gpu_reg =     APMU_CLKRSTGEN_REG_ADDR_OF(GPUClkConfig_ClkGenConfig);

    *ap_gpu_reg = APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_RSTN_REPLACE_VAL(*ap_gpu_reg,1);
    *gpu_bus_reg = APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RSTN_REPLACE_VAL(*gpu_bus_reg,1);
    *gpu_reg = APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RSTN_REPLACE_VAL(*gpu_reg,1);
    // wait for bit to set
    while (APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RSTN_MASK_SHIFT(*gpu_reg) != 1);
    return true;
}
static bool apmu_gpu_bulkpwrup(volatile uint32_t *picr, volatile uint32_t *pisr, apmu_pi_device_t device, bool power_on)
{
    // Set the BULKPWRUP field to 01.
    *picr = APMU_MISC_REG_GPUPWRISLAND_PICR_BULKPWRUP_REPLACE_VAL(*picr,1);
    // Wait at least 20 us (no maximum).
    APMU_DELAY_US(20);
    // Read the PISR until the BULKSTATUS field is 01.
    while  (APMU_MISC_REG_GPUPWRISLAND_PISR_BULKSTATUS_MASK_SHIFT(*pisr) != 1);
    // Set the BULKPWRUP field to 11.
    *picr = APMU_MISC_REG_GPUPWRISLAND_PICR_BULKPWRUP_REPLACE_VAL(*picr,3);
    // Read the PISR until the BULKSTATUS field is 11.
    while  (APMU_MISC_REG_GPUPWRISLAND_PISR_BULKSTATUS_MASK_SHIFT(*pisr) != 3);
    return true;
}
static bool apmu_gpu_bulkpwrdown(volatile uint32_t *picr, volatile uint32_t *pisr, apmu_pi_device_t device, bool power_on)
{
    // Set the BULKPWRUP field to 0.
    *picr = APMU_MISC_REG_GPUPWRISLAND_PICR_BULKPWRUP_REPLACE_VAL(*picr,0);
    // Wait at least 20 us (no maximum).
    APMU_DELAY_US(20);
    // Read the PISR until the BULKSTATUS field is 00.
    while  (APMU_MISC_REG_GPUPWRISLAND_PISR_BULKSTATUS_MASK_SHIFT(*pisr) != 0);
    return true;
}
static bool apmu_gpu_gate_clock(bool gate)
{
    volatile uint32_t *config = APMU_CLKRSTGEN_REG_ADDR_OF(APBusClk_GPUConfig_ClkGenConfig);
    volatile uint32_t *status = APMU_CLKRSTGEN_REG_ADDR_OF(APBusClk_GPUConfig_ClkGenStatus);
    // gate = false --> enable clocks, stop gating
    // verilog shows 3 clocks gated:    apbusclk_gpu, gpuclk, gpubusclk
    // first: apbusclk_gpu
    uint32_t clk_en = APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(*config);
    uint32_t clk_st = APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status);
    if (!clk_en && !clk_st)
    {
        *config = APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,1);
    }
    *config = APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,!gate);
    // wait for bit
#ifndef EMULATION
    while (APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status) != gate);
#endif
    // now: gpuclk
    config = APMU_CLKRSTGEN_REG_ADDR_OF(GPUClkConfig_ClkGenConfig);
    status = APMU_CLKRSTGEN_REG_ADDR_OF(GPUClkConfig_ClkGenStatus);
    *config = APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,!gate);
    // wait for bit
#ifndef EMULATION
    while (APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status) != gate);
#endif
    // now: gpubusclk
    config = APMU_CLKRSTGEN_REG_ADDR_OF(GPUBusClkConfig_ClkGenConfig);
    status = APMU_CLKRSTGEN_REG_ADDR_OF(GPUBusClkConfig_ClkGenStatus);
    clk_en = APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(*config);
    clk_st = APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status);
    if (!clk_en && !clk_st)
    {
        *config = APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,1);
    }
    *config = APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,!gate);
    // wait for bit
#ifndef EMULATION
    while (APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status) != gate);
#endif
    return true;
}
static bool apmu_gpu_enable_clocks(volatile uint32_t *picr, volatile uint32_t *pisr, apmu_pi_device_t device, bool power_on)
{
    return apmu_gpu_gate_clock(false);
}
static bool apmu_gpu_disable_clocks(volatile uint32_t *picr, volatile uint32_t *pisr, apmu_pi_device_t device, bool power_on)
{
    return apmu_gpu_gate_clock(true);
}
static bool apmu_gpu_deisolate(volatile uint32_t *picr, volatile uint32_t *pisr, apmu_pi_device_t device, bool power_on)
{
    // Set the ISOB field to 1 to de-isolate the island.
    *picr = APMU_MISC_REG_GPUPWRISLAND_PICR_ISOB_REPLACE_VAL(*picr,1);
    // Read the PISR until the ISOBSTATUS field is 1.
    while  (APMU_MISC_REG_GPUPWRISLAND_PISR_ISOB_MASK_SHIFT(*pisr) != 1);
    return true;
}
static bool apmu_gpu_isolate(volatile uint32_t *picr, volatile uint32_t *pisr, apmu_pi_device_t device, bool power_on)
{
    // Clear the ISOB field to 0 to isolate the inteface to the island.
    *picr = APMU_MISC_REG_GPUPWRISLAND_PICR_ISOB_REPLACE_VAL(*picr,0);
    // Read the PISR until the ISOBSTATUS field is 0.
    while  (APMU_MISC_REG_GPUPWRISLAND_PISR_ISOB_MASK_SHIFT(*pisr) != 0);
    return true;
}

static bool apmu_gpu_change_sram_power(volatile uint32_t *picr, volatile uint32_t *pisr, apmu_pi_device_t device, bool power_on)
{
    volatile uint32_t *sramctrl = &(((APMU_MISC_REG_REGS_t*)AP_APMU_APMU_MISC_REG_BASE)->GC400LT_SRAMPdwn_SPCTL);
    // Set bits to 0 to bring out of lowleakage when powering up. Set to 1 to enter lowleakage when powering down
    *sramctrl = APMU_MISC_REG_GC400LT_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(*sramctrl,!power_on);
    *sramctrl = APMU_MISC_REG_GC400LT_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(*sramctrl,!power_on);

    return true;
}

/*
 * APMU empty routine - does nothing very well
 */
static bool apmu_empty(volatile uint32_t *picr, volatile uint32_t *pisr, apmu_pi_device_t device, bool power_on)
{
    // intentionally empty
    return true;
}

typedef bool (*PFN_HANDLER)(volatile uint32_t *picr, volatile uint32_t *pisr, apmu_pi_device_t device, bool power_on);
typedef struct
{
    PFN_HANDLER     assert_resets;
    PFN_HANDLER     deassert_resets;
    PFN_HANDLER     bulkpwrup;
    PFN_HANDLER     bulkpwrdown;
    PFN_HANDLER     enable_clocks;
    PFN_HANDLER     disable_clocks;
    PFN_HANDLER     deisolate;
    PFN_HANDLER     isolate;
    PFN_HANDLER     change_sram_power;
    PFN_HANDLER     fuse_bits_repair;
} power_island_function_table_t;
/*
 * Generic sequence for APMU power island ON:
 * 1) assert resets
 * 2) bulkpwrup
 * 3) enable clocks
 * 4) disable clocks
 * 5) de-isolate
 * 6) enable SRAMs
 * 7) enable clocks
 * 8) de-assert resets
 *
 * Generic sequence for APMU power island OFF:
 * 1) disable clocks
 * 2) Put SRAMs in low power state
 * 3) isolate
 * 4) bulkpwrdown
 *
 * Sequence of power islands also matters. This file does NOT govern the sequence of power islands.
 * To bring up first core:
 * 1) power on A53OTHER
 * 2) power on A53_L2
 * 3) power on A53_Core0...
 *
 * To bring down last core, it appears the proper sequence is:
 * 1) power off A53 OTHER
 * 2) power off A53 L2
 * 3) power off last A53 Core
 */
static power_island_function_table_t pi_handler[] =
    {
/*core 0 */    {.bulkpwrup = apmu_core_x_bulkpwrup, .bulkpwrdown = apmu_core_x_bulkpwrdown, .enable_clocks = apmu_other_enable_clocks , .disable_clocks = apmu_core_x_disable_clocks, .deisolate = apmu_core_x_deisolate, .isolate = apmu_core_x_isolate, .deassert_resets = apmu_core_x_deassert_reset, .assert_resets = apmu_core_x_assert_reset, .change_sram_power = apmu_empty                , .fuse_bits_repair = apmu_empty                   },
/*core 1 */    {.bulkpwrup = apmu_core_x_bulkpwrup, .bulkpwrdown = apmu_core_x_bulkpwrdown, .enable_clocks = apmu_other_enable_clocks , .disable_clocks = apmu_core_x_disable_clocks, .deisolate = apmu_core_x_deisolate, .isolate = apmu_core_x_isolate, .deassert_resets = apmu_core_x_deassert_reset, .assert_resets = apmu_core_x_assert_reset, .change_sram_power = apmu_empty                , .fuse_bits_repair = apmu_empty                   },
/*core 2 */    {.bulkpwrup = apmu_core_x_bulkpwrup, .bulkpwrdown = apmu_core_x_bulkpwrdown, .enable_clocks = apmu_other_enable_clocks , .disable_clocks = apmu_core_x_disable_clocks, .deisolate = apmu_core_x_deisolate, .isolate = apmu_core_x_isolate, .deassert_resets = apmu_core_x_deassert_reset, .assert_resets = apmu_core_x_assert_reset, .change_sram_power = apmu_empty                , .fuse_bits_repair = apmu_empty                   },
/*core 3 */    {.bulkpwrup = apmu_core_x_bulkpwrup, .bulkpwrdown = apmu_core_x_bulkpwrdown, .enable_clocks = apmu_other_enable_clocks , .disable_clocks = apmu_core_x_disable_clocks, .deisolate = apmu_core_x_deisolate, .isolate = apmu_core_x_isolate, .deassert_resets = apmu_core_x_deassert_reset, .assert_resets = apmu_core_x_assert_reset, .change_sram_power = apmu_empty                , .fuse_bits_repair = apmu_empty                   },
/*L2     */    {.bulkpwrup = apmu_l2_bulkpwrup    , .bulkpwrdown = apmu_l2_bulkpwrdown    , .enable_clocks = apmu_other_enable_clocks , .disable_clocks = apmu_other_disable_clocks , .deisolate = apmu_empty           , .isolate = apmu_empty         , .deassert_resets = apmu_l2_deassert_reset    , .assert_resets = apmu_l2_assert_reset    , .change_sram_power = apmu_L2_change_sram_power , .fuse_bits_repair = apmu_L2_fuse_bits_repair     },
/*Other  */    {.bulkpwrup = apmu_core_x_bulkpwrup, .bulkpwrdown = apmu_core_x_bulkpwrdown, .enable_clocks = apmu_other_enable_clocks , .disable_clocks = apmu_other_disable_clocks , .deisolate = apmu_other_deisolate , .isolate = apmu_other_isolate , .deassert_resets = apmu_other_deassert_reset , .assert_resets = apmu_other_assert_reset , .change_sram_power = apmu_empty                , .fuse_bits_repair = apmu_empty                   },
/*GPU    */    {.bulkpwrup = apmu_gpu_bulkpwrup   , .bulkpwrdown = apmu_gpu_bulkpwrdown   , .enable_clocks = apmu_gpu_enable_clocks   , .disable_clocks = apmu_gpu_disable_clocks   , .deisolate = apmu_gpu_deisolate   , .isolate = apmu_gpu_isolate   , .deassert_resets = apmu_gpu_deassert_reset   , .assert_resets = apmu_gpu_assert_reset   , .change_sram_power = apmu_gpu_change_sram_power, .fuse_bits_repair = apmu_empty                   },
    };


bool apmu_power_island(apmu_pi_device_t device, bool power_on)
{
    volatile uint32_t *picr = NULL;
    volatile uint32_t *pisr = NULL;

    APMU_PRINTF("%s: APMU_ASIC_ID = %x, APMU_ASIC_REV\r\n",__FUNCTION__,APMU_ASIC_ID(),APMU_ASIC_REV());
    APMU_PRINTF("%s: device %d; on=%d\r\n",__FUNCTION__,device,power_on);
    switch (device)
    {
    case apmu_pi_device_core0:
        picr = &(((APMU_MISC_REG_REGS_t*)AP_APMU_APMU_MISC_REG_BASE)->A53Core0PwrIsland_PICR);
        pisr = &(((APMU_MISC_REG_REGS_t*)AP_APMU_APMU_MISC_REG_BASE)->A53Core0PwrIsland_PISR);
        break;
    case apmu_pi_device_core1:
        picr = &(((APMU_MISC_REG_REGS_t*)AP_APMU_APMU_MISC_REG_BASE)->A53Core1PwrIsland_PICR);
        pisr = &(((APMU_MISC_REG_REGS_t*)AP_APMU_APMU_MISC_REG_BASE)->A53Core1PwrIsland_PISR);
        break;
    case apmu_pi_device_core2:
        picr = &(((APMU_MISC_REG_REGS_t*)AP_APMU_APMU_MISC_REG_BASE)->A53Core2PwrIsland_PICR);
        pisr = &(((APMU_MISC_REG_REGS_t*)AP_APMU_APMU_MISC_REG_BASE)->A53Core2PwrIsland_PISR);
        break;
    case apmu_pi_device_core3:
        picr = &(((APMU_MISC_REG_REGS_t*)AP_APMU_APMU_MISC_REG_BASE)->A53Core3PwrIsland_PICR);
        pisr = &(((APMU_MISC_REG_REGS_t*)AP_APMU_APMU_MISC_REG_BASE)->A53Core3PwrIsland_PISR);
        break;
    case apmu_pi_device_l2:
        picr = &(((APMU_MISC_REG_REGS_t*)AP_APMU_APMU_MISC_REG_BASE)->A53_L2PwrIsland_PICR);
        pisr = &(((APMU_MISC_REG_REGS_t*)AP_APMU_APMU_MISC_REG_BASE)->A53_L2PwrIsland_PISR);
        break;
    case apmu_pi_device_other:
        picr = &(((APMU_MISC_REG_REGS_t*)AP_APMU_APMU_MISC_REG_BASE)->A53OtherPwrIsland_PICR);
        pisr = &(((APMU_MISC_REG_REGS_t*)AP_APMU_APMU_MISC_REG_BASE)->A53OtherPwrIsland_PISR);
        break;
    case apmu_pi_device_gpu:
        picr = &(((APMU_MISC_REG_REGS_t*)AP_APMU_APMU_MISC_REG_BASE)->GPUPwrIsland_PICR);
        pisr = &(((APMU_MISC_REG_REGS_t*)AP_APMU_APMU_MISC_REG_BASE)->GPUPwrIsland_PISR);
        break;
    default:
        APMU_ASSERT(0);
        return false;
    }
    if (power_on)
    {
        // .../APMU_reg.html#APMU_misc_reg::A53Core0PwrIsland_PICR
        // .../APMU_reg.html#APMU_misc_reg::A53OtherPwrIsland_PICR
        // .../APMU_reg.html#APMU_misc_reg::A53_L2PwrIsland_PICR
        // .../APMU_reg.html#APMU_misc_reg::GPUPwrIsland_PICR
        // using gr2_power.sv verilog for simulations as 'C' code reference. --> islandPowerUp()
        // TBD - do we need to detect if island is already on???
        APMU_PRINTF("assert\r\n");
        pi_handler[device].assert_resets(picr,pisr,device,power_on);
        APMU_PRINTF("bulkup\r\n");
        pi_handler[device].bulkpwrup(picr,pisr,device,power_on);
        APMU_PRINTF("enclk\r\n");
        pi_handler[device].enable_clocks(picr,pisr,device,power_on);
        APMU_PRINTF("wait10us\r\n");
        APMU_DELAY_US(10); // wait 10us to propagate resets
        APMU_PRINTF("disclk\r\n");
        pi_handler[device].disable_clocks(picr,pisr,device,power_on);
        APMU_PRINTF("deiso\r\n");
        pi_handler[device].deisolate(picr,pisr,device,power_on);
        APMU_PRINTF("ensram\r\n");//Bring SRAMs out of low leakage mode
        pi_handler[device].change_sram_power(picr,pisr,device,power_on);
        APMU_PRINTF("enclk\r\n");
        pi_handler[device].enable_clocks(picr,pisr,device,power_on);
        APMU_PRINTF("fuseBitRepair\r\n");//toggle sram repair via fuse bits
        pi_handler[device].fuse_bits_repair(picr,pisr,device,power_on);
        APMU_PRINTF("deass\r\n");
        pi_handler[device].deassert_resets(picr,pisr,device,power_on);
        APMU_PRINTF("exit\r\n");
        return true;
    }
    // using gr2_power.sv verilog for simulations as 'C' code reference.  --> islandPowerDown()
    APMU_PRINTF("disclk\r\n");
    pi_handler[device].disable_clocks(picr,pisr,device,power_on);
    APMU_PRINTF("iso\r\n");
    pi_handler[device].isolate(picr,pisr,device,power_on);
    APMU_PRINTF("dissram\r\n");//place SRAM in low leakage mode
    pi_handler[device].change_sram_power(picr,pisr,device,power_on);
    APMU_PRINTF("bulkd\r\n");
    pi_handler[device].bulkpwrdown(picr,pisr,device,power_on);
    APMU_PRINTF("fuseBitRepair\r\n");//toggle sram repair via fuse bits
    pi_handler[device].fuse_bits_repair(picr,pisr,device,power_on);
    APMU_PRINTF("exit\r\n");
    return true;
}

