/*
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
*/
/**
 *
 * \file UH_PMU_Dividers.h
 *
 **/

#ifndef __UH_PMU_DIVIDERS_H
#define __UH_PMU_DIVIDERS_H

#include <stdint.h>


//--------------------------------------------------------------
// Defines
//--------------------------------------------------------------

/* PMUs */
typedef enum
{
    MPMU_DIV_DOMAIN = 1,
    APMU_DIV_DOMAIN,
    IPMU_DIV_DOMAIN
} pmu_div_domain_t;

/* Clock dividers */
typedef enum
{
    /* MPMU Clock dividers */
    MFPI_DEGLITCH_CLK_DIV = 1,
    HIPWM_CLK_DIV,

    /* APMU Clock dividers */
    CPU_DBG_CLK_DIV,
    TPIU_CLK_DIV,
    AP_BUS_CLK_DIV,
    I2C_CLK1_DIV,
    I2C_CLK2_DIV,
    I2C_CLK3_DIV,
    SSP_CLK1_DIV,
    SSP_CLK2_DIV,
    SSP_CLK3_DIV,
    SMMU_CLK_DIV,
    BCM_CPU_CLK_DIV,
    AP_CPU_CLK_DIV,
    GPU_BUS_CLK_DIV,
    GPU_CLK_DIV,
    SDMMC_CLK_DIV,
    QSPI_CLK_DIV,
    LCDX_CLK_DIV,
    UART0_CLK_DIV,
    UART1_CLK_DIV,
    UART2_CLK_DIV,
    UART3_CLK_DIV,
    LAN_MAC_CLK2X_DIV,
    LAN_MAC_CLK_DIV,
    DDR_CLK2X_DIV,
    DDR_CLK_DIV,

    /* IPMU Clock dividers */
    IPS_BUS_CLK_DIV,
    AUDIO_CLK_PRE_DIV,
    AUDIO_CLK_DIV,
    ADC_TSEN_CLK_DIV,
    FAST_NSS_CLK_DIV,
    NSS_CLK_DIV,
    VCF_CLK_DIV,
    XCPU_CLK_DIV,
    XIO_CLK_DIV,
    LVDS_AFE_REF_CLK_DIV,
    SCAN_CLK_DIV
} pmu_clk_div_t;

/* Clock sources */
typedef enum
{
    DDR_PLL = 1,
    SYSTEM_PLL,
    SYSTEM_PLL_1GHZ,
    CORE_PLL,
    SILABS_PLL,
    SCAN_PLL,
    LVDSAFE_PLL,
    HIPS_CORE_CLK,
    LAN_RX_CLK_PHY,
    LAN_MAC_CLK,
    REF_CLK
} pmu_div_clk_src_t;

typedef struct pmu_div_config_s
{
    pmu_div_domain_t pmu_div_domain;
    pmu_clk_div_t clk_div;
    pmu_div_clk_src_t clk_src;
    uint32_t div_value;
    uint32_t prediven;
    uint32_t fract_num;
    uint32_t fract_denom;
} pmu_div_config_t;

/*------------------------------------------------------------------------
            Function Prototypes
-------------------------------------------------------------------------*/

/**
 *
 * \brief Configures the PMU Clk dividers. This API can configure one or more
 * dividers at a time.
 *
 * \param [in] div_configs     An array of PMU divider configs.
 * \param [in] num_divs        The number of configs in the array.
 *
 * \return [out] int           -1=FAIL, 0=SUCCESS.
 *
 */
int set_PMU_dividers(pmu_div_config_t *div_configs, uint32_t num_divs);

/**
 *
 * \brief Sets the specified PMU Clk divider.
 *
 * \param [in] pmu_div_domain  PMU domain of the divider (MPMU, APMU, or IPMU).
 * \param [in] clk_div         Clock divider to operate on.
 * \param [in] clk_src         Specified clock source.
 * \param [in] div_val         Desired divider value.
 * \param [in] prediven        Pre-divider enable value (if applicable).
 * \param [in] fract_num       Fractional numerator value (if applicable).
 * \param [in] fract_denom     Fractional denominator value (if applicable).
 *
 * \return [out] int           -1=FAIL, 0=SUCCESS.
 *
 */
int PMU_set_clk_div(pmu_div_domain_t pmu_div_domain,
                    pmu_clk_div_t clk_div,
                    pmu_div_clk_src_t clk_src,
                    uint32_t div_value,
                    uint32_t prediven,
                    uint32_t fract_num,
                    uint32_t fract_denom);

#endif //__UH_PMU_DIVIDERS_H
