/*
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
*/

/*------------------------------------------------------------------------
    UH PMU Dividers
------------------------------------------------------------------------*/    

/*------------------------------------------------------------------------
            Includes
-------------------------------------------------------------------------*/
//#include "UTF.h"
#include "regAddrs.h"
#include "MPMU_regheaders.h"
#include "APMU_regheaders.h"
#include "IPMU_regheaders.h"
#include "UH_PMU_Dividers.h"
#include "apb_top_regheaders.h"
#include "pmu_internal.h"
#include "ID_utils.h"


/*------------------------------------------------------------------------
            Defines
-------------------------------------------------------------------------*/
#ifdef DEBUG
#define DUMP_REG(_x_) minPrintf( #_x_ "(0x%08x)=0x%08x\n", &(_x_), reg_val)
#else
#define DUMP_REG(_x_) while(0)
#endif // DEBUG

/*------------------------------------------------------------------------
            Function Prototypes
-------------------------------------------------------------------------*/
static int MPMU_set_clk_div(pmu_clk_div_t clk_div,
                            pmu_div_clk_src_t clk_src,
                            uint32_t div_value,
                            uint32_t prediven,
                            uint32_t fract_num,
                            uint32_t fract_denom);
static int APMU_set_clk_div(pmu_clk_div_t clk_div,
                            pmu_div_clk_src_t clk_src,
                            uint32_t div_value,
                            uint32_t prediven,
                            uint32_t fract_num,
                            uint32_t fract_denom);
static int IPMU_set_clk_div(pmu_clk_div_t clk_div,
                            pmu_div_clk_src_t clk_src,
                            uint32_t div_value,
                            uint32_t prediven,
                            uint32_t fract_num,
                            uint32_t fract_denom);
static void calc_dividers(uint32_t div_value,
                          uint32_t *hi_div,
                          uint32_t *lo_div);

/*------------------------------------------------------------------------
            External Function Prototypes
-------------------------------------------------------------------------*/

/*------------------------------------------------------------------------
            Global Variables
-------------------------------------------------------------------------*/


/*------------------------------------------------------------------------
            set_PMU_dividers
-------------------------------------------------------------------------*/
/**
 *
 * See UH_PMU_Dividers.h for details.
 *
**/
int set_PMU_dividers(pmu_div_config_t *div_configs, uint32_t num_divs)
{
    uint32_t i;
    int ret;

    ret = 0;

    for (i=0; i<num_divs; i++)
    {
        ret = PMU_set_clk_div(div_configs->pmu_div_domain,
                              div_configs->clk_div,
                              div_configs->clk_src,
                              div_configs->div_value,
                              div_configs->prediven,
                              div_configs->fract_num,
                              div_configs->fract_denom);
        if (ret < 0) goto done;
        div_configs++;
    }

done:
    return ret; 
}

/*------------------------------------------------------------------------
            PMU_set_clk_div
-------------------------------------------------------------------------*/
/**
 *
 * See UH_PMU_Dividers.h for details.
 *
**/
int PMU_set_clk_div(pmu_div_domain_t pmu_div_domain,
                    pmu_clk_div_t clk_div,
                    pmu_div_clk_src_t clk_src,
                    uint32_t div_value,
                    uint32_t prediven,
                    uint32_t fract_num,
                    uint32_t fract_denom)
{
    int ret;

    ret = 0;

    switch (pmu_div_domain)
    {
        case MPMU_DIV_DOMAIN:
            ret = MPMU_set_clk_div(clk_div,
                                   clk_src,
                                   div_value,
                                   prediven,
                                   fract_num,
                                   fract_denom);
            break;
        case APMU_DIV_DOMAIN:
            ret = APMU_set_clk_div(clk_div,
                                   clk_src,
                                   div_value,
                                   prediven,
                                   fract_num,
                                   fract_denom);
            break;
        case IPMU_DIV_DOMAIN:
            ret = IPMU_set_clk_div(clk_div,
                                   clk_src,
                                   div_value,
                                   prediven,
                                   fract_num,
                                   fract_denom);
            break;
    }

    return ret;
}

/*------------------------------------------------------------------------
            MPMU_set_clk_div
-------------------------------------------------------------------------*/
/**
 * 
 * \brief Sets the specified MPMU Clk divider.
 *
 * \param [in] clk_div  Clock divider to operate on.
 * \param [in] clk_src  Specified clock source.
 * \param [in] div_val  Desired divider value.
 * \param [in] prediven        Pre-divider enable value (if applicable).
 * \param [in] fract_num       Fractional numerator value (if applicable).
 * \param [in] fract_denom     Fractional denominator value (if applicable).
 *
 * \return [out] int    -1=FAIL, 0=SUCCESS.
 *
 */
static int MPMU_set_clk_div(pmu_clk_div_t clk_div,
                            pmu_div_clk_src_t clk_src,
                            uint32_t div_value,
                            uint32_t prediven,
                            uint32_t fract_num,
                            uint32_t fract_denom)
{
    uint32_t reg_val;
	uint32_t hi_div;
	uint32_t lo_div;
	uint32_t src_sel;
    int ret;
    MPMU_CLKRSTGEN_REGS_t *mpmu_clkrstgen_regs = (MPMU_CLKRSTGEN_REGS_t *)AP_MPMU_MPMU_CLKRSTGEN_BASE;

    ret = 0;
	calc_dividers(div_value, &hi_div, &lo_div);

    switch (clk_div)
    {
        case MFPI_DEGLITCH_CLK_DIV:
            // 0 : 25 MHz reference clock 
            // 1 : Core PLL Clock
            if (REF_CLK == clk_src)
            {
                src_sel = 0;
            }
            else if (CORE_PLL == clk_src)
            {
                src_sel = 1;
            }
            else
            {
                return -1;
            }
            //mfpideglitchclk
            reg_val = mpmu_clkrstgen_regs->MFPIDeglitchClkConfig_ClkGenConfig;
            reg_val = MPMU_CLKRSTGEN_MFPIDEGLITCHCLKCONFIG_CLKGENCONFIG_SRCSEL_REPLACE_VAL(reg_val, src_sel);
            reg_val = MPMU_CLKRSTGEN_MFPIDEGLITCHCLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg_val, hi_div);
            reg_val = MPMU_CLKRSTGEN_MFPIDEGLITCHCLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg_val, lo_div);
            mpmu_clkrstgen_regs->MFPIDeglitchClkConfig_ClkGenConfig = reg_val;
            DUMP_REG(mpmu_clkrstgen_regs->MFPIDeglitchClkConfig_ClkGenConfig);
            break;
        case HIPWM_CLK_DIV:
            // 0 = select the HiPS core generated clock as the source for the hipwm clock
            // 1 = select the System PLL clock as the source for the hipwm clock
            if (HIPS_CORE_CLK == clk_src)
            {
                src_sel = 0;
            }
            else if (SYSTEM_PLL == clk_src)
            {
                src_sel = 1;
            }
            else
            {
                return -1;
            }
            //hipwmclk
            reg_val = mpmu_clkrstgen_regs->HiPWMClkConfig;
            reg_val = MPMU_CLKRSTGEN_HIPWMCLKCONFIG_SRCSEL_REPLACE_VAL(reg_val, src_sel);
            mpmu_clkrstgen_regs->HiPWMClkConfig = reg_val;
            DUMP_REG(mpmu_clkrstgen_regs->HiPWMClkConfig);
            break;
        default:
            return -1;
    }

    return ret;
}

/*------------------------------------------------------------------------
            APMU_set_clk_div
-------------------------------------------------------------------------*/
/**
 * 
 * \brief Sets the specified APMU Clk divider.
 *
 * \param [in] clk_div  Clock divider to operate on.
 * \param [in] clk_src  Specified clock source.
 * \param [in] div_val  Desired divider value.
 * \param [in] prediven        Pre-divider enable value (if applicable).
 * \param [in] fract_num       Fractional numerator value (if applicable).
 * \param [in] fract_denom     Fractional denominator value (if applicable).
 *
 * \return [out] int    -1=FAIL, 0=SUCCESS.
 *
 */

static int APMU_set_clk_div(pmu_clk_div_t clk_div,
                            pmu_div_clk_src_t clk_src,
                            uint32_t div_value,
                            uint32_t prediven,
                            uint32_t fract_num,
                            uint32_t fract_denom)
{
    uint32_t reg_val;
	uint32_t hi_div;
	uint32_t lo_div;
	uint32_t src_sel;
	uint32_t div_sel;
    int ret = 0;
    APMU_CLKRSTGEN_REGS_t *apmu_clkrstgen_regs = (APMU_CLKRSTGEN_REGS_t *)AP_APMU_APMU_CLKRSTGEN_BASE;

	calc_dividers(div_value, &hi_div, &lo_div);

    if (is_NOT_Gr2_RevA())
    {
        switch (clk_div)
        {
            case CPU_DBG_CLK_DIV:
            case TPIU_CLK_DIV:
            case DDR_CLK_DIV:
            case I2C_CLK1_DIV:
            default:
                apmu_clkrstgen_regs = (APMU_CLKRSTGEN_REGS_t *)(AP_APMU_APMU_CLKRSTGEN_BASE + 0x200);
                break;

            case SSP_CLK1_DIV:
                apmu_clkrstgen_regs = (APMU_CLKRSTGEN_REGS_t *)(AP_APMU_APMU_CLKRSTGEN_BASE + 0x228);
                break;

            case AP_BUS_CLK_DIV:
                apmu_clkrstgen_regs = (APMU_CLKRSTGEN_REGS_t *)(AP_APMU_APMU_CLKRSTGEN_BASE + 0x238);
                break;

            case SMMU_CLK_DIV:
            case GPU_BUS_CLK_DIV:
            case GPU_CLK_DIV:
            case SDMMC_CLK_DIV:
            case QSPI_CLK_DIV:
            case LCDX_CLK_DIV:
            case UART0_CLK_DIV:
            case UART1_CLK_DIV:
            case UART2_CLK_DIV:
            case UART3_CLK_DIV:
            case LAN_MAC_CLK2X_DIV:
            case LAN_MAC_CLK_DIV:
            case AP_CPU_CLK_DIV:
                apmu_clkrstgen_regs = (APMU_CLKRSTGEN_REGS_t *)(AP_APMU_APMU_CLKRSTGEN_BASE + 0x230);
                break;

            case BCM_CPU_CLK_DIV:
                apmu_clkrstgen_regs = (APMU_CLKRSTGEN_REGS_t *)(AP_APMU_APMU_CLKRSTGEN_BASE + 0x280);
                break;
        }
    }

    switch (clk_div)
    {
        case CPU_DBG_CLK_DIV:
            // 0 : Core PLL Clock 
            // 1 : DDR PLL Clock
            if (CORE_PLL == clk_src)
            {
                src_sel = 0;
            }
            else if (DDR_PLL == clk_src)
            {
                src_sel = 1;
            }
            else
            {
                return -1;
            }
            reg_val = apmu_clkrstgen_regs->CPUDbgClkConfig;
            reg_val = APMU_CLKRSTGEN_CPUDBGCLKCONFIG_SRCSEL_REPLACE_VAL(reg_val, src_sel);
            reg_val = APMU_CLKRSTGEN_CPUDBGCLKCONFIG_HIDIV_REPLACE_VAL(reg_val, hi_div);
            reg_val = APMU_CLKRSTGEN_CPUDBGCLKCONFIG_LODIV_REPLACE_VAL(reg_val, lo_div);
            apmu_clkrstgen_regs->CPUDbgClkConfig = reg_val;
            DUMP_REG(apmu_clkrstgen_regs->CPUDbgClkConfig);
            break;
        case TPIU_CLK_DIV:
            // 0 : Core PLL Clock 
            // 1 : DDR PLL Clock
            if (CORE_PLL == clk_src)
            {
                src_sel = 0;
            }
            else if (DDR_PLL == clk_src)
            {
                src_sel = 1;
            }
            else
            {
                return -1;
            }
            //tpiuclk    (/1, /2, /3 ... /32) default /1
            reg_val = apmu_clkrstgen_regs->TPIUClkConfig_ClkGenConfig;
            reg_val = APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_SRCSEL_REPLACE_VAL(reg_val, src_sel);
            reg_val = APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg_val, hi_div);
            reg_val = APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg_val, lo_div);
            apmu_clkrstgen_regs->TPIUClkConfig_ClkGenConfig = reg_val;
            DUMP_REG(apmu_clkrstgen_regs->TPIUClkConfig_ClkGenConfig);
            break;
        case AP_BUS_CLK_DIV:
            // 0 : Core PLL Clock 
            // 1 : DDR PLL Clock
            if (CORE_PLL == clk_src)
            {
                src_sel = 0;
            }
            else if (DDR_PLL == clk_src)
            {
                src_sel = 1;
            }
            else
            {
                return -1;
            }
            //apbusclk   (/1, /2, /3 ... /32) default /1
            reg_val = apmu_clkrstgen_regs->APBusClkConfig_ClkGenConfig;
            reg_val = APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_SRCSEL_REPLACE_VAL(reg_val, src_sel);
            reg_val = APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg_val, hi_div);
            reg_val = APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg_val, lo_div);
            apmu_clkrstgen_regs->APBusClkConfig_ClkGenConfig = reg_val;
            DUMP_REG(apmu_clkrstgen_regs->APBusClkConfig_ClkGenConfig);
            break;
        case I2C_CLK1_DIV:
            {
                int i, num_i2c = 1;

                if (is_NOT_Gr2_RevA())
                    num_i2c = 6;

                for (i=0;i < num_i2c;i++)
                {
                    apmu_clkrstgen_regs = (APMU_CLKRSTGEN_REGS_t *)((uint32_t) apmu_clkrstgen_regs + (i * 8));
                    //i2cclk     (apbusclk/N,  N = 1-16)
                    reg_val = apmu_clkrstgen_regs->I2CClkConfig_ClkGenConfig;
                    reg_val = APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg_val, hi_div);
                    reg_val = APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg_val, lo_div);
                    apmu_clkrstgen_regs->I2CClkConfig_ClkGenConfig = reg_val;
                    DUMP_REG(apmu_clkrstgen_regs->I2CClkConfig_ClkGenConfig);
                }
            }
            break;
        case SSP_CLK1_DIV:
            //sspclk     (apbusclk/N,  N = 1-16)
            reg_val = apmu_clkrstgen_regs->SSPClkConfig_ClkGenConfig;
            reg_val = APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg_val, hi_div);
            reg_val = APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg_val, lo_div);
            apmu_clkrstgen_regs->SSPClkConfig_ClkGenConfig = reg_val;
            DUMP_REG(apmu_clkrstgen_regs->SSPClkConfig_ClkGenConfig);
            break;
        case SSP_CLK2_DIV:
	{
            //sspclk     (apbusclk/N,  N = 1-16)
	    APMU_CLKRSTGEN_REGS_REVB_t *apmu_clkrstgen_revb =
		(APMU_CLKRSTGEN_REGS_REVB_t *)apmu_clkrstgen_regs;
            reg_val = apmu_clkrstgen_revb->SSP2ClkConfig_ClkGenConfig;
            reg_val = APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg_val, hi_div);
            reg_val = APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg_val, lo_div);
            apmu_clkrstgen_revb->SSP2ClkConfig_ClkGenConfig = reg_val;
            DUMP_REG(apmu_clkrstgen_revb->SSP2ClkConfig_ClkGenConfig);
            break;
	}
        case SSP_CLK3_DIV:
	{
            //sspclk     (apbusclk/N,  N = 1-16)
	    APMU_CLKRSTGEN_REGS_REVB_t *apmu_clkrstgen_revb =
		(APMU_CLKRSTGEN_REGS_REVB_t *)apmu_clkrstgen_regs;
            reg_val = apmu_clkrstgen_revb->SSP3ClkConfig_ClkGenConfig;
            reg_val = APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg_val, hi_div);
            reg_val = APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg_val, lo_div);
            apmu_clkrstgen_revb->SSP3ClkConfig_ClkGenConfig = reg_val;
            DUMP_REG(apmu_clkrstgen_revb->SSP3ClkConfig_ClkGenConfig);
            break;
	}
        case SMMU_CLK_DIV:
            // 0 : Core PLL Clock 
            // 1 : DDR PLL Clock
            if (CORE_PLL == clk_src)
            {
                src_sel = 0;
            }
            else if (DDR_PLL == clk_src)
            {
                src_sel = 1;
            }
            else
            {
                return -1;
            }
            //smmuclk    (/1, /2, /3 ... /64)  default /1
            reg_val = apmu_clkrstgen_regs->SMMUClkConfig_ClkGenConfig;
            reg_val = APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_SRCSEL_REPLACE_VAL(reg_val, src_sel);
            reg_val = APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg_val, hi_div);
            reg_val = APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg_val, lo_div);
            reg_val = APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_PREDIVEN_REPLACE_VAL(reg_val, prediven);
            apmu_clkrstgen_regs->SMMUClkConfig_ClkGenConfig = reg_val;
            DUMP_REG(apmu_clkrstgen_regs->SMMUClkConfig_ClkGenConfig);
            break;
        case BCM_CPU_CLK_DIV:
            // 0 : Core PLL Clock 
            // 1 : DDR PLL Clock
            if (CORE_PLL == clk_src)
            {
                src_sel = 0;
            }
            else if (DDR_PLL == clk_src)
            {
                src_sel = 1;
            }
            else
            {
                return -1;
            }
            //bcmcpuclk  (/1, /2, /3 ... /128) default /1
            reg_val = apmu_clkrstgen_regs->BCMCPUClkConfig_ClkGenConfig;
            reg_val = APMU_CLKRSTGEN_BCMCPUCLKCONFIG_CLKGENCONFIG_SRCSEL_REPLACE_VAL(reg_val, src_sel);
            reg_val = APMU_CLKRSTGEN_BCMCPUCLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg_val, hi_div);
            reg_val = APMU_CLKRSTGEN_BCMCPUCLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg_val, lo_div);
            reg_val = APMU_CLKRSTGEN_BCMCPUCLKCONFIG_CLKGENCONFIG_PREDIVEN_REPLACE_VAL(reg_val, prediven);
			apmu_clkrstgen_regs->BCMCPUClkConfig_ClkGenConfig = reg_val;
			DUMP_REG(apmu_clkrstgen_regs->BCMCPUClkConfig_ClkGenConfig);
            break;
        case AP_CPU_CLK_DIV:
            /* 1) Set the clock source. COREPLL is the only source that is currently supported. */
            if (CORE_PLL == clk_src)
            {
                /* SrcSel for Core PLL changes after Rev A Si!! */
                if (is_NOT_Gr2_RevA())
                    src_sel = 0;
                else
                    src_sel = 1;
            }
            else
            {
                return -1;
            }
            reg_val = apmu_clkrstgen_regs->APCPUClkControl;
            reg_val = APMU_CLKRSTGEN_APCPUCLKCONTROL_CORECLK_SRCSEL_REPLACE_VAL(reg_val, src_sel);
            apmu_clkrstgen_regs->APCPUClkControl = reg_val;

            /* 2) Wait 1us; */
            UTF_spin_delay(1);

            /* 3) Select the Core AP clock divisor value. */
            reg_val = apmu_clkrstgen_regs->APCPUClkControl;
            reg_val = APMU_CLKRSTGEN_APCPUCLKCONTROL_COREAPCKL_DIV_REPLACE_VAL(reg_val, div_value);
            apmu_clkrstgen_regs->APCPUClkControl = reg_val;

            /* 4) Wait 1us; */
            UTF_spin_delay(1);

            /* 5) Select the ACLK divisor value. This field controls the AXI speed at the A53
             * and should be no less than 2 when the coreapckl_Div value is a 1 and COREPLL
             * is at max speed. */
            reg_val = apmu_clkrstgen_regs->APCPUClkControl;
            reg_val = APMU_CLKRSTGEN_APCPUCLKCONTROL_ACLKENM_DIV_REPLACE_VAL(reg_val, 2);   /* Hardcode to 2 */
            apmu_clkrstgen_regs->APCPUClkControl = reg_val;

            /* 6) Wait 1 us; */
            UTF_spin_delay(1);

            /* 7) Select the ATCLK divisor value. This field controls the debug speed at the
             * A53 and should be no less than 3 when the coreapckl_Div value is a 1 and
             * COREPLL is at max speed. */
            reg_val = apmu_clkrstgen_regs->APCPUClkControl;
            reg_val = APMU_CLKRSTGEN_APCPUCLKCONTROL_ATCLKEN_DIV_REPLACE_VAL(reg_val, 3);   /* Hardcode to 3 */
            apmu_clkrstgen_regs->APCPUClkControl = reg_val;

            /* 8) Wait 1us; */
            UTF_spin_delay(1);
            DUMP_REG(apmu_clkrstgen_regs->APCPUClkControl);
            break;
        case GPU_BUS_CLK_DIV:
            // 0 : Core PLL Clock 
            // 1 : DDR PLL Clock
            if (CORE_PLL == clk_src)
            {
                src_sel = 0;
            }
            else if (DDR_PLL == clk_src)
            {
                src_sel = 1;
            }
            else
            {
                return -1;
            }
            //gpubusclk  (/1, /2, /3 ... /16) defaults off
            reg_val = apmu_clkrstgen_regs->GPUBusClkConfig_ClkGenConfig;
            reg_val = APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_SRCSEL_REPLACE_VAL(reg_val, src_sel);
            reg_val = APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg_val, hi_div);
            reg_val = APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg_val, lo_div);
            apmu_clkrstgen_regs->GPUBusClkConfig_ClkGenConfig = reg_val;
            DUMP_REG(apmu_clkrstgen_regs->GPUBusClkConfig_ClkGenConfig);
            break;
        case GPU_CLK_DIV:
            // 0 : Core PLL Clock 
            // 1 : DDR PLL Clock
            if (CORE_PLL == clk_src)
            {
                src_sel = 0;
            }
            else if (DDR_PLL == clk_src)
            {
                src_sel = 1;
            }
            else
            {
                return -1;
            }
            //gpuclk     (/1, /2, /3 ... /16) defaults off
            reg_val = apmu_clkrstgen_regs->GPUClkConfig_ClkGenConfig;
            reg_val = APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_SRCSEL_REPLACE_VAL(reg_val, src_sel);
            reg_val = APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg_val, hi_div);
            reg_val = APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg_val, lo_div);
            apmu_clkrstgen_regs->GPUClkConfig_ClkGenConfig = reg_val;
            DUMP_REG(apmu_clkrstgen_regs->GPUClkConfig_ClkGenConfig);
            break;
        case SDMMC_CLK_DIV:
            // 0 : Core PLL Clock 
            // 1 : DDR PLL Clock
            if (CORE_PLL == clk_src)
            {
                src_sel = 0;
            }
            else if (DDR_PLL == clk_src)
            {
                src_sel = 1;
            }
            else
            {
                return -1;
            }
            //sdmmcclk   (/1, /2, /3 ... /32) default /1
            reg_val = apmu_clkrstgen_regs->SDMMCClkConfig_ClkGenConfig;
            reg_val = APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_PREDIVEN_REPLACE_VAL(reg_val, prediven);
            reg_val = APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_SRCSEL_REPLACE_VAL(reg_val, src_sel);
            reg_val = APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg_val, hi_div);
            reg_val = APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg_val, lo_div);
            apmu_clkrstgen_regs->SDMMCClkConfig_ClkGenConfig = reg_val;
            DUMP_REG(apmu_clkrstgen_regs->SDMMCClkConfig_ClkGenConfig);
            break;
        case QSPI_CLK_DIV:
            // 0 : Core PLL Clock 
            // 1 : DDR PLL Clock
            if (CORE_PLL == clk_src)
            {
                src_sel = 0;
            }
            else if (DDR_PLL == clk_src)
            {
                src_sel = 1;
            }
            else
            {
                return -1;
            }
            //qspiclk    (/1, /2, /3 ... /32) default /1
            reg_val = apmu_clkrstgen_regs->QSPIClkConfig_ClkGenConfig;
            reg_val = APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_SRCSEL_REPLACE_VAL(reg_val, src_sel);
            reg_val = APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg_val, hi_div);
            reg_val = APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg_val, lo_div);
            apmu_clkrstgen_regs->QSPIClkConfig_ClkGenConfig = reg_val;
            DUMP_REG(apmu_clkrstgen_regs->QSPIClkConfig_ClkGenConfig);
            break;
        case LCDX_CLK_DIV:
            // 0 : Core PLL Clock 
            // 1 : DDR PLL Clock
            if (CORE_PLL == clk_src)
            {
                src_sel = 0;
            }
            else if (DDR_PLL == clk_src)
            {
                src_sel = 1;
            }
            else
            {
                return -1;
            }
            //lcdxclk    (PLL/(2*D[7:0]+1))   D defaults to 7 i.e. PLL/16 but is off
            reg_val = apmu_clkrstgen_regs->LCDXClkConfig_ClkGenConfig;
            reg_val = APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_SRCSEL_REPLACE_VAL(reg_val, src_sel);
            reg_val = APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg_val, hi_div);
            reg_val = APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg_val, lo_div);
            reg_val = APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_PREDIVEN_REPLACE_VAL(reg_val, prediven);
            apmu_clkrstgen_regs->LCDXClkConfig_ClkGenConfig = reg_val;
            DUMP_REG(apmu_clkrstgen_regs->LCDXClkConfig_ClkGenConfig);
            break;
        case UART0_CLK_DIV:
            // 00 : 25 MHz reference clock 
            // 01 : System PLL Clock 
            // 10 : DDR PLL Clock 
            // 11 : Clock source 3
            if (REF_CLK == clk_src)
            {
                src_sel = 0;
            }
            else if (SYSTEM_PLL == clk_src)
            {
                src_sel = 1;
            }
            else if (DDR_PLL == clk_src)
            {
                src_sel = 2;
            }
            else
            {
                return -1;
            }

            //uartclk    (M/N)                default /1
            reg_val = apmu_clkrstgen_regs->UART0ClkConfig_ClkGenConfig;
            reg_val = APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_SRCSEL_REPLACE_VAL(reg_val, src_sel);
            reg_val = APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg_val, hi_div);
            reg_val = APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg_val, lo_div);
            reg_val = APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_PREDIVEN_REPLACE_VAL(reg_val, prediven);
            apmu_clkrstgen_regs->UART0ClkConfig_ClkGenConfig = reg_val;
            DUMP_REG(apmu_clkrstgen_regs->UART0ClkConfig_ClkGenConfig);
            //Above setting should give 416.667 MHz as the fractional divider base
            //clock. Shooting for 14.7456 MHz on uartclk
            reg_val = apmu_clkrstgen_regs->UART0ClkFracDivCfg;
            reg_val = APMU_CLKRSTGEN_UART0CLKFRACDIVCFG_NUMERATOR_REPLACE_VAL(reg_val, fract_num);
            reg_val = APMU_CLKRSTGEN_UART0CLKFRACDIVCFG_DENOMINATOR_REPLACE_VAL(reg_val, fract_denom);
            apmu_clkrstgen_regs->UART0ClkFracDivCfg = reg_val;
            DUMP_REG(apmu_clkrstgen_regs->UART0ClkFracDivCfg);
            break;
        case UART1_CLK_DIV:
            // 00 : 25 MHz reference clock 
            // 01 : System PLL Clock 
            // 10 : DDR PLL Clock 
            // 11 : Clock source 3
            if (REF_CLK == clk_src)
            {
                src_sel = 0;
            }
            else if (SYSTEM_PLL == clk_src)
            {
                src_sel = 1;
            }
            else if (DDR_PLL == clk_src)
            {
                src_sel = 2;
            }
            else
            {
                return -1;
            }
            //uartclk    (M/N)                default /1
            reg_val = apmu_clkrstgen_regs->UART1ClkConfig_ClkGenConfig;
            reg_val = APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_SRCSEL_REPLACE_VAL(reg_val, src_sel);
            reg_val = APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg_val, hi_div);
            reg_val = APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg_val, lo_div);
            reg_val = APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_PREDIVEN_REPLACE_VAL(reg_val, prediven);
            apmu_clkrstgen_regs->UART1ClkConfig_ClkGenConfig = reg_val;
            DUMP_REG(apmu_clkrstgen_regs->UART1ClkConfig_ClkGenConfig);
            //Above setting should give 416.667 MHz as the fractional divider base
            //clock. Shooting for 14.7456 MHz on uartclk
            reg_val = apmu_clkrstgen_regs->UART1ClkFracDivCfg;
            reg_val = APMU_CLKRSTGEN_UART1CLKFRACDIVCFG_NUMERATOR_REPLACE_VAL(reg_val, fract_num);
            reg_val = APMU_CLKRSTGEN_UART1CLKFRACDIVCFG_DENOMINATOR_REPLACE_VAL(reg_val, fract_denom);
            apmu_clkrstgen_regs->UART1ClkFracDivCfg = reg_val;
            DUMP_REG(apmu_clkrstgen_regs->UART1ClkFracDivCfg);
            break;
        case UART2_CLK_DIV:
            // 00 : 25 MHz reference clock 
            // 01 : System PLL Clock 
            // 10 : DDR PLL Clock 
            // 11 : Clock source 3
            if (REF_CLK == clk_src)
            {
                src_sel = 0;
            }
            else if (SYSTEM_PLL == clk_src)
            {
                src_sel = 1;
            }
            else if (DDR_PLL == clk_src)
            {
                src_sel = 2;
            }
            else
            {
                return -1;
            }
            //uartclk    (M/N)                default /1
            reg_val = apmu_clkrstgen_regs->UART2ClkConfig_ClkGenConfig;
            reg_val = APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_SRCSEL_REPLACE_VAL(reg_val, src_sel);
            reg_val = APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg_val, hi_div);
            reg_val = APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg_val, lo_div);
            reg_val = APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_PREDIVEN_REPLACE_VAL(reg_val, prediven);
            apmu_clkrstgen_regs->UART2ClkConfig_ClkGenConfig = reg_val;
            DUMP_REG(apmu_clkrstgen_regs->UART2ClkConfig_ClkGenConfig);
            //Above setting should give 416.667 MHz as the fractional divider base
            //clock. Shooting for 14.7456 MHz on uartclk
            reg_val = apmu_clkrstgen_regs->UART2ClkFracDivCfg;
            reg_val = APMU_CLKRSTGEN_UART2CLKFRACDIVCFG_NUMERATOR_REPLACE_VAL(reg_val, fract_num);
            reg_val = APMU_CLKRSTGEN_UART2CLKFRACDIVCFG_DENOMINATOR_REPLACE_VAL(reg_val, fract_denom);
            apmu_clkrstgen_regs->UART2ClkFracDivCfg = reg_val;
            DUMP_REG(apmu_clkrstgen_regs->UART2ClkFracDivCfg);
            break;
        case UART3_CLK_DIV:
            // 00 : 25 MHz reference clock 
            // 01 : System PLL Clock 
            // 10 : DDR PLL Clock 
            // 11 : Clock source 3
            if (REF_CLK == clk_src)
            {
                src_sel = 0;
            }
            else if (SYSTEM_PLL == clk_src)
            {
                src_sel = 1;
            }
            else if (DDR_PLL == clk_src)
            {
                src_sel = 2;
            }
            else
            {
                return -1;
            }
            //uartclk    (M/N)                default /1
            reg_val = apmu_clkrstgen_regs->UART3ClkConfig_ClkGenConfig;
            reg_val = APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_SRCSEL_REPLACE_VAL(reg_val, src_sel);
            reg_val = APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg_val, hi_div);
            reg_val = APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg_val, lo_div);
            reg_val = APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_PREDIVEN_REPLACE_VAL(reg_val, prediven);
            apmu_clkrstgen_regs->UART3ClkConfig_ClkGenConfig = reg_val;
            DUMP_REG(apmu_clkrstgen_regs->UART3ClkConfig_ClkGenConfig);
            //Above setting should give 416.667 MHz as the fractional divider base
            //clock. Shooting for 14.7456 MHz on uartclk
            reg_val = apmu_clkrstgen_regs->UART3ClkFracDivCfg;
            reg_val = APMU_CLKRSTGEN_UART3CLKFRACDIVCFG_NUMERATOR_REPLACE_VAL(reg_val, fract_num);
            reg_val = APMU_CLKRSTGEN_UART3CLKFRACDIVCFG_DENOMINATOR_REPLACE_VAL(reg_val, fract_denom);
            apmu_clkrstgen_regs->UART3ClkFracDivCfg = reg_val;
            DUMP_REG(apmu_clkrstgen_regs->UART3ClkFracDivCfg);
            break;
        case LAN_MAC_CLK2X_DIV:
            // The source frequency from the System PLL is normally 2.5 GHz. If desired for some low power states,
            // the System PLL can be programmed for 1 GHz, in which case this bit should be set so that the
            // dividers can be adjusted properly.
            if (SYSTEM_PLL == clk_src)
            {
                src_sel = 0;
            }
            else if (SYSTEM_PLL_1GHZ == clk_src)
            {
                src_sel = 1;
            }
            else
            {
                return -1;
            }
            //lanmacclk2x expecting 2.5G
            reg_val = apmu_clkrstgen_regs->LANMacClk2xConfig;
            reg_val = APMU_CLKRSTGEN_LANMACCLK2XCONFIG_SRCFREQ_REPLACE_VAL(reg_val, src_sel);
            apmu_clkrstgen_regs->LANMacClk2xConfig = reg_val;
            DUMP_REG(apmu_clkrstgen_regs->LANMacClk2xConfig);
            break;
        case LAN_MAC_CLK_DIV:
            // A '0' in this field selects the normal LAN Rx Clock from the PHY. 
            // A '1' in this field selects the LANMacClk as the source of this clock.
            if (LAN_RX_CLK_PHY == clk_src)
            {
                src_sel = 0;
            }
            else if (LAN_MAC_CLK == clk_src)
            {
                src_sel = 1;
            }
            else
            {
                return -1;
            }
            //lanmacclk ??Rx??
            reg_val = apmu_clkrstgen_regs->LANRxClkGenConfig;
            reg_val = APMU_CLKRSTGEN_LANRXCLKGENCONFIG_SRCSEL_REPLACE_VAL(reg_val, src_sel);
            apmu_clkrstgen_regs->LANRxClkGenConfig = reg_val;
            DUMP_REG(apmu_clkrstgen_regs->LANRxClkGenConfig);
            break;
        case DDR_CLK2X_DIV:
            /* Don't do this on Rev B Si!! ddrclk2x clkgenconfig doesn't exist. */
            if (is_NOT_Gr2_RevA())
            {
                break;
            }

            // 0 : DDR PLL Clock 
            // 1 : Core PLL Clock
            if (DDR_PLL == clk_src)
            {
                src_sel = 0;
            }
            else if (CORE_PLL == clk_src)
            {
                src_sel = 1;
            }
            else
            {
                return -1;
            }

            if (is_Gr2() && is_RevA())
            {
                //ddrclk2x
                reg_val = apmu_clkrstgen_regs->DDRClk2xConfig_ClkGenConfig;
                reg_val = APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_SRCSEL_REPLACE_VAL(reg_val, src_sel);
                reg_val = APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg_val, hi_div);
                reg_val = APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg_val, lo_div);

                /* The ddrclk2x clkgen register won't exist in RevB, and so the associated gating
                 * capability won't be available. So we enable ddrclk2x here for RevB
                 * compatibility. */
                reg_val = APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(reg_val, 1);
                apmu_clkrstgen_regs->DDRClk2xConfig_ClkGenConfig = reg_val;
                DUMP_REG(apmu_clkrstgen_regs->DDRClk2xConfig_ClkGenConfig);

                /* Wait for the ddrclk2x status to show as not gated */
                reg_val = apmu_clkrstgen_regs->DDRClk2xConfig_ClkGenStatus;
                while (APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(reg_val) == 1)
                {
                    reg_val = apmu_clkrstgen_regs->DDRClk2xConfig_ClkGenStatus;
                }
                DUMP_REG(apmu_clkrstgen_regs->DDRClk2xConfig_ClkGenStatus);
            }
            break;
        case DDR_CLK_DIV:
            if (is_Gr2() && is_RevA())
            {
                // Selects if the DDR clock is divide by 2, or divide by 4 from the DDR PHY clock.
                // 1 = divide by 2
                // 0 = divide by 4
                if (4 == div_value)
                {
                    div_sel = 0;
                }
                else if (2 == div_value)
                {
                    div_sel = 1;
                }
                else
                {
                    return -1;
                }
                //ddrclk
                reg_val = apmu_clkrstgen_regs->DDRClkConfig;
                reg_val = APMU_CLKRSTGEN_DDRCLKCONFIG_DIVSEL_REPLACE_VAL(reg_val, div_sel);
                apmu_clkrstgen_regs->DDRClkConfig = reg_val;
                DUMP_REG(apmu_clkrstgen_regs->DDRClkConfig);
            } else
            {
                // Selects if the DDR clock is divide by 2, or divide by 4 from the DDR PHY clock.
                // Set both hi and lo to 1 to get a div 4, both to 0 for a div 2
                if (4 == div_value)
                {
                    hi_div = 1;
                    lo_div = 1;
                }
                else if (2 == div_value)
                {
                    hi_div = 0;
                    lo_div = 0;
                }
                else
                {
                    return -1;
                }

                //ddrclk
                reg_val = apmu_clkrstgen_regs->DDRClkConfig;
                reg_val = APMU_CLKRSTGEN_DDRCLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg_val, hi_div);
                reg_val = APMU_CLKRSTGEN_DDRCLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg_val, lo_div);
                apmu_clkrstgen_regs->DDRClkConfig = reg_val;
                DUMP_REG(apmu_clkrstgen_regs->DDRClkConfig);
            }
            break;
        default:
            return -1;
    }

    return ret;
}

/*------------------------------------------------------------------------
            IPMU_set_clk_div
-------------------------------------------------------------------------*/
/**
 * 
 * \brief Sets the specified IPMU Clk divider.
 *
 * \param [in] clk_div  Clock divider to operate on.
 * \param [in] clk_src  Specified clock source.
 * \param [in] div_val  Desired divider value.
 * \param [in] prediven        Pre-divider enable value (if applicable).
 * \param [in] fract_num       Fractional numerator value (if applicable).
 * \param [in] fract_denom     Fractional denominator value (if applicable).
 *
 * \return [out] int    -1=FAIL, 0=SUCCESS.
 *
 */
static int IPMU_set_clk_div(pmu_clk_div_t clk_div,
                            pmu_div_clk_src_t clk_src,
                            uint32_t div_value,
                            uint32_t prediven,
                            uint32_t fract_num,
                            uint32_t fract_denom)
{
    uint32_t reg_val;
	uint32_t hi_div;
	uint32_t lo_div;
	uint32_t src_sel;
    int ret;
    IPMU_CLKRSTGEN_REGS_t *ipmu_clkrstgen_regs = (IPMU_CLKRSTGEN_REGS_t *)IPS_IPMU_IPMU_CLKRSTGEN_BASE;

    ret = 0;
	calc_dividers(div_value, &hi_div, &lo_div);

	if (is_Ge2())
	{
		switch (clk_div)
		{
		case XCPU_CLK_DIV:
		case XIO_CLK_DIV:
		case IPS_BUS_CLK_DIV:
			ipmu_clkrstgen_regs = (IPMU_CLKRSTGEN_REGS_t *)(IPS_IPMU_IPMU_CLKRSTGEN_BASE + 0x08);
			break;
		default:
			break;
		}
	}

    switch (clk_div)
    {
        case IPS_BUS_CLK_DIV:
            // 0 : Core PLL Clock 
            // 1 : DDR PLL Clock
            if (CORE_PLL == clk_src)
            {
                src_sel = 0;
            }
            else if (DDR_PLL == clk_src)
            {
                src_sel = 1;
            }
            else
            {
                return -1;
            }
            //ipsbusclk  (/1, /2, /3 ... /32) default /1
            reg_val = ipmu_clkrstgen_regs->IPSBusClkConfig_ClkGenConfig;
            reg_val = IPMU_CLKRSTGEN_IPSBUSCLKCONFIG_CLKGENCONFIG_SRCSEL_REPLACE_VAL(reg_val, src_sel);
            reg_val = IPMU_CLKRSTGEN_IPSBUSCLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg_val, hi_div);
            reg_val = IPMU_CLKRSTGEN_IPSBUSCLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg_val, lo_div);
            ipmu_clkrstgen_regs->IPSBusClkConfig_ClkGenConfig = reg_val;
            DUMP_REG(ipmu_clkrstgen_regs->IPSBusClkConfig_ClkGenConfig);
            break;
        case AUDIO_CLK_PRE_DIV:
            // 0 : System PLL Clock 
            // 1 : 25MHz reference clock
            if (SYSTEM_PLL == clk_src)
            {
                src_sel = 0;
            }
            else if (REF_CLK == clk_src)
            {
                src_sel = 1;
            }
            else
            {
                return -1;
            }
            //audioclk  (not sure of the div value?)
            reg_val = ipmu_clkrstgen_regs->AudioClkPreConfig_ClkGenConfig;
            reg_val = IPMU_CLKRSTGEN_AUDIOCLKPRECONFIG_CLKGENCONFIG_PREDIVEN_REPLACE_VAL(reg_val, prediven);
            reg_val = IPMU_CLKRSTGEN_AUDIOCLKPRECONFIG_CLKGENCONFIG_SRCSEL_REPLACE_VAL(reg_val, src_sel);
            reg_val = IPMU_CLKRSTGEN_AUDIOCLKPRECONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg_val, hi_div);
            reg_val = IPMU_CLKRSTGEN_AUDIOCLKPRECONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg_val, lo_div);
            ipmu_clkrstgen_regs->AudioClkPreConfig_ClkGenConfig = reg_val;
            DUMP_REG(ipmu_clkrstgen_regs->AudioClkPreConfig_ClkGenConfig);
            break;
        case AUDIO_CLK_DIV:
            reg_val = ipmu_clkrstgen_regs->AudioClkConfig_ClkGenConfig;
            reg_val = IPMU_CLKRSTGEN_AUDIOCLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg_val, hi_div);
            reg_val = IPMU_CLKRSTGEN_AUDIOCLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg_val, lo_div);
            ipmu_clkrstgen_regs->AudioClkConfig_ClkGenConfig = reg_val;
            DUMP_REG(ipmu_clkrstgen_regs->AudioClkConfig_ClkGenConfig);
            break;
        case ADC_TSEN_CLK_DIV:
            //adctsenclk
            reg_val = ipmu_clkrstgen_regs->AdcTsenClkConfig_ClkGenConfig;
            reg_val = IPMU_CLKRSTGEN_ADCTSENCLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg_val, hi_div);
            reg_val = IPMU_CLKRSTGEN_ADCTSENCLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg_val, lo_div);
            ipmu_clkrstgen_regs->AdcTsenClkConfig_ClkGenConfig = reg_val;
            DUMP_REG(ipmu_clkrstgen_regs->AdcTsenClkConfig_ClkGenConfig);
            break;
        case FAST_NSS_CLK_DIV:
            //fastnssclk
            reg_val = ipmu_clkrstgen_regs->FastNSSClkConfig_ClkGenConfig;
            reg_val = IPMU_CLKRSTGEN_FASTNSSCLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg_val, hi_div);
            reg_val = IPMU_CLKRSTGEN_FASTNSSCLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg_val, lo_div);
            ipmu_clkrstgen_regs->FastNSSClkConfig_ClkGenConfig = reg_val;
            DUMP_REG(ipmu_clkrstgen_regs->FastNSSClkConfig_ClkGenConfig);
            break;
        case NSS_CLK_DIV:
            //nssclk
            reg_val = ipmu_clkrstgen_regs->NSSClkConfig_ClkGenConfig;
            reg_val = IPMU_CLKRSTGEN_NSSCLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg_val, hi_div);
            reg_val = IPMU_CLKRSTGEN_NSSCLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg_val, lo_div);
            ipmu_clkrstgen_regs->NSSClkConfig_ClkGenConfig = reg_val;
            DUMP_REG(ipmu_clkrstgen_regs->NSSClkConfig_ClkGenConfig);
            break;
        case VCF_CLK_DIV:
            //vcfclk , ??not sure where this lives??
            break;
        case XCPU_CLK_DIV:
            // 0 : DDR PLL Clock 
            // 1 : Core PLL Clock
            if (DDR_PLL == clk_src)
            {
                src_sel = 0;
            }
            else if (CORE_PLL == clk_src)
            {
                src_sel = 1;
            }
            else
            {
                return -1;
            }
            //xcpuclk
            reg_val = ipmu_clkrstgen_regs->XCPUClkConfig_ClkGenConfig;
            reg_val = IPMU_CLKRSTGEN_XCPUCLKCONFIG_CLKGENCONFIG_SRCSEL_REPLACE_VAL(reg_val, src_sel);
            reg_val = IPMU_CLKRSTGEN_XCPUCLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg_val, hi_div);
            reg_val = IPMU_CLKRSTGEN_XCPUCLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg_val, lo_div);
            ipmu_clkrstgen_regs->XCPUClkConfig_ClkGenConfig = reg_val;
            DUMP_REG(ipmu_clkrstgen_regs->XCPUClkConfig_ClkGenConfig);
            break;
        case XIO_CLK_DIV:
            // 0 : DDR PLL Clock 
            // 1 : Core PLL Clock
            if (DDR_PLL == clk_src)
            {
                src_sel = 0;
            }
            else if (CORE_PLL == clk_src)
            {
                src_sel = 1;
            }
            else
            {
                return -1;
            }
            //xioclk
            reg_val = ipmu_clkrstgen_regs->XIOClkConfig_ClkGenConfig;
            reg_val = IPMU_CLKRSTGEN_XIOCLKCONFIG_CLKGENCONFIG_SRCSEL_REPLACE_VAL(reg_val, src_sel);
            reg_val = IPMU_CLKRSTGEN_XIOCLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg_val, hi_div);
            reg_val = IPMU_CLKRSTGEN_XIOCLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg_val, lo_div);
            ipmu_clkrstgen_regs->XIOClkConfig_ClkGenConfig = reg_val;
            DUMP_REG(ipmu_clkrstgen_regs->XIOClkConfig_ClkGenConfig);
            break;
        case LVDS_AFE_REF_CLK_DIV:
            //lvdsaferefclk
            reg_val = ipmu_clkrstgen_regs->LVDSAFERefClkConfig_ClkGenConfig;
            reg_val = IPMU_CLKRSTGEN_LVDSAFEREFCLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg_val, hi_div);
            reg_val = IPMU_CLKRSTGEN_LVDSAFEREFCLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg_val, lo_div);
            ipmu_clkrstgen_regs->LVDSAFERefClkConfig_ClkGenConfig = reg_val;
            DUMP_REG(ipmu_clkrstgen_regs->LVDSAFERefClkConfig_ClkGenConfig);
            break;
        case SCAN_CLK_DIV:
            // 00 : Scan PLL Clock 
            // 01 : System PLL Clock 
            // 10 : Core PLL Clock 
            // 11 : Reserved
            if (SCAN_PLL == clk_src)
            {
                src_sel = 0;
            }
            else if (SYSTEM_PLL == clk_src)
            {
                src_sel = 1;
            }
            else if (CORE_PLL == clk_src)
            {
                src_sel = 2;
            }
            else
            {
                return -1;
            }
            //scanclk
            reg_val = ipmu_clkrstgen_regs->ScanClkConfig_ClkGenConfig;
            reg_val = IPMU_CLKRSTGEN_SCANCLKCONFIG_CLKGENCONFIG_SRCSEL_REPLACE_VAL(reg_val, src_sel);
            reg_val = IPMU_CLKRSTGEN_SCANCLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg_val, hi_div);
            reg_val = IPMU_CLKRSTGEN_SCANCLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg_val, lo_div);
            ipmu_clkrstgen_regs->ScanClkConfig_ClkGenConfig = reg_val;
            DUMP_REG(ipmu_clkrstgen_regs->ScanClkConfig_ClkGenConfig);
            break;
        default:
            return -1;
    }

    return ret;
}

static void calc_dividers(uint32_t div_value,
                          uint32_t *hi_div,
                          uint32_t *lo_div)
{
    if (div_value > 1)
	{
		*hi_div = (div_value+1) / 2;
		*lo_div = div_value - *hi_div;
	}
	else
	{
	    /* Divide by 1 */
		*hi_div = 0;
		*lo_div = 0;
	}
}

