/*
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
*/

/**
 *
 * \file UH_APMU_SRAMs.c
 *
 * \brief This header file contains APMU specific routines for
 * power up/down the SRAMs.
 *
 *
 *
 **/

#include <stdint.h>
#include <stdbool.h>
#include <string.h>

#include "UH_PMU_Device.h"
#include "ID_utils.h"


#define APMU_CLKRSTGEN_REG_ADDR_OF(reg)                             \
    (is_Gr2() && is_RevA()) ?                                       \
    &(((APMU_CLKRSTGEN_REGS_t*)AP_APMU_APMU_CLKRSTGEN_BASE)->reg) : \
    &(((APMU_CLKRSTGEN_REGS_REVB_t*)(AP_APMU_APMU_CLKRSTGEN_BASE + 0x200))->reg)
#define APMU_CLKRSTGEN_REVB_REG_ADDR_OF(reg)			\
    &(((APMU_CLKRSTGEN_REGS_REVB_t*)(AP_APMU_APMU_CLKRSTGEN_BASE + 0x200))->reg)


// Clock Functions
static bool apmu_lcd_clocks_enable(bool power_on)
{
#ifndef PMU_DEVICE_EMULATION
    // clocks to enable LCD
    volatile uint32_t *config = APMU_CLKRSTGEN_REG_ADDR_OF(APBusClk_LCDConfig_ClkGenConfig);
    volatile uint32_t *status = APMU_CLKRSTGEN_REG_ADDR_OF(APBusClk_LCDConfig_ClkGenStatus);

    // power_on = true --> enable clocks, and wait until nolonger gating
    // verilog shows 3 clocks power_ond:    apbusclk_lcd, lcdclk, lcdxclk
    // first: apbusclk_lcd
    uint32_t clk_en = APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(*config);
    uint32_t clk_st = APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status);
    if (!clk_en && !clk_st)
    {
        *config = APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,1);
    }
    *config = APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,power_on);
    // wait for bit
    while (APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status) == power_on);

    // now:  lcdclk  Note unconventional naming
    config = APMU_CLKRSTGEN_REG_ADDR_OF(LCDClkConfig);
    status = APMU_CLKRSTGEN_REG_ADDR_OF(LCDClkStatus);

    *config = APMU_CLKRSTGEN_LCDCLKCONFIG_CLKEN_REPLACE_VAL(*config,power_on);
    // wait for bit
    while (APMU_CLKRSTGEN_LCDCLKSTATUS_CLKGATED_MASK_SHIFT(*status) == power_on);

    // now: lcdxclk
    config = APMU_CLKRSTGEN_REG_ADDR_OF(LCDXClkConfig_ClkGenConfig);
    status = APMU_CLKRSTGEN_REG_ADDR_OF(LCDXClkConfig_ClkGenStatus);

    *config = APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,power_on);
    // wait for bit
    while (APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status) == power_on);
#endif

    return true;
}

static bool apmu_AP_APB_clocks_enable(bool power_on)
{
#ifndef PMU_DEVICE_EMULATION
    // clocks to enable actualy only one AP_APB


    volatile uint32_t *config = APMU_CLKRSTGEN_REG_ADDR_OF(APBusClk_APBConfig_ClkGenConfig);
    volatile uint32_t *status = APMU_CLKRSTGEN_REG_ADDR_OF(APBusClk_APBConfig_ClkGenStatus);

    // power_on = true --> enable clocks, and wait until nolonger gating
    // verilog shows 3 clocks power_ond:    apbusclk_lcd, lcdclk, lcdxclk
    // first: apbusclk_lcd
    uint32_t clk_en = APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(*config);
    uint32_t clk_st = APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status);
    if (!clk_en && !clk_st)
    {
        *config = APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,1);
    }
    *config = APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,power_on);
    // wait for bit
    while (APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status) == power_on);
#endif
    //TODO: Should I also turn on UART, SPI, and I2C as well?

    return true;
}

static bool apmu_uart_clocks_enable(bool power_on)
{
#ifndef PMU_DEVICE_EMULATION
    // power_on = true --> enable clocks, and wait until nolonger gating
    // UART0
    volatile uint32_t *config = APMU_CLKRSTGEN_REG_ADDR_OF(UART0ClkConfig_ClkGenConfig);
    volatile uint32_t *status = APMU_CLKRSTGEN_REG_ADDR_OF(UART0ClkConfig_ClkGenStatus);

    *config = APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,power_on);
    // wait for bit
    while (APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status) == power_on);


    // UART1
    config = APMU_CLKRSTGEN_REG_ADDR_OF(UART1ClkConfig_ClkGenConfig);
    status = APMU_CLKRSTGEN_REG_ADDR_OF(UART1ClkConfig_ClkGenStatus);

    *config = APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,power_on);
    // wait for bit
    while (APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status) == power_on);


    // UART2
    config = APMU_CLKRSTGEN_REG_ADDR_OF(UART2ClkConfig_ClkGenConfig);
    status = APMU_CLKRSTGEN_REG_ADDR_OF(UART2ClkConfig_ClkGenStatus);

    *config = APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,power_on);
    // wait for bit
    while (APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status) == power_on);


    // UART3
    config = APMU_CLKRSTGEN_REG_ADDR_OF(UART3ClkConfig_ClkGenConfig);
    status = APMU_CLKRSTGEN_REG_ADDR_OF(UART3ClkConfig_ClkGenStatus);

    *config = APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,power_on);
    // wait for bit
    while (APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status) == power_on);

#endif //End of if Emulation
    return true;
}

static bool apmu_spi_clocks_enable(bool power_on)
{
#ifndef PMU_DEVICE_EMULATION
    // clocks to enable SPI clocks
    volatile uint32_t *config = APMU_CLKRSTGEN_REG_ADDR_OF(SSPClkConfig_ClkGenConfig);
    volatile uint32_t *status = APMU_CLKRSTGEN_REG_ADDR_OF(SSPClkConfig_ClkGenStatus);

    // power_on = true --> enable clocks, and wait until nolonger gating
    *config = APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,power_on);
    // wait for bit
    while (APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status) == power_on);
#endif //End of if Emulation
    return true;

}
/* for SPI2 */
static bool apmu_spi2_clocks_enable(bool power_on)
{
#ifndef PMU_DEVICE_EMULATION
    // clocks to enable SPI2 clocks
    volatile uint32_t *config = APMU_CLKRSTGEN_REVB_REG_ADDR_OF(SSP2ClkConfig_ClkGenConfig);
    volatile uint32_t *status = APMU_CLKRSTGEN_REVB_REG_ADDR_OF(SSP2ClkConfig_ClkGenStatus);

    // power_on = true --> enable clocks, and wait until nolonger gating
    *config = APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,power_on);
    // wait for bit
    while (APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status) == power_on);
#endif //End of if Emulation
    return true;
}
/* for SPI3 */
static bool apmu_spi3_clocks_enable(bool power_on)
{
#ifndef PMU_DEVICE_EMULATION
    // clocks to enable SPI3 clocks
    volatile uint32_t *config = APMU_CLKRSTGEN_REVB_REG_ADDR_OF(SSP3ClkConfig_ClkGenConfig);
    volatile uint32_t *status = APMU_CLKRSTGEN_REVB_REG_ADDR_OF(SSP3ClkConfig_ClkGenStatus);

    // power_on = true --> enable clocks, and wait until nolonger gating
    *config = APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,power_on);
    // wait for bit
    while (APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status) == power_on);
#endif //End of if Emulation
    return true;
}

static bool apmu_i2c_clocks_enable(bool power_on)
{
#ifndef PMU_DEVICE_EMULATION
    // clocks to enable i2c
    volatile uint32_t *config = APMU_CLKRSTGEN_REG_ADDR_OF(I2CClkConfig_ClkGenConfig);
    volatile uint32_t *status = APMU_CLKRSTGEN_REG_ADDR_OF(I2CClkConfig_ClkGenStatus);

    // power_on = true --> enable clocks, and wait until nolonger gating
    *config = APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,power_on);
    // wait for bit

    while (APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status) == power_on);

    if (is_NOT_Gr2_RevA()) {
        config = &(((APMU_CLKRSTGEN_REGS_REVB_t*)(AP_APMU_APMU_CLKRSTGEN_BASE + 0x200))->I2C2ClkConfig_ClkGenConfig);
        status = &(((APMU_CLKRSTGEN_REGS_REVB_t*)(AP_APMU_APMU_CLKRSTGEN_BASE + 0x200))->I2C2ClkConfig_ClkGenStatus);

        // power_on = true --> enable clocks, and wait until nolonger gating
        *config = APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,power_on);
        // wait for bit

        while (APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status) == power_on);
    }

#endif //End of if Emulation
    return true;
}

static bool apmu_sata_clocks_enable(bool power_on)
{
#ifndef PMU_DEVICE_EMULATION
    // power_on = true --> enable clocks, and wait until nolonger gating
    // only on clock Sata
    volatile uint32_t *config = APMU_CLKRSTGEN_REG_ADDR_OF(APBusClk_SATAConfig_ClkGenConfig);
    volatile uint32_t *status = APMU_CLKRSTGEN_REG_ADDR_OF(APBusClk_SATAConfig_ClkGenStatus);

    // first: apbusclk_sata
    uint32_t clk_en = APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(*config);
    uint32_t clk_st = APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status);
    if (!clk_en && !clk_st)
    {
        *config = APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,1);
    }
    *config = APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,power_on);
    // wait for bit
    while (APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status) == power_on);
#endif

    return true;
}


static bool apmu_pcie_dm2_clocks_enable(bool power_on)
{

    // power_on = true --> enable clocks, and wait until nolonger gating
    // only on clock Sata
    volatile uint32_t *config = APMU_CLKRSTGEN_REG_ADDR_OF(APBusClk_PCIe_DM2Config_ClkGenConfig);
    volatile uint32_t *status = APMU_CLKRSTGEN_REG_ADDR_OF(APBusClk_PCIe_DM2Config_ClkGenStatus);

    // first: apbusclk_sata
    uint32_t clk_en = APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(*config);
    uint32_t clk_st = APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status);
    if (!clk_en && !clk_st)
    {
        *config = APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,1);
    }
    *config = APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,power_on);
    // wait for bit
#ifndef EMULATION
    while (APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status) == power_on);
#endif

    return true;
}

static bool apmu_pcie_dm4_clocks_enable(bool power_on)
{
#ifndef EMULATION
    // power_on = true --> enable clocks, and wait until nolonger gating
    // only on clock Sata
    volatile uint32_t *config = APMU_CLKRSTGEN_REG_ADDR_OF(APBusClk_PCIe_DM4Config_ClkGenConfig);
    volatile uint32_t *status = APMU_CLKRSTGEN_REG_ADDR_OF(APBusClk_PCIe_DM4Config_ClkGenStatus);

    // first: apbusclk_sata
    uint32_t clk_en = APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(*config);
    uint32_t clk_st = APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status);
    if (!clk_en && !clk_st)
    {
        *config = APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,1);
    }
    *config = APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,power_on);
    // wait for bit
    while (APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status) == power_on);
#endif

    return true;
}

static bool apmu_sdmmc_clocks_enable(bool power_on)
{
#ifndef EMULATION
    // power_on = true --> enable clocks, and wait until nolonger gating
    // clocks to enable SDMMC
    volatile uint32_t *config = APMU_CLKRSTGEN_REG_ADDR_OF(SDMMCClkConfig_ClkGenConfig);
    volatile uint32_t *status = APMU_CLKRSTGEN_REG_ADDR_OF(SDMMCClkConfig_ClkGenStatus);

    *config = APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,power_on);
    // wait for bit to change
    while (APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status) == power_on);
#endif
    return true;
}

static bool apmu_ndsmc_clocks_enable(bool power_on)
{
#ifndef EMULATION
    // power_on = true --> enable clocks, and wait until nolonger gating
    // clocks to enable NDSMC
    volatile uint32_t *config = APMU_CLKRSTGEN_REG_ADDR_OF(APBusClkDiv2Config_ClkGenConfig);
    volatile uint32_t *status = APMU_CLKRSTGEN_REG_ADDR_OF(APBusClkDiv2Config_ClkGenStatus);

    *config = APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,power_on);
    // wait for bit to change
    while (APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status) == power_on);
#endif
    return true;
}

static bool apmu_usb3_clocks_enable(bool power_on)
{
#ifndef EMULATION
    // power_on = true --> enable clocks, and wait until nolonger gating
    // clocks to enable apbusclk_usb3d and apbusclk_usb3phy
    // apbusclk_usb3d
    volatile uint32_t *config = APMU_CLKRSTGEN_REG_ADDR_OF(APBusClk_USB3DConfig_ClkGenConfig);
    volatile uint32_t *status = APMU_CLKRSTGEN_REG_ADDR_OF(APBusClk_USB3DConfig_ClkGenStatus);

    uint32_t clk_en = APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(*config);
    uint32_t clk_st = APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status);
    if (!clk_en && !clk_st)
    {
        *config = APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,1);
    }
    *config = APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,power_on);
    // wait for bit
    while (APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status) == power_on);

    // apbusclk_usb3phy
    config = APMU_CLKRSTGEN_REG_ADDR_OF(APBusClk_USB3PHYConfig_ClkGenConfig);
    status = APMU_CLKRSTGEN_REG_ADDR_OF(APBusClk_USB3PHYConfig_ClkGenStatus);

    clk_en = APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(*config);
    clk_st = APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status);
    if (!clk_en && !clk_st)
    {
        *config = APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,1);
    }
    *config = APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,power_on);
    // wait for bit
    while (APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status) == power_on);
#endif

    return true;
}


static bool apmu_usb2_clocks_enable(bool power_on)
{
#ifndef EMULATION
    // power_on = true --> enable clocks, and wait until nolonger gating
    // clocks to enable apbusclk_usb2d and apbusclk_usb2phy
    // apbusclk_usb2d
    volatile uint32_t *config = APMU_CLKRSTGEN_REG_ADDR_OF(APBusClk_USB2AConfig_ClkGenConfig);
    volatile uint32_t *status = APMU_CLKRSTGEN_REG_ADDR_OF(APBusClk_USB2AConfig_ClkGenStatus);

    uint32_t clk_en = APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(*config);
    uint32_t clk_st = APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status);
    if (!clk_en && !clk_st)
    {
        *config = APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,1);
    }
    *config = APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,power_on);
    // wait for bit
    while (APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status) == power_on);

    // apbusclk_usb2phy
    config = APMU_CLKRSTGEN_REG_ADDR_OF(APBusClk_USB2PHYConfig_ClkGenConfig);
    status = APMU_CLKRSTGEN_REG_ADDR_OF(APBusClk_USB2PHYConfig_ClkGenStatus);

    clk_en = APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(*config);
    clk_st = APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status);
    if (!clk_en && !clk_st)
    {
        *config = APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,1);
    }
    *config = APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,power_on);
    // wait for bit
    while (APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status) == power_on);
#endif

    return true;
}

static bool apmu_lan_clocks_enable(bool power_on)
{
#ifndef EMULATION
    volatile uint32_t apbusclk_lan_en_state = 0;
    // power_on = true --> enable clocks, and wait until nolonger gating
    // clocks to enable apbusclk_lan,lanmacclk, lanmacclk2x, and lanrxclk
    // apbusclk_lan
    volatile uint32_t *config = APMU_CLKRSTGEN_REG_ADDR_OF(APBusClk_LANConfig_ClkGenConfig);
    volatile uint32_t *status = APMU_CLKRSTGEN_REG_ADDR_OF(APBusClk_LANConfig_ClkGenStatus);

    volatile uint32_t *lanLPI = &(((MAD_REGS_t*)AP_IO_WRAP_LAN_MAD_BASE)->reserved1[0]); //Accessing reserved address at 0xD0701028 will probably be labeled MAD:R10
    *config = APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,1);
    // Inorder to prevent LAN from blocking clock changes the bit need to be set.
    *lanLPI = (*lanLPI | 0x80000000); // set bit 31 (AXI BUS MODE Register - En_LPI bit)
    *config = APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,power_on);
    // wait for bit
    while (APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status) == power_on);



//lanmacclk
    config = APMU_CLKRSTGEN_REG_ADDR_OF(LANMacClk2xConfig);
    status = APMU_CLKRSTGEN_REG_ADDR_OF(LANMacClk2xStatus);

    uint32_t clk_en = APMU_CLKRSTGEN_LANMACCLK2XCONFIG_CLKEN_MASK_SHIFT(*config);
    uint32_t clk_st = APMU_CLKRSTGEN_LANMACCLK2XSTATUS_CLKGATED_MASK_SHIFT(*status);
    if (!clk_en && !clk_st)
    {
        *config = APMU_CLKRSTGEN_LANMACCLK2XCONFIG_CLKEN_REPLACE_VAL(*config,1);
    }
    *config = APMU_CLKRSTGEN_LANMACCLK2XCONFIG_CLKEN_REPLACE_VAL(*config,power_on);
    // wait for bit
    while (APMU_CLKRSTGEN_LANMACCLK2XSTATUS_CLKGATED_MASK_SHIFT(*status) == power_on);

//lanmacclk2x
    // store APBusClk_LAN state for later use in lanmacclk2x
    config = APMU_CLKRSTGEN_REG_ADDR_OF(APBusClk_LANConfig_ClkGenConfig);
    apbusclk_lan_en_state = APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(*config);

    volatile uint32_t *lanExtSB = &(((EXT_REGS_t*)AP_IO_WRAP_LAN_EXT_BASE)->SB);
    *lanExtSB = EXT_SB_TXC_CLOCK_STOP_FORCE_REPLACE_VAL(*lanExtSB,!power_on);

    //hit lanmacclk again?
    config = APMU_CLKRSTGEN_REG_ADDR_OF(LANMacClk2xConfig);
    status = APMU_CLKRSTGEN_REG_ADDR_OF(LANMacClk2xStatus);

    clk_en = APMU_CLKRSTGEN_LANMACCLK2XCONFIG_CLKEN_MASK_SHIFT(*config);
    clk_st = APMU_CLKRSTGEN_LANMACCLK2XSTATUS_CLKGATED_MASK_SHIFT(*status);
    if (!clk_en && !clk_st)
    {
        *config = APMU_CLKRSTGEN_LANMACCLK2XCONFIG_CLKEN_REPLACE_VAL(*config,1);
    }
    *config = APMU_CLKRSTGEN_LANMACCLK2XCONFIG_CLKEN_REPLACE_VAL(*config,power_on);
    // wait for bit
    while (APMU_CLKRSTGEN_LANMACCLK2XSTATUS_CLKGATED_MASK_SHIFT(*status) == power_on);
    // restor APBusCLK_LAN state
    config = APMU_CLKRSTGEN_REG_ADDR_OF(APBusClk_LANConfig_ClkGenConfig);

    *config = APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,apbusclk_lan_en_state);

//lanrxclk
    config = APMU_CLKRSTGEN_REG_ADDR_OF(LANRxClkGenConfig);
    status = APMU_CLKRSTGEN_REG_ADDR_OF(LANRxClkGenStatus);

    clk_en = APMU_CLKRSTGEN_LANRXCLKGENCONFIG_CLKEN_MASK_SHIFT(*config);
    clk_st = APMU_CLKRSTGEN_LANRXCLKGENSTATUS_CLKGATED_MASK_SHIFT(*status);
    if (!clk_en && !clk_st)
    {
        *config = APMU_CLKRSTGEN_LANRXCLKGENCONFIG_CLKEN_REPLACE_VAL(*config,1);
    }
    *config = APMU_CLKRSTGEN_LANRXCLKGENCONFIG_CLKEN_REPLACE_VAL(*config,power_on);
    // wait for bit
    while (APMU_CLKRSTGEN_LANRXCLKGENSTATUS_CLKGATED_MASK_SHIFT(*status) == power_on);
#endif

    return true;
}

static bool apmu_io_wrap_clocks_enable(bool power_on)
{
#ifndef EMULATION
    // power_on = true --> enable clocks, and wait until nolonger gating
    // clocks to enable apbusclk_iowrap and qspiclk
    // first: apbusclk_iowrap
    volatile uint32_t *config = APMU_CLKRSTGEN_REG_ADDR_OF(APBusClk_IOWrapConfig_ClkGenConfig);
    volatile uint32_t *status = APMU_CLKRSTGEN_REG_ADDR_OF(APBusClk_IOWrapConfig_ClkGenStatus);

    uint32_t clk_en = APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(*config);
    uint32_t clk_st = APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status);
    if (!clk_en && !clk_st)
    {
        *config = APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,1);
    }
    *config = APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,power_on);
    // wait for bit
    while (APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status) == power_on);

    // now: qspiclk
    config = APMU_CLKRSTGEN_REG_ADDR_OF(QSPIClkConfig_ClkGenConfig);
    status = APMU_CLKRSTGEN_REG_ADDR_OF(QSPIClkConfig_ClkGenStatus);

    clk_en = APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(*config);
    clk_st = APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status);
    if (!clk_en && !clk_st)
    {
        *config = APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,1);
    }
    *config = APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,power_on);
    // wait for bit
    while (APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status) == power_on);
#endif

    return true;
}

static bool apmu_cdma_clocks_enable(bool power_on)
{
    // power_on = true --> enable clocks, and wait until nolonger gating
    // clocks to enable apbusclk_iowrap and qspiclk
    // first: apbusclk_iowrap
    volatile uint32_t *config = APMU_CLKRSTGEN_REG_ADDR_OF(APBusClk_CDMAConfig_ClkGenConfig);
    volatile uint32_t *status = APMU_CLKRSTGEN_REG_ADDR_OF(APBusClk_CDMAConfig_ClkGenStatus);

    uint32_t clk_en = APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(*config);
    uint32_t clk_st = APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status);
    if (!clk_en && !clk_st)
    {
        *config = APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,1);
    }
    *config = APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,power_on);
    // wait for bit
#ifndef EMULATION
    while (APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status) == power_on);
#endif



    return true;
}

static bool apmu_power_up_MC5_Clocks(void)
{
#ifndef EMULATION
    // Turning on MC5 requires some additional steps in sequence to enable correctly

    //Step#1 enable ddr clock
    volatile uint32_t *config = APMU_CLKRSTGEN_REG_ADDR_OF(DDRClkConfig);
    volatile uint32_t *status = APMU_CLKRSTGEN_REG_ADDR_OF(DDRClkStatus);

    *config = APMU_CLKRSTGEN_DDRCLKCONFIG_CLKEN_REPLACE_VAL(*config,1);
    while (APMU_CLKRSTGEN_DDRCLKSTATUS_CLKGATED_MASK_SHIFT(*status) == 1);


    //step#2  Power up MC_PHY
    config = &(((APMU_MISC_REG_REGS_t*)AP_APMU_APMU_MISC_REG_BASE)->GENERIC_CTRL);
    *config = APMU_MISC_REG_GENERIC_CTRL_MC_PHY_PWRDWN_REPLACE_VAL(*config,0);

    //step#3 syncronize after frequency changes
    volatile uint32_t *phy_ctrl_6 = &(((MC_REGS_t*)AP_MC_BASE)->CH0_PHY_Control_6);
    volatile uint32_t *phy_ctrl_9 = &(((MC_REGS_t*)AP_MC_BASE)->CH0_PHY_Control_9);
    *phy_ctrl_6 = MC_CH0_PHY_CONTROL_6_MC_PHY_SYNC_EN_SEL_REPLACE_VAL(*phy_ctrl_6,0);
    *phy_ctrl_9 = MC_CH0_PHY_CONTROL_9_PHY_SYNC_EN_REPLACE_VAL(*phy_ctrl_9,1);
    *phy_ctrl_6 = MC_CH0_PHY_CONTROL_6_MC_PHY_SYNC_EN_SEL_REPLACE_VAL(*phy_ctrl_6,1);
    *phy_ctrl_9 = MC_CH0_PHY_CONTROL_9_PHY_SYNC_EN_REPLACE_VAL(*phy_ctrl_9,1);
    *phy_ctrl_6 = MC_CH0_PHY_CONTROL_6_MC_PHY_SYNC_EN_SEL_REPLACE_VAL(*phy_ctrl_6,0);

    //step#4 If any parts are in self-refresh then need to Exit self-refresh (Not Needed for booting)
    volatile uint32_t *phy_ctrl = &(((MC_REGS_t*)AP_MC_BASE)->DRAM_STATUS);
    if((MC_DRAM_STATUS_STATE_SR00_MASK || MC_DRAM_STATUS_STATE_SR10_MASK ||
        MC_DRAM_STATUS_STATE_SR01_MASK || MC_DRAM_STATUS_STATE_SR11_MASK ||
        MC_DRAM_STATUS_STATE_SR02_MASK || MC_DRAM_STATUS_STATE_SR12_MASK ||
        MC_DRAM_STATUS_STATE_SR03_MASK || MC_DRAM_STATUS_STATE_SR13_MASK )
        && *phy_ctrl)
    {
        phy_ctrl = &(((MC_REGS_t*)AP_MC_BASE)->USER_COMMAND_0);
        volatile uint32_t cmd =  0;
        cmd = MC_USER_COMMAND_0_CH_REPLACE_VAL(cmd, 0x1);
        cmd = MC_USER_COMMAND_0_CS_REPLACE_VAL(cmd, 0xF);
        cmd = MC_USER_COMMAND_0_SR_REQ_REPLACE_VAL(cmd, 0x2);
        cmd = MC_USER_COMMAND_0_SDRAM_INIT_REQ_REPLACE_VAL(cmd, 0x0);
        *phy_ctrl = cmd;
    }


    // wait 1us
    PMU_DEVICE_US(1);

    //step#5 Enable mcbus guard clock
    config = APMU_CLKRSTGEN_REG_ADDR_OF(MCBusGuardConfig_ClkGenConfig);
    status = APMU_CLKRSTGEN_REG_ADDR_OF(MCBusGuardConfig_ClkGenStatus);

    *config = APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,1);
    while (APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status) == 1);
#endif
    return true;
}

static bool apmu_power_down_MC5_Clocks(void)
{
#ifndef EMULATION
    // Turning off MC5 requires some additional steps in sequence to dissable correctly

    // step #1 dissable mcBus gaud clock
    volatile uint32_t *config = APMU_CLKRSTGEN_REG_ADDR_OF(MCBusGuardConfig_ClkGenConfig);
    volatile uint32_t *status = APMU_CLKRSTGEN_REG_ADDR_OF(MCBusGuardConfig_ClkGenStatus);

    *config = APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,0);
    while (APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status) == 0);

    volatile uint32_t *phy_ctrl = &(((MC_REGS_t*)AP_MC_BASE)->USER_COMMAND_0);
    // step #2 Flush MC buffers to preserve DRAM
    volatile uint32_t cmd =  0;
    cmd = MC_USER_COMMAND_0_CH_REPLACE_VAL(cmd, 0x1);
    cmd = MC_USER_COMMAND_0_CS_REPLACE_VAL(cmd, 0xF);
    cmd = MC_USER_COMMAND_0_WCB_DRAIN_REQ_REPLACE_VAL(cmd, 0x1);
    cmd = MC_USER_COMMAND_0_SDRAM_INIT_REQ_REPLACE_VAL(cmd, 0x0);
    *phy_ctrl = cmd;

    // step #3 Enter Self refresh
    cmd =  0;
    cmd = MC_USER_COMMAND_0_CH_REPLACE_VAL(cmd, 0x1);
    cmd = MC_USER_COMMAND_0_CS_REPLACE_VAL(cmd, 0xF);
    cmd = MC_USER_COMMAND_0_SR_REQ_REPLACE_VAL(cmd, 0x1);
    cmd = MC_USER_COMMAND_0_SDRAM_INIT_REQ_REPLACE_VAL(cmd, 0x0);
    *phy_ctrl = cmd;

    // wait 1us
    PMU_DEVICE_US(1);

    //step#4  Power down MC_PHY
    config = &(((APMU_MISC_REG_REGS_t*)AP_APMU_APMU_MISC_REG_BASE)->GENERIC_CTRL);
    *config = APMU_MISC_REG_MC5_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(*config,1);
    while(APMU_MISC_REG_MC5_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(*config,0));

    // step # 4 dissable ddrclk
    config = APMU_CLKRSTGEN_REG_ADDR_OF(DDRClkConfig);
    status = APMU_CLKRSTGEN_REG_ADDR_OF(DDRClkStatus);

    *config = APMU_CLKRSTGEN_DDRCLKCONFIG_CLKEN_REPLACE_VAL(*config,0);
    while (APMU_CLKRSTGEN_DDRCLKSTATUS_CLKGATED_MASK_SHIFT(*status) == 0);
#endif

    return true;
}

static bool apmu_mc5_clocks_enable(bool power_on)
{
    //Turning on memory controler clocks is more complicated than other devices.
    //Therefore, special power up and power down sequences are required
    if(power_on)
        return apmu_power_up_MC5_Clocks();
    else
        return apmu_power_down_MC5_Clocks();

    return true;
}

static bool ipmu_cdma_clocks_enable(bool power_on)
{
    // power_on = true --> enable clocks, and wait until nolonger gating
    // clocks to enable ipsbusclk_mvdo, hipsclk, hipwmclk
    // ipsbusclk_dec
    volatile uint32_t *config = &(((IPMU_CLKRSTGEN_REGS_t*)IPS_IPMU_IPMU_CLKRSTGEN_BASE)->IPSBusClk_CDMAConfig_ClkGenConfig);
    volatile uint32_t *status = &(((IPMU_CLKRSTGEN_REGS_t*)IPS_IPMU_IPMU_CLKRSTGEN_BASE)->IPSBusClk_CDMAConfig_ClkGenStatus);
    uint32_t clk_en = IPMU_CLKRSTGEN_IPSBUSCLK_CDMACONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(*config);
    uint32_t clk_st = IPMU_CLKRSTGEN_IPSBUSCLK_CDMACONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status);
    if (!clk_en && !clk_st)
    {
        *config = IPMU_CLKRSTGEN_IPSBUSCLK_CDMACONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,1);
    }
    *config = IPMU_CLKRSTGEN_IPSBUSCLK_CDMACONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,power_on);
    // wait for bit
#ifndef EMULATION
    while (IPMU_CLKRSTGEN_IPSBUSCLK_CDMACONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status) == power_on);
#endif

    return true;
}

static bool pmu_smmu_clocks_enable(bool power_on)
{
    /// power_on = true --> enable clocks, and wait until nolonger gating
    volatile uint32_t *config = APMU_CLKRSTGEN_REG_ADDR_OF(SMMUClkConfig_ClkGenConfig);
    volatile uint32_t *status = APMU_CLKRSTGEN_REG_ADDR_OF(SMMUClkConfig_ClkGenStatus);


    uint32_t clk_en = APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(*config);
    uint32_t clk_st = APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status);
    if (!clk_en && !clk_st)
    {
        *config = APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,1);
    }

    *config = APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,power_on);
    // wait for bit
#ifndef EMULATION
    while (APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status) == power_on);
#endif
    return true;
}

static bool ipmu_dec_clocks_enable(bool power_on)
{
    // power_on = true --> enable clocks, and wait until nolonger gating
    // clocks to enable ipsbusclk_dec, fastnssclk, nssclk
    // ipsbusclk_dec
    volatile uint32_t *config = &(((IPMU_CLKRSTGEN_REGS_t*)IPS_IPMU_IPMU_CLKRSTGEN_BASE)->IPSBusClk_DECConfig_ClkGenConfig);
    volatile uint32_t *status = &(((IPMU_CLKRSTGEN_REGS_t*)IPS_IPMU_IPMU_CLKRSTGEN_BASE)->IPSBusClk_DECConfig_ClkGenStatus);
    uint32_t clk_en = IPMU_CLKRSTGEN_IPSBUSCLK_DECCONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(*config);
    uint32_t clk_st = IPMU_CLKRSTGEN_IPSBUSCLK_DECCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status);
    if (!clk_en && !clk_st)
    {
        *config = IPMU_CLKRSTGEN_IPSBUSCLK_DECCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,1);
    }
    *config = IPMU_CLKRSTGEN_IPSBUSCLK_DECCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,power_on);
    // wait for bit
#ifndef EMULATION
    while (IPMU_CLKRSTGEN_IPSBUSCLK_DECCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status) == power_on);
#endif

    // fastnssclk
    config = &(((IPMU_CLKRSTGEN_REGS_t*)IPS_IPMU_IPMU_CLKRSTGEN_BASE)->FastNSSClkConfig_ClkGenConfig);
    status = &(((IPMU_CLKRSTGEN_REGS_t*)IPS_IPMU_IPMU_CLKRSTGEN_BASE)->FastNSSClkConfig_ClkGenStatus);

    *config = IPMU_CLKRSTGEN_FASTNSSCLKCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,power_on);
    // wait for bit
#ifndef EMULATION
    while (IPMU_CLKRSTGEN_FASTNSSCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status) == power_on);
#endif

    // nssclk //TODO: order matter in this case ipsbusclk_dec needs to be enabled?
    config = &(((IPMU_CLKRSTGEN_REGS_t*)IPS_IPMU_IPMU_CLKRSTGEN_BASE)->FastNSSClkConfig_ClkGenConfig);
    status = &(((IPMU_CLKRSTGEN_REGS_t*)IPS_IPMU_IPMU_CLKRSTGEN_BASE)->FastNSSClkConfig_ClkGenStatus);
    clk_en = IPMU_CLKRSTGEN_FASTNSSCLKCONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(*config);
    clk_st = IPMU_CLKRSTGEN_FASTNSSCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status);
    if (!clk_en && !clk_st)
    {
        *config = IPMU_CLKRSTGEN_FASTNSSCLKCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,1);
    }
    *config = IPMU_CLKRSTGEN_FASTNSSCLKCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,power_on);
    // wait for bit
#ifndef EMULATION
    while (IPMU_CLKRSTGEN_FASTNSSCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status) == power_on);
#endif

    return true;

}

static bool ipmu_mvdo_clocks_enable(bool power_on)
{
    // power_on = true --> enable clocks, and wait until nolonger gating
    // clocks to enable ipsbusclk_mvdo, hipsclk, hipwmclk
    // ipsbusclk_dec
    volatile uint32_t *config = &(((IPMU_CLKRSTGEN_REGS_t*)IPS_IPMU_IPMU_CLKRSTGEN_BASE)->IPSBusClk_MVDOConfig_ClkGenConfig);
    volatile uint32_t *status = &(((IPMU_CLKRSTGEN_REGS_t*)IPS_IPMU_IPMU_CLKRSTGEN_BASE)->IPSBusClk_MVDOConfig_ClkGenStatus);
    uint32_t clk_en = IPMU_CLKRSTGEN_IPSBUSCLK_MVDOCONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(*config);
    uint32_t clk_st = IPMU_CLKRSTGEN_IPSBUSCLK_MVDOCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status);
    if (!clk_en && !clk_st)
    {
        *config = IPMU_CLKRSTGEN_IPSBUSCLK_MVDOCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,1);
    }
    *config = IPMU_CLKRSTGEN_IPSBUSCLK_MVDOCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,power_on);
    // wait for bit
#ifndef EMULATION
    while (IPMU_CLKRSTGEN_IPSBUSCLK_MVDOCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status) == power_on);
#endif

    // hipsclk
    config = &(((MPMU_CLKRSTGEN_REGS_t*)AP_MPMU_MPMU_CLKRSTGEN_BASE)->HipsClkConfig);
    status = &(((MPMU_CLKRSTGEN_REGS_t*)AP_MPMU_MPMU_CLKRSTGEN_BASE)->HipsClkStatus);
    *config = MPMU_CLKRSTGEN_HIPSCLKCONFIG_CLKEN_REPLACE_VAL(*config,power_on);

    // hipwmclk
    config = &(((MPMU_CLKRSTGEN_REGS_t*)AP_MPMU_MPMU_CLKRSTGEN_BASE)->HiPWMClkConfig);
    status = &(((MPMU_CLKRSTGEN_REGS_t*)AP_MPMU_MPMU_CLKRSTGEN_BASE)->HiPWMClkStatus);
    *config = MPMU_CLKRSTGEN_HIPWMCLKCONFIG_CLKEN_REPLACE_VAL(*config,power_on);
    // wait for bit
#ifndef EMULATION
    while (MPMU_CLKRSTGEN_HIPWMCLKSTATUS_CLKGATED_MASK_SHIFT(*status) == power_on);
#endif

    return true;
}

static bool ipmu_ips_apb_clocks_enable(bool power_on)
{
    // power_on = true --> enable clocks, and wait until nolonger gating
    // clocks to enable ipsbusclk_apb, adctsenclk, audioclk
    // ipsbusclk_apb
    volatile uint32_t *config = &(((IPMU_CLKRSTGEN_REGS_t*)IPS_IPMU_IPMU_CLKRSTGEN_BASE)->IPSBusClk_APBConfig_ClkGenConfig);
    volatile uint32_t *status = &(((IPMU_CLKRSTGEN_REGS_t*)IPS_IPMU_IPMU_CLKRSTGEN_BASE)->IPSBusClk_APBConfig_ClkGenStatus);
    uint32_t clk_en = IPMU_CLKRSTGEN_IPSBUSCLK_APBCONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(*config);
    uint32_t clk_st = IPMU_CLKRSTGEN_IPSBUSCLK_APBCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status);
    if (!clk_en && !clk_st)
    {
        *config = IPMU_CLKRSTGEN_IPSBUSCLK_APBCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,1);
    }
    *config = IPMU_CLKRSTGEN_IPSBUSCLK_APBCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,power_on);
    // wait for bit
#ifndef EMULATION
    while (IPMU_CLKRSTGEN_IPSBUSCLK_APBCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status) == power_on);
#endif

    //adctsenclk
    config = &(((IPMU_CLKRSTGEN_REGS_t*)IPS_IPMU_IPMU_CLKRSTGEN_BASE)->AdcTsenClkConfig_ClkGenConfig);
    status = &(((IPMU_CLKRSTGEN_REGS_t*)IPS_IPMU_IPMU_CLKRSTGEN_BASE)->AdcTsenClkConfig_ClkGenStatus);
    *config = IPMU_CLKRSTGEN_ADCTSENCLKCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,power_on);
    // wait for bit
#ifndef EMULATION
    while (IPMU_CLKRSTGEN_ADCTSENCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status) == power_on);
#endif

    //audioclk
    config = &(((IPMU_CLKRSTGEN_REGS_t*)IPS_IPMU_IPMU_CLKRSTGEN_BASE)->AudioClkConfig_ClkGenConfig);
    status = &(((IPMU_CLKRSTGEN_REGS_t*)IPS_IPMU_IPMU_CLKRSTGEN_BASE)->AudioClkConfig_ClkGenStatus);
    *config = IPMU_CLKRSTGEN_AUDIOCLKCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(*config,power_on);
    // wait for bit
#ifndef EMULATION
    while (IPMU_CLKRSTGEN_AUDIOCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(*status) == power_on);
#endif

    return true;
}


// SRAM functions
static bool apmu_lcd_sram_leakage(bool power_on)
{
    volatile uint32_t *sramctrl = &(((APMU_MISC_REG_REGS_t*)AP_APMU_APMU_MISC_REG_BASE)->lcd_SRAMPdwn_SPCTL);
    // Set bits to 0 to bring out of lowleakage when powering up. Set to 1 to enter lowleakage when powering down
    *sramctrl = APMU_MISC_REG_LCD_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(*sramctrl,!power_on);
    *sramctrl = APMU_MISC_REG_LCD_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(*sramctrl,!power_on);

    return true;
}

static bool apmu_ap_apb_sram_leakage(bool power_on)
{

    // Set bits to 0 to bring out of lowleakage when powering up. Set to 1 to enter lowleakage when powering down
    volatile uint32_t *sramctrl = &(((APMU_MISC_REG_REGS_t*)AP_APMU_APMU_MISC_REG_BASE)->PDMA0_SRAMPdwn_SPCTL);
    *sramctrl = APMU_MISC_REG_PDMA0_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(*sramctrl,!power_on);
    *sramctrl = APMU_MISC_REG_PDMA0_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(*sramctrl,!power_on);

    sramctrl = &(((APMU_MISC_REG_REGS_t*)AP_APMU_APMU_MISC_REG_BASE)->PDMA1_SRAMPdwn_SPCTL);
    *sramctrl = APMU_MISC_REG_PDMA1_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(*sramctrl,!power_on);
    *sramctrl = APMU_MISC_REG_PDMA1_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(*sramctrl,!power_on);

    return true;
}

static bool apmu_sata_sram_leakage(bool power_on)
{
    volatile uint32_t *sramctrl = &(((APMU_MISC_REG_REGS_t*)AP_APMU_APMU_MISC_REG_BASE)->SATA_SRAMPdwn_SPCTL);
    // Set bits to 0 to bring out of lowleakage when powering up. Set to 1 to enter lowleakage when powering down
    *sramctrl = APMU_MISC_REG_SATA_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(*sramctrl,!power_on);
    *sramctrl = APMU_MISC_REG_SATA_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(*sramctrl,!power_on);

    return true;
}

static bool apmu_pcie_dm2_sram_leakage(bool power_on)
{
    volatile uint32_t *sramctrl = &(((APMU_MISC_REG_REGS_t*)AP_APMU_APMU_MISC_REG_BASE)->PCIe_dm2_SRAMPdwn_SPCTL);
    // Set bits to 0 to bring out of lowleakage when powering up. Set to 1 to enter lowleakage when powering down
    *sramctrl = APMU_MISC_REG_PCIE_DM2_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(*sramctrl,!power_on);
    *sramctrl = APMU_MISC_REG_PCIE_DM2_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(*sramctrl,!power_on);

    return true;
}

static bool apmu_pcie_dm4_sram_leakage(bool power_on)
{
    volatile uint32_t *sramctrl = &(((APMU_MISC_REG_REGS_t*)AP_APMU_APMU_MISC_REG_BASE)->PCIe_dm4_SRAMPdwn_SPCTL);
    // Set bits to 0 to bring out of lowleakage when powering up. Set to 1 to enter lowleakage when powering down
    *sramctrl = APMU_MISC_REG_PCIE_DM4_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(*sramctrl,!power_on);
    *sramctrl = APMU_MISC_REG_PCIE_DM4_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(*sramctrl,!power_on);

    return true;
}

static bool apmu_sdmmc_sram_leakage(bool power_on)
{
    volatile uint32_t *sramctrl = &(((APMU_MISC_REG_REGS_t*)AP_APMU_APMU_MISC_REG_BASE)->SDMMC_SRAMPdwn_SPCTL);
    // Set bits to 0 to bring out of lowleakage when powering up. Set to 1 to enter lowleakage when powering down
    *sramctrl = APMU_MISC_REG_SDMMC_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(*sramctrl,!power_on);
    *sramctrl = APMU_MISC_REG_SDMMC_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(*sramctrl,!power_on);

    return true;
}

// apbus iowrap funcitons
static bool apmu_ndsmc_sram_leakage(bool power_on)
{
    // Set bits to 0 to bring out of lowleakage when powering up. Set to 1 to enter lowleakage when powering down
    // NAND sram is associated on apbusclkdiv2 and both are associated with NDSMC device
    volatile uint32_t *sramctrl = &(((APMU_MISC_REG_REGS_t*)AP_APMU_APMU_MISC_REG_BASE)->NAND_SRAMPdwn_SPCTL);
    *sramctrl = APMU_MISC_REG_NAND_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(*sramctrl,!power_on);
    *sramctrl = APMU_MISC_REG_NAND_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(*sramctrl,!power_on);

    return true;
}

static bool apmu_usb3_sram_leakage(bool power_on)
{
    volatile uint32_t *sramctrl = &(((APMU_MISC_REG_REGS_t*)AP_APMU_APMU_MISC_REG_BASE)->usb3d_SRAMPdwn_SPCTL);
    // Set bits to 0 to bring out of lowleakage when powering up. Set to 1 to enter lowleakage when powering down
    *sramctrl = APMU_MISC_REG_USB3D_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(*sramctrl,!power_on);
    *sramctrl = APMU_MISC_REG_USB3D_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(*sramctrl,!power_on);

    //TODO: is the usb3h register connected to anything?

    return true;
}

static bool apmu_usb2_sram_leakage(bool power_on)
{
    // Set bits to 0 to bring out of lowleakage when powering up. Set to 1 to enter lowleakage when powering down
    volatile uint32_t *sramctrl = &(((APMU_MISC_REG_REGS_t*)AP_APMU_APMU_MISC_REG_BASE)->usbdm_SRAMPdwn_SPCTL);
    *sramctrl = APMU_MISC_REG_USBDM_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(*sramctrl,!power_on);
    *sramctrl = APMU_MISC_REG_USBDM_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(*sramctrl,!power_on);

    sramctrl = &(((APMU_MISC_REG_REGS_t*)AP_APMU_APMU_MISC_REG_BASE)->usbh_SRAMPdwn_SPCTL);
    *sramctrl = APMU_MISC_REG_USBH_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(*sramctrl,!power_on);
    *sramctrl = APMU_MISC_REG_USBH_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(*sramctrl,!power_on);

    return true;
}

static bool apmu_lan_sram_leakage(bool power_on)
{
    // Set bits to 0 to bring out of lowleakage when powering up. Set to 1 to enter lowleakage when powering down
    volatile uint32_t *sramctrl = &(((APMU_MISC_REG_REGS_t*)AP_APMU_APMU_MISC_REG_BASE)->LANRX_SRAMPdwn_SPCTL);
    *sramctrl = APMU_MISC_REG_LANRX_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(*sramctrl,!power_on);
    *sramctrl = APMU_MISC_REG_LANRX_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(*sramctrl,!power_on);

    sramctrl = &(((APMU_MISC_REG_REGS_t*)AP_APMU_APMU_MISC_REG_BASE)->LANTX_SRAMPdwn_SPCTL);
    *sramctrl = APMU_MISC_REG_LANTX_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(*sramctrl,!power_on);
    *sramctrl = APMU_MISC_REG_LANTX_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(*sramctrl,!power_on);
    return true;
}

// apbus iowrap funcitons
static bool apmu_iowrap_sram_leakage(bool power_on)
{
    // Set bits to 0 to bring out of lowleakage when powering up. Set to 1 to enter lowleakage when powering down
    // io wrap clock has 5 srams acosiated with it- NAND, SCCP, M2M0, M2M1, SDMMC
    // NAND SDMMC are also handled by there own devices so probably should be removed here.
    volatile uint32_t *sramctrl = &(((APMU_MISC_REG_REGS_t*)AP_APMU_APMU_MISC_REG_BASE)->NAND_SRAMPdwn_SPCTL);
    *sramctrl = APMU_MISC_REG_NAND_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(*sramctrl,!power_on);
    *sramctrl = APMU_MISC_REG_NAND_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(*sramctrl,!power_on);

    sramctrl = &(((APMU_MISC_REG_REGS_t*)AP_APMU_APMU_MISC_REG_BASE)->SCCP_SRAMPdwn_SPCTL);
    *sramctrl = APMU_MISC_REG_SCCP_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(*sramctrl,!power_on);
    *sramctrl = APMU_MISC_REG_SCCP_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(*sramctrl,!power_on);

    sramctrl = &(((APMU_MISC_REG_REGS_t*)AP_APMU_APMU_MISC_REG_BASE)->M2M0_SRAMPdwn_SPCTL);
    *sramctrl = APMU_MISC_REG_M2M0_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(*sramctrl,!power_on);
    *sramctrl = APMU_MISC_REG_M2M0_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(*sramctrl,!power_on);

    sramctrl = &(((APMU_MISC_REG_REGS_t*)AP_APMU_APMU_MISC_REG_BASE)->M2M1_SRAMPdwn_SPCTL);
    *sramctrl = APMU_MISC_REG_M2M1_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(*sramctrl,!power_on);
    *sramctrl = APMU_MISC_REG_M2M1_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(*sramctrl,!power_on);

    sramctrl = &(((APMU_MISC_REG_REGS_t*)AP_APMU_APMU_MISC_REG_BASE)->SDMMC_SRAMPdwn_SPCTL);
    *sramctrl = APMU_MISC_REG_SDMMC_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(*sramctrl,!power_on);
    *sramctrl = APMU_MISC_REG_SDMMC_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(*sramctrl,!power_on);

    return true;
}


static bool apmu_mc5_sram_leakage(bool power_on)
{
    volatile uint32_t *sramctrl = &(((APMU_MISC_REG_REGS_t*)AP_APMU_APMU_MISC_REG_BASE)->MC5_SRAMPdwn_SPCTL);
    // Set bits to 0 to bring out of lowleakage when powering up. Set to 1 to enter lowleakage when powering down
    *sramctrl = APMU_MISC_REG_MC5_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(*sramctrl,!power_on);
    *sramctrl = APMU_MISC_REG_MC5_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(*sramctrl,!power_on);

    return true;
}

static bool pmu_smmu_sram_leakage(bool power_on)
{
    volatile uint32_t *sramctrl = &(((APMU_MISC_REG_REGS_t*)AP_APMU_APMU_MISC_REG_BASE)->SMMU_SRAMPdwn_SPCTL);
    // Set bits to 0 to bring out of lowleakage when powering up. Set to 1 to enter lowleakage when powering down
    *sramctrl = APMU_MISC_REG_SMMU_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(*sramctrl,!power_on);
    *sramctrl = APMU_MISC_REG_SMMU_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(*sramctrl,!power_on);

    sramctrl = &(((IPMU_MISC_REG_REGS_t*)IPS_IPMU_IPMU_MISC_REG_BASE)->IPSSMMUSRAMPdwn_SPCTL);
    *sramctrl = IPMU_MISC_REG_IPSSMMUSRAMPDWN_SPCTL_PDWN_REPLACE_VAL(*sramctrl,!power_on);
    *sramctrl = IPMU_MISC_REG_IPSSMMUSRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(*sramctrl,!power_on);

    return true;
}


static bool ipmu_ips_apb_sram_leakage(bool power_on)
{
    volatile uint32_t *sramctrl = &(((IPMU_MISC_REG_REGS_t*)IPS_IPMU_IPMU_MISC_REG_BASE)->IPSSRAMPdwn_SPCTL);
    // Set bits to 0 to bring out of lowleakage when powering up. Set to 1 to enter lowleakage when powering down
    *sramctrl = IPMU_MISC_REG_IPSSRAMPDWN_SPCTL_PDWN_REPLACE_VAL(*sramctrl,!power_on);
    *sramctrl = IPMU_MISC_REG_IPSSRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(*sramctrl,!power_on);

    return true;
}

/*
 * APMU empty routine - does nothing very well
 */
static bool pmu_device_empty(bool power_on)
{
    // intentionally empty
    return true;
}

typedef bool (*PDFT_HANDLER)(bool power_on);
typedef struct
{
    PDFT_HANDLER    clocks_enable;
    PDFT_HANDLER    sram_leakage;
    PDFT_HANDLER    need_device_check;
    PDFT_HANDLER    depend_device_check;
} power_device_function_table_t;

/*
 * Generic sequence for turning PMU Device ON:
 * 1) Checkthat all required devices are On
 * 2) Check Enable SRAM power
 * 3) enable clocks
 *
 * Generic sequence for turning PMU Device OFF:
 * 1) Check that all dependent devices are Off
 * 2) Disable clocks
 * 3) Put SRAMs in low power state
 *
 * This file does NOT govern the sequence of device turn on user needs turn devices and island in correct order.
 *
 */
static power_device_function_table_t device_handler[] =
{
[APMU_DEVICE_LCD]      = {.clocks_enable = apmu_lcd_clocks_enable      , .sram_leakage = apmu_lcd_sram_leakage      , .need_device_check = pmu_device_empty   , .depend_device_check = pmu_device_empty},
[APMU_DEVICE_AP_APB]   = {.clocks_enable = apmu_AP_APB_clocks_enable   , .sram_leakage = apmu_ap_apb_sram_leakage   , .need_device_check = pmu_device_empty   , .depend_device_check = pmu_device_empty},
[APMU_DEVICE_UART]     = {.clocks_enable = apmu_uart_clocks_enable     , .sram_leakage = pmu_device_empty           , .need_device_check = pmu_device_empty   , .depend_device_check = pmu_device_empty},
[APMU_DEVICE_SPI]      = {.clocks_enable = apmu_spi_clocks_enable      , .sram_leakage = pmu_device_empty           , .need_device_check = pmu_device_empty   , .depend_device_check = pmu_device_empty},
[APMU_DEVICE_SPI2]      = {.clocks_enable = apmu_spi2_clocks_enable      , .sram_leakage = pmu_device_empty           , .need_device_check = pmu_device_empty   , .depend_device_check = pmu_device_empty},
[APMU_DEVICE_SPI3]      = {.clocks_enable = apmu_spi3_clocks_enable      , .sram_leakage = pmu_device_empty           , .need_device_check = pmu_device_empty   , .depend_device_check = pmu_device_empty},
[APMU_DEVICE_I2C]      = {.clocks_enable = apmu_i2c_clocks_enable      , .sram_leakage = pmu_device_empty           , .need_device_check = pmu_device_empty   , .depend_device_check = pmu_device_empty},
[APMU_DEVICE_SATA]     = {.clocks_enable = apmu_sata_clocks_enable     , .sram_leakage = apmu_sata_sram_leakage     , .need_device_check = pmu_device_empty   , .depend_device_check = pmu_device_empty},
[APMU_DEVICE_PCIe_dm2] = {.clocks_enable = apmu_pcie_dm2_clocks_enable , .sram_leakage = apmu_pcie_dm2_sram_leakage , .need_device_check = pmu_device_empty   , .depend_device_check = pmu_device_empty},
[APMU_DEVICE_PCIe_dm4] = {.clocks_enable = apmu_pcie_dm4_clocks_enable , .sram_leakage = apmu_pcie_dm4_sram_leakage , .need_device_check = pmu_device_empty   , .depend_device_check = pmu_device_empty},
[APMU_DEVICE_SDMMC]    = {.clocks_enable = apmu_sdmmc_clocks_enable    , .sram_leakage = apmu_sdmmc_sram_leakage    , .need_device_check = pmu_device_empty   , .depend_device_check = pmu_device_empty},
[APMU_DEVICE_NDSMC]    = {.clocks_enable = apmu_ndsmc_clocks_enable    , .sram_leakage = apmu_ndsmc_sram_leakage    , .need_device_check = pmu_device_empty   , .depend_device_check = pmu_device_empty},
[APMU_DEVICE_USB3]     = {.clocks_enable = apmu_usb3_clocks_enable     , .sram_leakage = apmu_usb3_sram_leakage     , .need_device_check = pmu_device_empty   , .depend_device_check = pmu_device_empty},
[APMU_DEVICE_USB2]     = {.clocks_enable = apmu_usb2_clocks_enable     , .sram_leakage = apmu_usb2_sram_leakage     , .need_device_check = pmu_device_empty   , .depend_device_check = pmu_device_empty},
[APMU_DEVICE_LAN]      = {.clocks_enable = apmu_lan_clocks_enable      , .sram_leakage = apmu_lan_sram_leakage      , .need_device_check = pmu_device_empty   , .depend_device_check = pmu_device_empty},
[APMU_DEVICE_IO_WRAP]  = {.clocks_enable = apmu_io_wrap_clocks_enable  , .sram_leakage = apmu_iowrap_sram_leakage   , .need_device_check = pmu_device_empty   , .depend_device_check = pmu_device_empty},
[APMU_DEVICE_CDMA]     = {.clocks_enable = apmu_cdma_clocks_enable     , .sram_leakage = pmu_device_empty           , .need_device_check = pmu_device_empty   , .depend_device_check = pmu_device_empty},
[PMU_DEVICE_SMMU]      = {.clocks_enable = pmu_smmu_clocks_enable      , .sram_leakage = pmu_smmu_sram_leakage      , .need_device_check = pmu_device_empty   , .depend_device_check = pmu_device_empty},
[IPMU_DEVICE_DEC]      = {.clocks_enable = ipmu_dec_clocks_enable      , .sram_leakage = pmu_device_empty           , .need_device_check = pmu_device_empty   , .depend_device_check = pmu_device_empty},
[IPMU_DEVICE_MVDO]     = {.clocks_enable = ipmu_mvdo_clocks_enable     , .sram_leakage = pmu_device_empty           , .need_device_check = pmu_device_empty   , .depend_device_check = pmu_device_empty},
[IPMU_DEVICE_IPS_APB]  = {.clocks_enable = ipmu_ips_apb_clocks_enable  , .sram_leakage = ipmu_ips_apb_sram_leakage  , .need_device_check = pmu_device_empty   , .depend_device_check = pmu_device_empty},
[IPMU_DEVICE_MC5]      = {.clocks_enable = apmu_mc5_clocks_enable      , .sram_leakage = apmu_mc5_sram_leakage      , .need_device_check = pmu_device_empty   , .depend_device_check = pmu_device_empty},
[IPMU_DEVICE_CDMA]     = {.clocks_enable = ipmu_cdma_clocks_enable     , .sram_leakage = pmu_device_empty           , .need_device_check = pmu_device_empty   , .depend_device_check = pmu_device_empty},
};

bool pmu_power_device(pmu_device_t device, bool power_on)
{

    PMU_DEVICE_PRINTF("%s: device %d; on=%d\r\n",__FUNCTION__,device,power_on);
    if (power_on)
    {
        // using XXX_power.sv verilog for simulations as 'C' code reference. --> islandPowerUp()
        // TBD - do we need to detect if island is already on???
        PMU_DEVICE_PRINTF("need_check\r\n");
        if(!device_handler[device].need_device_check(power_on))
        {
             PMU_DEVICE_PRINTF("needs to be turned on\r\n");
            return false;
        }

        PMU_DEVICE_PRINTF("ensram\r\n");//Bring SRAMs out of low leakage mode
        device_handler[device].sram_leakage(power_on);
        PMU_DEVICE_US(1);
        PMU_DEVICE_PRINTF("enclk\r\n");
        device_handler[device].clocks_enable(power_on);
        PMU_DEVICE_US(1);
        PMU_DEVICE_PRINTF("exit\r\n");
        return true;
    }
    PMU_DEVICE_PRINTF("depend_check\r\n");
    if(!device_handler[device].depend_device_check(power_on))
    {
        PMU_DEVICE_PRINTF("would be killed.So cannot turn off\r\n");
        return false;
    }
    // using XXX_power.sv verilog for simulations as 'C' code reference.  --> islandPowerDown()
    PMU_DEVICE_PRINTF("disclk\r\n");
    device_handler[device].clocks_enable(power_on);
    PMU_DEVICE_PRINTF("dissram\r\n");//place SRAM in low leakage mode
    device_handler[device].sram_leakage(power_on);
    PMU_DEVICE_PRINTF("exit\r\n");
    return true;
}
