/*
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
*/

#include "pll_internal.h"
#include "pll_api.h"
#include "assert.h"
#include "error_types.h"
#include "board_types.h"
#include "regAddrs.h"
#include "SSPLL3P0G_regheaders.h"
#include "SSPLL3P0G_api.h"
#include "ID_utils.h"

// Static Declarations
static error_type_t setupCorePLL( pll_freq_t freq );
static error_type_t setupDDRPLL ( pll_freq_t freq );
static error_type_t setupSYSPLL ( pll_freq_t freq );
static error_type_t setupSCANPLL( pll_freq_t freq );

error_type_t initialize_PLL( board_types_e board_type, pll_freq_t PLL_frequency )
{
    if (is_FPGA())
    {
        return STATUS_OK;
    }
    else
    {
        switch (PLL_frequency)
        {
            case eCORE_PLL_BYPASS:
            case eCORE_PLL_800MHZ:/* RICOH ADD */
            case eCORE_PLL_2400MHZ:
            case eCORE_PLL_2000MHZ:
                return setupCorePLL( PLL_frequency );
                break;
            case eDDR_PLL_2133MHZ:
            case eDDR_PLL_1600MHZ:
            case eDDR_PLL_1200MHZ:
            case eDDR_PLL_800MHZ:
            case eDDR_PLL_1333MHZ:
                return setupDDRPLL( PLL_frequency );
                break;
		    case eSYS_PLL_2500MHZ:
		    case eSYS_PLL_1000MHZ:
                return setupSYSPLL( PLL_frequency );
			    break;
		    case eSCAN_PLL_3000MHZ:
		    case eSCAN_PLL_1500MHZ:
                return setupSCANPLL( PLL_frequency );
			    break;
            default:
                assert(0);
        }

    }
    return STATUS_OK;
}



static error_type_t setupCorePLL( pll_freq_t freq )
{
    uint32_t ret;
    SSPLL3P0G_cfg_p core_pll_cfg_ptr;
    SSPLL3P0G_REGS_t *core_pll_regs_ptr;

    core_pll_regs_ptr = (SSPLL3P0G_REGS_t *)AP_MPMU_CORE_PLL_BASE;

    switch (freq)
    {
        case eCORE_PLL_BYPASS:
            return STATUS_OK;
            break;

        case eCORE_PLL_800MHZ:
        case eCORE_PLL_2400MHZ:
            core_pll_cfg_ptr = (freq == eCORE_PLL_800MHZ) ?
				/* Core PLL: RICOH setting &SSPLL3P0G_CFG_800MHZ_8_25MHZ_SSC_0KHZ_0_CENTER_OFFSET_0_DOWN, Core PLL Config 800 MHz (post div 2), bypass OK */
				mrvl_index_get_cfg_ptr_SSPLL3P0G(SSPLL3P0G_CFG_800MHZ_8_25MHZ_SSC_0KHZ_0_CENTER_OFFSET_0_DOWN) :
				/* Core PLL: &cfg_2400MHz_8_25MHz_SSC_0KHz_0_CENTER_OFFSET_0_DOWN, Core PLL Config 1200 MHz (post div 2), bypass OK */
				mrvl_index_get_cfg_ptr_SSPLL3P0G(SSPLL3P0G_CFG_2400MHZ_8_25MHZ_SSC_0KHZ_0_CENTER_OFFSET_0_DOWN);

            /* Program all the PLL registers (Quick LOCK?) */
            ret = mrvl_prg_bypass_SSPLL3P0G(core_pll_cfg_ptr, core_pll_regs_ptr, NULL, NULL);
            if (ret == 0) 
                return STATUS_OK;
            else
                return FAIL;
            break;

        case eCORE_PLL_2000MHZ:
        	/* Core PLL: &cfg_2000MHz_8_25MHz_SSC_0KHz_0_CENTER_OFFSET_0_DOWN, Core PLL Config 2000 MHz, bypass OK */
            core_pll_cfg_ptr = mrvl_index_get_cfg_ptr_SSPLL3P0G(SSPLL3P0G_CFG_2000MHZ_8_25MHZ_SSC_0KHZ_0_CENTER_OFFSET_0_DOWN);

            /* Program all the PLL registers (Quick LOCK?) */
            ret = mrvl_prg_bypass_SSPLL3P0G(core_pll_cfg_ptr, core_pll_regs_ptr, NULL, NULL);
            if (ret == 0)
                return STATUS_OK;
            else
                return FAIL;
            break;

        // TODO Add in more cases for other Freqs
        default:
            assert(0);
    }
    return STATUS_OK;
}

static error_type_t setupDDRPLL( pll_freq_t freq )
{
    uint32_t ret;
    SSPLL3P0G_cfg_t *ddr_pll_cfg_ptr = NULL;
    SSPLL3P0G_REGS_t *ddr_pll_regs_ptr = NULL;

    ddr_pll_regs_ptr = (SSPLL3P0G_REGS_t *)AP_MPMU_DDR_PLL_BASE;

    switch(freq)
    {
        case eDDR_PLL_2133MHZ:
            /* DDR PLL: &cfg_2133MHz_8_25MHz_SSC_0KHz_0_CENTER_OFFSET_0_DOWN, DDR PLL Config 1, bypass OK */
            ddr_pll_cfg_ptr = mrvl_index_get_cfg_ptr_SSPLL3P0G(SSPLL3P0G_CFG_2133MHZ_8_25MHZ_SSC_0KHZ_0_CENTER_OFFSET_0_DOWN);
            break;
        case eDDR_PLL_1600MHZ:
            /* DDR PLL: &cfg_1600MHz_8_25MHz_SSC_0KHz_0_CENTER_OFFSET_0_DOWN, DDR PLL Config 1, bypass OK */
            ddr_pll_cfg_ptr = mrvl_index_get_cfg_ptr_SSPLL3P0G(SSPLL3P0G_CFG_1600MHZ_8_25MHZ_SSC_0KHZ_0_CENTER_OFFSET_0_DOWN);
            break;
        case eDDR_PLL_1200MHZ:
        	/* DDR PLL: &cfg_2400MHz_8_25MHz_SSC_0KHz_0_CENTER_OFFSET_0_DOWN, Core/DDR PLL Config 1200 MHz (post div 2), bypass OK */
            ddr_pll_cfg_ptr = mrvl_index_get_cfg_ptr_SSPLL3P0G(SSPLL3P0G_CFG_2400MHZ_8_25MHZ_SSC_0KHZ_0_CENTER_OFFSET_0_DOWN);
            break;
        case eDDR_PLL_800MHZ:
            /* DDR PLL: &cfg_800MHz_8_25MHz_SSC_0KHz_0_CENTER_OFFSET_0_DOWN, DDR PLL Config 4 for bring-up, bypass OK */
            ddr_pll_cfg_ptr = mrvl_index_get_cfg_ptr_SSPLL3P0G(SSPLL3P0G_CFG_800MHZ_8_25MHZ_SSCON_0KHZ_0_CENTER_OFFSET_0_DOWN);
            break;
        case eDDR_PLL_1333MHZ:
            /* DDR PLL: &cfg_1333MHz_8_25MHz_SSC_0KHz_0_CENTER_OFFSET_0_UP, DDR PLL Config 8 for bring-up, bypass OK */
            ddr_pll_cfg_ptr = mrvl_index_get_cfg_ptr_SSPLL3P0G(SSPLL3P0G_CFG_1333MHz_8_25MHz_SSC_0KHz_0_CENTER_OFFSET_0_UP);
            break;
        default:
            assert(0); // Only support these three!
    }

    /* Program all the PLL registers (Quick LOCK?) */
    ret = mrvl_prg_bypass_SSPLL3P0G(ddr_pll_cfg_ptr, ddr_pll_regs_ptr, NULL, NULL);
    if (ret != 0) {
        return FAIL;
    }

    if (ddr_pll_cfg_ptr->SSC_CLK_EN) {
        ricoh_ssc_enable(ddr_pll_regs_ptr, NULL, NULL);
    }

    return STATUS_OK;
}

static error_type_t setupSYSPLL ( pll_freq_t freq )
{
    int cnt = 0;
    const unsigned long *pll_setting = NULL;
    volatile unsigned long *syspll_reg_head = (unsigned long *)AP_MPMU_SYS_PLL_BASE;
    const unsigned long c_2500mhz_setting[9] = {
        0x00000003, 0x004b0000, 0x0000000e, 0x0000306c, 0x00000000,
        0x04900004, 0x000a0006, 0x00000000, 0x00010000,
    };
    const unsigned long c_1000mhz_setting[9] = {
        0x00000003, 0x003c0101, 0x0000000c, 0x00002c6c, 0x00000000,
        0x04900004, 0x000A0006, 0x00000000, 0x00010000,
    };

    if ((freq == eSYS_PLL_2500MHZ) || (freq == eSYS_PLL_1000MHZ)) {
        pll_setting = (freq == eSYS_PLL_2500MHZ) ? c_2500mhz_setting : c_1000mhz_setting;
        for (cnt = 0; cnt < 9; cnt++) {
            syspll_reg_head[cnt] = pll_setting[cnt];
        }
    } else {
        return FAIL;
    }

    return STATUS_OK;
}

static error_type_t setupSCANPLL( pll_freq_t freq )
{
    uint32_t ret;
    SSPLL3P0G_cfg_t *scan_pll_cfg_ptr = NULL;
    SSPLL3P0G_REGS_t *scan_pll_regs_ptr = NULL;

    scan_pll_regs_ptr = (SSPLL3P0G_REGS_t *)AP_MPMU_SCAN_PLL_BASE;

    switch(freq) {
    case eSCAN_PLL_3000MHZ:
        scan_pll_cfg_ptr = mrvl_index_get_cfg_ptr_SSPLL3P0G(SSPLL3P0G_CFG_3000MHZ_8_25MHZ_SSC_0KHZ_0_CENTER_OFFSET_0_DOWN);
        break;
    case eSCAN_PLL_1500MHZ:
        scan_pll_cfg_ptr = mrvl_index_get_cfg_ptr_SSPLL3P0G(SSPLL3P0G_CFG_1500MHZ_8_25MHZ_SSC_30KHZ_0_CENTER_OFFSET_0_DOWN);
        break;
    default:
        assert(0);
        break;
    }

    ret = mrvl_prg_bypass_SSPLL3P0G(scan_pll_cfg_ptr, scan_pll_regs_ptr, NULL, NULL);
    if (ret) {
        return FAIL;
    }

    if (scan_pll_cfg_ptr->SSC_CLK_EN) {
        ricoh_ssc_enable(scan_pll_regs_ptr, NULL, NULL);
    }

    return STATUS_OK;
}
