/*
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
*/

#include <stdint.h>
#include <string.h>
#include "pll_internal.h"
#if 0
#include "system_functions.h"
//#include "API_REMAP_utils.c"
//#define msg(ARGS) mrvl_msg ARGS
//#define mrvl_msg(...) ((void)0)
#define mrvl_msg(...) ml_msg(__VA_ARGS__)
#define mrvl_regread32(...) ml_regread32(__VA_ARGS__)
#define mrvl_regwrite32(...) ml_regwrite32(__VA_ARGS__)
#define mrvl_sysAlloc(...) ml_sysAlloc(__VA_ARGS__)
#define mrvl_wait_for_us(...) ml_wait_for_us(__VA_ARGS__)
#endif
#define SSPLL3P0G_DEBUG 0
#include "SSPLL3P0G_regheaders.h"
#include "SSPLL3P0G_api.h"


/*
 *----------------------------------------------------------------------
 * mrvl_prg_SSPLL3P0G(*cfg, *regs, *callback, *cb_data)
 *      &
 * mrvl_prg_bypass_SSPLL3P0G(*cfg, *regs, *callback, *cb_data)
 *----------------------------------------------------------------------
 *
 * This task implements the PLL programming sequence with the settings
 * in the SSPLL3P0G_cfg_p struct instance pointed to by the cfg pointer.
 *
 * The PLL registers are written using the register struct instance pointed
 * to by the reg pointer.
 *
 * Returns a non-zero value if the PLL does not lock.
 *
 * Inputs: Name of SSPLL3P0G_cfg_g struct instance, Name of SSPLL3P0G_REGS_t struct instance
 * Outputs: return value,
 *
 *
 * The implemented programming sequence:
 * 1.  ONLY IF *_bypass_* function called:   Place the PLL in BYPASS mode.
 * 2.  Assert the PLL RESET signal.
 * 3.  Program the PLL registers for desired function
	as defined in cfg (ref/output frequency, dividers, dithering, offest, etc).
 * 4.  De-assert the PLL RESET signal.
 * 5.  Check for lock
 *       - if no lock return non-zero and exit.
 *       - if lock continue.
 * 6.  Assert the SSC block reset signals.
 * 7.  De-assert the SSC block reset signals.
 * 8.  Setup fixed clock output from SSC block.
 * 9.  Setup fixed clock output from SSC block.
 * 10. Exit BYPASS mode.
 * 11. High-speed fixed clock available.
 *
 * This example programs the PLL pointed to by regs_chandle with the the
 * my_pll_cfg_ptr values.
 * Usage: mrvl_prg_SSPLL3P0G( my_pll_cfg_ptr , regs_chandle , callback, cb_data);
 */



uint32_t prg_SSPLL3P0G(
  SSPLL3P0G_cfg_p cfg,
  SSPLL3P0G_REGS_t *regs,
  void (*callback)(uint32_t, void *),
  void *cb_data
  ) {
  uint32_t regVal;
  uint32_t i;

  mrvl_msg(STATUS, "SSPLL3P0G", "Configuring SSPLL_3P0G, regs:0x%08x.\n", regs);

//Assert Reset
  regVal = 0;
  regVal = mrvl_regread32((addr_p)&(regs->rst_prediv));
  regVal =  SSPLL3P0G_RST_PREDIV_REFDIV_REPLACE_VAL(regVal, cfg->REFDIV);
  regVal =  SSPLL3P0G_RST_PREDIV_RESET_REPLACE_VAL(regVal, cfg->RESET);
  regVal =  SSPLL3P0G_RST_PREDIV_RESET_PI_REPLACE_VAL(regVal, cfg->RESET_PI);
  regVal =  SSPLL3P0G_RST_PREDIV_RESET_SSC_REPLACE_VAL(regVal, cfg->RESET_SSC);
  regVal =  SSPLL3P0G_RST_PREDIV_RESET_REPLACE_VAL(regVal, 1);
  mrvl_regwrite32((addr_p)&(regs->rst_prediv), regVal);

//Program registers

  regVal = 0;
  regVal = mrvl_regread32((addr_p)&(regs->mult_postdiv));
  regVal = SSPLL3P0G_MULT_POSTDIV_FBDIV_REPLACE_VAL(regVal, cfg->FBDIV);
  regVal = SSPLL3P0G_MULT_POSTDIV_CLKOUT_DIFF_DIV_SEL_REPLACE_VAL(regVal, cfg->CLKOUT_DIFF_DIV_SEL);
  regVal = SSPLL3P0G_MULT_POSTDIV_CLKOUT_SE_DIV_SEL_REPLACE_VAL(regVal, cfg->CLKOUT_SE_DIV_SEL);
  mrvl_regwrite32((addr_p)&(regs->mult_postdiv), regVal);

  regVal = 0;
  regVal = mrvl_regread32((addr_p)&(regs->kvco));
  regVal = SSPLL3P0G_KVCO_KVCO_REPLACE_VAL(regVal, cfg->KVCO);
  mrvl_regwrite32((addr_p)&(regs->kvco), regVal);

  regVal = 0;
  regVal = mrvl_regread32((addr_p)&(regs->misc));
     regVal = SSPLL3P0G_MISC_PLL_BW_SEL_REPLACE_VAL(regVal, cfg->PLL_BW_SEL);
     regVal = SSPLL3P0G_MISC_CTUNE_REPLACE_VAL(regVal, cfg->CTUNE);
     regVal = SSPLL3P0G_MISC_INTPI_REPLACE_VAL(regVal, cfg->INTPI);
     regVal = SSPLL3P0G_MISC_ICP_REPLACE_VAL(regVal, cfg->ICP);
     regVal = SSPLL3P0G_MISC_VDDM_REPLACE_VAL(regVal, cfg->VDDM);
     regVal = SSPLL3P0G_MISC_VDDL_REPLACE_VAL(regVal, cfg->VDDL);
     mrvl_regwrite32((addr_p)&(regs->misc), regVal);

     regVal = 0;
     regVal = mrvl_regread32((addr_p)&(regs->feedback_mode_deskew));
     regVal = SSPLL3P0G_FEEDBACK_MODE_DESKEW_FBCDLY_REPLACE_VAL(regVal, cfg->FBCDLY);
     regVal = SSPLL3P0G_FEEDBACK_MODE_DESKEW_FBCLK_EXT_SEL_REPLACE_VAL(regVal, cfg->FBCLK_EXT_SEL);
     mrvl_regwrite32((addr_p)&(regs->feedback_mode_deskew), regVal);

     regVal = 0;
     regVal = mrvl_regread32((addr_p)&(regs->offset_mode));
     regVal = SSPLL3P0G_OFFSET_MODE_INTPR_REPLACE_VAL(regVal, cfg->INTPR);
     regVal = SSPLL3P0G_OFFSET_MODE_FD_REPLACE_VAL(regVal, cfg->FD);
     regVal = SSPLL3P0G_OFFSET_MODE_PI_LOOP_MODE_REPLACE_VAL(regVal, cfg->PI_LOOP_MODE);
     regVal = SSPLL3P0G_OFFSET_MODE_FREQ_OFFSET_REPLACE_VAL(regVal, cfg->FREQ_OFFSET);
     regVal = SSPLL3P0G_OFFSET_MODE_FREQ_OFFSET_MODE_SELECTION_REPLACE_VAL(regVal, cfg->FREQ_OFFSET_MODE_SELECTION);
     regVal = SSPLL3P0G_OFFSET_MODE_FREQ_OFFSET_VALID_REPLACE_VAL(regVal, cfg->FREQ_OFFSET_VALID);
#if 0 /* For ricoh system, always set FREQ_OFFSET_EN to 0 */
     regVal = SSPLL3P0G_OFFSET_MODE_FREQ_OFFSET_EN_REPLACE_VAL(regVal, cfg->FREQ_OFFSET_EN);
#endif /* 0 */
     mrvl_regwrite32((addr_p)&(regs->offset_mode), regVal);

     regVal = 0;
     regVal = mrvl_regread32((addr_p)&(regs->fixed_mode_ssc_mode));
     regVal = SSPLL3P0G_FIXED_MODE_SSC_MODE_AVDD1815_SEL_REPLACE_VAL(regVal, cfg->AVDD1815_SEL);
     regVal = SSPLL3P0G_FIXED_MODE_SSC_MODE_CLKOUT_SE_GATING_EN_REPLACE_VAL(regVal, cfg->CLKOUT_SE_GATING_EN);
     regVal = SSPLL3P0G_FIXED_MODE_SSC_MODE_PU_REPLACE_VAL(regVal, cfg->PU);
//     regVal = SSPLL3P0G_FIXED_MODE_SSC_MODE_BYPASS_EN_REPLACE_VAL(regVal, cfg->BYPASS_EN);
     regVal = SSPLL3P0G_FIXED_MODE_SSC_MODE_SSC_MODE_REPLACE_VAL(regVal, cfg->SSC_MODE);
     regVal = SSPLL3P0G_FIXED_MODE_SSC_MODE_CLK_DET_EN_REPLACE_VAL(regVal, cfg->CLK_DET_EN);
     regVal = SSPLL3P0G_FIXED_MODE_SSC_MODE_PI_EN_REPLACE_VAL(regVal, cfg->PI_EN);
     regVal = SSPLL3P0G_FIXED_MODE_SSC_MODE_SSC_CLK_EN_REPLACE_VAL(regVal, cfg->SSC_CLK_EN);
     mrvl_regwrite32((addr_p)&(regs->fixed_mode_ssc_mode), regVal);

     regVal = 0;
     regVal = mrvl_regread32((addr_p)&(regs->ssc_freq_ssc_range));
     regVal = SSPLL3P0G_SSC_FREQ_SSC_RANGE_SSC_FREQ_DIV_REPLACE_VAL(regVal, cfg->SSC_FREQ_DIV);
     regVal = SSPLL3P0G_SSC_FREQ_SSC_RANGE_SSC_RNGE_REPLACE_VAL(regVal, cfg->SSC_RNGE);
     mrvl_regwrite32((addr_p)&(regs->ssc_freq_ssc_range), regVal);

     regVal = 0;
     regVal = mrvl_regread32((addr_p)&(regs->clk_ctrl_marvell_test));
     regVal = SSPLL3P0G_CLK_CTRL_MARVELL_TEST_CLKOUT_DIFF_EN_REPLACE_VAL(regVal, cfg->CLKOUT_DIFF_EN);
     regVal = SSPLL3P0G_CLK_CTRL_MARVELL_TEST_CLKOUT_SOURCE_SEL_REPLACE_VAL(regVal, cfg->CLKOUT_SOURCE_SEL);
     regVal = SSPLL3P0G_CLK_CTRL_MARVELL_TEST_RESERVE_IN_REPLACE_VAL(regVal, cfg->RESERVE_IN);
     regVal = SSPLL3P0G_CLK_CTRL_MARVELL_TEST_TEST_ANA_REPLACE_VAL(regVal, cfg->TEST_ANA);
     mrvl_regwrite32((addr_p)&(regs->clk_ctrl_marvell_test), regVal);



   // Make sure reset pulse is 2us
     mrvl_wait_for_us((uint32_t)2);

   // De-assert reset
     regVal = 0;
     regVal = mrvl_regread32((addr_p)&(regs->rst_prediv));
     regVal =  SSPLL3P0G_RST_PREDIV_RESET_REPLACE_VAL(regVal, 0);
     mrvl_regwrite32((addr_p)&(regs->rst_prediv), regVal);

   //Wait for PLL lock sequence
      mrvl_wait_for_us((uint32_t)50);
   //TEMP: Quick LOCK for simulation only. Wait for PLL lock sequence
   //   mrvl_wait_for_us((uint32_t)1);


   //Wait for PLL Lock signal
      regVal = SSPLL3P0G_LOCK_STATUS_PLL_LOCK_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->lock_status)));
      for (i =0;  ((i < 500) && (regVal != 1)) ; i++) {
         regVal = SSPLL3P0G_LOCK_STATUS_PLL_LOCK_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->lock_status)));
         mrvl_wait_for_us((uint32_t)1);
         mrvl_msg(STATUS, "SSPLL3P0G", "Check for lock.  Iteration = %d\n", i);
      } // while ( regVal != 1 );

      if (1 != regVal)
      {
         mrvl_msg(STATUS, "SSPLL3P0G", "\n\n!! PLL lock checking loop timeout !!  Iteration = %d\n\n", i);
         assert(1 == regVal);
      }

    if (( regVal == 1 ) | SSPLL3P0G_DEBUG ) {

        mrvl_msg(STATUS, "SSPLL3P0G", "Power-up and program the PI\n");
        // Power-up and program the PI
        regVal = 0;
        regVal = mrvl_regread32((addr_p)&(regs->fixed_mode_ssc_mode));
        regVal = SSPLL3P0G_FIXED_MODE_SSC_MODE_PI_EN_REPLACE_VAL(regVal, 1);
        mrvl_regwrite32((addr_p)&(regs->fixed_mode_ssc_mode), regVal);
        //Wait for PI power-up lock sequence
        mrvl_wait_for_us((uint32_t)1);

        mrvl_msg(STATUS, "SSPLL3P0G", "De-Assert SSC Reset signals\n");
        // De-Assert SSC Reset signals
        regVal = 0;
        regVal = mrvl_regread32((addr_p)&(regs->rst_prediv));
        regVal = SSPLL3P0G_RST_PREDIV_RESET_PI_REPLACE_VAL(regVal, 0);
        regVal = SSPLL3P0G_RST_PREDIV_RESET_SSC_REPLACE_VAL(regVal, 0);
        mrvl_regwrite32((addr_p)&(regs->rst_prediv), regVal);
        //Wait for PI to reset
        mrvl_wait_for_us((uint32_t)1);

#if 0 /* For ricoh system, always set FREQ_OFFSET_EN to 0 */
        mrvl_msg(STATUS, "SSPLL3P0G", "Enable FREQ_OFFSET MODE\n");
        //Enable FREQ_OFFSET MODE
        regVal = 0;
        regVal = mrvl_regread32((addr_p)&(regs->offset_mode));
        regVal = SSPLL3P0G_OFFSET_MODE_FREQ_OFFSET_EN_REPLACE_VAL(regVal, 1);
        mrvl_regwrite32((addr_p)&(regs->offset_mode), regVal);
        //Wait for PI to stabilize
        mrvl_wait_for_us((uint32_t)1);
#endif /* 0 */

        mrvl_msg(STATUS, "SSPLL3P0G", "Place PI in loop\n");
        //Place PI in loop
        //TODO:  move into main programming
        regVal = 0;
        regVal = mrvl_regread32((addr_p)&(regs->offset_mode));
        regVal = SSPLL3P0G_OFFSET_MODE_PI_LOOP_MODE_REPLACE_VAL(regVal, 1);
        mrvl_regwrite32((addr_p)&(regs->offset_mode), regVal);
        //Wait for loop to stabilize
        mrvl_wait_for_us((uint32_t)2);


        mrvl_msg(STATUS, "SSPLL3P0G", "Exit Bypass mode\n");
        // Exit Bypass mode
        regVal = 0;
        regVal = mrvl_regread32((addr_p)&(regs->fixed_mode_ssc_mode));
        regVal = SSPLL3P0G_FIXED_MODE_SSC_MODE_BYPASS_EN_REPLACE_VAL(regVal, 0);
        mrvl_regwrite32((addr_p)&(regs->fixed_mode_ssc_mode), regVal);
        mrvl_msg(STATUS, "SSPLL3P0G", "PLL high speed clock available. \n","" );

        //Why was this HERE???
        //mrvl_wait_for_us((uint32_t)40);


        if (callback) {
          callback(0, cb_data);
        }
     }
     else {
         if (callback) { //If pll fails to LOCK, flag return value 2, don't abort
            mrvl_msg(WARNING, "SSPLL3P0G", "PLL failed to lock in %s.\n", __func__);
            callback(2, cb_data);
         }
     }

  return 0;
}

uint32_t mrvl_prg_bypass_SSPLL3P0G(
  SSPLL3P0G_cfg_p cfg,
  SSPLL3P0G_REGS_t *regs,
  void (*callback)(uint32_t, void *),
  void *cb_data
  ) {
  uint32_t regVal;
  uint32_t returnval;

  mrvl_msg(STATUS, "SSPLL3P0G", "Put SSPLL_3P0G in Bypass.\n");
  // Write the SSPLL3P0G Configuration Registers
// Enter Bypass mode
  regVal = 0;
  regVal = mrvl_regread32((addr_p)&(regs->fixed_mode_ssc_mode));
  regVal = SSPLL3P0G_FIXED_MODE_SSC_MODE_BYPASS_EN_REPLACE_VAL(regVal, 1);
  mrvl_regwrite32((addr_p)&(regs->fixed_mode_ssc_mode), regVal);
  mrvl_msg(STATUS, "SSPLL3P0G", "Entering Bypass Mode. \n","" );

  returnval = prg_SSPLL3P0G(cfg, regs, callback, cb_data);
  return returnval;
}

uint32_t mrvl_prg_SSPLL3P0G(
  SSPLL3P0G_cfg_p cfg,
  SSPLL3P0G_REGS_t *regs,
  void (*callback)(uint32_t, void *),
  void *cb_data
  ) {
  uint32_t regVal;
  uint32_t returnval;

  mrvl_msg(STATUS, "SSPLL3P0G", "Put SSPLL_3P0G out of  Bypass.\n");
  // Write the SSPLL3P0G Configuration Registers
// Enter Bypass mode
  regVal = 0;
  regVal = mrvl_regread32((addr_p)&(regs->fixed_mode_ssc_mode));
  regVal = SSPLL3P0G_FIXED_MODE_SSC_MODE_BYPASS_EN_REPLACE_VAL(regVal, 0);
  mrvl_regwrite32((addr_p)&(regs->fixed_mode_ssc_mode), regVal);
  mrvl_msg(STATUS, "SSPLL3P0G", "De-assert Bypass Mode. \n","" );

  returnval = prg_SSPLL3P0G(cfg, regs, callback, cb_data);
  return returnval;
}

/*
 *----------------------------------------------------------------------
 * mrvl_ssc_enable_SSPLL3P0G(*regs, *callback, *cb_data)
 *----------------------------------------------------------------------
 *
 * This function starts the SSC function (SSC_EN 0->1) in the PLL struct
 * instance pointed to by the regs pointer.
 *
 * If the PLL is not locked it returns a non-zero.
 *
 * Inputs: Name of SSPLL3P0G_t struct instance pointer.
 * Outputs:return value
 *
 * This example turns SSC on for the PLL pointed to by regs_chandle.
 * Usage:mrvl_ssc_enable_SSPLL3P0G( regs_chandle , callback, cb_data);
 */

uint32_t mrvl_ssc_enable_SSPLL3P0G(
  SSPLL3P0G_REGS_t *regs,
  void (*callback)(uint32_t, void *),
  void *cb_data
  ) {
  uint32_t regVal;

//Check PLL Lock signal
   regVal = 0;
   regVal = SSPLL3P0G_LOCK_STATUS_PLL_LOCK_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->lock_status)));


   if (( regVal == 1 ) | SSPLL3P0G_DEBUG ) {
//    Enable SSC
     regVal = 0;
     regVal = mrvl_regread32((addr_p)&(regs->fixed_mode_ssc_mode));
     regVal = SSPLL3P0G_FIXED_MODE_SSC_MODE_SSC_CLK_EN_REPLACE_VAL(regVal, 1);
     mrvl_regwrite32((addr_p)&(regs->fixed_mode_ssc_mode), regVal);
     regVal = SSPLL3P0G_LOCK_STATUS_PLL_LOCK_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->lock_status)));

     if (callback) {
       callback(0, cb_data);
     }
   } else {
     mrvl_msg(ERROR, "SSPLL3P0G", "No lock signal detected LOCK_STATUS Reg =  %08x.\n", mrvl_regread32((addr_p)&(regs->lock_status)));
     if (callback) {
         callback(1, cb_data);
      }

   }
  return 0;
}

/*
 *----------------------------------------------------------------------
 * mrvl_ssc_disable_SSPLL3P0G(*regs, *callback, *cb_data)
 *----------------------------------------------------------------------
 *
 * This function ends the SSC function (SSC_EN 1->0) in the PLL struct
 * instance pointed to by the regs pointer.
 *
 *
 * Inputs: Name of SSPLL3P0G_t struct instance pointer.
 * Outputs:return value
 *
 * This example turns SSC off for the PLL pointed to by regs_chandle.
 * Usage:mrvl_ssc_disable_SSPLL3P0G( regs_chandle , callback, cb_data);
 */

uint32_t mrvl_ssc_disable_SSPLL3P0G(
  SSPLL3P0G_REGS_t *regs,
  void (*callback)(uint32_t, void *),
  void *cb_data
  ) {
  uint32_t regVal;

  regVal = 0;
  regVal = mrvl_regread32((addr_p)&(regs->fixed_mode_ssc_mode));
  regVal = SSPLL3P0G_FIXED_MODE_SSC_MODE_SSC_CLK_EN_REPLACE_VAL(regVal, 0);
  mrvl_regwrite32((addr_p)&(regs->fixed_mode_ssc_mode), regVal);
  regVal = SSPLL3P0G_LOCK_STATUS_PLL_LOCK_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->lock_status)));

  if (callback) {
    callback(0, cb_data);
  }
  return 0;
}



uint32_t mrvl_dump_cfg_SSPLL3P0G(
	SSPLL3P0G_cfg_p cfg
) {
  mrvl_msg(STATUS, "SSPLL3P0G", "RESET = %0d\n", cfg->RESET);
  mrvl_msg(STATUS, "SSPLL3P0G", "RESET_PI = %0d\n", cfg->RESET_PI);
  mrvl_msg(STATUS, "SSPLL3P0G", "RESET_SSC = %0d\n", cfg->RESET_SSC);
  mrvl_msg(STATUS, "SSPLL3P0G", "REFDIV = %0d\n", cfg->REFDIV);

  mrvl_msg(STATUS, "SSPLL3P0G", "FBDIV = %0d\n", cfg->FBDIV);
  mrvl_msg(STATUS, "SSPLL3P0G", "CLKOUT_DIFF_DIV_SEL = %0d\n", cfg->CLKOUT_DIFF_DIV_SEL);
  mrvl_msg(STATUS, "SSPLL3P0G", "CLKOUT_SE_DIV_SEL = %0d\n", cfg->CLKOUT_SE_DIV_SEL);

  mrvl_msg(STATUS, "SSPLL3P0G", "KVCO = %0d\n", cfg->KVCO);

  mrvl_msg(STATUS, "SSPLL3P0G", "PLL_BW_SEL = %0d\n", cfg->PLL_BW_SEL);
  mrvl_msg(STATUS, "SSPLL3P0G", "CTUNE = %0d\n", cfg->CTUNE);
  mrvl_msg(STATUS, "SSPLL3P0G", "INTPI = %0d\n", cfg->INTPI);
  mrvl_msg(STATUS, "SSPLL3P0G", "ICP = %0d\n", cfg->ICP);
  mrvl_msg(STATUS, "SSPLL3P0G", "VDDM = %0d\n", cfg->VDDM);
  mrvl_msg(STATUS, "SSPLL3P0G", "VDDL = %0d\n", cfg->VDDL);

  mrvl_msg(STATUS, "SSPLL3P0G", "FBCDLY= %0d\n", cfg->FBCDLY);
  mrvl_msg(STATUS, "SSPLL3P0G", "FBCLK_EXT_SEL = %0d\n", cfg->FBCLK_EXT_SEL);

  mrvl_msg(STATUS, "SSPLL3P0G", "INTPR = %0d\n", cfg->INTPR);
  mrvl_msg(STATUS, "SSPLL3P0G", "FD = %0d\n", cfg->FD);
  mrvl_msg(STATUS, "SSPLL3P0G", "PI_LOOP_MODE = %0d\n", cfg->PI_LOOP_MODE);
  mrvl_msg(STATUS, "SSPLL3P0G", "FREQ_OFFSET = %0d\n", cfg->FREQ_OFFSET);
  mrvl_msg(STATUS, "SSPLL3P0G", "FREQ_OFFSET_MODE_SELECTION = %0d\n", cfg->FREQ_OFFSET_MODE_SELECTION);
  mrvl_msg(STATUS, "SSPLL3P0G", "FREQ_OFFSET_VALID = %0d\n", cfg->FREQ_OFFSET_VALID);
  mrvl_msg(STATUS, "SSPLL3P0G", "FREQ_OFFSET_EN = %0d\n", cfg->FREQ_OFFSET_EN);


  mrvl_msg(STATUS, "SSPLL3P0G", "AVDD1815_SEL = %0d\n", cfg->AVDD1815_SEL);
  mrvl_msg(STATUS, "SSPLL3P0G", "CLKOUT_SE_GATING_EN = %0d\n", cfg->CLKOUT_SE_GATING_EN);
  mrvl_msg(STATUS, "SSPLL3P0G", "PU = %0d\n", cfg->PU);
  mrvl_msg(STATUS, "SSPLL3P0G", "BYPASS_EN = %0d\n", cfg->BYPASS_EN);
  mrvl_msg(STATUS, "SSPLL3P0G", "SSC_MODE = %0d\n", cfg->SSC_MODE);
  mrvl_msg(STATUS, "SSPLL3P0G", "CLK_DET_EN = %0d\n", cfg->CLK_DET_EN);
  mrvl_msg(STATUS, "SSPLL3P0G", "PI_EN = %0d\n", cfg->PI_EN);
  mrvl_msg(STATUS, "SSPLL3P0G", "SSC_CLK_EN = %0d\n", cfg->SSC_CLK_EN);
  mrvl_msg(STATUS, "SSPLL3P0G", "SSC_FREQ_DIV = %0d\n", cfg->SSC_FREQ_DIV);
  mrvl_msg(STATUS, "SSPLL3P0G", "SSC_RNGE = %0d\n", cfg->SSC_RNGE);

  mrvl_msg(STATUS, "SSPLL3P0G", "CLKOUT_DIFF_EN = %0d\n", cfg->CLKOUT_DIFF_EN);
  mrvl_msg(STATUS, "SSPLL3P0G", "CLKOUT_SOURCE_SEL = %0d\n", cfg->CLKOUT_SOURCE_SEL);
  mrvl_msg(STATUS, "SSPLL3P0G", "RESERVE_IN = %0d\n", cfg->RESERVE_IN);
  mrvl_msg(STATUS, "SSPLL3P0G", "TEST_ANA = %0d\n", cfg->TEST_ANA);

  mrvl_msg(STATUS, "SSPLL3P0G", "PLL_LOCK = %0d\n", cfg->PLL_LOCK);


  mrvl_msg(STATUS, "SSPLL3P0G", "RESERVE_OUT = %0d\n", cfg->RESERVE_OUT);

  return 0;
}

/*
 *----------------------------------------------------------------------
 * mrvl_query_cfg_SSPLL3P0G(*cfg, *regs, *callback, *cb_data)
 *----------------------------------------------------------------------
 *
 * This function puts the PLL registers values pointed to by *regs into
 * the config struct instance pointed to by *cfg.
 * It also prints out all the final config values.
 *
 * Inputs: Name of SSPLL3P0G_t struct instance pointer.
 * Outputs:return value
 */
uint32_t mrvl_query_cfg_SSPLL3P0G(
	SSPLL3P0G_cfg_p cfg,
    SSPLL3P0G_REGS_t *regs,
	void (*callback)(uint32_t, void *),
	void *cb_data
) {
  cfg->RESET				= SSPLL3P0G_RST_PREDIV_RESET_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->rst_prediv)));
  cfg->RESET_PI				= SSPLL3P0G_RST_PREDIV_RESET_PI_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->rst_prediv)));
  cfg->RESET_SSC			= SSPLL3P0G_RST_PREDIV_RESET_SSC_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->rst_prediv)));
  cfg->REFDIV				= SSPLL3P0G_RST_PREDIV_REFDIV_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->rst_prediv)));

  cfg->FBDIV				= SSPLL3P0G_MULT_POSTDIV_FBDIV_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->mult_postdiv)));
  cfg->CLKOUT_DIFF_DIV_SEL		= SSPLL3P0G_MULT_POSTDIV_CLKOUT_DIFF_DIV_SEL_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->mult_postdiv)));
  cfg->CLKOUT_SE_DIV_SEL 		= SSPLL3P0G_MULT_POSTDIV_CLKOUT_SE_DIV_SEL_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->mult_postdiv)));

  cfg->KVCO				= (mrvl_regread32((addr_p)&(regs->kvco)));

  cfg->PLL_BW_SEL                       = SSPLL3P0G_MISC_PLL_BW_SEL_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->misc)));
  cfg->CTUNE				= SSPLL3P0G_MISC_CTUNE_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->misc)));
  cfg->INTPI				= SSPLL3P0G_MISC_INTPI_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->misc)));
  cfg->ICP				= SSPLL3P0G_MISC_ICP_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->misc)));
  cfg->VDDM			        = SSPLL3P0G_MISC_VDDM_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->misc)));
  cfg->VDDL				= SSPLL3P0G_MISC_VDDL_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->misc)));

  cfg->FBCDLY				= SSPLL3P0G_FEEDBACK_MODE_DESKEW_FBCDLY_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->feedback_mode_deskew)));
  cfg->FBCLK_EXT_SEL			= SSPLL3P0G_FEEDBACK_MODE_DESKEW_FBCLK_EXT_SEL_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->feedback_mode_deskew)));

  cfg->INTPR				= SSPLL3P0G_OFFSET_MODE_INTPR_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->offset_mode)));
  cfg->FD				= SSPLL3P0G_OFFSET_MODE_FD_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->offset_mode)));
  cfg->PI_LOOP_MODE			= SSPLL3P0G_OFFSET_MODE_PI_LOOP_MODE_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->offset_mode)));
  cfg->FREQ_OFFSET			= SSPLL3P0G_OFFSET_MODE_FREQ_OFFSET_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->offset_mode)));
  cfg->FREQ_OFFSET_MODE_SELECTION	= SSPLL3P0G_OFFSET_MODE_FREQ_OFFSET_MODE_SELECTION_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->offset_mode)));
  cfg->FREQ_OFFSET_VALID		= SSPLL3P0G_OFFSET_MODE_FREQ_OFFSET_VALID_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->offset_mode)));
  cfg->FREQ_OFFSET_EN			= SSPLL3P0G_OFFSET_MODE_FREQ_OFFSET_EN_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->offset_mode)));


  cfg->AVDD1815_SEL			= SSPLL3P0G_FIXED_MODE_SSC_MODE_AVDD1815_SEL_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->fixed_mode_ssc_mode)));
  cfg->CLKOUT_SE_GATING_EN		= SSPLL3P0G_FIXED_MODE_SSC_MODE_CLKOUT_SE_GATING_EN_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->fixed_mode_ssc_mode)));
  cfg->PU				= SSPLL3P0G_FIXED_MODE_SSC_MODE_PU_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->fixed_mode_ssc_mode)));
  cfg->BYPASS_EN			= SSPLL3P0G_FIXED_MODE_SSC_MODE_BYPASS_EN_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->fixed_mode_ssc_mode)));
  cfg->SSC_MODE				= SSPLL3P0G_FIXED_MODE_SSC_MODE_SSC_MODE_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->fixed_mode_ssc_mode)));
  cfg->CLK_DET_EN			= SSPLL3P0G_FIXED_MODE_SSC_MODE_CLK_DET_EN_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->fixed_mode_ssc_mode)));
  cfg->PI_EN				= SSPLL3P0G_FIXED_MODE_SSC_MODE_PI_EN_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->fixed_mode_ssc_mode)));
  cfg->SSC_CLK_EN			= SSPLL3P0G_FIXED_MODE_SSC_MODE_SSC_CLK_EN_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->fixed_mode_ssc_mode)));
  cfg->SSC_FREQ_DIV			= SSPLL3P0G_SSC_FREQ_SSC_RANGE_SSC_FREQ_DIV_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->ssc_freq_ssc_range)));
  cfg->SSC_RNGE				= SSPLL3P0G_SSC_FREQ_SSC_RANGE_SSC_RNGE_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->ssc_freq_ssc_range)));

  cfg->CLKOUT_DIFF_EN			= SSPLL3P0G_CLK_CTRL_MARVELL_TEST_CLKOUT_DIFF_EN_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->clk_ctrl_marvell_test)));
  cfg->CLKOUT_SOURCE_SEL		= SSPLL3P0G_CLK_CTRL_MARVELL_TEST_CLKOUT_SOURCE_SEL_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->clk_ctrl_marvell_test)));
  cfg->RESERVE_IN			= SSPLL3P0G_CLK_CTRL_MARVELL_TEST_RESERVE_IN_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->clk_ctrl_marvell_test)));
  cfg->TEST_ANA				= SSPLL3P0G_CLK_CTRL_MARVELL_TEST_TEST_ANA_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->clk_ctrl_marvell_test)));

  cfg->PLL_LOCK				= SSPLL3P0G_LOCK_STATUS_PLL_LOCK_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->lock_status)));

  cfg->RESERVE_OUT			= SSPLL3P0G_RESERVE_OUT_RESERVE_OUT_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->reserve_out)));

  mrvl_dump_cfg_SSPLL3P0G(cfg);

 if (callback) {
    callback(0, cb_data);
  }
  return 0;
}

uint32_t ricoh_ssc_enable(SSPLL3P0G_REGS_t *regs, void (*callback)(uint32_t, void *), void *cb_data)
{
	uint32_t regVal;

	/* Check PLL Lock signal */
	regVal = 0;
	regVal = SSPLL3P0G_LOCK_STATUS_PLL_LOCK_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->lock_status)));

	if (regVal == 1) {
		/**
		 * 1. SSC_CLK_EN : Lo
		 */
		regVal = 0;
		regVal = mrvl_regread32((addr_p)&(regs->fixed_mode_ssc_mode));
		regVal = SSPLL3P0G_FIXED_MODE_SSC_MODE_SSC_CLK_EN_REPLACE_VAL(regVal, 0);
		mrvl_regwrite32((addr_p)&(regs->fixed_mode_ssc_mode), regVal);
		mrvl_wait_for_us((uint32_t)1);
		/**
		 * 2. PI_EN : Hi
		 */
		regVal = 0;
		regVal = mrvl_regread32((addr_p)&(regs->fixed_mode_ssc_mode));
		regVal = SSPLL3P0G_FIXED_MODE_SSC_MODE_PI_EN_REPLACE_VAL(regVal,1);
		mrvl_regwrite32((addr_p)&(regs->fixed_mode_ssc_mode), regVal);
		/**
		 * 3. RESET_SSC&RESET_PI : Hi
		 */
		regVal = 0;
        regVal = mrvl_regread32((addr_p)&(regs->rst_prediv));
        regVal = SSPLL3P0G_RST_PREDIV_RESET_PI_REPLACE_VAL(regVal, 1);
        regVal = SSPLL3P0G_RST_PREDIV_RESET_SSC_REPLACE_VAL(regVal, 1);
        mrvl_regwrite32((addr_p)&(regs->rst_prediv), regVal);
		/**
		 * 4. 2us wait
		 */
		mrvl_wait_for_us((uint32_t)2);
		/**
		 * 5. RESET_SSC&RESET_PI : Lo
		 */
		regVal = 0;
        regVal = mrvl_regread32((addr_p)&(regs->rst_prediv));
        regVal = SSPLL3P0G_RST_PREDIV_RESET_PI_REPLACE_VAL(regVal, 0);
        regVal = SSPLL3P0G_RST_PREDIV_RESET_SSC_REPLACE_VAL(regVal, 0);
        mrvl_regwrite32((addr_p)&(regs->rst_prediv), regVal);
		/**
		 * 6. 2us wait
		 */
		mrvl_wait_for_us((uint32_t)2);
		/**
		 * 7. PI_LOOP_MODE : Hi
		 */
		regVal = 0;
        regVal = mrvl_regread32((addr_p)&(regs->offset_mode));
		regVal = SSPLL3P0G_OFFSET_MODE_PI_LOOP_MODE_REPLACE_VAL(regVal, 1);
        mrvl_regwrite32((addr_p)&(regs->offset_mode), regVal);
		/**
		 * 8. 2us wait
		 */
		mrvl_wait_for_us((uint32_t)2);
		/**
		 * 9. SSC_CLK_EN: Hi
		 */
		regVal = 0;
		regVal = mrvl_regread32((addr_p)&(regs->fixed_mode_ssc_mode));
		regVal = SSPLL3P0G_FIXED_MODE_SSC_MODE_SSC_CLK_EN_REPLACE_VAL(regVal, 1);
		mrvl_regwrite32((addr_p)&(regs->fixed_mode_ssc_mode), regVal);
		mrvl_wait_for_us((uint32_t)2);

		regVal = SSPLL3P0G_LOCK_STATUS_PLL_LOCK_MASK_SHIFT(mrvl_regread32((addr_p)&(regs->lock_status)));
		if (callback) {
			callback(0, cb_data);
		}
	} else {
		mrvl_msg(ERROR, "SSPLL3P0G", "No lock signal detected LOCK_STATUS Reg =  %08x.\n", mrvl_regread32((addr_p)&(regs->lock_status)));
		if (callback) {
			callback(1, cb_data);
		}
	}
	return 0;
}
