/*
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014-2016, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
*/

//#define DEBUG

#include <stdint.h>
#include <assert.h>
#include <string.h>
#include "error_types.h"
#include "i2c_api.h"
#include "minPrintf.h"
#include "APMU_regheaders.h"
#include "ID_utils.h"
#include "hex_dump.h"
#include "i2c-twsi.h"


error_type_t eeprom_write(void *handle, uint8_t i2c_address, uint8_t buffer[], uint32_t data_len);

static const uint32_t I2CRegBases[I2C_NUM_BLOCKS] = I2C_BASE_REGS;

static	void	*I2CCurrentContext;

error_type_t initialize_I2C(uint32_t bus)
{
	unsigned long	MfprSCLAddr=0, MfprSDAAddr=0;
	APMU_CLKRSTGEN_REGS_t *apmu_clk_rst_gen_regs = (APMU_CLKRSTGEN_REGS_t *) AP_APMU_APMU_CLKRSTGEN_BASE;
	ENTRY();
	dbg_printf("%s bus: %08x\n",__func__,bus);

	if ((bus < 1) || (bus > I2C_NUM_BLOCKS))
		return FAIL;


	I2CCurrentContext = i2c_twsi_init(I2CRegBases[bus-1], &MfprSCLAddr, &MfprSDAAddr,0,0,I2C_SLAVE_ADDR );
	if (I2CCurrentContext == NULL) {
		return FAIL;
	}
	return STATUS_OK;
}

error_type_t I2C_read( uint8_t chip_address, uint16_t part_address, uint16_t part_addr_len, uint16_t size, void* buffer )
{
	error_type_t	retval;
	uint8_t			slave_address = ((chip_address << 1) & 0xFE);
	uint8_t			cmd[2];

	ENTRY();

	if (2 < part_addr_len)
		return -1;

	if (I2C_ONE_BYTE_MODE == part_addr_len)
	{
		cmd[0] = (part_address & 0xFF);
	} else
	{
		cmd[0] = ((part_address >> 8) & 0xFF);
		cmd[1] = (part_address & 0xFF);
	}

	dbg_printf("\n\nCalling I2CMasterReceiveDataDirect\n");
	dbg_printf("%s chip_addr: %02x part_addr:%04x\n",__func__,chip_address,part_address);
	dbg_printf("part %x %x %x\n", part_address, part_addr_len, size);
	retval = eeprom_read ( I2CCurrentContext, slave_address, cmd, part_addr_len, buffer, size);
	if (size != retval)
	{
		dbg_printf("\n\nI2CMasterReceiveDataDirect returned: %d IDBR: %08x  ICR: %08x ISR: %08x ISAR: %08x\n",retval,
				*((volatile uint32_t*)0xd4011008),
				*((volatile uint32_t*)0xd4011010),
				*((volatile uint32_t*)0xd4011018),
				*((volatile uint32_t*)0xd4011020)
					);
		return FAIL;
	}

	return STATUS_OK;
}


error_type_t I2C_write( uint8_t chip_address, uint16_t part_address, uint8_t part_addr_len, uint16_t write_buf_size, void* write_buffer )
{
	error_type_t	retval;
	uint8_t			slave_address = ((chip_address << 1) & 0xFE);
	uint8_t			write_cmd_and_data[2+32]; // Space for 2 command bytes + 32 bytes payload

	ENTRY();
	if (32 < write_buf_size)
		return -1;

	if (2 < part_addr_len)
		return -1;

	dbg_hex_dump(write_buffer, write_buf_size);

	if (I2C_ONE_BYTE_MODE == part_addr_len)
	{
		write_cmd_and_data[0] = (part_address & 0xFF);
	} else
	{
		write_cmd_and_data[0] = ((part_address >> 8) & 0xFF);
		write_cmd_and_data[1] = (part_address & 0xFF);
	}
	memcpy(&write_cmd_and_data[part_addr_len], write_buffer, write_buf_size);
	dbg_hex_dump(write_cmd_and_data, write_buf_size+part_addr_len);

	dbg_printf("\n\nCalling I2CMasterSendDataDirect\n");
	dbg_printf("%s chip_addr: %02x part_addr:%04x\n",__func__,chip_address,part_address);
	retval = eeprom_write( I2CCurrentContext, slave_address, write_cmd_and_data, write_buf_size+part_addr_len);
	if (retval != write_buf_size + part_addr_len)
	{
		dbg_printf("\n\nI2CMasterSendDataDirect returned: %d IDBR: %08x  ICR: %08x ISR: %08x ISAR: %08x\n",retval,
				*((volatile uint32_t*)0xd4011008),
				*((volatile uint32_t*)0xd4011010),
				*((volatile uint32_t*)0xd4011018),
				*((volatile uint32_t*)0xd4011020)
					);
		return FAIL;
	}

	return STATUS_OK;
}

/* Used by vim and some versions of vi: set tabstop=4 shiftwidth=4: */
