/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2016, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/



//#define DEBUG

#include <stdint.h>
#include <assert.h>
#include <string.h>
#include "error_types.h"
#include "i2c_api.h"
#include "minPrintf.h"
#include "APMU_regheaders.h"
#include "apb_top_regheaders.h"
#include "ID_utils.h"
#include "hex_dump.h"


/**
 * \brief enum to determine the r/w bit setting at the end of 
 *        the part address.
 */
typedef enum
{
	WRITE = 0,
	READ
}i2c_rw_e;
/**
 * \brief enum for the write routine to specify if a stop 
 *        transaction will happen at the end of the transfer.
 */
typedef enum
{
	DO_STOP,
	NO_STOP
}i2c_stop_e;

/**
 * \brief Initialize a given i2c block
 * 
 * 
 * @param i2c_reg_base address of the register base
 * @param slave_addr slave address of the i2c, used only when we 
 *      	     are not master (not supported by us)
 * 
 * @return void* Handle to this device.
 */
void *i2c_twsi_init(uint32_t i2c_reg_base, uint32_t slave_addr )
{
	I2C1_REGS_t *i2c_regs;

	uint32_t temp;

	// for init, first reset the controller
	i2c_regs = (I2C1_REGS_t *) i2c_reg_base;
	if(i2c_regs->TWSI_ISR & I2C1_TWSI_ISR_UB_MASK) {
		dbg_printf("Block is active, get out\n");
		return NULL;	// block is busy, no reset allowed.
	}

	i2c_regs->TWSI_ICR = I2C1_TWSI_ICR_UR_REPLACE_VAL(i2c_regs->TWSI_ICR, 1);
	i2c_regs->TWSI_ICR = ~I2C1_TWSI_ICR_UR_MASK;		// clear everything but reset
	i2c_regs->TWSI_ICR = 0;		// now drop the reset

	// clear the isr bits
	i2c_regs->TWSI_ISR = 0xffffffff;
	dbg_printf("i2c registers %x slave addr %x\n", i2c_regs, slave_addr);
	dbg_printf("Insure status bits clear %x\n", i2c_regs->TWSI_ISR);
	// input the slave address into the block
	i2c_regs->TWSI_ISAR = slave_addr;	// could also use I2C1_TWSI_ISAR_SLAVE_ADDRESS_REPLACE_VAL
	// now enable the block.
	// clean out the control register.
	i2c_regs->TWSI_ICR = 0;
	temp = I2C1_TWSI_ICR_IUE_REPLACE_VAL(i2c_regs->TWSI_ICR, 1);
	i2c_regs->TWSI_ICR = I2C1_TWSI_ICR_SCLE_REPLACE_VAL(temp, 1);
	dbg_printf("control ergister %x\n", i2c_regs->TWSI_ICR);
	dbg_printf("tag registers %x %x\n", i2c_regs->REV0, i2c_regs->REV1);
	return i2c_regs;		//handle is the address of the register block.

}
/**
 * \brief wait for a given condition in the isr register.  Any 
 *        bit set in the mask will cause us to complete
 * 
 * @param i2c_regs address of the i2c block
 * @param mask Mask to look for 
 * 
 * @return error_type_t normal error type returns.
 */
static error_type_t i2c_wait_for_complete(I2C1_REGS_t *i2c_regs,uint32_t mask)
{
	uint32_t timeout=100;	// 100 msec timeout
	// wait for this to complete, 
	while(!(i2c_regs->TWSI_ISR & mask)){
		if(timeout-- == 0) {
			return FAIL;
		}
		Delay(1000);

	}
	dbg_printf("xmit receive %x\n", i2c_regs->TWSI_ISR);
	// clear all the interrupts
	i2c_regs->TWSI_ISR = I2C1_TWSI_ISR_ITE_MASK;
	i2c_regs->TWSI_ISR = I2C1_TWSI_ISR_UB_MASK;
	i2c_regs->TWSI_ISR = I2C1_TWSI_ISR_MSD_MASK;
	i2c_regs->TWSI_ISR = mask;		// make sure this one at least is clear
	return STATUS_OK;
}
/**
 * \brief send an i2c start condition and part address.
 * 
 * @param i2c_regs address of the i2c block
 * @param slave_addr address of the part in the start protocal
 * @param read_write bit for read/write for the next data
 * 
 * @return error_type_t standard error type returns.
 */
static error_type_t i2c_send_start(I2C1_REGS_t *i2c_regs, uint8_t slave_addr, i2c_rw_e read_write)
{
	uint32_t temp;
	// load the target slave address
	i2c_regs->TWSI_IDBR = slave_addr | read_write;	// r/nw = 0
	//
	// send a start and the part address
	//
	temp = I2C1_TWSI_ICR_START_REPLACE_VAL(i2c_regs->TWSI_ICR, 1);
	temp = I2C1_TWSI_ICR_STOP_REPLACE_VAL(temp, 0);
	temp = I2C1_TWSI_ICR_ALDIE_REPLACE_VAL(temp, 0);
	temp = I2C1_TWSI_ICR_ITEIE_REPLACE_VAL(temp, 1);
	i2c_regs->TWSI_ICR = I2C1_TWSI_ICR_TB_REPLACE_VAL(temp, 1);
	dbg_printf("icr register %x\n", i2c_regs->TWSI_ICR);
	// now wait for it to go on the bus.
	i2c_wait_for_complete(i2c_regs, I2C1_TWSI_ISR_ITE_MASK);
	return STATUS_OK;
}
/**
 * \brief Send the data in the data buffer to a given part 
 *        address.  Optionally stop xfer.
 * 
 * @param handle value returned from the init
 * @param slave_addr address of the i2c part on the bus to talk 
 *      	     to.
 * @param buffer Write data buffer
 * @param data_len Number of bytes in the data buffer
 * @param stop Stop or leave running bit.
 * 
 * @return error_type_t returns FAIL for bad or number of bytes 
 *         xfered.
 */
static error_type_t i2c_twsi_write(void *handle, uint8_t slave_addr, uint8_t *buffer, uint32_t data_len, i2c_stop_e stop)
{
	uint32_t temp;
	uint32_t i;
	I2C1_REGS_t *i2c_regs = (I2C1_REGS_t *)handle;
	uint32_t mask;
	dbg_printf("%s icr %x\n", __func__, i2c_regs->TWSI_ICR);

	//
	// send a start
	//
	i2c_send_start(i2c_regs,slave_addr, WRITE);
	dbg_printf("%s icr %x\n", __func__, i2c_regs->TWSI_ICR);
	//
	// now send the data.
	for(i = 0; i < data_len; i++) {
		// Stick the data byte into the register.
		i2c_regs->TWSI_IDBR = buffer[i];
		temp = I2C1_TWSI_ICR_START_REPLACE_VAL(i2c_regs->TWSI_ICR, 0);
		if((i == data_len -1) &&
		   (stop == DO_STOP)) {
			temp = I2C1_TWSI_ICR_STOP_REPLACE_VAL(temp, 1);
			temp = I2C1_TWSI_ICR_MSDE_REPLACE_VAL(temp, 1);
			temp = I2C1_TWSI_ICR_MSDIE_REPLACE_VAL(temp,1);
			mask = I2C1_TWSI_ISR_MSD_MASK;
		} else
		{
			temp = I2C1_TWSI_ICR_STOP_REPLACE_VAL(temp, 0);
			mask = I2C1_TWSI_ISR_ITE_MASK;
		}
		temp = I2C1_TWSI_ICR_ALDIE_REPLACE_VAL(temp, 0);
		temp = I2C1_TWSI_ICR_ITEIE_REPLACE_VAL(temp, 1);
		// start the xfer
		i2c_regs->TWSI_ICR = I2C1_TWSI_ICR_TB_REPLACE_VAL(temp, 1);
		// wait for it to complete
		i2c_wait_for_complete(i2c_regs,mask);
	}
	dbg_printf("%s icr %x\n", __func__, i2c_regs->TWSI_ICR);
	// clear the stop bit in the control register.
	i2c_regs->TWSI_ICR = I2C1_TWSI_ICR_STOP_REPLACE_VAL(i2c_regs->TWSI_ICR, 0);
	dbg_printf("%s icr %x\n", __func__, i2c_regs->TWSI_ICR);

	return data_len;
}
/**
 * @brief Read from a
 * 
 * @author  (6/29/2016)
 * 
 * @param handle 
 * @param slave_addr 
 * @param buffer 
 * @param data_len 
 * 
 * @return error_type_t <0 error >0 number of bytes read.
 */
static error_type_t i2c_twsi_read(void *handle, uint8_t slave_addr, uint8_t *buffer, uint32_t data_len)
{
	int i;
	uint32_t stop_condition =0;
	I2C1_REGS_t *i2c_regs = (I2C1_REGS_t *)handle;
	uint32_t temp;
	dbg_printf("%s icr %x\n", __func__, i2c_regs->TWSI_ICR);
	//
	// send a start on the bus
	//
	i2c_send_start(i2c_regs,slave_addr, READ);
	dbg_printf("%s icr %x\n", __func__, i2c_regs->TWSI_ICR);

	//
	// now read however much data was requested.
	for(i = 0; i < data_len; i++) {

		// setup the read

		temp = I2C1_TWSI_ICR_ALDIE_REPLACE_VAL(i2c_regs->TWSI_ICR, 1);
		temp = I2C1_TWSI_ICR_START_REPLACE_VAL(temp, 0);
		if(i == data_len -1) {
			temp = I2C1_TWSI_ICR_ACKNAK_REPLACE_VAL(temp, 1);
			temp = I2C1_TWSI_ICR_STOP_REPLACE_VAL(temp, 1);
			temp = I2C1_TWSI_ICR_MSDE_REPLACE_VAL(temp, 1);
			temp = I2C1_TWSI_ICR_MSDIE_REPLACE_VAL(temp,1);
			stop_condition = I2C1_TWSI_ISR_MSD_MASK;
		} else
		{
			temp = I2C1_TWSI_ICR_ACKNAK_REPLACE_VAL(temp, 0);
			temp = I2C1_TWSI_ICR_STOP_REPLACE_VAL(temp, 0);
			stop_condition = I2C1_TWSI_ISR_IRF_MASK;
		}
		i2c_regs->TWSI_ICR = I2C1_TWSI_ICR_TB_REPLACE_VAL(temp, 1);
		// wait for a byte to come in.
		if(i2c_wait_for_complete(i2c_regs, stop_condition) == FAIL)
		{
			// no response, fail.
			return FAIL;
		}
		// get the data and store it in the buffer.
		buffer[i] = i2c_regs->TWSI_IDBR;	// read the data byte		
		// clean up a little.
		temp = I2C1_TWSI_ICR_STOP_REPLACE_VAL(i2c_regs->TWSI_ICR, 0);
		i2c_regs->TWSI_ICR = I2C1_TWSI_ICR_ACKNAK_REPLACE_VAL(temp,0);
	}
	dbg_printf("%s icr %x\n", __func__, i2c_regs->TWSI_ICR);
	return data_len;

}
/**
 * \brief read from a i2c device by writting out the address and 
 *        then turning around and reading all requested data.
 * If the address_length = 0 then this is a continue read and 
 * just does an i2c read. 
 * 
 * @param handle value returned from the init call
 * @param i2c_address Part address on the i2c bus
 * @param part_address Address within the part on the i2c bus
 * @param address_length Number of bytes in the address
 * @param buffer Buffer for the data
 * @param data_len Number of bytes in the buffer.
 * 
 * @return error_type_t <0 error >0 number of bytes written
 */
error_type_t eeprom_read(void *handle, uint8_t i2c_address, uint8_t part_address[], uint16_t address_length, uint8_t buffer[], uint32_t data_len)
{
	//
	// First send the address out if needed.
	//
	if(address_length > 0) {
		i2c_twsi_write(handle, i2c_address, part_address, address_length, NO_STOP);  // don't want to do a stop transaction here.
	}
	return i2c_twsi_read(handle, i2c_address, buffer,data_len);
}
/**
 * @brief Write the given data buffer to the i2c part.
 * 
 * @param handle handle returned from the init call.
 * @param i2c_address address on the i2c bus of the part to 
 *      	      write to.
 * @param buffer Data buffer to write
 * @param data_len How much data is in this buffer.
 * 
 * @return error_type_t <0 error >0 number of bytes written.
 */
error_type_t eeprom_write(void *handle, uint8_t i2c_address, uint8_t buffer[], uint32_t data_len)
{
	error_type_t retval;
	retval = i2c_twsi_write(handle,i2c_address,buffer,data_len, DO_STOP);

	dbg_printf("%s return val %d\n", __func__,retval);
	return retval;		
}
