/*
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
*/

#include <stdint.h>
#include <assert.h>
#include "regAddrs.h"
#include "minPrintf.h"
#include "flash_internal.h"
#include "flash_api.h"
#include "cpu_api.h"
#include "rom_types.h"
#include "ID_utils.h"
#include "SDMMC3_regheaders_2p.h"

#define ENABLE_INTS             (0)
#define TIM_MEDIA_ID_OFFSET     (0x0a)
#define WILDCARD_SIGNATURE      (0xfe)
#define TIMH_BASE_ADDR_FIELD    (0xD100003C)
#define FOUND_MEDIA_ID_FIELD    (0xD100001C)
#define SHADOW_REG_ADDR         (0xD401D85C)

#define SD_SPEED_IN_MHZ         (10)    // desired speed for SD when reading via bootrom (12 max)

typedef enum {
    eBlock1_Port0 = 0x07,
    eBlock0_Port0 = 0x08,
    eBlock0_Port1 = 0x09,
} eMMC_SD_port_type;

static const uint32_t eMMC_SD_Reg_Base[] = {
    [eBlock0_Port0] = AP_IO_WRAP_SDH0_BASE,
    [eBlock0_Port1] = (AP_IO_WRAP_SDH0_BASE + 0x800),
    [eBlock1_Port0] = AP_IO_WRAP_SDH1_BASE,
};

static const struct flash_ops flash_ops_cbs[] = {
    [eROM_RevC] = {
        .configure_flashes   = (pConfigure_Flashes_F)0xffe010a8,
        .ReadFlash           = (pReadFlash_F)        0xffe00f70,
    },
    [eROM_RevC_bSPI_Aspl] = {
        .configure_flashes   = (pConfigure_Flashes_F)0xfe0010a8,
        .ReadFlash           = (pReadFlash_F)        0xfe000f70,
    },
    [eROM_RevB_bSPI] = {
        .configure_flashes   = (pConfigure_Flashes_F)0xfe000f18,
        .ReadFlash           = (pReadFlash_F)        0xfe000de0,
    },
    [eROM_RevB_bSPI_Aspl] = {
        .configure_flashes   = (pConfigure_Flashes_F)0xfe000f18,
        .ReadFlash           = (pReadFlash_F)        0xfe000de0,
    },
    [eFPGA_RevA] = {
        .configure_flashes   = (pConfigure_Flashes_F)0xffe011d4,
        .ReadFlash           = (pReadFlash_F)        0xffe010c4,
    },
    [eROM_RevA] = {
        .configure_flashes   = (pConfigure_Flashes_F)0xffe01204,
        .ReadFlash           = (pReadFlash_F)        0xffe010f4,
    },
    [eROM_RevA_bSPI_Aspl] = {
        .configure_flashes   = (pConfigure_Flashes_F)0xfe001220,
        .ReadFlash           = (pReadFlash_F)        0xfe001110,
    },
    [eROM_RevA_bSPI] = {
        .configure_flashes   = (pConfigure_Flashes_F)0xfe001220,
        .ReadFlash           = (pReadFlash_F)        0xfe001110,
    },
    [eFPGA_RevB_19_16] = {
        .configure_flashes   = (pConfigure_Flashes_F)0xffe00eec,
        .ReadFlash           = (pReadFlash_F)        0xffe00db0,
    },
    [eROM_RevB1] = {
        .configure_flashes   = (pConfigure_Flashes_F)0xffe00f18,
        .ReadFlash           = (pReadFlash_F)        0xffe00de0,
    },
    [eFPGA_Ge2_RevA] = {
        .configure_flashes   = (pConfigure_Flashes_F)0xffe01078,
        .ReadFlash           = (pReadFlash_F)        0xffe00f40,
    },
    [eROM_Ge2_RevA] = {
        .configure_flashes   = (pConfigure_Flashes_F)0xffe010a8,
        .ReadFlash           = (pReadFlash_F)        0xffe00f70,
    },
    [eROM_Ge2_RevB] = {
        .configure_flashes   = (pConfigure_Flashes_F)0xffe010a8,
        .ReadFlash           = (pReadFlash_F)        0xffe00f6c,
    },
};

static const struct flash_ops *flash_ops_cb = NULL;


static void disable_USB_ints(void)
{
    uint32_t *ICU_conf_reg29;
    uint32_t *ICU_conf_reg32;


    if (is_Gr2() && is_RevA())
    {
        ICU_conf_reg29 = (uint32_t*)(AP_ICU_ICU1_IRQ_REG_BASE + (29 * 4));
        ICU_conf_reg32 = (uint32_t*)(AP_ICU_ICU1_IRQ_REG_BASE + (32 * 4));

        *ICU_conf_reg29 &= ~0x10;  // Clear IRQ bank bit 1 Enable
        *ICU_conf_reg32 &= ~0x10;  // Clear IRQ bank bit 1 Enable
        asm volatile ( "DSB ;ISB ;" :::);
    }
}


static void flash_adjust_SDeMMC_controller_speed(uint32_t bus_speed_mhz)
{
    uint8_t  load_flash_id  = get_media_source_id();
    SDMMC3_2P_REGS_t *sd_regs = ((SDMMC3_2P_REGS_t *)eMMC_SD_Reg_Base[load_flash_id]);
    volatile uint16_t *sd_clk_reg = &(sd_regs->SD_CLOCK_CTRL0);    
    uint16_t divider;

    // Disable the sd clock while we're reconfiguring for the new PLL speed
    *sd_clk_reg &= ~SDMMC3_2P_SD_CLOCK_CTRL0_SD_CLK_EN_MASK;

    // Set up the new divider value
    divider = bus_speed_mhz / (2 * SD_SPEED_IN_MHZ);

    *sd_clk_reg &= ~(SDMMC3_2P_SD_CLOCK_CTRL0_SD_FREQ_SEL0_MASK | SDMMC3_2P_SD_CLOCK_CTRL0_SD_FREQ_SEL1_MASK);
    *sd_clk_reg |= (divider & 0xFF) << SDMMC3_2P_SD_CLOCK_CTRL0_SD_FREQ_SEL0_SHIFT;
    *sd_clk_reg |= ((divider >> 8) & 0x3) << SDMMC3_2P_SD_CLOCK_CTRL0_SD_FREQ_SEL1_SHIFT;

    // Re-enable clocks and wait for them to be stable
    *sd_clk_reg |= SDMMC3_2P_SD_CLOCK_CTRL0_SD_CLK_EN_MASK;
    *sd_clk_reg |= SDMMC3_2P_SD_CLOCK_CTRL0_INT_CLK_EN_MASK;
    while (0 == (*sd_clk_reg & SDMMC3_2P_SD_CLOCK_CTRL0_INT_CLK_STABLE_MASK));
}

uint8_t get_media_source_id( void )
{
    uint8_t  media_source_id;
    rom_types_e rom_type        = check_rom_type();
    uint32_t *TIMH              = *((uint32_t **) (((eROM_RevA == rom_type)||(eROM_RevA_bSPI == rom_type)) ? TIMH_BASE_ADDR_FIELD-4 : TIMH_BASE_ADDR_FIELD));
    uint8_t  TIMH_flashnum      = TIMH[TIM_MEDIA_ID_OFFSET]&0xff;
    uint32_t *found_media_ptr   = (uint32_t*) FOUND_MEDIA_ID_FIELD;
    
    if ( WILDCARD_SIGNATURE == TIMH_flashnum )
    {
        // Media src ID will either be in the upper or lower slot depending on state.
        // And the opposite will field always be a zero.  OR them and get it one
        // way or the other.
        media_source_id = (((((*found_media_ptr) & 0x1FC00000)>>22) & 0x3F) |
                            (((*found_media_ptr)>>1) & 0x3F));
    }
    else
    {
        media_source_id = TIMH_flashnum;
    }
    return media_source_id;
}

    
void flash_callback_api_init(rom_types_e rom_type, uint32_t bus_speed_mhz) 
{
    MEDIA_T  media;
    uint32_t *TIMH          = *((uint32_t **) (((eROM_RevA == rom_type)||(eROM_RevA_bSPI == rom_type)) ? TIMH_BASE_ADDR_FIELD-4 : TIMH_BASE_ADDR_FIELD));
    uint8_t  TIMH_flashnum  = TIMH[TIM_MEDIA_ID_OFFSET]&0xff;
    uint8_t  load_flash_id  = get_media_source_id();
    uint32_t *shadowreg_ptr = (uint32_t*) SHADOW_REG_ADDR;
    flash_ops_cb            = &flash_ops_cbs[rom_type];
    uint32_t ret;
    
    dbg_printf("\nRestoring Interrupts to On\n");
    disable_USB_ints();
    cpu_restore_interrupts(ENABLE_INTS);

    minPrintf("\nCalling Configure_Flashes @ 0x%08x %d %02x %02x %02x\n", (uint32_t)flash_ops_cb->configure_flashes,rom_type, TIMH_flashnum, load_flash_id, *shadowreg_ptr);
    if ((ret=flash_ops_cb->configure_flashes(&media, load_flash_id )) != 0)
    {
        // error
        minPrintf("\nConfigure_Flashes failed!  returned: %d\n",ret);
        assert(0);
    }
    if (NULL != media.SetPartition)
    {
        media.SetPartition(0);
    }
}

uint32_t read_flash (rom_types_e rom_type, uint32_t FlashOffsetInBytes, uint32_t LocalBufferPtr, uint32_t SizeInBytes, uint32_t bus_speed_mhz)
{
    uint32_t DummyParam = 0;
    uint8_t  load_flash_id  = get_media_source_id();

    // Work around for Rev A BootROM issue - Pad out read to full sector size.
    if ((eFPGA_RevA == rom_type) || (eROM_RevA == rom_type))
    {
        if (( SizeInBytes % 0xBE800 ) > 0xAEA00 )
        {
            SizeInBytes += (512 - (SizeInBytes % 512));
        }
    }

    switch (load_flash_id)
    {
        // Adjust the SD/eMMC controller if we are using any SD/eMMC id
        case  7: case  8: case  9: case 11: case 12: case 24: case 25:
            flash_adjust_SDeMMC_controller_speed(bus_speed_mhz);
            break;
        default:
            break;
    }

    return flash_ops_cb->ReadFlash(FlashOffsetInBytes, LocalBufferPtr, SizeInBytes, DummyParam);
}
