/*
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
*/

//#define DEBUG

#include "assert.h"
#include "error_types.h"
#include "board_types.h"
#include "cpu_api.h"
#include "minPrintf.h"
#include "regAddrs.h"
#include "MC_regheaders.h"
#include "ddr_utils.h"
#include "ddr_config.h"
#include "spd_rom.h"
#include "pmu_api.h"
#include "ciu_regheaders.h"
#include "ID_utils.h"
#include "pll_api.h"
#include "i2c_api.h"
#include "hex_dump.h"
#include "GS2/ddr.h"
#include "GR2/ddr_gr2.h"
#include "Customer_DDR_Config.h"
#include "utils.h"

//static error_type_t	set_DDR_configuration(board_types_e board_type, const DDR_config_t *DDR_config_record, cs_sa_t *startAddrs);
//static error_type_t set_FPGA_DDR_configuration(const DDR_config_t *DDR_config_record);
//static error_type_t set_SI_DDR_configuration(const DDR_config_t *DDR_config_record, cs_sa_t *startAddrs);
//static void calDDRPHYPads(MC_REGS_t *mc_reg, uint32_t *zpr, uint32_t *znr);
//static void DDR_dynamic_adjustment(cs_sa_t *startAddrs, uint32_t mem_width, uint32_t clk_pad_val);

bool ddr4_memory = false;


// Values for when we can NOT find any board info and need a least risk mem config.
#define MEM_WIDTH_VAL   (32)
#define DDRCLKPD_VAL_800    (1250) // DDR 400MHz = 2500, 800MHz or 625MHz = 1250, 600MHz = 1667
#define DDRCLKPD_VAL_400    (2500) // DDR 400MHz = 2500, 800MHz or 625MHz = 1250, 600MHz = 1667
static const DDR_config_t _32bit400m1024_config = Customer_400_x32_1024_INIT_DATA;
static const DDR_config_t _32bit400m512_config = Customer_400_x32_512_INIT_DATA;
static const DDR_config_t _32bit400m2048_config = Customer_400_x32_2048_INIT_DATA;
static const DDR_config_t _ddr4_32bit800m2048_config = Customer_mt40a512m16jy_083e_800_32_INIT_DATA;

int get_memory_size(void)
{
	/**
	 * GPIOA[13]   IOPAD013:0xd401e034/0x00018040
	 *             PDRA    :0xd4019004/0x00000000
	 *             PLRA    :0xd4019000/NA
	 * GPIOE[28]   IOPAD156:0xd401e290/0x00019840
	 * GPIOE[10]   IOPAD138:0xd401e248/0x00018040
	 *             PDRE    :0xd4019404/0x00000000
	 *             PLRE    :0xd4019400/NA
	 */
	volatile unsigned long *iopad_013 = (unsigned long *)0xd401e034;
	volatile unsigned long *iopad_138 = (unsigned long *)0xd401e248;
	volatile unsigned long *iopad_156 = (unsigned long *)0xd401e290;
	volatile unsigned long *gpio_pdra = (unsigned long *)0xd4019004;
	volatile unsigned long *gpio_plra = (unsigned long *)0xd4019000;
	volatile unsigned long *gpio_pdre = (unsigned long *)0xd4019404;
	volatile unsigned long *gpio_plre = (unsigned long *)0xd4019400;
	int rc = 0;
	int bit2 = 0,  bit1 = 0, bit0 = 0;
	int type_id = 0;

	/* set our configuration */
	*iopad_013 = *iopad_138 = *iopad_156 = 0x00018040;
	*gpio_pdra = *gpio_pdre = 0x00000000;/* set all dir to in */
	Delay(5000);

	bit2 = ((*gpio_plra) & 0x00002000) ? 1 : 0;
	bit1 = ((*gpio_plre) & 0x10000000) ? 1 : 0;
	bit0 = ((*gpio_plre) & 0x00000400) ? 1 : 0;

	type_id = (bit2 << 2) | (bit1 << 1) | (bit0);

	switch (type_id) {
	case 3:/* Cheetah 2048MB */
	case 1:/* Pumbaa 2048MB */
		rc = 2048;
		break;
	case 7:/* 4L 1024MB */
	case 2:/* Cheetah 1024MB */
	case 0:/* Pumbaa 1024MB */
		rc = 1024;
		break;
	case 6:/* 4L 512MB */
		rc = 512;
		break;
	case 5:
	case 4:
	default:
		rc = -1;
		break;
	}

	rc = 2048;
	return rc;
}

void initialize_DDR(board_types_e board_type)
{
	cs_sa_t startAddrs;
	MC_REGS_t *mem_ctrl_regs = (MC_REGS_t*) AP_MC_BASE;

	ENTRY();

	if (is_Gr2())
	{
		//minPrintf("this is a gr2\n");
		PMU_enable(ePMU_DEVICE_SMMU);
	}
	PMU_enable(ePMU_DEVICE_MC5);

	minPrintf("\ninitializing with RICOH's setting\n");
	// If we want to use DDR, we need the DDR PLL spun up.
	if (is_Gr2()) {
		switch (get_memory_size()) {
		case 512:
			/* SFP : 32bit 400MHz 512MB */
			initialize_PLL(board_type, eDDR_PLL_800MHZ);
			ddr_config_and_init_gr2(&_32bit400m512_config, &startAddrs, MEM_WIDTH_VAL,
			                        mem_ctrl_regs, DDRCLKPD_VAL_400);
			break;
		case 1024:
			/* MFP : 32bit 400MHz 1024MB */
			initialize_PLL(board_type, eDDR_PLL_800MHZ);
			ddr_config_and_init_gr2(&_32bit400m1024_config, &startAddrs, MEM_WIDTH_VAL,
			                        mem_ctrl_regs, DDRCLKPD_VAL_400);
			break;
		case 2048:
			/* MFP w Cheetah : DDR4 32bit 800MHz 2048MB */
			ddr4_memory = true;
			initialize_PLL(board_type, eDDR_PLL_1600MHZ);
			ddr_config_and_init_gr2(&_ddr4_32bit800m2048_config, &startAddrs, MEM_WIDTH_VAL,
			                        mem_ctrl_regs, DDRCLKPD_VAL_800);
			break;
		default:
			break;
		}
	}
}
/* Used by vim and some versions of vi: set tabstop=4 shiftwidth=4: */
