/*
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2011-2014, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
*/
/*------------------------------------------------------------------------
            Include Files
------------------------------------------------------------------------*/
#include <stdarg.h>
#include <string.h>

#include "ciu_regheaders.h"
#include "MC_regheaders.h"

/*------------------------------------------------------------------------
        Defines
------------------------------------------------------------------------*/

 #define MAX(a,b) \
   ({ __typeof__ (a) _a = (a); \
       __typeof__ (b) _b = (b); \
     _a > _b ? _a : _b; })

  #define MIN(a,b) \
   ({ __typeof__ (a) _a = (a); \
       __typeof__ (b) _b = (b); \
     _a < _b ? _a : _b; })

//#define msg(x,y,...) minPrintf(__VA_ARGS__)
#define msg(x,y,...) 
/*------------------------------------------------------------------------
        Globals
------------------------------------------------------------------------*/


/*------------------------------------------------------------------------
        write_level
------------------------------------------------------------------------*/
/**
 *
 * \brief Write Leveling on the MC
 *
 * \param const MC_REGS_t *mc_reg - pointer to memory controller registers
 *        int cs - chip select for device being leveled
 *        int mem_width - width of ddr part being leveeled
 * \return void (no return value)
 *
 *
 */
void write_level(MC_REGS_t *mc_reg, int cs, int mem_width)
{
    int i, j, k;
    uint32_t dq_old[2];
    uint32_t dq_new = 0;
    uint32_t done = 0;
    uint32_t DRAM_Control_2_ODT0;
    uint32_t DRAM_Control_2_ODT1;
    uint32_t DRAM_Control_2_ODT2;
    uint32_t DRAM_Control_2_ODT3;
    uint32_t halfClkTap = 128;
    uint32_t wlVal[4] = {0,0,0,0};
    uint32_t wlValNeg[4] = {0,0,0,0};
    uint32_t minVal, maxVal;

    // Activate WL mode
    mc_reg->MC_Control_0 = MC_MC_CONTROL_0_WRITE_LEVEL_MODE_REPLACE_VAL(mc_reg->MC_Control_0, 1);
    mc_reg->CH0_PHY_WL_RL_Control = MC_CH0_PHY_WL_RL_CONTROL_PHY_WL_CS_SEL_REPLACE_VAL( \
                                    mc_reg->CH0_PHY_WL_RL_Control, 0x1<<cs);
    mc_reg->CH0_DRAM_Config_2 = MC_CH0_DRAM_CONFIG_2_WRITE_LEVEL_EN_REPLACE_VAL( \
                                mc_reg->CH0_DRAM_Config_2, 1);
    mc_reg->USER_COMMAND_0 = MC_USER_COMMAND_0_CH0_REPLACE_VAL(0, 1) |
                             MC_USER_COMMAND_0_CS0_REPLACE_VAL(0, 0x1) |
                             MC_USER_COMMAND_0_LMR1_REQ_REPLACE_VAL(0, 1);
    DRAM_Control_2_ODT0 = MC_CH0_DRAM_CONTROL_2_ODT0_SWITCH_MODE_MASK_SHIFT( \
                          mc_reg->CH0_DRAM_Control_2);
    DRAM_Control_2_ODT1 = MC_CH0_DRAM_CONTROL_2_ODT1_SWITCH_MODE_MASK_SHIFT( \
                          mc_reg->CH0_DRAM_Control_2);
    DRAM_Control_2_ODT2 = MC_CH0_DRAM_CONTROL_2_ODT2_SWITCH_MODE_MASK_SHIFT( \
                          mc_reg->CH0_DRAM_Control_2);
    DRAM_Control_2_ODT3 = MC_CH0_DRAM_CONTROL_2_ODT3_SWITCH_MODE_MASK_SHIFT( \
                          mc_reg->CH0_DRAM_Control_2);
    mc_reg->CH0_DRAM_Control_2 = MC_CH0_DRAM_CONTROL_2_ODT3_SWITCH_MODE_REPLACE_VAL( \
                          MC_CH0_DRAM_CONTROL_2_ODT2_SWITCH_MODE_REPLACE_VAL( \
                          MC_CH0_DRAM_CONTROL_2_ODT1_SWITCH_MODE_REPLACE_VAL( \
                          MC_CH0_DRAM_CONTROL_2_ODT0_SWITCH_MODE_REPLACE_VAL( \
                          MC_CH0_DRAM_CONTROL_2_ODT0_REPLACE_VAL( \
                          mc_reg->CH0_DRAM_Control_2, 0x1), 0), 0), 0), 0);

    dq_old[0] = 0xffffffff;
    dq_old[1] = 0xffffffff;

    switch ( mem_width )
    {
        case 8:  done = ~0x1; break;
        case 16: done = ~0x3; break;
        case 32: done = ~0xf; break;
    }

    for ( i = 0; (i < 128) && (done != 0xffffffff); i++ )
    {
        if ((i%4 == 0) || (i%4 == 1)) 
        {           
            switch ( cs )
            {
                case 0:
                    if(!(done&0x1))
                    {
                        mc_reg->CH0_PHY_WL_DATA_Control_CS0_B0 =
                            MC_CH0_PHY_WL_DATA_CONTROL_CS0_B0_PHY_WL_WCK_QSC_DLY_REPLACE_VAL( \
                            MC_CH0_PHY_WL_DATA_CONTROL_CS0_B0_PHY_WL_WCK_DQC_DLY_REPLACE_VAL( \
                            MC_CH0_PHY_WL_DATA_CONTROL_CS0_B0_PHY_WL_WCK_QS_DLY_REPLACE_VAL( \
                            MC_CH0_PHY_WL_DATA_CONTROL_CS0_B0_PHY_WL_WCK_DQ_DLY_REPLACE_VAL( \
                            mc_reg->CH0_PHY_WL_DATA_Control_CS0_B0, i|0x1), i|0x1), i&(~0x1)), i&(~0x1));
                    }
                    if ( mem_width > 8 )
                    {
                        if(!(done&0x2))
                        {
                            mc_reg->CH0_PHY_WL_DATA_Control_CS0_B1 =
                                MC_CH0_PHY_WL_DATA_CONTROL_CS0_B1_PHY_WL_WCK_QSC_DLY_REPLACE_VAL( \
                                MC_CH0_PHY_WL_DATA_CONTROL_CS0_B1_PHY_WL_WCK_DQC_DLY_REPLACE_VAL( \
                                MC_CH0_PHY_WL_DATA_CONTROL_CS0_B1_PHY_WL_WCK_QS_DLY_REPLACE_VAL( \
                                MC_CH0_PHY_WL_DATA_CONTROL_CS0_B1_PHY_WL_WCK_DQ_DLY_REPLACE_VAL( \
                                mc_reg->CH0_PHY_WL_DATA_Control_CS0_B1, i|0x1), i|0x1), i&(~0x1)), i&(~0x1));
                        }
                    }

                    //msg(MSG_DEBUG, RAW_DATA, "write leveling\r\nmc_reg->CH0_PHY_WL_DATA_Control_CS0_B0  0x%08x\r\nmc_reg->CH0_PHY_WL_DATA_Control_CS0_B1  0x%08x\r\nmc_reg->CH0_PHY_WL_DATA_Control_CS0_B2  0x%08x\r\nmc_reg->CH0_PHY_WL_DATA_Control_CS0_B3  0x%08x\r\n", mc_reg->CH0_PHY_WL_DATA_Control_CS0_B0, mc_reg->CH0_PHY_WL_DATA_Control_CS0_B1, mc_reg->CH0_PHY_WL_DATA_Control_CS0_B2,mc_reg->CH0_PHY_WL_DATA_Control_CS0_B3 );
                    break;
            }
            mc_reg->CH0_PHY_Control_10 = MC_CH0_PHY_CONTROL_10_PHY_WL_DQS_PULSE_REPLACE_VAL( \
                                         mc_reg->CH0_PHY_Control_10, 1);
            dq_new = MC_CH0_PHY_CONTROL_10_PHY_WL_DQ_OUT_MASK_SHIFT(mc_reg->CH0_PHY_Control_10);
            //if (cs == 0)
              msg(MSG_DEBUG, RAW_DATA, "i = 0x%02x, dq_new = 0x%02x, dq_old[0] = 0x%02x, dq_old[1] = 0x%02x, done = 0x%0x\r\n", i, dq_new, dq_old[0], dq_old[1], done);
    
            for ( j = 0; j < (mem_width>>3); j++ )
            {
                //Falling edge detect must have 2 adjacent low values preceded by a high value.
                if ( !(dq_new&(0x1<<j)) && !(dq_old[0]&(0x1<<j)) && (dq_old[1]&(0x1<<j)) && !(done&(0x1<<j)) ) {
                    wlValNeg[j] = i - 1;  // Back up one value due to 2 tap glitch filter.
                }
                // Rising edge detect must have 2 adjacent high values preceded by a low value.
                if ( (dq_new&(0x1<<j)) && (dq_old[0]&(0x1<<j)) && !(dq_old[1]&(0x1<<j)) && !(done&(0x1<<j)) ) {
                    //msg(MSG_DEBUG, RAW_DATA, "\tFound rising edge of DQ at i = %0d, byte lane %0d !\r\n", i,j);
                    done |= (0x1<<j);
                    wlVal[j] = i-1;   // Back up one value due to 2 tap glitch filter.
                }
            }
            // Store 2 old stages so that we can filter glitches.
            dq_old[1] = dq_old[0];
            dq_old[0] = dq_new;
        }
    }

    for ( k = 0; k < (mem_width>>3); k++ )
    {
        if ((wlValNeg[k] != 0) && (wlVal[k] != 0))
        {
            if (wlValNeg[k] < wlVal[k]) 
            {
                halfClkTap = wlVal[k] - wlValNeg[k];
            }
            else
            {
                halfClkTap = wlValNeg[k] - wlVal[k];
            }
        }
        msg(MSG_DEBUG, RAW_DATA, "BL = %d, wlVal = 0x%02x, wlValNeg = 0x%02x, halfClkTap = 0x%02x\r\n", k, wlVal[k], wlValNeg[k], halfClkTap);
    }

    // Find min & max WL values
    minVal = 128;
    maxVal = 0;
    for ( k = 0; k < (mem_width>>3); k++ )
    {
        minVal = MIN(minVal,wlVal[k]);
        maxVal = MAX(maxVal,wlVal[k]);
    }
 
    msg(MSG_DEBUG, RAW_DATA, "Min Val = %02x, Max Val = 0x%02x\r\n", minVal, maxVal);

    // If delta from smallest to largest wlVal is > 1/2 clock period, find outlier and use 0 instead.

    wlVal[0] = ((wlValNeg[0] < wlVal[0]) && (wlValNeg[0] > (halfClkTap >> 1))) ? 0x0 : wlVal[0];
    wlVal[1] = ((wlValNeg[1] < wlVal[1]) && (wlValNeg[1] > (halfClkTap >> 1))) ? 0x0 : wlVal[1];
    
    // Write final WL values to registers.
    switch ( cs )
    {
        case 0:
            mc_reg->CH0_PHY_WL_DATA_Control_CS0_B0 =
                MC_CH0_PHY_WL_DATA_CONTROL_CS0_B0_PHY_WL_WCK_QSC_DLY_REPLACE_VAL( \
                MC_CH0_PHY_WL_DATA_CONTROL_CS0_B0_PHY_WL_WCK_DQC_DLY_REPLACE_VAL( \
                MC_CH0_PHY_WL_DATA_CONTROL_CS0_B0_PHY_WL_WCK_QS_DLY_REPLACE_VAL( \
                MC_CH0_PHY_WL_DATA_CONTROL_CS0_B0_PHY_WL_WCK_DQ_DLY_REPLACE_VAL( \
                mc_reg->CH0_PHY_WL_DATA_Control_CS0_B0, wlVal[0]|0x1), wlVal[0]|0x1), wlVal[0]&(~0x1)), wlVal[0]&(~0x1));
            mc_reg->CH0_PHY_WL_DATA_Control_CS0_B1 =
                MC_CH0_PHY_WL_DATA_CONTROL_CS0_B1_PHY_WL_WCK_QSC_DLY_REPLACE_VAL( \
                MC_CH0_PHY_WL_DATA_CONTROL_CS0_B1_PHY_WL_WCK_DQC_DLY_REPLACE_VAL( \
                MC_CH0_PHY_WL_DATA_CONTROL_CS0_B1_PHY_WL_WCK_QS_DLY_REPLACE_VAL( \
                MC_CH0_PHY_WL_DATA_CONTROL_CS0_B1_PHY_WL_WCK_DQ_DLY_REPLACE_VAL( \
                mc_reg->CH0_PHY_WL_DATA_Control_CS0_B1, wlVal[1]|0x1), wlVal[1]|0x1), wlVal[1]&(~0x1)), wlVal[1]&(~0x1));
            break;
    }

    mc_reg->CH0_DRAM_Control_2 = MC_CH0_DRAM_CONTROL_2_ODT3_SWITCH_MODE_REPLACE_VAL( \
                                 MC_CH0_DRAM_CONTROL_2_ODT2_SWITCH_MODE_REPLACE_VAL( \
                                 MC_CH0_DRAM_CONTROL_2_ODT1_SWITCH_MODE_REPLACE_VAL( \
                                 MC_CH0_DRAM_CONTROL_2_ODT0_SWITCH_MODE_REPLACE_VAL( \
                                 MC_CH0_DRAM_CONTROL_2_ODT0_REPLACE_VAL( \
                                 mc_reg->CH0_DRAM_Control_2, 0x0), \
                                 DRAM_Control_2_ODT0), \
                                 DRAM_Control_2_ODT1), \
                                 DRAM_Control_2_ODT2), \
                                 DRAM_Control_2_ODT3);
    mc_reg->CH0_DRAM_Config_2 = MC_CH0_DRAM_CONFIG_2_WRITE_LEVEL_EN_REPLACE_VAL( \
                                mc_reg->CH0_DRAM_Config_2, 0);
    mc_reg->CH0_PHY_WL_RL_Control = MC_CH0_PHY_WL_RL_CONTROL_PHY_WL_CS_SEL_REPLACE_VAL( \
                                     mc_reg->CH0_PHY_WL_RL_Control, 0x0);
    mc_reg->USER_COMMAND_0 = MC_USER_COMMAND_0_CH0_REPLACE_VAL(0, 1) |
                             MC_USER_COMMAND_0_CS0_REPLACE_VAL(0, 0x1<<cs) |
                             MC_USER_COMMAND_0_LMR1_REQ_REPLACE_VAL(0, 1);
    mc_reg->MC_Control_0 = MC_MC_CONTROL_0_WRITE_LEVEL_MODE_REPLACE_VAL(mc_reg->MC_Control_0, 0);

    return;
}

