/*
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2011-2014, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
*/
/*------------------------------------------------------------------------
            Include Files
------------------------------------------------------------------------*/
#include <stdarg.h>
#include <string.h>

#include "regAddrs.h"
#include "MC_regheaders.h"
#include "../ddr_utils.h"
#include "ciu_regheaders.h"
#include "ID_utils.h"

/*------------------------------------------------------------------------
        Defines
------------------------------------------------------------------------*/


//#define msg(x,y,...) minPrintf(__VA_ARGS__)
#define msg(x,y,...) 

#define SEED 0x3A8F05C5
#define MIN_DLL_WINDOW 10

#define UTF_waitForMicroSecs(x) cpu_spin_delay(x)
//#define IRL  //Internal Read Leveling Routine.  Only on RevC

/*------------------------------------------------------------------------
        Globals
------------------------------------------------------------------------*/
extern uint32_t __start_of_free_DDR__;  //from ld file


/*------------------------------------------------------------------------
        Prototypes
------------------------------------------------------------------------*/
static void insertValue(uint16_t value, uint16_t* windowArray, uint16_t* indexArray, uint8_t index );
static void setReadLevelingDelays(MC_REGS_t *mc_reg, int cs, int mem_width, uint32_t CDly, uint32_t TDly);


#if defined(IRL)
/*------------------------------------------------------------------------
        read_leveling_internal (no common CDelay requirement)
------------------------------------------------------------------------*/
/**
 *
 * \brief Attempt to adjust DQS gate to optimal value
 *
 * \param const MC_REGS_t *mc_reg - pointer to memory controller registers
 * \return uint8_t return readLevelingFail = 0 for pass and readLevelingFail = 1 for fail
 *
 *
 */

uint8_t read_leveling(MC_REGS_t *mc_reg, uint16_t mem_width, cs_sa_t *startAddrs)
{
    int still_searching = 1;
    int still_scanning = 1;
    int status;
    int BL;
    int done;
    uint32_t mapRecord[3];
    volatile uint32_t *testAddress[3];
    uint32_t mask = 0x000000ff;
    uint32_t c_delay = 0x000000ff;  //max tap delay
    uint32_t t_delay;
    uint32_t RL_setting;
    uint8_t i, csVld[3];
    uint8_t readLevelingFail = 0;
    volatile uint32_t rldata;
    uint32_t act[4];
    uint32_t data[4];
    CIU_REGS_t *ciu_regs = (CIU_REGS_t*) AP_CIU_BASE;


    msg(MSG_DEBUG, RAW_DATA, "Enable Read Leveling\r\n");
    mc_reg->CH0_PHY_WL_RL_Control = MC_CH0_PHY_WL_RL_CONTROL_PHY_RL_ENABLE_REPLACE_VAL(mc_reg->CH0_PHY_WL_RL_Control,0x1);
    msg(MSG_DEBUG, RAW_DATA, "CH0 PHY Write Leveling Read Leveling Control written with 0x%08x",
                                      mc_reg->CH0_PHY_WL_RL_Control);

    msg(MSG_DEBUG, RAW_DATA, "\r\n\r\n\r\nread Leveling 2\r\n");
    //msg(MSG_DEBUG, RAW_DATA, "0x%08x 0x%08x 0x%08x \r\n\r\n\r\n", freeDDR_CS[0], freeDDR_CS[1], freeDDR_CS[2]);

    //record map register
    mapRecord[0] = mc_reg->CH0_MMAP0;

    //set new mapping
    mc_reg->CH0_MMAP0 = 0x000d0001;

    testAddress[0] = &__start_of_free_DDR__;

    act[0] = 0xAA5555AA;
    act[1] = 0xFF0000FF;
    act[2] = 0x00FFFF00;
    act[3] = 0x11223344;

    // Determine which CS are valid
    for (i=0; i<1; i++)
    {
        if (startAddrs->sa[i] != 0xDEADBEEF)
        {
            csVld[i] = 1;
            still_searching = 1;
            still_scanning = 1;  

            if (mem_width == 16) 
            {
                done = 0x3;
            }
            else
            {
                done = 0xf;
            }

            //write test data
            msg(MSG_INFO, RAW_DATA, "Starting Internal Read Leveling on CS%d.\r\n", i);
            msg(MSG_DEBUG, RAW_DATA, "testAddress 0x%08x\r\n", testAddress[i]);
            xfer_DMA_DATA(act, testAddress[i], 16);
            msg(MSG_DEBUG, RAW_DATA, "Wrote test data\r\n");

            c_delay = 0x00000F00;  //max tap delay
            while (still_searching == 1)
            {
                t_delay = 0x0000007f;  //max tap delay
                still_scanning = 1;
                int state7[4] = {0,0,0,0};
                int stateF[4] = {0,0,0,0};
                int stateE[4] = {0,0,0,0};
  
                while (still_scanning == 1)
                {
                    // clear OutP and OutN flops, set delays
                    //msg(MSG_DEBUG, RAW_DATA, "Setting QS Gate Clear.\r\n");
                    Delay(50);
                    ciu_regs->MC_PRIORITY3 &= !CIU_MC_PRIORITY3_QS_GATE_CLR_MASK;
                    //msg(MSG_DEBUG, RAW_DATA, "Clearing QS Gate Clear.\r\n");
                    Delay(50);
                    ciu_regs->MC_PRIORITY3 |= CIU_MC_PRIORITY3_QS_GATE_CLR_MASK;
                    //msg(MSG_DEBUG, RAW_DATA, "Clearing QS Gate Clear.\r\n");
                    Delay(50);
                    RL_setting = c_delay | t_delay;

                    switch(i)
                    {
                        case 0:
                            //msg(MSG_DEBUG, RAW_DATA, "Read Leveling on CS0.  RL_Setting:  0x%08x\r\n", RL_setting);
                            if(done & 0x1) 
                            {
                                mc_reg->CH0_PHY_RL_Control_CS0_B0 = RL_setting;
                                //msg(MSG_DEBUG, RAW_DATA, "1\r\n");
                            }
  
                            if(done & 0x2) 
                            {
                                mc_reg->CH0_PHY_RL_Control_CS0_B1 = RL_setting;
                                //msg(MSG_DEBUG, RAW_DATA, "2\r\n");
                            }
  
                            break;
                    }
  
                    //msg(MSG_DEBUG, RAW_DATA, "Read Leveling delays set\r\n");
                    //xfer_DMA_DATA(&rldata, testAddress, 4);
                    rldata = *testAddress[i];
  
                    //msg(MSG_DEBUG, RAW_DATA, "Read Leveling data read\r\n");
                    // read OutP and OutN
                    rldata = *(uint32_t*)0xD0610040;

                    //msg(MSG_DEBUG, RAW_DATA, "c_delay 0x%08x t_delay 0x%08x outp 0x%08x\r\n",c_delay, t_delay, rldata);
    
                    for(BL = 0; BL < mem_width/8; BL++)
                    {
                        //$display("rldataF %x, shift %x",rldata & (32'h0000000F << 4*BL), (32'h0000000E << 4*BL));
                        if((done & (0x1 << BL)) == (0x1 << BL))
                        {  
                            mask = 0x000000FF << (8*BL);
    
                            if ((rldata & (0x0000000F << 4*BL)) == (0x00000007 << 4*BL))
                            {
                                state7[BL] = 1;
                                stateF[BL] = 0;
                                stateE[BL] = 0;
                                //msg(MSG_DEBUG, RAW_DATA, "Found 7 on BL%d of CS%d\r\n", BL, i);
                            }
                            else if ((rldata & (0x0000000F << 4*BL)) == (0x0000000F << 4*BL)) 
                            {
                                //stateF = 1;
                                if ((state7[BL] == 1) && (stateE[BL] == 0)) 
                                {
                                    //state7 = 1;
                                    stateF[BL] = 1;
                                    stateE[BL] = 0;
                                    //msg(MSG_DEBUG, RAW_DATA, "Found 7-F sequence on BL%d of CS%d\r\n", BL, i);
                                }
                                else
                                {
                                    state7[BL] = 0;
                                    stateF[BL] = 0;
                                    stateE[BL] = 0;
                                }
    
                            }
                            else if ((rldata & (0x0000000F << 4*BL)) == (0x0000000E << 4*BL)) 
                            {
                                if ((state7[BL] == 1) && (stateF[BL] == 1)) 
                                {
                                    stateE[BL] = 1;
                                    //msg(MSG_DEBUG, RAW_DATA, "Found 7-F-E sequence on BL%d of CS%d, testing data\r\n", BL, i);
    
                                    xfer_DMA_DATA(testAddress[i], data, 16);
    
                                    if(((data[0] & mask) == (act[0] & mask))
                                       && ((data[1] & mask) == (act[1] & mask))
                                       && ((data[2] & mask) == (act[2] & mask))
                                       && ((data[3] & mask) == (act[3] & mask)))
                                    {  
                                        done &= ~(0x1 << BL);
                                        msg(MSG_DEBUG, RAW_DATA, "SUCCESS: It looks like we're done for CS%d, BL %d.  Data 0x%08x 0x%08x 0x%08x 0x%08x\r\n\r\n", i, BL, data[0], data[1], data[2], data[3]);
                                    }
                                    else
                                    {
                                        msg(MSG_DEBUG, RAW_DATA, "FAIL: Found an E, but data did not compare for CS%d, BL %d.  Data 0x%08x 0x%08x 0x%08x 0x%08x\r\n\r\n", i, BL, data[0], data[1], data[2], data[3]);
                                    }
                                }
                                else
                                {
                                    state7[BL] = 0;
                                    stateF[BL] = 0;
                                    stateE[BL] = 0;
                                }
                            }
                            else
                            {
                                state7[BL] = 0;
                                stateF[BL] = 0;
                                stateE[BL] = 0;
                            }
                        }
                    }


                    status = (((((rldata & 0x0000000F) == 0x7) || ((rldata & 0x0000000F) == 0xF) || ((rldata & 0x0000000F) == 0xE))&& ((done & 0x1) == 0x1))
                              || ((((rldata & 0x000000F0) == 0x70) || ((rldata & 0x000000F0) == 0x000000F0) || ((rldata & 0x000000F0) == 0x000000E0)) && ((done & 0x2) == 0x2))
                              || ((((rldata & 0x00000F00) == 0x700) || ((rldata & 0x00000F00) == 0x00000F00) || ((rldata & 0x00000F00) == 0x00000E00)) && ((done & 0x4) == 0x4))
                              || ((((rldata & 0x0000F000) == 0x7000) || ((rldata & 0x0000F000) == 0x0000F000) || ((rldata & 0x0000F000) == 0x0000E000)) && ((done & 0x8) == 0x8)));
    
                    //msg(MSG_DEBUG, RAW_DATA, "INFO: status %d   rldata 0x%08x   done 0x%01x\r\n", status, rldata, done);
                    if (status == 1)
                    {
                        if (t_delay > 0)
                        {
                            t_delay = t_delay - 1;
                        }
                        else
                        {
                            still_scanning = 0;
                        }
                    }
                    else
                    {
                        if (t_delay > 16)
                        {
                            t_delay = t_delay - 16;
                        }
                        else
                        {
                            still_scanning = 0;
                        }
                    }
                }
    
                if (done == 0)
                {
                    still_searching = 0;
                }
    
                if (c_delay > 0)
                {
                    c_delay = c_delay - 0x00000100;
                }
                else
                {
                    still_searching = 0;
                    readLevelingFail = 1;
                    msg(MSG_DEBUG, RAW_DATA, "FAILURE! No valid read leveling values found for CS%d.", i);
                }
            }
        }
        else
        {
            csVld[i] = 0;
        }
    }

    for (i=0; i<1; i++)
    {
        if (csVld[i] == 1) 
        {
            switch(i)
            {
                case 0:
                    msg(MSG_STATUS, RAW_DATA, "mc_reg->CH0_PHY_RL_Control_CS0_B0  0x%08x\r\nmc_reg->CH0_PHY_RL_Control_CS0_B1  0x%08x\r\n", mc_reg->CH0_PHY_RL_Control_CS0_B0, mc_reg->CH0_PHY_RL_Control_CS0_B1);
                    break;
                default:
                    msg(MSG_STATUS, RAW_DATA, "Error: No valid CS found!" );
                    readLevelingFail = 1;
                    break;
            }
        }
    }

    mc_reg->CH0_MMAP0 = mapRecord[0];
    return readLevelingFail;

}

#else

/*------------------------------------------------------------------------
        read_leveling (no common CDelay requirement)
------------------------------------------------------------------------*/
/**
 *
 * \brief Attempt to adjust DQS gate to optimal value
 *
 * \param const MC_REGS_t *mc_reg - pointer to memory controller registers
 * \return uint8_t return readLevelingFail = 0 for pass and readLevelingFail = 1 for fail
 *
 *
 */
uint8_t read_leveling(MC_REGS_t *mc_reg, uint16_t mem_width, cs_sa_t *startAddrs)
{
    uint32_t i = 0;
    uint8_t j = 0;
    uint8_t k = 0;
    uint8_t m = 0;
    uint16_t firstPass[4] = {0xffff,0xffff,0xffff,0xffff};
    uint16_t lastPass[4] = {0xffff,0xffff,0xffff,0xffff};
    uint8_t passArray[4][128];
    uint32_t dataMask = 0x000000FF;
    uint8_t numBL = 2;
    uint32_t mapRecord[3];
    uint32_t tmp;
    uint16_t window[2][16];
    uint16_t chosen[2][16];
    uint16_t chosenWindows[2];
    uint16_t windowOrder[2][16];
    uint32_t CS_Reg_Span = 8;
    uint8_t csVld[1];
    volatile uint32_t* freeDDR_CS[1];
    volatile uint32_t* PHY_B0_Reg = &mc_reg->CH0_PHY_RL_Control_CS0_B0;
    volatile uint32_t* PHY_B1_Reg = &mc_reg->CH0_PHY_RL_Control_CS0_B1;
    uint32_t act[8];
    uint32_t data[8];
    uint8_t cdMax = 8;

    //set initial data background
    act[0] = SEED;
    for(i=0;i<7;i++)
    {
        act[i+1] = act[i]*SEED;
    }

    //free space in DDR to use for test data
    freeDDR_CS[0] = &__start_of_free_DDR__;

    msg(MSG_DEBUG, RAW_DATA, "\r\n\r\n\r\nRead Leveling\r\n");
    msg(MSG_DEBUG, RAW_DATA, "0x%08x \r\n\r\n\r\n", freeDDR_CS[0]);

    //record map register
    mapRecord[0] = mc_reg->CH0_MMAP0;

    //set new mapping
    mc_reg->CH0_MMAP0 = 0x000d0001;

    // Determine which CS are valid
    for (i=0; i<1; i++)
    {
        if (startAddrs->sa[i] != 0xDEADBEEF)
            csVld[i] = 1;
        else
            csVld[i] = 0;
    }
    //enable read leveling
    mc_reg->CH0_PHY_WL_RL_Control = MC_CH0_PHY_WL_RL_CONTROL_PHY_RL_ENABLE_REPLACE_VAL(mc_reg->CH0_PHY_WL_RL_Control, 1);

    //run for each chip select
    for(m=0; m<1; m++)
    {
        //set all window data to 0
        memset(window,0x0,sizeof(window));
        memset(chosen,0x0,sizeof(chosen));
        memset(windowOrder,0x0,sizeof(windowOrder));
        memset(chosenWindows,0x0,sizeof(chosenWindows));
        
        if(csVld[m])  //if cs is valid run read leveling routine
        {
            msg(MSG_DEBUG, RAW_DATA, "pre read leveilng CS%d\r\n", m);
            if (m==0)
                msg(MSG_DEBUG, RAW_DATA, "mc_reg->CH0_PHY_RL_Control_CS0_B0  0x%08x\r\nmc_reg->CH0_PHY_RL_Control_CS0_B1  0x%08x\r\n", mc_reg->CH0_PHY_RL_Control_CS0_B0, mc_reg->CH0_PHY_RL_Control_CS0_B1);

            for( j=16; j>0; j-- )
            {
                memset(passArray,0x1,sizeof(passArray));

                for( i = 1; i < 128; i++ )
                {
                    //create pseudo-random background data
                    act[0] = act[7]+SEED;
                    for(k=0;k<7;k++)
                    {
                        act[k+1] = act[k]+SEED;
                    }

                    xfer_DMA_DATA(act, freeDDR_CS[m], 32);

                    //set cycle delays
                    setReadLevelingDelays(mc_reg, m, mem_width, (j-1), i);

                    //read data
                    xfer_DMA_DATA(freeDDR_CS[m], data, 32);

                    //verify byte lanes
                    if (mem_width == 16)
                        numBL = 2;
                    else
                        numBL = 4;

                    for(k=0; k<numBL; k++)
                    {
                        dataMask = 0x000000FF << 8*k;
                        //msg(MSG_DEBUG, RAW_DATA, "datamask  0x%08x\r\n", dataMask );

                        if( ((data[0] & dataMask) == (act[0] & dataMask)) && ((data[1] & dataMask) == (act[1] & dataMask)) && \
                            ((data[2] & dataMask) == (act[2] & dataMask)) && ((data[3] & dataMask) == (act[3] & dataMask)) && \
                            ((data[4] & dataMask) == (act[4] & dataMask)) && ((data[5] & dataMask) == (act[5] & dataMask)) && \
                            ((data[6] & dataMask) == (act[6] & dataMask)) && ((data[7] & dataMask) == (act[7] & dataMask)) )
                        {
                            passArray[k][i] = 0;
                        }
                        else
                        {
                            passArray[k][i] = 1;
                        }

                        //msg(MSG_DEBUG, RAW_DATA, "Pass Array %0d, %0d, %0d, is %0d\r\n", j, k, i, passArray[k][i]);
                    }

                    //set cycle delays
                    setReadLevelingDelays(mc_reg, m, mem_width, 0, 0);

                    //read data to make sure gate is closed
                    xfer_DMA_DATA(freeDDR_CS[m], data, 32);

                }
                // For each cycle delay and tap delay of each byte lane find the biggest window
                // using the biggest windows found for each byte lane of each cycle, determine the
                // smallest of those windows in each cycle.  Compare all the smallest windows from
                // each cycle delay to find the largest of those.  Use the taps associated with
                //  that cycle delay.
                for(i=0; i<128; i++)
                {
                    for(k=0; k<numBL; k++)
                    {
                        //did point pass?
                        if(passArray[k][i] == 0)
                        {
                             //msg(MSG_DEBUG, RAW_DATA, "Point Passed\r\n");
                            //set start of window
                            if(firstPass[k] == 0xffff)
                            {
                                //msg(MSG_DEBUG, RAW_DATA, "Found firstPass[%0d] at 0x%0x\r\n",k,i);
                                msg(MSG_DEBUG, RAW_DATA, "Found firstPass[%0d] at 0x%0x\r\n",k,i);
                                firstPass[k] = i;
                            }
                            else if (i == 127)  // Last tap delay value is good.
                            {
                                lastPass[k] = 127;
                                tmp = lastPass[k] - firstPass[k];
                                //msg(MSG_DEBUG, RAW_DATA, "Found lastPass[%0d][%0d][%d] tmp %0d  window %0d\r\n",k,j-1,i, tmp, window[k][j-1]);
                                msg(MSG_DEBUG, RAW_DATA, "Found lastPass[%0d][%0d][%d] tmp %0d  window %0d\r\n",k,j-1,i, tmp, window[k][j-1]);
                                if(tmp > window[k][j-1])
                                {
                                    window[k][j-1] = tmp;
                                    // Use 50% window for DIMM
                                    if (m == 0)
                                      chosen[k][j-1] = firstPass[k] + (uint16_t)(tmp*63/100);
                                    else
                                      chosen[k][j-1] = firstPass[k] + (uint16_t)(tmp*37/100);
                                      //chosen[k][j-1] = firstPass[k] + (uint16_t)(tmp*50/100);

                                    msg(MSG_DEBUG, RAW_DATA, "Window[%0d][%0d]  %03d    %02x - %02x    tmp=0x%0x, chosen[%0d]=0x%0x\r\n",
                                        k,(j-1),window[k][j-1],firstPass[k],lastPass[k],tmp,k,chosen[k][j-1]);
                                    firstPass[k] = 0xffff;
                                    lastPass[k] = 0xffff;
                                }
                            }
                        }
                        else
                        {

                             //msg(MSG_DEBUG, RAW_DATA, "Point Failed\r\n");
                            //point failed, set end of window if applicable
                            if((firstPass[k] != 0xffff) && (lastPass[k] == 0xffff))
                            {
                                lastPass[k] = i-1;
                                tmp = lastPass[k] - firstPass[k];
                                //msg(MSG_DEBUG, RAW_DATA, "Found lastPass[%0d][%0d][%d] tmp %0d  window %0d\r\n",k,j-1,i, tmp, window[k][j-1]);
                                msg(MSG_DEBUG, RAW_DATA, "Found lastPass[%0d][%0d][%d] tmp %0d  window %0d\r\n",k,j-1,i, tmp, window[k][j-1]);
                                if(tmp > window[k][j-1])
                                {
                                    window[k][j-1] = tmp;

                                    //Choose 50% window
                                    chosen[k][j-1] = firstPass[k] + (uint16_t)(tmp*50/100);

                                    msg(MSG_DEBUG, RAW_DATA, "Window[%0d][%0d]  %03d    %02x - %02x    tmp=0x%0x, chosen[%0d]=0x%0x\r\n",
                                        k,(j-1),window[k][j-1],firstPass[k],lastPass[k],tmp,k,chosen[k][j-1]);
                                }
                                firstPass[k] = 0xffff;
                                lastPass[k] = 0xffff;
                            }
                        }
                    }
                }
                //reset all the variables
                firstPass[0] = 0xffff;
                lastPass[0] = 0xffff;
                firstPass[1] = 0xffff;
                lastPass[1] = 0xffff;
                if (mem_width == 32)
                {
                    firstPass[2] = 0xffff;
                    lastPass[2] = 0xffff;
                    firstPass[3] = 0xffff;
                    lastPass[3] = 0xffff;
                }

            }
            for(i = 0; i<2; i++)
            {
                // JPS - Limit CD < 8 unless revC
                cdMax = 16;                  
                for(j = 0; j<cdMax; j++)
                {
                    insertValue( window[i][j], window[i], windowOrder[i], j);
                }
            }
            for(i = 0; i<2; i++)
            {
                // JPS - Limit CD < 8 unless revC
                cdMax = 16;
                for(j = 0; j<cdMax; j++)
                {
                    msg(MSG_DEBUG, RAW_DATA, "windowOrder[%d][%d] %03d %03d\r\n", i,j,windowOrder[i][j],window[i][windowOrder[i][j]]);
                }
            }

            // JPS - Limit CD < 8 unless revC
            cdMax = 15;
            chosenWindows[0] = windowOrder[0][cdMax];
            chosenWindows[1] = windowOrder[1][cdMax];
            //chooseWindows(windowOrder[0],windowOrder[1],windowOrder[2],windowOrder[3],chosenWindows, mem_width);

            //set final cycle and tap delays
            *PHY_B0_Reg = MC_CH0_PHY_RL_CONTROL_CS0_B0_PHY_RL_CYCLE_DLY_REPLACE_VAL(*PHY_B0_Reg, chosenWindows[0]);
            *PHY_B1_Reg = MC_CH0_PHY_RL_CONTROL_CS0_B1_PHY_RL_CYCLE_DLY_REPLACE_VAL(*PHY_B1_Reg, chosenWindows[1]);
            *PHY_B0_Reg = MC_CH0_PHY_RL_CONTROL_CS0_B0_PHY_RL_TAP_DLY_REPLACE_VAL(*PHY_B0_Reg, chosen[0][chosenWindows[0]]);
            *PHY_B1_Reg = MC_CH0_PHY_RL_CONTROL_CS0_B1_PHY_RL_TAP_DLY_REPLACE_VAL(*PHY_B1_Reg, chosen[1][chosenWindows[1]]);
        }

        PHY_B0_Reg += CS_Reg_Span;
        PHY_B1_Reg += CS_Reg_Span;
    }
 
    //restore map register
    mc_reg->CH0_MMAP0 = mapRecord[0];

    msg(MSG_INFO, RAW_DATA, "POST read leveling\r\n");
    if(csVld[0])  //if cs is valid run read leveling routine
    {
        msg(MSG_INFO, RAW_DATA, "mc_reg->CH0_PHY_RL_Control_CS0_B0  0x%08x\r\nmc_reg->CH0_PHY_RL_Control_CS0_B1  0x%08x\r\n", mc_reg->CH0_PHY_RL_Control_CS0_B0, mc_reg->CH0_PHY_RL_Control_CS0_B1);
    }

    if (((mc_reg->CH0_PHY_RL_Control_CS0_B0 == 0x00000F00) || (mc_reg->CH0_PHY_RL_Control_CS0_B0 == 0x00000000)) || 
        ((mc_reg->CH0_PHY_RL_Control_CS0_B1 == 0x00000F00) || (mc_reg->CH0_PHY_RL_Control_CS0_B1 == 0x00000000)))
    {
        msg(MSG_INFO, RAW_DATA, "Retrying Read Leveling\r\n");
        return 1;
    }
    else
    {
        msg(MSG_INFO, RAW_DATA, "Passed Read Leveling\r\n");
        msg(MSG_INFO, RAW_DATA, "\r\n");
        return 0;
    }
}
#endif

/*------------------------------------------------------------------------
        insertValue
------------------------------------------------------------------------*/
/**
 *
 * \brief Use the value/index pair  to insert the index into the array in
 *        in order to maintain a sorted list
 *
 * \param uint16_t  windowSize - passing window
 *        uint16_t* windowSizeArray - pointer to sorted array of window sizes
 *        uint16_t* cdArray - sorted list cycle delays
 *        uint8_t   cd - cycle delay
 *
 *
 */
void insertValue(uint16_t windowSize, uint16_t* windowSizeArray, uint16_t* cdArray, uint8_t cd )
{
    uint16_t j;
    //msg(MSG_DEBUG, RAW_DATA, "cd %d  windowSizeArray %d    cdArray %d \r\n", cd,windowSizeArray[cd],cdArray[cd]);
    // JPS mod:  use <=, which will bias the selection to the lower cd in cases where they are equal
    for (j = cd; j > 0 && windowSize < windowSizeArray[cdArray[j-1]]; j--)
    //for (j = cd; j > 0 && windowSize <= windowSizeArray[cdArray[j-1]]; j--)
    {
        cdArray[j] = cdArray[j - 1];
        //msg(MSG_DEBUG, RAW_DATA, "after change cd %d   windowSizeArray %d    cdArray %d  \r\n", j,windowSizeArray[cdArray[j]],cdArray[j]);
    }
    cdArray[j] = cd;
    //msg(MSG_DEBUG, RAW_DATA, "after insert cd %d  cdArray %d  \r\n", j, cdArray[j]);
}

/*------------------------------------------------------------------------
        setReadLevelingDelays
------------------------------------------------------------------------*/
/**
 *
 * \abstract writing of registers to allow for up to 3 chip selects
 *
 *
 */
static void setReadLevelingDelays(MC_REGS_t *mc_reg, int cs, int mem_width, uint32_t CDly, uint32_t TDly)
{
    volatile uint32_t* PHY_B0_Reg = &mc_reg->CH0_PHY_RL_Control_CS0_B0;
    volatile uint32_t* PHY_B1_Reg = &mc_reg->CH0_PHY_RL_Control_CS0_B1;
    uint32_t CS_Reg_Span = 0x20;

    PHY_B0_Reg += cs*CS_Reg_Span;
    PHY_B1_Reg += cs*CS_Reg_Span;

    switch ( cs )
    {
        case 0:
            if (TDly == 1) /* Only write debug message on first tap delay setting */
                msg(MSG_DEBUG, RAW_DATA, "Setting CS%0d CDly = %0d\r\n",cs,CDly);
            //set cycle delays
            mc_reg->CH0_PHY_RL_Control_CS0_B0 = MC_CH0_PHY_RL_CONTROL_CS0_B0_PHY_RL_CYCLE_DLY_REPLACE_VAL(mc_reg->CH0_PHY_RL_Control_CS0_B0, CDly);
            mc_reg->CH0_PHY_RL_Control_CS0_B1 = MC_CH0_PHY_RL_CONTROL_CS0_B1_PHY_RL_CYCLE_DLY_REPLACE_VAL(mc_reg->CH0_PHY_RL_Control_CS0_B1, CDly);
            //set tap delay
            mc_reg->CH0_PHY_RL_Control_CS0_B0 = MC_CH0_PHY_RL_CONTROL_CS0_B0_PHY_RL_TAP_DLY_REPLACE_VAL(mc_reg->CH0_PHY_RL_Control_CS0_B0, TDly);
            mc_reg->CH0_PHY_RL_Control_CS0_B1 = MC_CH0_PHY_RL_CONTROL_CS0_B1_PHY_RL_TAP_DLY_REPLACE_VAL(mc_reg->CH0_PHY_RL_Control_CS0_B1, TDly);

            //ensure write is complete
            while((MC_CH0_PHY_RL_CONTROL_CS0_B0_PHY_RL_CYCLE_DLY_MASK_SHIFT(mc_reg->CH0_PHY_RL_Control_CS0_B0)) != CDly);
            while((MC_CH0_PHY_RL_CONTROL_CS0_B1_PHY_RL_CYCLE_DLY_MASK_SHIFT(mc_reg->CH0_PHY_RL_Control_CS0_B1)) != CDly);
            while((MC_CH0_PHY_RL_CONTROL_CS0_B0_PHY_RL_TAP_DLY_MASK_SHIFT(mc_reg->CH0_PHY_RL_Control_CS0_B0)) != TDly);
            while((MC_CH0_PHY_RL_CONTROL_CS0_B1_PHY_RL_TAP_DLY_MASK_SHIFT(mc_reg->CH0_PHY_RL_Control_CS0_B1)) != TDly);

            break;
    }
}


