/*
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2011-2014, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
*/
/*------------------------------------------------------------------------
            Include Files
------------------------------------------------------------------------*/
#include <string.h>
#include "regAddrs.h"
#include "MC_regheaders.h"
#include "../ddr_utils.h"
#include "ddr4_training.h"

/*------------------------------------------------------------------------
        Defines
------------------------------------------------------------------------*/

#define SEED 0x3A8F05C5
//#define msg(x,y,...) minPrintf(__VA_ARGS__)
#define msg(x,y,...)

/*------------------------------------------------------------------------
        Globals
------------------------------------------------------------------------*/

extern uint32_t __start_of_free_DDR__;  //from ld file


/*------------------------------------------------------------------------
        dram_vref_training
------------------------------------------------------------------------*/
/**
 *
 * \brief Attempt to train DDR4 Vref (only supports 1 CS)
 *
 * \param const MC_REGS_t *mc_reg - pointer to memory controller registers
 * \param const uint16_t mem_width - width or ddr data bus
 * \return void (no return value)
 *
 *
 */
void dram_vref_training(MC_REGS_t *mc_reg, uint16_t mem_width, uint16_t winPos)
{
    uint32_t i,k;
    uint8_t passArray[64];
    uint32_t dataMask = 0xFFFFFFFF;
    uint8_t vrefDQ = 0;
    uint16_t vrefDQChosen = 0;
    uint8_t numVrefDqSteps = 64;  // 6 bit value
    uint32_t phyCtrl2Save;
    uint32_t dramCfg3Save;
    volatile uint32_t* freeDDR_CS0;
    uint32_t act[8] = {0,0,0,0,0,0,0,0};
    uint32_t data[8] = {0,0,0,0,0,0,0,0};


    //set initial data background
    act[0] = SEED;
    for(i=0;i<7;i++)
    {
        act[i+1] = act[i]*SEED;
    }

    //free space in DDR to use for test data
    freeDDR_CS0 = &__start_of_free_DDR__;

    msg(MSG_DEBUG, RAW_DATA, "\r\n\r\nStart Dram Vref Training\r\n");
    //msg(MSG_DEBUG, RAW_DATA, "freeDDR_CS0 = 0x%08x \r\n\r\n", freeDDR_CS0);
    //msg(MSG_DEBUG, RAW_DATA, "mc_reg->CH0_PHY_Control_2  0x%08x, mc_reg->CH0_PHY_Control_4  0x%08x\r\n", mc_reg->CH0_PHY_Control_2, mc_reg->CH0_PHY_Control_4 );

    vrefDQ = 0;

    msg(MSG_DEBUG, RAW_DATA, "mc_reg->CH0_DRAM_Config_3  0x%08x\r\n", mc_reg->CH0_DRAM_Config_3 );

    memset(passArray,0x1,sizeof(passArray));

    // Store current CH0_PHY_Control_2 & CH0_DRAM_Config_3 values
    phyCtrl2Save = mc_reg->CH0_PHY_Control_2;
    dramCfg3Save = mc_reg->CH0_DRAM_Config_3;
    // Set Zptrm to 0x1, Zntrm to 0x0
    mc_reg->CH0_PHY_Control_2 =
        MC_CH0_PHY_CONTROL_2_PHY_DQ_ZPTRM_REPLACE_VAL( \
        MC_CH0_PHY_CONTROL_2_PHY_DQ_ZNTRM_REPLACE_VAL( \
        phyCtrl2Save, 0x0), 0x1);
    msg(MSG_DEBUG, RAW_DATA, "mc_reg->CH0_PHY_Control_2  0x%08x, mc_reg->CH0_PHY_Control_4  0x%08x\r\n", mc_reg->CH0_PHY_Control_2, mc_reg->CH0_PHY_Control_4 );

    for( i = 0; i < numVrefDqSteps; i++ ) {
        //msg(MSG_DEBUG, RAW_DATA, "vrefDQ = 0x%0x\r\nact[0:7], followed by data[0:7]\r\n  ", vrefDQ);
        //create pseudo-random background data
        act[0] = act[7]+SEED;
        for(k=0;k<7;k++)
        {
            act[k+1] = act[k]+SEED;
            //msg(MSG_DEBUG, RAW_DATA, "0x%08x, ", act[k]);
        }
        //msg(MSG_DEBUG, RAW_DATA, "\r\n  ");

        //enable dram vref training, set range to 1, and set verfDQ value to 0
        mc_reg->CH0_DRAM_Config_3 =
            MC_CH0_DRAM_CONFIG_3_VREF_TRAINING_RANGE_REPLACE_VAL( \
            MC_CH0_DRAM_CONFIG_3_VREF_TRAINING_REPLACE_VAL( \
            MC_CH0_DRAM_CONFIG_3_VREF_TRAINING_VALUE_REPLACE_VAL( \
            mc_reg->CH0_DRAM_Config_3, vrefDQ), 0x1), 0x1);
        // Send LMR6 command on Channel 0, CS 0
        mc_reg->USER_COMMAND_0 =
            MC_USER_COMMAND_0_CH0_REPLACE_VAL( \
            MC_USER_COMMAND_0_CS0_REPLACE_VAL( \
            MC_USER_COMMAND_0_LMR6_REQ_REPLACE_VAL( \
            mc_reg->USER_COMMAND_0, 0x1), 0x1), 0x1);

        // write data
        xfer_DMA_DATA(act, freeDDR_CS0, 32);
        //read data
        xfer_DMA_DATA(freeDDR_CS0, data, 32);
        //for(k=0;k<7;k++)
        //    msg(MSG_DEBUG, RAW_DATA, "0x%08x, ", data[k]);
        //msg(MSG_DEBUG, RAW_DATA, "\r\n");

        // Since all byte lanes use same vrefDQ, can look at the whole word, regardless of mem_width
        dataMask = 0xFFFFFFFF;
        //msg(MSG_DEBUG, RAW_DATA, "datamask  0x%08x\r\n", dataMask );

        if( ((data[0] & dataMask) == (act[0] & dataMask)) && ((data[1] & dataMask) == (act[1] & dataMask)) && \
            ((data[2] & dataMask) == (act[2] & dataMask)) && ((data[3] & dataMask) == (act[3] & dataMask)) && \
            ((data[4] & dataMask) == (act[4] & dataMask)) && ((data[5] & dataMask) == (act[5] & dataMask)) && \
            ((data[6] & dataMask) == (act[6] & dataMask)) && ((data[7] & dataMask) == (act[7] & dataMask)) )
        {
            passArray[i] = 0;
        }
        else
        {
            passArray[i] = 1;
        }

        //msg(MSG_DEBUG, RAW_DATA, "passArray[%0d] = %0d\r\n", i, passArray[i]);
        vrefDQ++;
    }

    // Find middle of biggest window
    vrefDQChosen = selValue(passArray, numVrefDqSteps, winPos);

    if (vrefDQChosen != 0xffff)  // Valid window found.
    {
        msg(MSG_DEBUG, RAW_DATA, "POST dram vref training, vrefDQChosen = 0x%0x, winPos = %0d%%\r\n", vrefDQChosen, winPos );
        //write final dram vref training value (vrefDQ must still be 1 for value to take hold).
        mc_reg->CH0_DRAM_Config_3 =
            MC_CH0_DRAM_CONFIG_3_VREF_TRAINING_RANGE_REPLACE_VAL( \
            MC_CH0_DRAM_CONFIG_3_VREF_TRAINING_REPLACE_VAL( \
            MC_CH0_DRAM_CONFIG_3_VREF_TRAINING_VALUE_REPLACE_VAL( \
            mc_reg->CH0_DRAM_Config_3, vrefDQChosen), 0x1), 0x1);
        // Send LMR6 command on Channel 0, CS 0
        mc_reg->USER_COMMAND_0 =
            MC_USER_COMMAND_0_CH0_REPLACE_VAL( \
            MC_USER_COMMAND_0_CS0_REPLACE_VAL( \
            MC_USER_COMMAND_0_LMR6_REQ_REPLACE_VAL( \
            mc_reg->USER_COMMAND_0, 0x1), 0x1), 0x1);

        //Set verfDQ value to 0 to complete training
        mc_reg->CH0_DRAM_Config_3 = MC_CH0_DRAM_CONFIG_3_VREF_TRAINING_REPLACE_VAL(mc_reg->CH0_DRAM_Config_3, 0x0);
        // Send LMR6 command on Channel 0, CS 0
        mc_reg->USER_COMMAND_0 =
            MC_USER_COMMAND_0_CH0_REPLACE_VAL( \
            MC_USER_COMMAND_0_CS0_REPLACE_VAL( \
            MC_USER_COMMAND_0_LMR6_REQ_REPLACE_VAL( \
            mc_reg->USER_COMMAND_0, 0x1), 0x1), 0x1);
    }
    else
    {
        mc_reg->CH0_DRAM_Config_3 = dramCfg3Save;
        msg(MSG_ERROR, RAW_DATA, "No valid DRAM vref window found!  Using default in config header.\r\n");
    }

    msg(MSG_DEBUG, RAW_DATA, "mc_reg->CH0_DRAM_Config_3  0x%08x\r\n", mc_reg->CH0_DRAM_Config_3 );

    // Restore current CH0_PHY_Control_2 values
    mc_reg->CH0_PHY_Control_2 = phyCtrl2Save;
    msg(MSG_DEBUG, RAW_DATA, "mc_reg->CH0_PHY_Control_2  0x%08x,  mc_reg->CH0_PHY_Control_4  0x%08x\r\n", mc_reg->CH0_PHY_Control_2, mc_reg->CH0_PHY_Control_4 );

    msg(MSG_INFO, RAW_DATA, "\r\n");
    return;
}

/*------------------------------------------------------------------------
        phy_vref_training
------------------------------------------------------------------------*/
/**
 *
 * \brief Attempt to train DDR4 Vref (only supports 1 CS)
 *
 * \param const MC_REGS_t *mc_reg - pointer to memory controller registers
 * \param const uint16_t mem_width - width or ddr data bus
 * \return void (no return value)
 *
 *
 */
void phy_vref_training(MC_REGS_t *mc_reg, uint16_t mem_width, uint16_t winPos)
{
    uint32_t i,k;
    uint8_t passArray[64];
    uint32_t dataMask = 0xFFFFFFFF;
    uint8_t vrefDQ = 0;
    uint16_t vrefDQChosen = 0;
    uint8_t numVrefDqSteps = 64;  // 6 bit value
    uint32_t phyCtrl15Save;
    //uint32_t phyCtrl4Save;
    volatile uint32_t* freeDDR_CS0;
    uint32_t act[8] = {0,0,0,0,0,0,0,0};
    uint32_t data[8] = {0,0,0,0,0,0,0,0};

    //set initial data background
    act[0] = SEED;
    for(i=0;i<7;i++)
    {
        act[i+1] = act[i]*SEED;
    }

    //free space in DDR to use for test data
    freeDDR_CS0 = &__start_of_free_DDR__;

    msg(MSG_DEBUG, RAW_DATA, "\r\n\r\nPHY Vref Training\r\n");
    msg(MSG_DEBUG, RAW_DATA, "freeDDR_CS0 = 0x%08x \r\n\r\n", freeDDR_CS0);
    //msg(MSG_DEBUG, RAW_DATA, "mc_reg->CH0_PHY_Control_2  0x%08x, mc_reg->CH0_PHY_Control_4  0x%08x\r\n", mc_reg->CH0_PHY_Control_2, mc_reg->CH0_PHY_Control_4 );

    vrefDQ = 0;

    msg(MSG_DEBUG, RAW_DATA, "start of PHY vref training\r\n");
    msg(MSG_DEBUG, RAW_DATA, "mc_reg->CH0_PHY_Control_15  0x%08x\r\n", mc_reg->CH0_PHY_Control_15 );

    memset(passArray,0x1,sizeof(passArray));

    // Store current CH0_PHY_Control_15 value
    phyCtrl15Save = mc_reg->CH0_PHY_Control_15;

    for( i = 0; i < numVrefDqSteps; i++ ) {
        //msg(MSG_DEBUG, RAW_DATA, "vrefDQ = 0x%0x\r\nact[0:7], followed by data[0:7]\r\n  ", vrefDQ);
        //create pseudo-random background data
        act[0] = act[7]+SEED;
        for(k=0;k<7;k++)
        {
            act[k+1] = act[k]+SEED;
            //msg(MSG_DEBUG, RAW_DATA, "0x%08x, ", act[k]);
        }
        //msg(MSG_DEBUG, RAW_DATA, "\r\n  ");

        //enable PHY vref training, set range to 1, and set verfDQ value
        mc_reg->CH0_PHY_Control_15 =
            MC_CH0_PHY_CONTROL_15_PHY_VREFINT_EN_REPLACE_VAL( \
            MC_CH0_PHY_CONTROL_15_PHY_REFBUF_TRI_REPLACE_VAL( \
            MC_CH0_PHY_CONTROL_15_PHY_VREF_CTRL_REPLACE_VAL( \
            phyCtrl15Save, vrefDQ), 0x1), 0x0);

        // write data
        xfer_DMA_DATA(act, freeDDR_CS0, 32);

        //read data
        xfer_DMA_DATA(freeDDR_CS0, data, 32);

        //for(k=0;k<7;k++)
        //    msg(MSG_DEBUG, RAW_DATA, "0x%08x, ", data[k]);
        //msg(MSG_DEBUG, RAW_DATA, "\r\n");

        // Since all byte lanes use same vrefDQ, can look at the whole word, regardless of mem_width
        dataMask = 0xFFFFFFFF;
        //msg(MSG_DEBUG, RAW_DATA, "datamask  0x%08x\r\n", dataMask );

        if( ((data[0] & dataMask) == (act[0] & dataMask)) && ((data[1] & dataMask) == (act[1] & dataMask)) && \
            ((data[2] & dataMask) == (act[2] & dataMask)) && ((data[3] & dataMask) == (act[3] & dataMask)) && \
            ((data[4] & dataMask) == (act[4] & dataMask)) && ((data[5] & dataMask) == (act[5] & dataMask)) && \
            ((data[6] & dataMask) == (act[6] & dataMask)) && ((data[7] & dataMask) == (act[7] & dataMask)) )
        {
            passArray[i] = 0;
        }
        else
        {
            passArray[i] = 1;
        }

        //msg(MSG_DEBUG, RAW_DATA, "passArray[%0d] = %0d\r\n", i, passArray[i]);
        vrefDQ++;
    }

    // Find middle of biggest window
    vrefDQChosen = selValue(passArray, numVrefDqSteps, winPos);

    if (vrefDQChosen != 0xffff)  // Valid window found.
            {
        msg(MSG_DEBUG, RAW_DATA, "POST dram vref training, vrefDQChosen = 0x%0x, winPos = %0d%%\r\n", vrefDQChosen, winPos );
        mc_reg->CH0_PHY_Control_15 =
            MC_CH0_PHY_CONTROL_15_PHY_VREFINT_EN_REPLACE_VAL( \
            MC_CH0_PHY_CONTROL_15_PHY_REFBUF_TRI_REPLACE_VAL( \
            MC_CH0_PHY_CONTROL_15_PHY_VREF_CTRL_REPLACE_VAL( \
            phyCtrl15Save, vrefDQChosen), 0x1), 0x0);
    }
    else
    {
        mc_reg->CH0_PHY_Control_15 = phyCtrl15Save;
        msg(MSG_ERROR, RAW_DATA, "No valid PHY vref window found!  Using default in config header.\r\n");
    }

    msg(MSG_DEBUG, RAW_DATA, "mc_reg->CH0_PHY_Control_15  0x%08x\r\n", mc_reg->CH0_PHY_Control_15 );

    msg(MSG_INFO, RAW_DATA, "\r\n");
    return;
}

/*------------------------------------------------------------------------
        selValue
------------------------------------------------------------------------*/
/**
 *
 * \brief Select appropriate value from series of possible windows
 *
 * \param uint8_t passArray[64] - pointer to array of pass / fail indicators
 *        uint8_t numSteps - number of elements in the passArray to parse
 *        uint16_t winPos - indicates position in window to select value (50 = midpoint)
 * \return uint16_t
 *
 */
uint16_t selValue (uint8_t passArray[64], uint8_t numSteps, uint16_t winPos)
{
    uint16_t value;
    vref_twin_t windows;
    int i, j;

    // Initialize struct
    for (i=0; i<DDR_INIT_NUM_WIN; i++) {
        windows.size[i] = 0xffff;
        windows.firstPass[i] = 0xffff;
        windows.lastPass[i] = 0xffff;
    }
    j = 0;  // Start off with first window

    // Find the window of working vrefDQ values
    for(i=0; i<numSteps; i++)
    {
        //did point pass?
        if(passArray[i] == 0)
        {
            //set start of window
            if(windows.firstPass[j] == 0xffff)
            {
                windows.firstPass[j] = i;
                msg(MSG_DEBUG, RAW_DATA, "Found firstPass at 0x%0x, window = %0d\r\n",windows.firstPass[j],j);
            }
            else if ((i == (numSteps - 1)) && (windows.lastPass[j] == 0xffff))  // Last vrefDq is good.
            {
                windows.lastPass[j] = i;
                windows.size[j] = windows.lastPass[j] - windows.firstPass[j];
                msg(MSG_DEBUG, RAW_DATA, "Found lastPass at 0x%0x, as last setting was still passing, window = %0d.\r\n",windows.lastPass[j],j);
            }
        }
        else
        {
            //point failed, set end of window if applicable
            if((windows.firstPass[j] != 0xffff) && (windows.lastPass[j] == 0xffff))
            {
                windows.lastPass[j] = i-1;
                windows.size[j] = windows.lastPass[j] - windows.firstPass[j] + 1;  // +1 Since both values pass
                msg(MSG_DEBUG, RAW_DATA, "Found lastPass at 0x%0x, window = %0d\r\n",windows.lastPass[j],j);
                // Increase j to point to next window in case one exists
                j++;
            }
        }
    }

    if (windows.size[0] != 0xffff)
    {
        for (i=0; i<5; i++)
            msg(MSG_DEBUG, RAW_DATA, "windows.size[%0d] = %0d, windows.firstPass[%0d] = 0x%0x, windows.lastPass[%0d] = 0x%0x\r\n",
                i,windows.size[i],i,windows.firstPass[i],i,windows.lastPass[i] );
        if (windows.size[1] == 0xffff) // Only 1 window found
            value = windows.firstPass[0] + (uint16_t)(windows.size[0]*winPos/100);
        else if (windows.size[2] == 0xffff)  // 2 windows found
        {
            if (windows.size[1] < windows.size[0])
                value = windows.firstPass[0] + (uint16_t)(windows.size[0]*winPos/100);
            else
                value = windows.firstPass[1] + (uint16_t)(windows.size[1]*winPos/100);
        }
        else if (windows.size[3] == 0xffff)  // 3 windows found
        {
            if ((windows.size[1] <= windows.size[0]) && (windows.size[2] <= windows.size[0]))
                value = windows.firstPass[0] + (uint16_t)(windows.size[0]*winPos/100);
            else if ((windows.size[0] <= windows.size[1]) && (windows.size[2] <= windows.size[1]))
                value = windows.firstPass[1] + (uint16_t)(windows.size[1]*winPos/100);
            else
                value = windows.firstPass[2] + (uint16_t)(windows.size[2]*winPos/100);
        }
        else if (windows.size[4] == 0xffff) // 4 windows found
        {
            if ((windows.size[1] <= windows.size[0]) && (windows.size[2] <= windows.size[0]) && (windows.size[3] <= windows.size[0]))
                value = windows.firstPass[0] + (uint16_t)(windows.size[0]*winPos/100);
            else if ((windows.size[0] <= windows.size[1]) && (windows.size[2] <= windows.size[1]) && (windows.size[3] <= windows.size[1]))
                value = windows.firstPass[1] + (uint16_t)(windows.size[1]*winPos/100);
            else if ((windows.size[0] <= windows.size[2]) && (windows.size[1] <= windows.size[2]) && (windows.size[3] <= windows.size[2]))
                value = windows.firstPass[2] + (uint16_t)(windows.size[2]*winPos/100);
            else
                value = windows.firstPass[3] + (uint16_t)(windows.size[3]*winPos/100);
        }
        else // 5 windows found
        {
            if ((windows.size[1] <= windows.size[0]) && (windows.size[2] <= windows.size[0]) && (windows.size[3] <= windows.size[0]) && (windows.size[4] <= windows.size[0]))
                value = windows.firstPass[0] + (uint16_t)(windows.size[0]*winPos/100);
            else if ((windows.size[0] <= windows.size[1]) && (windows.size[2] <= windows.size[1]) && (windows.size[3] <= windows.size[1]) && (windows.size[4] <= windows.size[1]))
                value = windows.firstPass[1] + (uint16_t)(windows.size[1]*winPos/100);
            else if ((windows.size[0] <= windows.size[2]) && (windows.size[1] <= windows.size[2]) && (windows.size[3] <= windows.size[2]) && (windows.size[4] <= windows.size[2]))
                value = windows.firstPass[2] + (uint16_t)(windows.size[2]*winPos/100);
            else if ((windows.size[0] <= windows.size[3]) && (windows.size[1] <= windows.size[3]) && (windows.size[2] <= windows.size[3]) && (windows.size[4] <= windows.size[3]))
                value = windows.firstPass[3] + (uint16_t)(windows.size[3]*winPos/100);
            else
                value = windows.firstPass[4] + (uint16_t)(windows.size[4]*winPos/100);
        }
    }
    else // No valid window found.
      value = 0xffff;

    return value;
}

